/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "MetaMemory.h"

#include "CPU_REG.h"			// LowMem_GetGlobal, HWRegisters, Software::ForgetStackChunk
#include "DebugMgr.h"			// Debug::GetRoutineName
#include "Miscellaneous.h"		// FindFunctionName
#include "RAM_ROM.h"			// CEnableFullAccess
#include "TrapPatches.h"		// IsInSysBinarySearch, OSMajorMinorVersion

#pragma mark -


// Some handy macros lifted from MemoryPrvNew.h

#define	memHandleProtect(h)											\
			((VoidHand)((DWord)h | 0x80000000))
#define	memHandleUnProtect(h)										\
			((VoidHand)((DWord)h & 0x7FFFFFFF))


// ===========================================================================
//		 MetaMemory
// ===========================================================================

class FunctionRange
{
	public:
					FunctionRange	(const char* functionName);

		Bool		InRange			(uaecptr = m68k_getpc ());
		void		Reset			(void);

		Bool		HasRange		(void) { return fBegin != UAE_NULL; }

	private:
		void		GetRange		(uaecptr addr);

		const char*	fName;
		uaecptr		fBegin;
		uaecptr		fEnd;

		FunctionRange*			fNext;
		static FunctionRange*	fgRoot;
};


	// Note: can't use the full name of PrvMisAlignedBackwardInnerBitBlt, as
	// Debug::GetRoutineName returns only the first 31 characters.

FunctionRange*	FunctionRange::fgRoot;

static FunctionRange	gcj_kptkdelete						("cj_kptkdelete");
static FunctionRange	gBackspaceChar						("BackspaceChar");
static FunctionRange	gCrc16CalcBlock						("Crc16CalcBlock");
static FunctionRange	gDmWrite							("DmWrite");
static FunctionRange	gFindShowResults					("FindShowResults");
static FunctionRange	gFindSaveFindStr					("FindSaveFindStr");
static FunctionRange	gFldDelete							("FldDelete");
static FunctionRange	gFntDefineFont						("FntDefineFont");
static FunctionRange	gGrfProcessStroke					("GrfProcessStroke");
static FunctionRange	gMemMove							("MemMove");
static FunctionRange	gMenuHandleEvent					("MenuHandleEvent");
static FunctionRange	gNetPrvSettingSet					("NetPrvSettingSet");
static FunctionRange	gNetPrvTaskMain						("NetPrvTaskMain");
static FunctionRange	gPrvCompressedInnerBitBlt			("PrvCompressedInnerBitBlt");
static FunctionRange	gPrvFindMemoryLeaks					("PrvFindMemoryLeaks");
static FunctionRange	gPrvMisAlignedForwardInnerBitBlt	("PrvMisAlignedForwardInnerBitBlt");
static FunctionRange	gPrvMisAlignedBackwardInnerBitBlt	("PrvMisAlignedBackwardInnerBitBl");
static FunctionRange	gPrvSystemTimerProc					("PrvSystemTimerProc");
static FunctionRange	gSecPrvRandomSeed					("SecPrvRandomSeed");
static FunctionRange	gSysAppExit							("SysAppExit");


// ---------------------------------------------------------------------------
//		 MetaMemory::Initialize
// ---------------------------------------------------------------------------

void MetaMemory::Initialize (void)
{
}


// ---------------------------------------------------------------------------
//		 MetaMemory::Reset
// ---------------------------------------------------------------------------

void MetaMemory::Reset (void)
{
	gcj_kptkdelete						.Reset ();
	gBackspaceChar						.Reset ();
	gCrc16CalcBlock						.Reset ();
	gDmWrite							.Reset ();
	gFindShowResults					.Reset ();
	gFindSaveFindStr					.Reset ();
	gFldDelete							.Reset ();
	gFntDefineFont						.Reset ();
	gGrfProcessStroke					.Reset ();
	gMemMove							.Reset ();
	gMenuHandleEvent					.Reset ();
	gNetPrvSettingSet					.Reset ();
	gNetPrvTaskMain						.Reset ();
	gPrvCompressedInnerBitBlt			.Reset ();
	gPrvFindMemoryLeaks					.Reset ();
	gPrvMisAlignedForwardInnerBitBlt	.Reset ();
	gPrvMisAlignedBackwardInnerBitBlt	.Reset ();
	gPrvSystemTimerProc					.Reset ();
	gSecPrvRandomSeed					.Reset ();
	gSysAppExit							.Reset ();
}


// ---------------------------------------------------------------------------
//		 MetaMemory::Save
// ---------------------------------------------------------------------------

void MetaMemory::Save (SessionFile&)
{
}


// ---------------------------------------------------------------------------
//		 MetaMemory::Load
// ---------------------------------------------------------------------------

void MetaMemory::Load (SessionFile&)
{
}


// ---------------------------------------------------------------------------
//		 MetaMemory::Dispose
// ---------------------------------------------------------------------------

void MetaMemory::Dispose (void)
{
}


#if FOR_LATER
// ---------------------------------------------------------------------------
//		 MetaMemory::MarkUninitialized
// ---------------------------------------------------------------------------

void MetaMemory::MarkUninitialized (uaecptr begin, uaecptr end)
{
	MarkRange (begin, end, kUninitialized);
}
#endif


#if FOR_LATER
// ---------------------------------------------------------------------------
//		 MetaMemory::MoveUninitialized
// ---------------------------------------------------------------------------
//	Transfer the initialized/uninitialized state of a range of bytes to a
//	new location.  Called during MemMove, so that the state of the bytes
//	being moved can be preserved.

void MetaMemory::MoveUninitialized (uaecptr source, uaecptr dest, uae_u32 size)
{
	source = MASK (source);
	dest = MASK (dest);

	if (source > fgMetaMemorySize)
		return;

	if (dest > fgMetaMemorySize)
		return;

	uae_u8*	p0 = fgMetaMemory + source;
	uae_u8*	p1 = fgMetaMemory + dest;

	while (size--)
	{
		uae_u8	dValue = *p0;
		uae_u8	sValue = *p1;

		dValue = (dValue & ~kUninitialized) | (sValue & kUninitialized);

		*p0 = dValue;

		++p0;
		++p1;
	}
}
#endif


#if FOR_LATER
// ---------------------------------------------------------------------------
//		 MetaMemory::MarkInitialized
// ---------------------------------------------------------------------------

void MetaMemory::MarkInitialized (uaecptr begin, uaecptr end)
{
	UnmarkRange (begin, end, kUninitialized);
}
#endif


// ---------------------------------------------------------------------------
//		 MetaMemory::MemChunkNew
// ---------------------------------------------------------------------------
//	Mark all the newly allocated bytes as uninitialized.

void MetaMemory::MemChunkNew (VoidPtr p, UInt attributes)
{
	// Sync up with new free space and relocated handles.

	SyncDynamicHeap ();

	// Mark the newly created block.
	// We don't need to do this if the block is relocatable, as
	// those blocks are marked in SyncHeap.

	if ((attributes & memNewChunkFlagNonMovable) && ValidChunk (p))
	{
		DWord	size = GetChunkSize (p);
		uaecptr	ptr = (uaecptr) p;

		MarkTotalAccess (ptr, ptr + size);
	}

#if FOR_LATER
	if (attributes & memNewChunkFlagNonMovable)
	{
		if (ValidChunk (p))
		{
			MarkUninitialized (((uaecptr) p), ((uaecptr) p) + GetChunkSize (p));
		}
	}
	else
	{
		if (ValidHandle ((VoidHand) p))
		{
			p = PalmHeap::DerefHandle ((VoidHand) p);
			if (p)
			{
				MarkUninitialized (((uaecptr) p), ((uaecptr) p) + GetChunkSize (p));
			}
		}
	}
#endif
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MemChunkFree
// ---------------------------------------------------------------------------
//	Mark all the newly deallocated bytes as being in a free chunk.

void MetaMemory::MemChunkFree (VoidPtr p)
{
	if (ValidChunk (p))
	{
		uaecptr	ptr = (uaecptr) p;
		MarkFreeChunk (ptr, ptr + GetChunkSize (p));

		Software::ForgetStackChunk (ptr);
	}
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MemPtrNew
// ---------------------------------------------------------------------------
//	Mark all the newly allocated bytes as uninitialized.

void MetaMemory::MemPtrNew (VoidPtr p)
{
	// Sync up with new free space and relocated handles.

	SyncDynamicHeap ();

	// Mark the newly created block.

	if (ValidChunk (p))
	{
		DWord	size = GetChunkSize (p);
		uaecptr	ptr = (uaecptr) p;

		MarkTotalAccess (ptr, ptr + size);
	}

#if FOR_LATER
	if (ValidChunk (p))
	{
		MarkUninitialized (((uaecptr) p), ((uaecptr) p) + GetChunkSize (p));
	}
#endif
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MemPtrResize
// ---------------------------------------------------------------------------
//	Mark all the newly allocated bytes as uninitialized.

void MetaMemory::MemPtrResize (VoidPtr p, UInt oldSize)
{
	// Sync up with new free space and relocated handles.

	SyncDynamicHeap ();

	// Mark the newly resized block.

	if (ValidChunk (p))
	{
		DWord	newSize = GetChunkSize (p);

		if (newSize > oldSize)
		{
			uaecptr	ptr = (uaecptr) p;
			MarkTotalAccess (ptr + oldSize, ptr + newSize - oldSize);
		}
	}

#if FOR_LATER
	if (ValidChunk (p))
	{
		UInt	newSize = GetChunkSize (p);
		if (newSize > oldSize)
		{
			MarkUninitialized (((uaecptr) p), ((uaecptr) p) + newSize);
		}
	}
#endif
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MemHandleNew
// ---------------------------------------------------------------------------
//	Mark all the newly allocated bytes as uninitialized.

void MetaMemory::MemHandleNew (VoidHand h)
{
	UNUSED_PARAM(h)

	// Sync up with new free space and relocated handles.

	SyncDynamicHeap ();

#if 0
	// Mark the newly created block.
	// (Actually, since this is a relocatable block, it was
	//  marked in SyncHeap).
#endif

#if FOR_LATER
	if (ValidHandle (h))
	{
		VoidPtr	p = PalmHeap::DerefHandle (h);
		if (p)
		{
			MarkUninitialized (((uaecptr) p), ((uaecptr) p) + GetChunkSize (p));
		}
	}
#endif
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MemHandleResize
// ---------------------------------------------------------------------------
//	Mark all the newly allocated bytes as uninitialized.

void MetaMemory::MemHandleResize (VoidHand h, UInt oldSize)
{
	// Sync up with new free space and relocated handles.

	SyncDynamicHeap ();

	// Mark the newly resized block.
	// If the block was unlocked, it will have been marked in
	// SyncHeap, so we don't need to deal with those here.

	if (ValidHandle (h))
	{
		VoidPtr	p = PalmHeap::DerefHandle (h);
		if (p && GetChunkLockCount (p) > 0)
		{
			DWord	newSize = GetChunkSize (p);

			if (newSize > oldSize)
			{
				uaecptr	ptr = (uaecptr) p;

				MarkTotalAccess (ptr, ptr + newSize);
			}
		}
	}

#if FOR_LATER
	if (ValidHandle (h))
	{
		VoidPtr	p = PalmHeap::DerefHandle (h);
		if (p && GetChunkLockCount (p) > 0)
		{
			DWord	newSize = GetChunkSize (p);
			if (newSize > oldSize)
			{
				MarkUninitialized (((uaecptr) p) + oldSize, ((uaecptr) p) + newSize);
			}
		}
	}
#endif
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MemHandleFree
// ---------------------------------------------------------------------------
//	Mark all the newly deallocated bytes as being in a free chunk.

void MetaMemory::MemHandleFree (VoidHand h)
{
	if (ValidHandle (h))
	{
		VoidPtr	p = PalmHeap::DerefHandle (h);

		if (p)
		{
			uaecptr	ptr = (uaecptr) p;
			MarkFreeChunk (ptr, ptr + GetChunkSize (p));
		}
	}
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MemLocalIDToLockedPtr
// ---------------------------------------------------------------------------
//	Removed the "unlocked" access bit from the bytes in the chunk.

void MetaMemory::MemLocalIDToLockedPtr (VoidPtr p)
{
	if (ValidChunk (p))
	{
		uaecptr	ptr = (uaecptr) p;
		MarkTotalAccess (ptr, ptr + GetChunkSize (p));
	}
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MemHandleLock
// ---------------------------------------------------------------------------
//	Removed the "unlocked" access bit from the bytes in the chunk.

void MetaMemory::MemHandleLock (VoidHand h)
{
	if (ValidHandle (h))
	{
		VoidPtr	p = PalmHeap::DerefHandle (h);
		if (p)
		{
			uaecptr	ptr = (uaecptr) p;
			MarkTotalAccess (ptr, ptr + GetChunkSize (p));
		}
	}
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MemHandleUnlock
// ---------------------------------------------------------------------------
//	If the lockCount is now zero, mark all the bytes in this chunk as being
//	in an unlocked chunk.

void MetaMemory::MemHandleUnlock (VoidHand h)
{
	if (ValidHandle (h))
	{
		VoidPtr	p = PalmHeap::DerefHandle (h);
		if (p && GetChunkLockCount (p) == 0)
		{
			uaecptr	ptr = (uaecptr) p;
			MarkUnlockedChunk (ptr, ptr + GetChunkSize (p));
		}
	}
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MemHandleResetLock
// ---------------------------------------------------------------------------
//	Mark all the bytes in this chunk as being in an unlocked chunk.

void MetaMemory::MemHandleResetLock (VoidHand h)
{
	if (ValidHandle (h))
	{
		VoidPtr	p = PalmHeap::DerefHandle (h);
		if (p)
		{
			uaecptr	ptr = (uaecptr) p;
			MarkUnlockedChunk (ptr, ptr + GetChunkSize (p));
		}
	}
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MemPtrResetLock
// ---------------------------------------------------------------------------
//	Mark all the bytes in this chunk as being in an unlocked chunk.

void MetaMemory::MemPtrResetLock (VoidPtr p)
{
	if (ValidChunk (p))
	{
		uaecptr	ptr = (uaecptr) p;
		MarkUnlockedChunk (ptr, ptr + GetChunkSize (p));
	}
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MemPtrUnlock
// ---------------------------------------------------------------------------
//	If the lockCount is now zero, mark all the bytes in this chunk as being
//	in an unlocked chunk.

void MetaMemory::MemPtrUnlock (VoidPtr p)
{
	if (ValidChunk (p))
	{
		if (GetChunkLockCount (p) == 0)
		{
			uaecptr	ptr = (uaecptr) p;
			MarkUnlockedChunk (ptr, ptr + GetChunkSize (p));
		}
	}
}


// ---------------------------------------------------------------------------
//		 MetaMemory::SyncHeap
// ---------------------------------------------------------------------------
//	Find chunk headers and mark them as memory manager structures.  Mark
//	unlocked relocatable chunks as "unlocked" and "initialized" (as we don't
//	have any idea any more what their state was before they were moved).
//	Mark free blocks as "uninitialized".

void MetaMemory::SyncHeap (const PalmHeap::HeapInfo& heapInfo)
{
	// Gather information about all the allocated, free, and
	// unlocked blocks.

	PalmHeap::WalkChunks (heapInfo, SyncOneChunk, NULL);

	// Hack for startup time.  When MemInit is called, it creates
	// a free block spanning the entire dynamic heap.  However,
	// that's where the current stack happens to be.  We still need
	// access to that, so free it up.

	if (!Patches::UIInitialized ())
	{
		MarkTotalAccess (GetSysGlobalsEnd (), GetSysGlobalsEnd () + 0x1400);
	}

	// Mark the heap header itself, all the parts up to the master pointer table array.

	PalmHeap::MPTInfo		mptInfo;
	PalmHeap::GetMPTInfo (heapInfo.heapHdrStart + heapInfo.heapHdrSize,
							heapInfo, &mptInfo);

	MarkHeapHeader (heapInfo.heapHdrStart, heapInfo.heapHdrStart + heapInfo.heapHdrSize + mptInfo.mptHdrSize);

	// Mark the master pointer blocks as OK for the system.
	// A lot of the ROM uses a MemMgr macro to deref handles.

	while (1)
	{
		uaecptr	mptArray = mptInfo.mptHdrStart + mptInfo.mptHdrSize;

		MarkMPT (mptArray, mptArray + mptInfo.numEntries * sizeof (Ptr));

		if (mptInfo.nextTblOffset == 0)
			break;

		PalmHeap::GetMPTInfo (heapInfo.heapHdrStart + mptInfo.nextTblOffset,
								heapInfo, &mptInfo);
	}
}


// ---------------------------------------------------------------------------
//		 MetaMemory::SyncDynamicHeap
// ---------------------------------------------------------------------------

void MetaMemory::SyncDynamicHeap (void)
{
	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (PalmHeap::GetHeapHeader (0), &heapInfo);

	SyncHeap (heapInfo);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::SyncAllHeaps
// ---------------------------------------------------------------------------

void MetaMemory::SyncAllHeaps (void)
{
//	PalmHeap::WalkHeaps (&SyncOneHeap, NULL);

	uaecptr	heapHdr = PalmHeap::GetHeapHeader (0);
	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (heapHdr, &heapInfo);

	SyncOneHeap (heapInfo, NULL);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::AssertSynced
// ---------------------------------------------------------------------------

void MetaMemory::AssertSynced (void)
{
	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (PalmHeap::GetHeapHeader (0), &heapInfo);

	PalmHeap::WalkChunks (heapInfo, AssertChunkSynced, NULL);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::GetWhatHappened
// ---------------------------------------------------------------------------
//	Some memory access violation was detected.  Nail down more closely what
//	it was.  These checks can be expensive, as they only occur when we're
//	pretty sure we're about to report an error in a dialog.

Errors::EAccessType MetaMemory::GetWhatHappened (uaecptr iAddress, long size, Bool forRead)
{
	// If  we've whisked away all memory access checking, return now.

	if (CEnableFullAccess::AccessOK ())
		return Errors::kOKAccess;

	CEnableFullAccess	munge;	// Remove blocks on memory access.

	Errors::EAccessType	whatHappened = Errors::kUnknownAccess;

	// Now figure out what just happened.

	if (MetaMemory::IsLowMemory (iAddress, size))
	{
		whatHappened = Errors::kLowMemAccess;
	}

	else if (MetaMemory::IsSystemGlobal (iAddress, size))
	{
		whatHappened = Errors::kGlobalVarAccess;
	}

	else if (MetaMemory::IsScreenBuffer (iAddress, size))
	{
		whatHappened = Errors::kScreenAccess;
	}

	else if (MetaMemory::IsLowStack (iAddress, size))
	{
		whatHappened = Errors::kLowStackAccess;
	}

	else
	{
		WhatHappenedData	info;
		info.result			= Errors::kUnknownAccess;
		info.address		= iAddress;
		info.size			= size;
		info.forRead		= forRead;

//		PalmHeap::WalkHeaps (WhatHappenedHeapCallback, &info);

		uaecptr	heapHdr = PalmHeap::GetHeapHeader (0);
		PalmHeap::HeapInfo	heapInfo;
		PalmHeap::GetHeapInfo (heapHdr, &heapInfo);

		WhatHappenedHeapCallback (heapInfo, &info);

		if (info.result != Errors::kUnknownAccess)
		{
			whatHappened = info.result;
		}
	}

#if 0
	else if (MetaMemory::IsUninitialized (iAddress, size))
	{
		if (MetaMemory::IsStack (iAddress, size))
		{
			whatHappened = Errors::kUninitializedStack;
		}

		else if (MetaMemory::IsAllocatedChunk (iAddress, size))
		{
			whatHappened = Errors::kUninitializedChunk;
		}
	}
#endif

	whatHappened = AllowForBugs (iAddress, size, forRead, whatHappened);

	return whatHappened;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::AllowForBugs
// ---------------------------------------------------------------------------

Errors::EAccessType MetaMemory::AllowForBugs (uaecptr iAddress, long size, Bool forRead, Errors::EAccessType whatHappened)
{
	if (whatHappened != Errors::kOKAccess && forRead)
	{
		// Let PrvFindMemoryLeaks have the run of the show.

		if (gPrvFindMemoryLeaks.InRange ())
		{
			return Errors::kOKAccess;
		}

		// SecPrvRandomSeed calls Crc16CalcBlock (NULL, 0x1000, 0).

		uaecptr	a6 = m68k_areg (regs, 6);
		uaecptr	rtnAddr = get_long (a6 + 4);

		if (iAddress < 0x1000 &&
			gCrc16CalcBlock.InRange () &&
			gSecPrvRandomSeed.InRange (rtnAddr))
		{
			return Errors::kOKAccess;
		}

		if (whatHappened == Errors::kLowMemAccess)
		{
			// See if the the low-memory checksummer is at work.

			if (size == 4 &&
				gPrvSystemTimerProc.InRange ())
			{
				return Errors::kOKAccess;
			}

			// There's a bug in BackspaceChar (pre-3.2) that accesses the word at 0x0000.

			if (Patches::HasBackspaceCharBug () &&
				iAddress == 0x0000 && size == 2 &&
				gBackspaceChar.InRange ())
			{
				return Errors::kOKAccess;
			}

			// There's a bug in FldDelete (pre-3.2) that accesses the word at 0x0000.

			if (Patches::HasFldDeleteBug () &&
				iAddress == 0x0000 && size == 2 &&
				gFldDelete.InRange ())
			{
				return Errors::kOKAccess;
			}

			// There's a bug in GrfProcessStroke (pre-3.1) that accesses the word at 0x0002.

			if (Patches::HasGrfProcessStrokeBug () &&
				iAddress == 0x0002 && size == 2 &&
				gGrfProcessStroke.InRange ())
			{
				return Errors::kOKAccess;
			}

			// There's a bug in NetPrvTaskMain (pre-3.2) that accesses low-memory.

			if (Patches::HasNetPrvTaskMainBug () &&
				gNetPrvTaskMain.InRange ())
			{
				return Errors::kOKAccess;
			}
		}

		//	There's a bug in pre-3.0 versions of SysBinarySearch that cause it to
		//	call the user callback function with a pointer just past the array
		//	to search.

		if (Patches::HasSysBinarySearchBug () &&
			Patches::IsInSysBinarySearch ())
		{
			return Errors::kOKAccess;
		}
	}

	return whatHappened;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::GetLowMemoryBegin
// ---------------------------------------------------------------------------

uaecptr MetaMemory::GetLowMemoryBegin (void)
{
	return 0;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::GetLowMemoryEnd
// ---------------------------------------------------------------------------

uaecptr MetaMemory::GetLowMemoryEnd (void)
{
	return offsetof (LowMemHdrType, globals);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::GetSysGlobalsBegin
// ---------------------------------------------------------------------------

uaecptr MetaMemory::GetSysGlobalsBegin (void)
{
	return offsetof (LowMemHdrType, globals);
}


// ---------------------------------------------------------------------------
//		 MetaMemory::GetSysGlobalsEnd
// ---------------------------------------------------------------------------

uaecptr MetaMemory::GetSysGlobalsEnd (void)
{
	CEnableFullAccess	munge;	// Remove blocks on memory access.

	return LowMem_GetGlobal (sysDispatchTableP) + 
			LowMem_GetGlobal (sysDispatchTableSize) * (sizeof (void*));
}


// ---------------------------------------------------------------------------
//		 MetaMemory::GetScreenBegin
// ---------------------------------------------------------------------------

uaecptr MetaMemory::GetScreenBegin (void)
{
	CEnableFullAccess	munge;	// Remove blocks on memory access.

	return HWRegisters::GetLCDStartAddr ();
}


// ---------------------------------------------------------------------------
//		 MetaMemory::GetScreenEnd
// ---------------------------------------------------------------------------

uaecptr MetaMemory::GetScreenEnd (void)
{
	CEnableFullAccess	munge;	// Remove blocks on memory access.

	// Get the screen's current width, height, depth, and buffer size.

	uaecptr	screenStart = HWRegisters::GetLCDStartAddr ();
	int		rowBytes	= HWRegisters::GetLCDRowBytes ();
	int		height		= HWRegisters::GetLCDHeight ();
	long	screenSize	= height * rowBytes;

	if (screenSize > 32 * 1024L)
		screenSize = 32 * 1024L;

	return screenStart + screenSize;
}

// ---------------------------------------------------------------------------
//		 MetaMemory::GetHeapHdrBegin
// ---------------------------------------------------------------------------

uaecptr MetaMemory::GetHeapHdrBegin (UInt heapID)
{
	PalmHeap::HeapInfo	info;
	PalmHeap::GetHeapInfo (PalmHeap::GetHeapHeader (heapID), &info);

	return info.heapHdrStart;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::GetHeapHdrEnd
// ---------------------------------------------------------------------------

uaecptr MetaMemory::GetHeapHdrEnd (UInt heapID)
{
	PalmHeap::HeapInfo	info;
	PalmHeap::GetHeapInfo (PalmHeap::GetHeapHeader (heapID), &info);

	return info.firstChunk;
}


Bool MetaMemory::IsLowMemory (uaecptr testAddress, uae_u32 size)
{
	return testAddress >= GetLowMemoryBegin () && testAddress + size <= GetLowMemoryEnd ();
}


Bool MetaMemory::IsSystemGlobal (uaecptr testAddress, uae_u32 size)
{
	return testAddress >= GetSysGlobalsBegin () && testAddress + size <= GetSysGlobalsEnd ();
}


Bool MetaMemory::IsScreenBuffer (uaecptr testAddress, uae_u32 size)
{
	return IsScreenBuffer (get_meta_address (testAddress), size);
}


Bool MetaMemory::IsMemMgrData (uaecptr testAddress, uae_u32 size)
{
	ChunkCheck	check;
	check.testAddress	= testAddress;
	check.size			= size;
	check.result		= false;

	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (PalmHeap::GetHeapHeader (0), &heapInfo);

	PalmHeap::WalkChunks (heapInfo, CheckForChunkHeader, &check);

	return check.result;
}


Bool MetaMemory::IsUnlockedChunk (uaecptr testAddress, uae_u32 size)
{
	ChunkCheck	check;
	check.testAddress	= testAddress;
	check.size			= size;
	check.result		= false;

	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (PalmHeap::GetHeapHeader (0), &heapInfo);

	PalmHeap::WalkChunks (heapInfo, CheckForUnlockedChunk, &check);

	return check.result;
}


Bool MetaMemory::IsFreeChunk (uaecptr testAddress, uae_u32 size)
{
	ChunkCheck	check;
	check.testAddress	= testAddress;
	check.size			= size;
	check.result		= false;

	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (PalmHeap::GetHeapHeader (0), &heapInfo);

	PalmHeap::WalkChunks (heapInfo, CheckForFreeChunk, &check);

	return check.result;
}


Bool MetaMemory::IsUninitialized (uaecptr testAddress, uae_u32 size)
{
#if FOR_LATER
	return (fgMetaMemory [MASK(testAddress)] & kUninitialized) != 0;
#else
	UNUSED_PARAM(testAddress)
	UNUSED_PARAM(size)

	return false;
#endif
}


Bool MetaMemory::IsStack (uaecptr testAddress, uae_u32 size)
{
	UNUSED_PARAM(testAddress)
	UNUSED_PARAM(size)

	return false;
}


Bool MetaMemory::IsLowStack (uaecptr testAddress, uae_u32 size)
{
	UNUSED_PARAM(testAddress)
	UNUSED_PARAM(size)

	return false;
}


Bool MetaMemory::IsAllocatedChunk (uaecptr testAddress, uae_u32 size)
{
	ChunkCheck	check;
	check.testAddress	= testAddress;
	check.size			= size;
	check.result		= false;

	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (PalmHeap::GetHeapHeader (0), &heapInfo);

	PalmHeap::WalkChunks (heapInfo, CheckForAllocatedChunk, &check);

	return check.result;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkRange
// ---------------------------------------------------------------------------

void MetaMemory::MarkRange (uaecptr start, uaecptr end, uae_u8 v)
{
	uae_u8*	startP = get_meta_address (start);
	uae_u8*	endP = get_meta_address (end);
	uae_u8*	end4P = (uae_u8*) (((uae_u32) endP) & ~3);
	uae_u8*	p = startP;

#if 1
	// Optimization: if there are no middle longs to fill, just
	// do everything a byte at a time.

	if (end - start < 12)
	{
		while (p < endP)
		{
			*p++ &= v;
		}
	}
	else
	{
		uae_u32	longValue = v;
		longValue |= (longValue << 8);
		longValue |= (longValue << 16);

		while (((uae_u32) p) & 3)		// while there are leading bytes
		{
			*p++ |= v;
		}

		while (p < end4P)				// while there are middle longs
		{
			*(uae_u32*) p |= longValue;
			p += sizeof (uae_u32);
		}

		while (p < endP)				// while there are trailing bytes
		{
			*p++ |= v;
		}
	}
#else
	for (p = startP; p < endP; ++p)
	{
		*p |= v;
	}
#endif
}


// ---------------------------------------------------------------------------
//		 MetaMemory::UnmarkRange
// ---------------------------------------------------------------------------

void MetaMemory::UnmarkRange (uaecptr start, uaecptr end, uae_u8 v)
{
	uae_u8*	startP = get_meta_address (start);
	uae_u8*	endP = get_meta_address (end);
	uae_u8*	end4P = (uae_u8*) (((uae_u32) endP) & ~3);
	uae_u8*	p = startP;

	v = ~v;

#if 1
	// Optimization: if there are no middle longs to fill, just
	// do everything a byte at a time.

	if (end - start < 12)
	{
		while (p < endP)
		{
			*p++ &= v;
		}
	}
	else
	{
		uae_u32	longValue = v;
		longValue |= (longValue << 8);
		longValue |= (longValue << 16);

		while (((uae_u32) p) & 3)		// while there are leading bytes
		{
			*p++ &= v;
		}

		while (p < end4P)				// while there are middle longs
		{
			*(uae_u32*) p &= longValue;
			p += sizeof (uae_u32);
		}

		while (p < endP)				// while there are trailing bytes
		{
			*p++ &= v;
		}
	}
#else
	for (p = startP; p < endP; ++p)
	{
		*p &= v;
	}
#endif
}


// ---------------------------------------------------------------------------
//		 MetaMemory::MarkUnmarkRange
// ---------------------------------------------------------------------------

void MetaMemory::MarkUnmarkRange (uaecptr start, uaecptr end,
							uae_u8 andValue, uae_u8 orValue)
{
	uae_u8*	startP = get_meta_address (start);
	uae_u8*	endP = get_meta_address (end);
	uae_u8*	end4P = (uae_u8*) (((uae_u32) endP) & ~3);
	uae_u8*	p = startP;

	if (andValue == 0xFF)
	{
		while (p < endP)
		{
			*p++ |= orValue;
		}
	}
	else if (orValue == 0x00)
	{
		while (p < endP)
		{
			*p++ &= andValue;
		}
	}
	else
	{
#if 1
		// Optimization: if there are no middle longs to fill, just
		// do everything a byte at a time.

		if (end - start < 12)
		{
			while (p < endP)				// while there are trailing bytes
			{
				*p = (*p & andValue) | orValue;
				p += sizeof (char);
			}
		}
		else
		{
			uae_u32	longAnd = andValue;
			longAnd |= (longAnd << 8);
			longAnd |= (longAnd << 16);

			uae_u32	longOr = orValue;
			longOr |= (longOr << 8);
			longOr |= (longOr << 16);

			while (((uae_u32) p) & 3)		// while there are leading bytes
			{
				*p = (*p & andValue) | orValue;
				p += sizeof (char);
			}

			while (p < end4P)				// while there are middle longs
			{
				*(uae_u32*) p = ((*(uae_u32*) p) & longAnd) | longOr;
				p += sizeof (long);
			}

			while (p < endP)				// while there are trailing bytes
			{
				*p = (*p & andValue) | orValue;
				p += sizeof (char);
			}
		}
#else
		for (p = startP; p < endP; ++p)
		{
			*p = (*p & andValue) | orValue;
		}
#endif
	}
}


// ---------------------------------------------------------------------------
//		 MetaMemory::GetChunkSize
// ---------------------------------------------------------------------------

DWord MetaMemory::GetChunkSize (VoidPtr p)
{
#if 1
	// !!! Too many calls to PalmHeap here.  Should make a single PalmHeap
	// function to do all this.

	// Get information on the heap for this pointer.

	uaecptr				heapHdr = PalmHeap::FindHeapHeader ((uaecptr) p);

	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (heapHdr, &heapInfo);

	uaecptr				chunkHdr = PalmHeap::GetChunkHeader (p, heapInfo);

	PalmHeap::ChunkInfo	chunkInfo;
	PalmHeap::GetChunkInfo (chunkHdr, heapInfo, &chunkInfo);

	return chunkInfo.size - (chunkInfo.chunkHdrSize + chunkInfo.sizeAdj);
#else
	PalmHeap::ChunkInfo	chunkInfo = PalmHeap::GetChunkContaining ((uaecptr) p);

	return chunkInfo.size - (chunkInfo.chunkHdrSize + chunkInfo.sizeAdj);
#endif
}


// ---------------------------------------------------------------------------
//		 MetaMemory::GetChunkLockCount
// ---------------------------------------------------------------------------

DWord MetaMemory::GetChunkLockCount (VoidPtr p)
{
#if 1
	// !!! Too many calls to PalmHeap here.  Should make a single PalmHeap
	// function to do all this.

	// Get information on the heap for this pointer.

	uaecptr				heapHdr = PalmHeap::FindHeapHeader ((uaecptr) p);

	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (heapHdr, &heapInfo);

	uaecptr				chunkHdr = PalmHeap::GetChunkHeader (p, heapInfo);

	PalmHeap::ChunkInfo	chunkInfo;
	PalmHeap::GetChunkInfo (chunkHdr, heapInfo, &chunkInfo);

	return chunkInfo.lockCount;
#else
	PalmHeap::ChunkInfo	chunkInfo = PalmHeap::GetChunkContaining ((uaecptr) p);

	return chunkInfo.lockCount;
#endif
}


// ---------------------------------------------------------------------------
//		 MetaMemory::SyncOneHeap
// ---------------------------------------------------------------------------
//	Mark the chunk header, free chunks, and unlocked chunks.

PalmHeap::EWalkerProcResult MetaMemory::SyncOneHeap (	const PalmHeap::HeapInfo& heapInfo,
														void* userData)
{
	UNUSED_PARAM(userData)

	SyncHeap (heapInfo);

	return PalmHeap::kKeepIterating;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::SyncOneChunk
// ---------------------------------------------------------------------------
//	Mark the chunk header, free chunks, and unlocked chunks.

PalmHeap::EWalkerProcResult MetaMemory::SyncOneChunk (	const PalmHeap::HeapInfo& heapInfo,
														const PalmHeap::ChunkInfo& chunkInfo,
														void* userData)
{
	UNUSED_PARAM(heapInfo)
	UNUSED_PARAM(userData)

	uaecptr	ptr				= chunkInfo.chunkHdrStart;
	DWord	chunkBodySize	= chunkInfo.size - (chunkInfo.chunkHdrSize + chunkInfo.sizeAdj);

	// Set the access for the chunk header.

	MarkChunkHeader (ptr, ptr + chunkInfo.chunkHdrSize);
	ptr += chunkInfo.chunkHdrSize;

	// Set the access for the body of the chunk.

	// Check for free chunks.

	if (chunkInfo.free)
	{
		MarkFreeChunk (ptr, ptr + chunkBodySize);
	}

	// It's not a free chunk; see if it's unlocked.

	else if (chunkInfo.lockCount == 0)
	{
		MarkUnlockedChunk (ptr, ptr + chunkBodySize);
	}

	// It's an allocated, locked chunk.

	else
	{
		MarkTotalAccess (ptr, ptr + chunkBodySize);
	}

	// Set the access for any unallocated trailing bytes.

	if (chunkInfo.sizeAdj)
	{
		ptr += chunkBodySize;
		MarkChunkTrailer (ptr, ptr + chunkInfo.sizeAdj);
	}

	return PalmHeap::kKeepIterating;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::AssertChunkSynced
// ---------------------------------------------------------------------------
//	Mark the chunk header, free chunks, and unlocked chunks.

Bool MetaMemory::memfilled (const uae_u8* ptr, uae_u8 value, long len)
{
	while (len--)
	{
		uae_u8	b = *ptr++;

		if ((b & MetaMemory::kAccessBitMask) != value)
		{
			return false;
		}
	}

	return true;
}

PalmHeap::EWalkerProcResult MetaMemory::AssertChunkSynced (
										const PalmHeap::HeapInfo& heapInfo,
										const PalmHeap::ChunkInfo& chunkInfo,
										void* userData)
{
	UNUSED_PARAM(heapInfo)
	UNUSED_PARAM(userData)

	uae_u8*	ptr				= get_meta_address (chunkInfo.chunkHdrStart);
	DWord	chunkBodySize	= chunkInfo.size - (chunkInfo.chunkHdrSize + chunkInfo.sizeAdj);

	// Set the access for the chunk header.

	assert (memfilled (ptr, kMemStructBits, chunkInfo.chunkHdrSize));
	ptr += chunkInfo.chunkHdrSize;

	// Set the access for the body of the chunk.

	// Check for free chunks.

	if (chunkInfo.free)
	{
		// Don't mess with this one until we're assured to be past
		// the point where the stack resides in the middle of a free block.

		if (Patches::UIInitialized ())
		{
			assert (memfilled (ptr, kFreeChunkBits, chunkBodySize));
		}
	}

	// It's not a free chunk; see if it's unlocked.

	else if (chunkInfo.lockCount == 0)
	{
		assert (memfilled (ptr, kUnlockedChunkBits, chunkBodySize));
	}

	// It's an allocated, locked chunk.

	else
	{
		assert (memfilled (ptr, kFreeAccessBits, chunkBodySize));
	}

	// Set the access for any unallocated trailing bytes.

	if (chunkInfo.sizeAdj)
	{
		ptr += chunkBodySize;
		assert (memfilled (ptr, kMemStructBits, chunkInfo.sizeAdj));
	}

	return PalmHeap::kKeepIterating;
}

// ---------------------------------------------------------------------------
//		 MetaMemory::WhatHappenedCallback
// ---------------------------------------------------------------------------
//	Check to see if there was an access to memory manager data, an unlocked
//	block, or a free block.

PalmHeap::EWalkerProcResult MetaMemory::WhatHappenedHeapCallback (
										const PalmHeap::HeapInfo& heapInfo,
										void* userData)
{
	WhatHappenedData*	info = (WhatHappenedData*) userData;

	// Bail out if the address to test is not even in this heap.

	if (info->address < heapInfo.heapHdrStart &&
		info->address >= heapInfo.heapHdrStart + heapInfo.size)
	{
		return PalmHeap::kKeepIterating;
	}

	// See if we touched a memory manager structure.
	// Let's start with the headers.

	// Is it in the heap header?

	if (info->address >= heapInfo.heapHdrStart &&
		info->address < heapInfo.firstChunk)
	{
		info->result = Errors::kMemMgrAccess;
		return PalmHeap::kStopIterating;
	}

	// Is it in any of the master pointer tables?

	PalmHeap::MPTInfo		mptInfo;
	PalmHeap::GetMPTInfo (heapInfo.heapHdrStart + heapInfo.heapHdrSize, heapInfo, &mptInfo);

	while (1)
	{
		uaecptr	mptHdrStart = mptInfo.mptHdrStart;
		uaecptr	mptEnd = mptInfo.mptHdrStart + mptInfo.mptHdrSize + mptInfo.numEntries * sizeof (Ptr);

		if (info->address >= mptHdrStart && info->address < mptEnd)
		{
			info->result = Errors::kMemMgrAccess;
			return PalmHeap::kStopIterating;
		}

		if (mptInfo.nextTblOffset == 0)
			break;

		PalmHeap::GetMPTInfo (heapInfo.heapHdrStart + mptInfo.nextTblOffset, heapInfo, &mptInfo);
	}

	// Check against all of the memory manager chunks.

	PalmHeap::WalkChunks (heapInfo, WhatHappenedChunkCallback, info);

	if (info->result != Errors::kUnknownAccess)
	{
		return PalmHeap::kStopIterating;
	}

	return PalmHeap::kKeepIterating;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::WhatHappenedChunkCallback
// ---------------------------------------------------------------------------
//	Check to see if there was an access to memory manager data, an unlocked
//	block, or a free block.

PalmHeap::EWalkerProcResult MetaMemory::WhatHappenedChunkCallback (
											const PalmHeap::HeapInfo& heapInfo,
											const PalmHeap::ChunkInfo& chunkInfo,
											void* userData)
{
	UNUSED_PARAM(heapInfo)

	WhatHappenedData*	info			= (WhatHappenedData*) userData;

	PalmHeap::EWalkerProcResult result	= PalmHeap::kKeepIterating;

	uaecptr	addrStart	= info->address;
	uaecptr	addrEnd		= info->address + info->size;

	// Sort out some ranges we'll be comparing against.

	uaecptr	hdrStart	= chunkInfo.chunkHdrStart;
	uaecptr	bodyStart	= chunkInfo.chunkHdrStart + chunkInfo.chunkHdrSize;
	uaecptr	trlStart	= chunkInfo.chunkHdrStart + chunkInfo.size - chunkInfo.sizeAdj;
	uaecptr	chunkEnd	= chunkInfo.chunkHdrStart + chunkInfo.size;

	/*
		A note on the comparison below (I had to draw this on the whiteboard
		before I could figure it out).  I need to see if one range of memory
		(the test address, extending for 1, 2, or 4 byte) overlaps another
		range of memory (the chunk header, the chunk body, etc.).  I started
		by drawing the second range as follows:

			+---------+---------+---------+
			|    A    |    B    |    C    |
			+---------+---------+---------+

		"B" is the range I'm interested in testing against, "A" is everything
		before it, and "C" is everything after it.

		The range of memory I want to test to see if it overlaps can be thought
		of as starting in region A, B, or C, and ending in A, B, or C.  We can
		identify all possible ranges by labelling each range "xy", where x is
		the starting section and y is the ending section.  Thus, all the possible
		ranges are AA, AB, AC, BB, BC, and CC.  From inspection, we can see that
		only segments AA and CC don't overlap B in any way.  Looking at it
		another way, any segment ending in A or starting in C will not overlap
		B.  All other ranges will.
	*/

	// See if we hit this chunk at all.  If not, leave.

	if (addrEnd > hdrStart && addrStart < chunkEnd)
	{
		// See if the access was to the body of the block.

		if (addrEnd > bodyStart && addrStart < trlStart)
		{
			// See if this is a free (unallocated) block.

			if (chunkInfo.free)
			{
				info->result = Errors::kFreeChunkAccess;
				result = PalmHeap::kStopIterating;
			}

			// See if this is an unlocked block.

			else if (chunkInfo.lockCount == 0)
			{
				info->result = Errors::kUnlockedChunkAccess;
				result = PalmHeap::kStopIterating;
			}
		}

		// See if the access was to the chunk header.

		if (addrEnd > hdrStart && addrStart < bodyStart)
		{
			info->result = Errors::kMemMgrAccess;
			result = PalmHeap::kStopIterating;
		}

		// See if the access was to the chunk trailer.

		if (addrEnd > trlStart && addrStart < chunkEnd)
		{
			info->result = Errors::kMemMgrAccess;
			result = PalmHeap::kStopIterating;
		}


		// ============================== ALLOW FOR BUGS ==============================

		// Allow the screen driver some liberty.  Some BitBlt functions get a running
		// start or come to a skidding halt before or after an allocated chunk.  They
		// read these two bytes, but the logic is such that those bits get masked out
		// or shifted away.

		if (Patches::HasBlitBugs() &&
			info->result == Errors::kMemMgrAccess &&
			info->forRead && info->size == sizeof (Word) &&
			(gPrvCompressedInnerBitBlt.InRange () ||
			 gPrvMisAlignedForwardInnerBitBlt.InRange () ||
			 gPrvMisAlignedBackwardInnerBitBlt.InRange ()))
		{
			goto HideBug;
		}


		// There's a bug in MenuHandleEvent (pre-3.1 only) that causes it to read a
		// random long (actually, the random location is the result of using a -1 to index
		// into a menu array).

		if (Patches::HasMenuHandleEventBug () &&
			info->forRead && info->size == sizeof (WinHandle) &&	// MenuPullDownType.menuWin
			gMenuHandleEvent.InRange ())
		{
			goto HideBug;
		}


		// There's a bug in NetPrvSettingSet that causes it to read 0x020C bytes
		// from the beginning of its globals buffer when that buffer is only 8 bytes long.

		if (Patches::HasNetPrvSettingSetBug () &&
			info->forRead && info->size == 4 &&
			gNetPrvSettingSet.InRange ())
		{
			goto HideBug;
		}


		// SysAppExit deletes the block of memory containing the current task's stack
		// and TCB before the task has been deleted.  When SysTaskDelete is called,
		// there are many accesses to both the stack (as functions are called) and
		// the TCB (as tasks are scheduled and as the task to be deleted is finally
		// deleted).
		//
		// We need to detect when any of these SysTaskDelete accesses occurs and
		// allow them.  One hueristic that gets most of them is to see if there
		// are any TCBs with a pointer into the deleted memory chunk.  That will
		// work up until the point the TCB is marked as deleted.  This marking
		// occurs in the AMX function cj_kptkdelete.  At that point, we can just
		// check to see if the access occured in that function and allow it.

		if (info->result == Errors::kFreeChunkAccess &&
			Patches::HasDeletedStackBug ())
		{
			// First see if there is an active TCB with a stack pointer
			// pointing into this deleted memory chunk.

			SysKernelInfoType	taskInfo;

			taskInfo.selector	= sysKernelInfoSelTaskInfo;
			taskInfo.id			= 0;	// Ask for the first task

			// Get the first task.  Remember the task IDs so that we can
			// detect when we've looped through them all.  Remembering *all*
			// the IDs (instead of just the first one) is necessary in case
			// we're called with the linked list of TCBs is inconsistant
			// (that is, it's in the process of being updated -- when that's
			// happening, we may find a loop that doesn't necessarily involve
			// the first TCB).

			vector<DWord>	taskIDList;
			Err				err = SysKernelInfo (&taskInfo);

			while (err == 0)
			{
				// See if we've seen this task ID before.  If so, leave.

				vector<DWord>::iterator	iter = taskIDList.begin ();
				while (iter != taskIDList.end ())
				{
					if (*iter++ == taskInfo.param.task.id)
					{
						goto PlanB;	// I'd really like a break(2) here... :-)
					}
				}

				// We haven't looked at this task before; check the stack.
				// Check against "stackStart" instead of stackP, as the
				// latter sometimes contains values outside of the current
				// stack range (such as the faux stacks we set up when
				// doing ATrap calls).

				if (taskInfo.param.task.stackStart >= bodyStart &&
					taskInfo.param.task.stackStart < trlStart)
				{
					goto HideBug;
				}

				// Save this task ID so we can see if we've visited this
				// record before.

				taskIDList.push_back (taskInfo.param.task.id);

				// Get the next TCB.

				taskInfo.selector	= sysKernelInfoSelTaskInfo;
				taskInfo.id			= taskInfo.param.task.nextID;
				err					= SysKernelInfo (&taskInfo);
			}

			// Plan B...
PlanB:
			// If there is no TCB pointing into the deleted chunk, see if the current
			// function is cj_kptkdelete.  If not, see if it looks like we're in a
			// function called by cj_kptkdelete.

			if (	m68k_areg (regs, 7) >= bodyStart &&
					m68k_areg (regs, 7) < trlStart)
			{
				// See if we're currently in cj_kptkdelete.

				if (gcj_kptkdelete.InRange ())
				{
					info->result = Errors::kOKAccess;
					result = PalmHeap::kStopIterating;
				}

				// If not, see if it's in the current call chain.  Normally (!)
				// I'd walk the a6 stack frame links, but there's a lot of
				// assembly stuff going on here, and not all of them set up
				// stack frames.  a6 *should* point to a valid stack frame
				// somewhere up the line, but it may be cj_kptkdelete's
				// stack frame, resulting in our skipping over it.
				//
				// It's important that gcj_kptkdelete.InRange() has returned
				// true at least once before entering this section of code.
				// That's because if it hasn't, the cached range of the
				// function will be empty, causing InRange to look for the
				// beginning and ending of the function containing the address
				// passed to it.  Since we're passing random contents of the
				// stack, it's very unlikely that there is a function range
				// corresponding to the values passed to InRange, sending it
				// on wild goose chases.  Fortunately, "Plan B" goes into effect
				// only after the TCB has been marked as deleted, which occurs
				// in cj_kptkdelete.  The very next thing cj_kptkdelete does is
				// push a parameter on the stack, causing us to get here while
				// the PC is in cj_kptkdelete.  This will cause the call to InRange
				// immediately above to return true, satisfying the precondition
				// we need here.

				else
				{
					// Get the current stack pointer and use it to iterate
					// over the contents of the stack.  We examine every
					// longword on every even boundary to see if it looks
					// like a return address into cj_kptkdelete.

					uaecptr	a7 = m68k_areg (regs, 7);

					while (a7 >= bodyStart && a7 < trlStart)
					{
						if (gcj_kptkdelete.InRange (get_long (a7)))
						{
							goto HideBug;
						}

						a7 += 2;
					}
				}	// end: checking the stack for cj_kptkdelete
			}	// end: A7 is in the deleted chunk
		}	// end: accessed a deleted chunk


		// There's a bug in FindShowResults (pre-3.0) that accesses
		// a byte in an unlocked block.

		if (info->result == Errors::kUnlockedChunkAccess &&
			Patches::HasFindShowResultsBug () &&
			info->forRead && info->size == sizeof (Boolean) &&
			addrStart == bodyStart + sizeof (Word) * 2 /*offsetof (FindParamsType, more)*/ &&
			gFindShowResults.InRange ())
		{
			goto HideBug;
		}


		// There's a bug in FindSaveFindStr that causes it to possibly read
		// off the end of the source handle.  The invalid access will be
		// generated in MemMove, which is called by DmWrite, which is called
		// by FindSaveFindStr.  So look up the stack a bit to see who's calling us.

		uaecptr	a6_0 = m68k_areg (regs, 6);	// MemMove's A6 (points to caller's A6 and return address to caller)
		uaecptr	a6_1 = get_long (a6_0);		// DmWrite's (points to caller's A6 and return address to caller)
		
		if (Patches::HasFindSaveFindStrBug () && info->forRead &&
			gMemMove.InRange () &&							// See if we're in MemMove
			gDmWrite.InRange (get_long (a6_0 + 4)) &&		// See if DmWrite is MemMove's caller
			gFindSaveFindStr.InRange (get_long (a6_1 + 4)))	// See if FindSaveFindStr is DmWrite's caller
		{
			goto HideBug;
		}

		// There's a bug in FntDefineFont that causes it to possibly read
		// off the end of the source handle.  The invalid access will be
		// generated in MemMove, which is called by FntDefineFont.  So look
		// up the stack a bit to see who's calling us.

		if (Patches::HasFntDefineFontBug () && info->forRead &&
			gMemMove.InRange () &&							// See if we're in MemMove
			gFntDefineFont.InRange(get_long (a6_0 + 4)))	// See if FntDefineFont is MemMove's caller
		{
			goto HideBug;
		}
	}

	goto Exit;

HideBug:
	info->result = Errors::kOKAccess;
	result = PalmHeap::kStopIterating;

Exit:
	return result;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::CheckForChunkHeader
// ---------------------------------------------------------------------------

PalmHeap::EWalkerProcResult MetaMemory::CheckForChunkHeader (	const PalmHeap::HeapInfo& heapInfo,
																const PalmHeap::ChunkInfo& chunkInfo,
																void* userData)
{
	UNUSED_PARAM(heapInfo)

	ChunkCheck*	data = (ChunkCheck*) userData;

	if (data->testAddress >= chunkInfo.chunkHdrStart &&
		data->testAddress < chunkInfo.chunkHdrStart + chunkInfo.chunkHdrSize)
	{
		data->result = true;	// it was found in the test range
		return PalmHeap::kStopIterating;
	}

	return PalmHeap::kKeepIterating;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::CheckForUnlockedChunk
// ---------------------------------------------------------------------------

PalmHeap::EWalkerProcResult MetaMemory::CheckForUnlockedChunk (	const PalmHeap::HeapInfo& heapInfo,
																const PalmHeap::ChunkInfo& chunkInfo,
																void* userData)
{
	UNUSED_PARAM(heapInfo)

	ChunkCheck*	data = (ChunkCheck*) userData;

	if (chunkInfo.lockCount == 0
		&& data->testAddress >= chunkInfo.chunkHdrStart + chunkInfo.chunkHdrSize
		&& data->testAddress < chunkInfo.chunkHdrStart + chunkInfo.chunkHdrSize - chunkInfo.sizeAdj)
	{
		data->result = true;	// it was found in the test range
		return PalmHeap::kStopIterating;
	}

	return PalmHeap::kKeepIterating;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::CheckForFreeChunk
// ---------------------------------------------------------------------------

PalmHeap::EWalkerProcResult MetaMemory::CheckForFreeChunk (	const PalmHeap::HeapInfo& heapInfo,
															const PalmHeap::ChunkInfo& chunkInfo,
															void* userData)
{
	UNUSED_PARAM(heapInfo)

	ChunkCheck*	data = (ChunkCheck*) userData;

	if (chunkInfo.free
		&& data->testAddress >= chunkInfo.chunkHdrStart + chunkInfo.chunkHdrSize
		&& data->testAddress < chunkInfo.chunkHdrStart + chunkInfo.chunkHdrSize - chunkInfo.sizeAdj)
	{
		data->result = true;	// it was found in the test range
		return PalmHeap::kStopIterating;
	}

	return PalmHeap::kKeepIterating;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::CheckForAllocatedChunk
// ---------------------------------------------------------------------------

PalmHeap::EWalkerProcResult MetaMemory::CheckForAllocatedChunk (	const PalmHeap::HeapInfo& heapInfo,
																	const PalmHeap::ChunkInfo& chunkInfo,
																	void* userData)
{
	UNUSED_PARAM(heapInfo)

	ChunkCheck*	data = (ChunkCheck*) userData;

	if (!chunkInfo.free
		&& data->testAddress >= chunkInfo.chunkHdrStart + chunkInfo.chunkHdrSize
		&& data->testAddress < chunkInfo.chunkHdrStart + chunkInfo.chunkHdrSize - chunkInfo.sizeAdj)
	{
		data->result = true;	// it was found in the test range
		return PalmHeap::kStopIterating;
	}

	return PalmHeap::kKeepIterating;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::ValidChunk
// ---------------------------------------------------------------------------

Bool MetaMemory::ValidChunk (VoidPtr p)
{
	if (!p)
		return false;

	uaecptr	ptr = (uaecptr) p;
	if (!valid_address (ptr, 1))
		return false;

	// Get information on the heap for this pointer.

	uaecptr	heapHdr = PalmHeap::FindHeapHeader (ptr);
	if (!heapHdr)
		return false;

	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (heapHdr, &heapInfo);

	PalmHeap::EPalmHeapResult	result = PalmHeap::ValidatePointer (p, heapInfo);

	return result == PalmHeap::kChunkOK;
}


// ---------------------------------------------------------------------------
//		 MetaMemory::ValidHandle
// ---------------------------------------------------------------------------

Bool MetaMemory::ValidHandle (VoidHand h)
{
	if (!h)
		return false;

	uaecptr	hdl = (uaecptr) memHandleUnProtect(h);
	if (!valid_address (hdl, 4))
		return false;

	// Get information on the heap for this pointer.

	uaecptr	heapHdr = PalmHeap::FindHeapHeader (hdl);
	if (!heapHdr)
		return false;

	PalmHeap::HeapInfo	heapInfo;
	PalmHeap::GetHeapInfo (heapHdr, &heapInfo);

	PalmHeap::EPalmHeapResult	result = PalmHeap::ValidateHandle (h, heapInfo);

	return result == PalmHeap::kChunkOK;
}


// ---------------------------------------------------------------------------
//		 FunctionRange::FunctionRange
// ---------------------------------------------------------------------------

FunctionRange::FunctionRange (const char* functionName) :
	fName (functionName),
	fBegin (UAE_NULL),
	fEnd (UAE_NULL),
	fNext (fgRoot)
{
	fgRoot = this;
}


// ---------------------------------------------------------------------------
//		 FunctionRange::InRange
// ---------------------------------------------------------------------------

Bool FunctionRange::InRange (uaecptr addr)
{
	// It's not in our range if it's in someone else's.

	static Bool	walkingChain;

	if (!walkingChain)
	{
		walkingChain = true;

		FunctionRange*	range = fgRoot;
		while (range)
		{
			if (range != this && range->HasRange () && range->InRange (addr))
			{
				walkingChain = false;
				return false;
			}

			range = range->fNext;
		}

		walkingChain = false;
	}

	// If we haven't determine the range of the function for which we're
	// responsible, see if given address is in that function.  If so,
	// remember the range of that function.

	if (!HasRange ())
	{
		GetRange (addr);
	}

	// If we know the range of the function for which we are responsible,
	// test the address against that range.

	if (HasRange () && (addr >= fBegin) && (addr < fEnd))
	{
		return true;
	}

	return false;
}


// ---------------------------------------------------------------------------
//		 FunctionRange::Reset
// ---------------------------------------------------------------------------

void FunctionRange::Reset (void)
{
	fBegin = UAE_NULL;
	fEnd = UAE_NULL;
}


// ---------------------------------------------------------------------------
//		 FunctionRange::GetRange
// ---------------------------------------------------------------------------

void FunctionRange::GetRange (uaecptr addr)
{
	char		name[80];
	uaecptr		startAddr;
	uaecptr		endAddr;

	::FindFunctionName (addr, name, &startAddr, &endAddr);

	if (strcmp (name, fName) == 0)
	{
		fBegin = startAddr;
		fEnd = endAddr;
	}
}


/*
Bool FindAllAppChunks (uaecptr chunkHdr, int ver, void* userData)
{
	MemChunkHeaderUnionType	chunk;
	chunk = *(MemChunkHeaderUnionType*) chunkP;
	Canonical (chunk);

	if (memUChunkOwner (&chunk, ver) == (long) userData)
		MemChunkFree ((BytePtr) chunkP + memUSizeOfChunkHeader(ver));

	return false;
}
*/
