/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "CPU_REG.h"

#include "ATraps.h" 			// DoingCall
#include "Bank_Dummy.h"			// ValidAddress
#include "Bank_ROM.h"			// ValidAddress
#include "Bank_DRAM.h"			// ValidAddress
#include "Bank_SRAM.h"			// ValidAddress
#include "Bank_MC68328.h"		// MC68328Bank::Reset, GetInterruptLevel, Cycle, etc.
#include "Bank_MC68EZ328.h" 	// MC68EZ328Bank::Reset, GetInterruptLevel, Cycle, etc.
#include "Bank_SED1375.h"		// SED1375 video controller register access
#include "Bank_ROM.h"			// ROMBank::GetMemoryStart
#include "Byteswapping.h"		// Canonical
#include "DebugMgr.h"			// Debug::Initialize
#include "ErrorHandling.h"		// kError_ stuff
#include "Hordes.h"				// Hordes::Initialize, Hordes::IsOn, Hordes::AutoSaveState, etc.
#include "Logging.h"			// LogSystemCalls, LogAppendMsg
#include "MetaMemory.h" 		// MetaMemory
#include "Miscellaneous.h"		// GetFunctionAddress
#include "omnithread.h" 		// omni_mutex, omni_mutex_lock
#include "PalmHeap.h"			// PalmHeap
#include "Platform.h"			// Initialize, Reset, GetMilliseconds, etc.
#include "RAM_ROM.h"			// Memory::Initialize, Memory::Reset, CEnableFullAccess
#include "SessionFile.h"		// WriteDeviceType
#include "StringData.h" 		// kExceptionNames
#include "Strings.r.h"			// kStr_ values
#include "TrapPatches.h"		// Patches::Initialize, Patches::Reset, Patches::HandleTrap15
#include "UAE_Utils.h"			// uae_memset
#include "UART.h"				// UART::Initialize, Reset, Dispose

#if HAS_PROFILING
#include "Profiling.h"
#endif

#if __profile__
#include <Profiler.h>
#endif


// Types

#pragma mark Types


// Definitions of the stack frames used in Software::ProcessException.

/*
	Exception stack frame:

			15						 0
	sp ->	+------------------------+ <-+						 Higher Address
			| Function Code 		 |	 |								|
			+------------------------+	 |								|
			| Access Address (High)  |	 | Only for Address and 		|
			+------------------------+	 | Bus errors.					|
			| Access Address (Low)	 |	 |								|
			+------------------------+	 |								|
			| Instruction Register	 |	 |								|
			+------------------------+ <-+								|
			| Status Register		 |									|
			+------------------------+									|
			| Program Counter (High) |									|
			+------------------------+									|
			| Program Counter (Low)  |									|
			+------------------------+									v

	Function code:
		Bits		Function
		0-2 		Function Code
		3			I/N = Instruction = 0, Not = 1
		4			R/W = Write = 0, Read = 1
*/

#include "PalmPack.h"

struct ExceptionStackFrame1
{
	uae_u16 statusRegister;
	uae_u32 programCounter;
};

struct ExceptionStackFrame2
{
	uae_u16 functionCode;
	uae_u32 accessAddress;
	uae_u16 instructionRegister;
	uae_u16 statusRegister;
	uae_u32 programCounter;
};

#include "PalmPackPop.h"


// Globals.

#pragma mark Globals

static	uae_u32 	gBreakReason;
static	uae_u32 	gBreakReasonMask;
static	omni_mutex	gBreakReasonMutex;

static	Bool		gShouldLoop;

/*static*/	Bool		gBreakOnException[kException_LastException];
static	omni_mutex	gBreakOnExceptionMutex;

		TByteQueue	gKeyQueue;
static	uaecptr 	gLastTraceAddress;
		uaecptr 	gInstructionStart;
		Bool		gNeedPostLoad;
		Bool		gSuspended;

int areg_byteinc[] = { 1,1,1,1,1,1,1,2 };	// (normally in newcpu.c)
int imm8_table[] = { 8,1,2,3,4,5,6,7 }; 	// (normally in newcpu.c)

int movem_index1[256];						// (normally in newcpu.c)
int movem_index2[256];						// (normally in newcpu.c)
int movem_next[256];						// (normally in newcpu.c)

int fpp_movem_index1[256];					// (normally in newcpu.c)
int fpp_movem_index2[256];					// (normally in newcpu.c)
int fpp_movem_next[256];					// (normally in newcpu.c)

cpuop_func *cpufunctbl[65536];				// (normally in newcpu.c)

#if HAS_PROFILING
perfRec perftbl[65536];
#endif
											// (normally in newcpu.c)
uae_u16 last_op_for_exception_3;			/* Opcode of faulting instruction */
uaecptr last_addr_for_exception_3;			/* PC at fault time */
uaecptr last_fault_for_exception_3; 		/* Address that generated the exception */

struct regstruct regs;						// (normally in newcpu.c)
struct flag_struct regflags;				// (normally in support.c)


const uae_u16	kOpcode_JMP_Abs32 = 0x4EF9;


static uae_u32*	gConvert1To8;	// Used to convert a 4-bit nybble consisting of 4 1-bit pixels
								// to a 32-bit value containing 4 8-bit pixels.

static uae_u32*	gConvert2To8;	// Used to convert an 8-bit byte consisting of 4 2-bit pixels
								// to a 32-bit value containing 4 8-bit pixels.

static uae_u16*	gConvert4To8;	// Used to convert an 8-bit byte consisting of 2 4-bit pixels
								// to a 16-bit value containing 4 8-bit pixels.


#if REGISTER_HISTORY

	#define reg_history_size	512
	long	reg_history_index;
	struct regstruct	reg_history[reg_history_size];

#endif

#if EXCEPTION_HISTORY

	struct exception_history_rec
	{
		const char* 	name;
		uae_u32 		pc;
		uae_u32 		sp;
	};
	
	#define exception_history_size	512
	long	exception_history_index;
	exception_history_rec	exception_history[exception_history_size];

#endif


#if TIME_STARTUP
	uae_u32 gStart;
#endif


#if COUNT_TRAPS
	long*	gTrapCounts;
	long	gTotalTrapCount;
#endif




#pragma mark -


// ===========================================================================
//		 Emulator
// ===========================================================================

static DeviceType		gHardwareDevice;
#if TIME_STARTUP
static unsigned long	gInstructionCount;
#endif
static unsigned long	gCycleCount;
Bool					gIsEZDevice;


/***********************************************************************
 *
 * FUNCTION:	Emulator::Initialize
 *
 * DESCRIPTION: Initializes the emulator hardware, software, and memory.
 *				This function is called just once after a new ROM is
 *				loaded.
 *
 * PARAMETERS:	cfg - .
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void Emulator::Initialize (const Configuration& cfg)
{
	try
	{
		// Set the hardware device here. We need to have it set up now
		// for things like SRAMBank::Initialize and Memory::Initialize,
		// which need to know what mode they're in.

		gHardwareDevice = kDeviceUnspecified;
		SetHardwareDevice (cfg.fDeviceType);

		// Ideally, we can initialize sub-systems in any order.  However,
		// it's probably a good idea to initialize Memory first.
		// The call to Memory::Initialize also initializes address banks
		// and MetaMemory.

		// If ROM is an embedded resouce, use it.
		if (Platform::ROMResourcePresent ())
		{
			void*		rom;
			uae_s32 	romSize;
			Bool		resourceLoaded = false;

			resourceLoaded = Platform::GetROMResource (rom, romSize);
			assert (resourceLoaded);

			BufferHandle	romHandle (rom, romSize);

			Memory::Initialize (romHandle, cfg.fRAMSize);
		}

		// If ROM is not embedded, use the filespec.
		else
		{
			FileHandle romHandle (cfg.fROMFile, kOpenRead | kOpenExisting);
			Memory::Initialize (romHandle, cfg.fRAMSize);
		}

		Hardware::Initialize ();
		UART::Initialize (EZMode () ? UART::kUART_DragonballEZ : UART::kUART_Dragonball);
		Hordes::Initialize ();
		Patches::Initialize ();
		Platform::Initialize ();
		Debug::Initialize ();
		Screen::Initialize ();
		PalmHeap::Initialize ();
		Software::Initialize ();
		LowMem::Initialize ();
		Errors::Initialize ();

		Emulator::Reset ();

		// Remember a whole bunch of stuff about what we did for next time.

		Preference<Configuration>	pref1 (kPrefKeyLastConfiguration);
		pref1 = cfg;

		// Zap this one out, so that we give preference to a New operation
		// at startup and not an Open operation.

		Preference<FileReference>	pref2 (kPrefKeyLastPSF);
		pref2 = FileReference();
	}
	catch (...)
	{
		Emulator::Dispose ();
		throw;
	}
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::Reset
 *
 * DESCRIPTION: Reset the emulator.  This effects a "warm restart" of
 *				the ROM.  This function can be called any number of
 *				times after a ROM is loaded.  It is also called as part
 *				of the Initialize sequence.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Emulator::Reset (void)
{
	// Ideally, we can reset sub-systems in any order.	However,
	// it's probably a good idea to reset them in the same order
	// in which they were initialized.

	Memory::Reset ();	// Also resets address banks and MetaMemory.

	Hardware::Reset ();
	UART::Reset ();
	Hordes::Reset ();
	Patches::Reset ();
	Platform::Reset ();
	Debug::Reset ();
	Screen::Reset ();
	PalmHeap::Reset ();
	Software::Reset ();
	LowMem::Reset ();
	Errors::Reset ();

	// Now reset self.

#if TIME_STARTUP
	gInstructionCount = 0;
#endif
	gCycleCount = 0;
	gSuspended = false;

	Emulator::ClearBreakReason (kBreak_All);
	Emulator::SetBreakReasonMask (kBreak_DefaultMask);

#if REGISTER_HISTORY
	reg_history_index = 0;
#endif

#if EXCEPTION_HISTORY
	exception_history_index = 0;
#endif

#if COUNT_TRAPS
	Platform::DisposeMemory (gTrapCounts);
#endif

	// If the appropriate modifier key is down, install a temporary breakpoint
	// at the start of the Big ROM.

	if (Platform::StopOnResetKeyDown ())
	{
		uaecptr romStart = ROMBank::GetMemoryStart ();
		uaecptr headerVersion = get_long (romStart + offsetof (CardHeaderType, hdrVersion));
		long	bigROMOffset = 0x03000;
		if (headerVersion > 1)
		{
			bigROMOffset = get_long (romStart + offsetof (CardHeaderType, bigROMOffset));
			bigROMOffset &= 0x000FFFFF; 	// Allows for 1 Meg offset.
		}

		uaecptr resetVector = get_long (romStart + bigROMOffset + offsetof (CardHeaderType, resetVector));

		Debug::SetBreakpoint (dbgTempBPIndex, resetVector, NULL);
	}

	// All of meta-memory gets wiped out on reset; re-establish these.

	Emulator::InstallCPUBreaks ();
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::DoSave (private)
 *
 * DESCRIPTION: Does the work of saving a session.
 *				Saves any sub-system state to the given session file.
 *
 * PARAMETERS:	f - The file reference.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Emulator::DoSave (SessionFile& f)
{
	Preference<Configuration>	cfg (kPrefKeyLastConfiguration);

	// Write out the device type.

	assert (cfg->fDeviceType != kDeviceUnspecified);
	f.WriteDeviceType (cfg->fDeviceType);

	// !!! Save break-reason and break-reason-mask, too?

	// Ideally, we can save sub-systems in any order.  However,
	// it's probably a good idea to save them in the same order
	// in which they were initialized.

	Memory::Save (f);	// Also saves address banks and MetaMemory.

	Hardware::Save (f);
	UART::Save (f);
	Hordes::Save (f);
	Patches::Save (f);
	Platform::Save (f);
	Debug::Save (f);
	Screen::Save (f);
	PalmHeap::Save (f);
	Software::Save (f);
	LowMem::Save (f);
	Errors::Save (f);
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::DoLoad (private)
 *
 * DESCRIPTION: Does the work of loading a session file.
 *				Loads any sub-system state from the given session file.
 *
 * PARAMETERS:	f - The file reference.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Emulator::DoLoad (SessionFile& f)
{
	try
	{
		// Set the hardware device here. We need to have it set up now
		// for things like SRAMBank::Initialize and Memory::Initialize,
		// which need to know what mode they're in.

		DeviceType	deviceType = kDeviceUnspecified;
		gHardwareDevice = deviceType;
		if (f.ReadDeviceType (deviceType))
		{
			SetHardwareDevice (deviceType);
		}

		// Ideally, we can load sub-systems in any order.  However,
		// it's probably a good idea to load them in the same order
		// in which they were initialized.

		Memory::Load (f);	// Also loads address banks and MetaMemory.

		Hardware::Load (f);
		UART::Load (f);
		Hordes::Load (f);
		Patches::Load (f);
		Platform::Load (f);
		Debug::Load (f);
		Screen::Load (f);
		PalmHeap::Load (f);
		Software::Load (f);
		LowMem::Load (f);
		Errors::Load (f);
	}
	catch (...)
	{
		Emulator::Dispose ();
		throw;
	}
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Emulator::Dispose (void)
{
	// Ideally, we can dispose sub-systems in any order.  However,
	// it's probably a good idea to dispose them in the reverse
	// order in which they were initialized.

	Errors::Dispose ();
	LowMem::Dispose ();
	Software::Dispose ();
	PalmHeap::Dispose ();
	Screen::Dispose ();
	Debug::Dispose ();
	Platform::Dispose ();
	Patches::Dispose ();
	Hordes::Dispose ();
	UART::Dispose ();
	Hardware::Dispose ();

	Memory::Dispose (); // Also disposes address banks and MetaMemory.

	// Now dispose self.

	Platform::DisposeMemory (gConvert1To8);
	Platform::DisposeMemory (gConvert2To8);
	Platform::DisposeMemory (gConvert4To8);
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::Save
 *
 * DESCRIPTION: Standard save function for PSF state files.
 *				Saves any sub-system state to the given session file.
 *
 * PARAMETERS:	f - reference to file,
				updateLastPSF - setting flag
				updateMRU - setting flag
				updateDefaultConfiguration - setting flag
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Emulator::Save (const FileReference& f,
					 Bool updateLastPSF,
					 Bool updateMRU,
					 Bool updateDefaultConfiguration)
{
	FileHandle	fileHandle (f, kCreateAlways | kOpenReadWrite);
	ChunkFile	chunkFile (fileHandle);
	SessionFile sessionFile (chunkFile);

	Emulator::DoSave (sessionFile);

	Configuration	cfg = sessionFile.GetConfiguration ();

	if (updateDefaultConfiguration)
	{
		Preference<Configuration>	pref (kPrefKeyLastConfiguration);
		pref = cfg;
	}

	if (updateMRU)
		gEmuPrefs->UpdateRAMMRU (f);

	if (updateLastPSF)
	{
		Preference<FileReference>	pref (kPrefKeyLastPSF);
		pref = f;
	}
}

/***********************************************************************
 *
 * FUNCTION:	Emulator::Load
 *
 * DESCRIPTION: Standard load function, for loading from a memory buffer.
 *				Loads any sub-system state from a RAM buffer at the the
 *				given location
 *
 * PARAMETERS:	pPSF - location of RAM buffer.
 *				iSize - size of RAM buffer.
 *				updateDefaultConfiguration - setting flag.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Emulator::Load (VoidPtr pPSF,
					 uae_u32 iSize,
					 Bool updateDefaultConfiguration)
{
	BufferHandle	bufferHandle (pPSF, iSize);
	ChunkFile		chunkFile (bufferHandle);
	SessionFile 	sessionFile (chunkFile);

	Emulator::Load(sessionFile,
					updateDefaultConfiguration);
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::Load
 *
 * DESCRIPTION: Standard load function, for loading from a file.
 *				Loads any sub-system state from a file reference
 *
 * PARAMETERS:	f - The file to load.
 *				updateLastPSF - setting flag
 *				Bool updateMRU - setting flag
 *				Bool updateDefaultConfiguration - setting flag
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Emulator::Load (const FileReference& f,
					 Bool updateLastPSF,
					 Bool updateMRU,
					 Bool updateDefaultConfiguration)
{
	FileHandle	fileHandle (f, kOpenExisting | kOpenRead);
	ChunkFile	chunkFile (fileHandle);
	SessionFile sessionFile (chunkFile);

	if (!f.Exists())
		throw;

	Emulator::Load(sessionFile,
					updateDefaultConfiguration);

	if (updateMRU)
		gEmuPrefs->UpdateRAMMRU (f);

	if (updateLastPSF)
	{
		Preference<FileReference>	pref (kPrefKeyLastPSF);
		pref = f;
	}
}

/***********************************************************************
 *
 * FUNCTION:	Emulator::Load
 *
 * DESCRIPTION: Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	sessionFile - sessionFile to load
 *				updateDefaultConfiguration - setting flag
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Emulator::Load (SessionFile& sessionFile,
					 Bool updateDefaultConfiguration)
{
	// Load enough information so that we can initialize the system

	Configuration	cfg;
	if (!sessionFile.ReadConfiguration (cfg))
		return;

	// Initialize (and reset) the system.

	Emulator::Initialize (cfg);

	// Load the saved state from the session file.	First, set the flag
	// that says whether or not we can successfully restart from the
	// information in this file.  As parts are loaded, the various
	// sub-systems will have a chance to veto this optimistic assumption.

	sessionFile.SetCanReload (true);
	Emulator::DoLoad (sessionFile);

	if (!sessionFile.GetCanReload ())
	{
		Emulator::Reset ();
		gNeedPostLoad = false;
	}
	else
	{
		// Argh!  Can't do this here.  This would cause the Gremlin
		// control window to be created in the context of the "main"
		// thread, when we want/need it to be created in the context
		// of the "UI" thread.

#if 0
		if (Hordes::IsOn ())
		{
			Hordes::TurnOn(false);
			Platform::GCW_Open ();
		}
#endif

		// Argh!  Can't do *this* here!  Calling PostLoad here on
		// Windows is done in the context of the "main" thread, not
		// the UI or CPU threads.  The stack of the main thread is
		// located at 0x00130000 or so.  This overlaps with the
		// emulated Palm OS range when emulating something bigger than
		// a 1 Meg device.	Since PostLoad results in calls to the ROM,
		// and since calls to the ROM involve mapping in a range of
		// the host stack space into Palm emulated space, we end up
		// with an overlap, which is flagged in DummyBank::MapPhysicalMemory.
		//
		// Therefore, I just set a flag here and check it when we enter
		// the CPU loop.

			// Arrrrggghhh!  Can't do it in the CPU loop!  Patches::PostLoad
			// will try to call Palm OS functions.  But there's no guarantee
			// that the emulated state is in any shape to accept a Palm OS
			// call (it may be in the middle of the kernel, for example).
			// Soooo....I now check gNeedPostLoad in Patches::HandleSystemCall.
			// That should be nice and safe...

#if 0
		// Do some stuff that is normally taken care of during
		// the process of resetting the device (autoloading
		// applications, setting the device date, installing
		// the HotSync user-name, and setting the 'gdbS' feature).

		Patches::PostLoad ();
#endif

		gNeedPostLoad = true;
	}


	// Save all this good info for next time.

	if (updateDefaultConfiguration)
	{
		Preference<Configuration>	pref (kPrefKeyLastConfiguration);
		pref = cfg;
	}
}

/***********************************************************************
 *
 * FUNCTION:	Emulator::Execute
 *
 * DESCRIPTION: Processes instructions until an event occurs.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	The reason why the loop exited.
 *
 ***********************************************************************/

 /*
		Comments on interrupt handling:

		- SPCFLAG_INT is set at the end of MakeFromSR.	MakeFromSR is
		  called at the end of any opcode that changes it.

		- SPCFLAG_INT is also set at the end of Registers::UpdateInterrupts,
		  which is called when we want to emulate an interrupt occuring.

		- SPCFLAG_INT is set at the end of Software::ProcessInterrupt.
		  Software::ProcessInterrupt is called from the main CPU loop if
		  the requested interrupt is greater than the interrupt level.

		- In the main CPU loop, if SPCFLAG_INT or SPCFLAG_DOINT are set
		  clear them.  If the interrupt is greater than the interrupt
		  level, call Software::ProcessInterrupt, set regs.stopped to zero,
		  and clear the SPCFLAG_STOP flag.

		- Later in the main CPU loop if SPCFLAG_DOINT is set, clear it.
		  If the interrupt is greater than the interrupt level, call
		  Software::ProcessInterrupt and set regs.stopped to zero. Note
		  that SPCFLAG_STOP is NOT cleared (I don't know why).

		- After that, if SPCFLAG_INT is set, clear it and set SPCFLAG_DOINT.


		Comments on trace handling:

		- In MakeFromSR, if t0 or t1 are set, the SPCFLAG_TRACE is
		  set. Otherwise, SPCFLAG_TRACE and SPCFLAG_DOTRACE are cleared.
		  MakeFromSR is called at the end of any opcode that changes it.

		- In main CPU loop after an opcode has been executed, if
		  SPCFLAG_DOTRACE is set, call Software::ProcessException (kException_Trace)

		- At the end of Software::ProcessException(), SPCFLAG_TRACE and
		  SPCFLAG_DOTRACE are cleared.

		- Near end of main CPU loop, if SPCFLAG_TRACE is set, SPCFLAG_TRACE
		  is cleared and SPCFLAG_DOTRACE is set.
*/

long Emulator::Execute (void)
{
	// Make sure that we're entering this function with a clean slate.
	// Still allow kBreak_StopRequest, as it's possible that someone
	// started up the CPU thread and then immediately asked it to stop.

	assert ((gBreakReason & (gBreakReasonMask & ~kBreak_StopRequest)) == 0);

	// Declare these here (instead of where it's actually used), so that VC++ doesn't
	// give us any lip about jumping over it with our goto statement.

	uae_u32 				opcode;
	register Bool*			shouldLoop = &gShouldLoop;
	register uaecptr*		instructionStart = &gInstructionStart;
	register unsigned long* cycleCount = &gCycleCount;
	register cpuop_func**	functable = cpufunctbl;

#if HAS_PROFILING_DEBUG
	UInt64	readCyclesSaved = 0;
	UInt64	writeCyclesSaved = 0;
	UInt64	clockCyclesSaved = 0;
#endif

#if TIME_STARTUP
	static calledOnce;

	if (!calledOnce)
	{
		calledOnce = true;

		gStart = Platform::GetMilliseconds ();
	}
#endif

#if defined (macintosh) && defined (_DEBUG)
	// Put in a little dead-man's switch. If this function doesn't
	// exit for a long time, let us get into the debugger.
	uae_u32 deadManStart;
	if ((gBreakReasonMask & kBreak_CheckForEvents) == 0)
	{
		deadManStart = Platform::GetMilliseconds ();
	}
#endif

	// Check for the stopped flag before entering the "execute an opcode"
	// section.  It could be that we last exited the loop while still in
	// stop mode, and we need to wind our way back down to that spot.

	if ((regs.spcflags & SPCFLAG_STOP) != 0)
		goto StoppedLoop;

	while (*shouldLoop)
	{
#if REGISTER_HISTORY
//		if (!ATrap::DoingCall ())
		{
			// Save the registers for the post-mortem, but don't record the
			// instructions we generate when calling the ROM as a subroutine.
			// We want those to be as transparent as possible.	In particular,
			// we don't want any functions that we call as part of figuring
			// out why a problem occured to knock the problem-causing registers
			// off of our array.

			++reg_history_index;
			reg_history[reg_history_index & (reg_history_size - 1)] = regs;
		}
#endif

#if defined (macintosh) && defined (_DEBUG)
		// Put in a little dead-man's switch. If this function doesn't
		// exit for a long time, let us get into the debugger.
		if ((gBreakReasonMask & kBreak_CheckForEvents) == 0)
		{
			uae_u32 deadManNow = Platform::GetMilliseconds ();
			if ((deadManNow - deadManStart) > 5000)
			{
				Platform::Debugger ();
			}
		}
#endif

		// Remember the starting point of this opcode.	Later, if an
		// exception occurs, we may want to reset the PC to this point.

//		*instructionStart = m68k_getpc ();
		int drift;
		drift = regs.pc_p - regs.pc_oldp;
		*instructionStart = regs.pc + drift;

		// See if we need to halt CPU execution at this location.  We could
		// need to do this for several reasons, including hitting soft
		// breakpoints or needing to execute tailpatches.

//		if (MetaMemory::IsCPUBreak (gInstructionStart))
		if (MetaMemory::IsCPUBreak (regs.pc_meta_oldp + drift))
		{
			HandleCPUBreak ();
		}

#if HAS_PROFILING
		// Turn gProfilingCounted on here so the GET_OPCODE fetch is counted.
		if (gProfilingEnabled)
		{
			// if detailed, log instruction here
			if (gProfilingDetailed)
				ProfileInstructionEnter(*instructionStart);

#if HAS_PROFILING_DEBUG
			readCyclesSaved = gReadCycles;
			writeCyclesSaved = gWriteCycles;
			clockCyclesSaved = gClockCycles;
#endif

			// Turn gProfilingCounted on here so the GET_OPCODE fetch below is counted.
			gProfilingCounted = true;
		}
#endif


		opcode = get_iword(0);
#if TIME_STARTUP
		++gInstructionCount;
#endif

		// Execute the instruction.  Catch any exceptions thrown because of
		// address/bus errors.
		//
		// In CodeWarrior, the overhead of this try/catch block is minimal:
		// the SP is saved, and there's a branch around the catch block.
		// According to Greg, the overhead in VC++ is similar.	Timing
		// tests show no appreciable difference with this try/catch
		// block in place.

		try
		{
		
#if HAS_PROFILING
		if (gProfilingEnabled) {

			// Add in the extra time taken to execute the instruction.
			ProfileIncrementClock (perftbl[opcode].extraCycles);
		}
	
#endif

			*cycleCount += (*functable[opcode]) (opcode);


#if HAS_PROFILING
			if (gProfilingEnabled)
			{
				// Detail (instruction level) profiling support
				if (gProfilingDetailed)
					ProfileInstructionExit(*instructionStart);

#if HAS_PROFILING_DEBUG
				// Validity check on RAM_ROM stuff.

				Boolean tryAgain = false;

				if (perftbl[opcode].readCycles != 0xFF && gReadCycles - readCyclesSaved != perftbl[opcode].readCycles)
				{
					gReadMismatch += gReadCycles - readCyclesSaved - perftbl[opcode].readCycles;
				}

				if (perftbl[opcode].writeCycles != 0xFF && gWriteCycles - writeCyclesSaved != perftbl[opcode].writeCycles)
				{
					gWriteMismatch += gWriteCycles - writeCyclesSaved - perftbl[opcode].writeCycles;
				}

				if (tryAgain)
				{
					(*functable[opcode]) (opcode);
				}
#endif
			}
#endif

		}
		catch (ErrCode errCode)
		{
			// By the time we reach here, the break code that caused the
			// exception to be thrown should be recorded in gBreakReason.
			// Any error should also have already been reported.

			assert (errCode == kError_NoError);

			// Actually, this is not a good assert. We don't always want to
			// break if we throw an exception. An example is when we reset
			// the device, and want to abort anything else that's going on
			// after we get the state into the right state.
//			assert (gBreakReason != 0);
		}

		// Perform periodic tasks.

		Hardware::Cycle (false);

StoppedLoop:

		// Handle special conditions.  NB: the code reached by calling
		// Emulator::ExecuteSpecial used to be inline in this function.  Moving it
		// out (thus simplifying both Emulator::Execute and Emulator::ExecuteSpecial)
		// sped up the CPU loop by 9%!

		if (regs.spcflags)
		{
			Emulator::ExecuteSpecial ();
		}

#if HAS_PROFILING
		gProfilingCounted = false;
#endif
	}	// while (...)

	// Clear the reason for our breaking, and return that reason to the caller.

	uae_u32 breakReason = gBreakReason;

	ClearBreakReason (gBreakReasonMask);

	return breakReason & gBreakReasonMask;
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::ExecuteSpecial
 *
 * DESCRIPTION: Deal with rarer CPU emulation activities, such as
 *				processing interruptes, handling trace mode, etc.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Emulator::ExecuteSpecial (void)
{
	if (regs.spcflags & SPCFLAG_RESET)
	{
		Emulator::Reset ();
		return;
	}

	// If we're tracing, execute the trace vector.

	if ((regs.spcflags & SPCFLAG_DOTRACE) && !(regs.spcflags & SPCFLAG_STOP))
	{
		Software::ProcessException (kException_Trace, gLastTraceAddress, m68k_getpc ());
	}

	if (regs.spcflags & SPCFLAG_STOP)
	{
		Emulator::ExecuteStoppedLoop ();

		// If we're still stopped, exit now; we have to handle events.

		if (regs.spcflags & SPCFLAG_STOP)
			return;
	}

	// Do trace-mode stuff (do_trace from newcpu.c does more,
	// but it's only needed for CPU_LEVEL > 0)

	if (regs.spcflags & SPCFLAG_TRACE)
	{
		gLastTraceAddress = m68k_getpc ();
		regs.spcflags &= ~SPCFLAG_TRACE;
		regs.spcflags |= SPCFLAG_DOTRACE;
	}

	if (regs.spcflags & SPCFLAG_DOINT)
	{
		uae_s32 interruptLevel = HWRegisters::GetInterruptLevel ();
		regs.spcflags &= ~SPCFLAG_DOINT;	// was ~(SPCFLAG_INT | SPCFLAG_DOINT) in Greg and Craig, but the latest UAE has this
		if ((interruptLevel != -1) && (interruptLevel > regs.intmask))
		{
			Software::ProcessInterrupt (interruptLevel);
			regs.stopped = 0;
		}
	}

	if (regs.spcflags & SPCFLAG_INT)
	{
		regs.spcflags &= ~SPCFLAG_INT;
		regs.spcflags |= SPCFLAG_DOINT;
	}

	if (regs.spcflags & SPCFLAG_BRK)
	{
		regs.spcflags &= ~SPCFLAG_BRK;
		Emulator::SetBreakReason (kException_Trace);
	}

	if (regs.spcflags & SPCFLAG_SAVE_STATE)
	{
		regs.spcflags &= ~SPCFLAG_SAVE_STATE;
		Hordes::AutoSaveState();
	}

	if (regs.spcflags & SPCFLAG_SAVE_SUSPENDED_STATE)
	{
		regs.spcflags &= ~SPCFLAG_SAVE_SUSPENDED_STATE;
		Hordes::SaveSuspendedState();
	}

	if (regs.spcflags & SPCFLAG_SAVE_ROOT_STATE)
	{
		regs.spcflags &= ~SPCFLAG_SAVE_ROOT_STATE;
		Hordes::SaveRootState();
	}

	if (regs.spcflags & SPCFLAG_LOAD_ROOT_STATE)
	{
		regs.spcflags &= ~SPCFLAG_LOAD_ROOT_STATE;
		Hordes::LoadRootState();
	}

	if (regs.spcflags & SPCFLAG_NEXT_GREMLIN_FROM_ROOT_STATE)
	{
		regs.spcflags &= ~SPCFLAG_NEXT_GREMLIN_FROM_ROOT_STATE;

		if ( Hordes::LoadRootState() == errNone )
			Hordes::StartGremlinFromLoadedRootState();

		else
			Hordes::TurnOn(false);
	}

	if (regs.spcflags & SPCFLAG_NEXT_GREMLIN_FROM_SUSPENDED_STATE)
	{
		regs.spcflags &= ~SPCFLAG_NEXT_GREMLIN_FROM_ROOT_STATE;

		if ( Hordes::LoadSuspendedState() == errNone )
			Hordes::StartGremlinFromLoadedSuspendedState();

		else
			Hordes::TurnOn(false);
	}

	if (regs.spcflags & SPCFLAG_RESUME_HORDES_FROM_FILE)
	{
		regs.spcflags &= ~SPCFLAG_RESUME_HORDES_FROM_FILE;

		// TODO: IMPLEMENT GREMLINS SEARCH RESUME.
	}

	if (regs.spcflags & SPCFLAG_RESET_BANKS)
	{
		regs.spcflags &= ~SPCFLAG_RESET_BANKS;

		Memory::ResetRAMBankHandlers ();
	}
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::ExecuteStoppedLoop
 *
 * DESCRIPTION: Handle the CPU state after it's executed the STOP opcode.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Emulator::ExecuteStoppedLoop (void)
{
	assert (!ATrap::DoingCall ());

	// While the CPU is stopped (because a STOP instruction was
	// executed) do some idle tasks.

#if defined (macintosh) && defined (_DEBUG)
	// Put in a little dead-man's switch. If this function doesn't
	// exit for a long time, let us get into the debugger.
	uae_u32 deadManStart;
	if ((gBreakReasonMask & kBreak_CheckForEvents) == 0)
	{
		deadManStart = Platform::GetMilliseconds ();
	}
#endif

	do {
#if defined (macintosh) && defined (_DEBUG)
		// Put in a little dead-man's switch. If this function doesn't
		// exit for a long time, let us get into the debugger.
		if ((gBreakReasonMask & kBreak_CheckForEvents) == 0)
		{
			uae_u32 deadManNow = Platform::GetMilliseconds ();
			if ((deadManNow - deadManStart) > 5000)
			{
				Platform::Debugger ();
			}
		}
#endif

		// Slow down processing so that the timer used
		// to increment the tickcount doesn't run too quickly.

#if __profile__
	short	oldStatus = ProfilerGetStatus ();
	ProfilerSetStatus (false);
#endif

		Platform::Delay ();

#if __profile__
	ProfilerSetStatus (oldStatus);
#endif

		// Perform periodic tasks.

		Hardware::Cycle (true);

		// Process an interrupt (see if it's time to wake up).

		if (regs.spcflags & (SPCFLAG_INT | SPCFLAG_DOINT))
		{
			uae_s32 interruptLevel = HWRegisters::GetInterruptLevel ();

			regs.spcflags &= ~(SPCFLAG_INT | SPCFLAG_DOINT);

			if ((interruptLevel != -1) && (interruptLevel > regs.intmask))
			{
				Software::ProcessInterrupt (interruptLevel);
				regs.stopped = 0;
				regs.spcflags &= ~SPCFLAG_STOP;
			}
		}
	} while ((regs.spcflags & SPCFLAG_STOP) && gShouldLoop);
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::ExecuteUntilATrap
 *
 * DESCRIPTION: Perform CPU emulation until a TRAP $F / $Axxx sequence
 *				is reached.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	The reason why the loop exited.
 *
 ***********************************************************************/

long Emulator::ExecuteUntilATrap (void)
{
	// Tell the CPU emulator to exit on the next "ATrap".

	CBreakOnException	breaker (kException_Trap15);

	// Run the processor until we break.

	uae_u32 oldBreakMask = Emulator::ClearBreakReasonMask (kBreak_StopRequest | kBreak_CheckForEvents);
	long	reason = Emulator::Execute ();
	Emulator::SetBreakReasonMask (oldBreakMask);

	// If a request to handle updates came in while we were running, ignore
	// it by clearing it.  We can handle it when we get back to the main
	// CPU loop.  We _don't_ want this break reason to remain registered,
	// as we're not in a position to handle it right now (we're probably
	// setting up to make a ROM subroutine call) and leaving it registered
	// will just cause the CPU loop to exit again.

	Emulator::ClearBreakReason (kBreak_CheckForEvents);

	return reason;
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::ExecuteUntilBreak
 *
 * DESCRIPTION: Processes instructions until a break occurs.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	The reason why the loop exited.
 *
 ***********************************************************************/

long Emulator::ExecuteUntilBreak (void)
{
	// ExecuteUntilBreak is called when we're calling ROM functions as
	// subroutines.  In that context, we DON'T want to break if another
	// thread wants to break in by telling us to break on an "ATrap".

	Bool	oldBreakOn15 = SetBreakOnException (kException_Trap15, false);

	// Similarly, we don't want to break to handle events or because another
	// thread asks us to.  Our call into the ROM at this point should be
	// pretty much atomic (modulo real exceptions like bus errors).

	uae_u32 oldBreakMask = Emulator::ClearBreakReasonMask (kBreak_StopRequest | kBreak_CheckForEvents);

	// Run the processor until we break.

	long	reason;

	do {
		reason = Emulator::Execute ();

		// See if we broke on an ATrap anyway.	If we did, then it's because
		// another thread called SetBreakOnException (kException_Trap15, true)
		// after we set it to false.  Remember the fact that the other thread
		// wants to break on the next ATrap (by setting oldBreakOn15 to true),
		// and set that break flag back to false.

		if ((reason & kBreak_Exception) == kException_Trap15)
		{
			reason &= ~kBreak_Exception;
			oldBreakOn15 = true;
			SetBreakOnException (kException_Trap15, false);
		}
	}
	while (reason == 0);

	// Restore the old settings.

	Emulator::SetBreakReasonMask (oldBreakMask);
	(void) SetBreakOnException (kException_Trap15, oldBreakOn15);

	// If a request to handle updates came in while we were running, ignore
	// it by clearing it.  We can handle it when we get back to the main
	// CPU loop.  We _don't_ want this break reason to remain registered,
	// as we're not in a position to handle it right now (we're probably
	// setting up to make a ROM subroutine call) and leaving it registered
	// will just cause the CPU loop to exit again.

	Emulator::ClearBreakReason (kBreak_CheckForEvents);

	return reason;
}


/***********************************************************************
 *
 * FUNCTION:    Emulator::Runable
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

Bool Emulator::Runable (void)
{
	return !Suspended() && !Debug::InDebugger ();
}


/***********************************************************************
 *
 * FUNCTION:    Emulator::Suspended
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

Bool Emulator::Suspended (void)
{
	return gSuspended;
}


/***********************************************************************
 *
 * FUNCTION:    Emulator::Suspend
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

ErrCode Emulator::Suspend (void)
{
	gSuspended = true;
	SetBreakReason (kBreak_Suspended);
	return errNone;
}


/***********************************************************************
 *
 * FUNCTION:    Emulator::Resume
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

ErrCode Emulator::Resume (void)
{
	// This function is only called from an external script.  External
	// scripts are called within the contenxt of a CPUStopper object.
	// When that object is destructed, it will restart the CPU thread
	// if gSuspended is false (and there are no other impediments to
	// restarting the thread).

	gSuspended = false;

	return errNone;
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::HandleCPUBreak
 *
 * DESCRIPTION: Give all interested parties a crack at determining if
 *				they need to carry out any actions at this breakpoint.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void Emulator::HandleCPUBreak (void)
{
	// Give interested parties a crack at this break.

	Patches::HandleCPUBreak ();
	Debug::HandleCPUBreak ();
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::InstallCPUBreaks
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void Emulator::InstallCPUBreaks (void)
{
	// Clear the bit saying to break here.

//	MetaMemory::UnmarkCPUBreak (m68k_getpc ());

	Patches::RemoveCPUBreaks ();
	Debug::RemoveCPUBreaks ();

	// Give interested parties a chance to re-install the break.

	Patches::InstallCPUBreaks ();
	Debug::InstallCPUBreaks ();
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::SetBreakReason
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uae_u32 Emulator::SetBreakReason (uae_u32 setBits)
{
	omni_mutex_lock lock (gBreakReasonMutex);

	uae_u32 oldReason = gBreakReason;

	gBreakReason |= setBits;

	gShouldLoop = (gBreakReason & gBreakReasonMask) == 0;

	return oldReason;
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::SetBreakReasonMask
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uae_u32 Emulator::SetBreakReasonMask (uae_u32 setBits)
{
	uae_u32 oldReasonMask = gBreakReasonMask;

	gBreakReasonMask |= setBits;

	return oldReasonMask;
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::ClearBreakReason
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uae_u32 Emulator::ClearBreakReason (uae_u32 clearBits)
{
	uae_u32 oldReason = gBreakReason;

	gBreakReason &= ~clearBits;

	gShouldLoop = (gBreakReason & gBreakReasonMask) == 0;

	return oldReason;
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::ClearBreakReasonMask
 *
 * DESCRIPTION: 
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uae_u32 Emulator::ClearBreakReasonMask (uae_u32 clearBits)
{
	uae_u32 oldReasonMask = gBreakReasonMask;

	gBreakReasonMask &= ~clearBits;

	return oldReasonMask;
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::SetBreakOnException
 *
 * DESCRIPTION: Specify whether or not the CPU loop should exit when
 *				the given exception is encountered.  If so, then the
 *				CPU loop exits with the PC pointing to the instruction
 *				that caused the loop to exit (and not 2, 4, or 6 bytes
 *				beyond it).
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Emulator::SetBreakOnException (uae_s32 exceptionNumber, Bool doBreak)
{
	omni_mutex_lock lock (gBreakOnExceptionMutex);

	Bool	oldBreak = gBreakOnException[exceptionNumber];
	gBreakOnException[exceptionNumber] = doBreak;
	return oldBreak;
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::GetHardwareDevice
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

DeviceType Emulator::GetHardwareDevice (void)
{
	assert (gHardwareDevice != kDeviceUnspecified);
	return gHardwareDevice;
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::SetHardwareDevice
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Emulator::SetHardwareDevice (DeviceType hardwareDevice)
{
	assert (gHardwareDevice == kDeviceUnspecified);
	gHardwareDevice = hardwareDevice;

// -*- NEW DEVICE -*-

	gIsEZDevice =	(gHardwareDevice == kDevicePalmIIIx) ||
					(gHardwareDevice == kDeviceAustin) ||
					(gHardwareDevice == kDevicePalmVIIEZ) ||
					(gHardwareDevice == kDevicePalmV);
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::HasFlash
 *
 * DESCRIPTION: Returns whether or not the currently emulated device
 *				contains Flash RAM or not.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	True if the device has Flash instead of ROM.
 *
 ***********************************************************************/

Bool Emulator::HasFlash (void)
{
// -*- NEW DEVICE -*-

	DeviceType	type = GetHardwareDevice();
	return (type == kDevicePalmVII) ||
			(type == kDevicePalmV) ||
			(type == kDeviceAustin) ||
			(type == kDevicePalmVIIEZ) ||
			(type == kDevicePalmIIIx);
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::HasPLD
 *
 * DESCRIPTION: Returns whether or not the currently emulated device
 *				contains Flash RAM or not.
 * *
 * PARAMETERS:	None
 *
 * RETURNED:	True if the device has Flash instead of ROM.
 *
 ***********************************************************************/

Bool Emulator::HasPLD (void)
{
// -*- NEW DEVICE -*-

	DeviceType	type = GetHardwareDevice();
	return (type == kDevicePalmVIIEZ);
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::HasSED
 *
 * DESCRIPTION: Returns whether or not the currently emulated device
 *				contains Flash RAM or not.
 * *
 * PARAMETERS:	None
 *
 * RETURNED:	True if the device has Flash instead of ROM.
 *
 ***********************************************************************/

Bool Emulator::HasSED (void)
{
// -*- NEW DEVICE -*-

	DeviceType	type = GetHardwareDevice();
	return (type == kDeviceAustin);
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::HandleDlgButton
 *
 * DESCRIPTION: Common code to handle the result of Errors::DoDialog
 *				(the Continue/Debug/Reset dialog).  Called by:
 *
 *					Software::CheckStackPointer
 *					Debug::BreakIfConnected
 *					Errors::ReportInvalidSystemCall
 *					PalmHeap::WalkChunks
 *					Memory::PreventedAccess
 *
 *				If the user clicked on Debug, we fire up the Debug
 *				sub-system and tell CPU::Execute to halt by setting
 *				a "break reason".
 *
 *				If the user clicked on Reset, we reset the emulator
 *				state and then throw a silent exception to get us back
 *				up to CPU::Execute without having the intermediate
 *				functions get in the way (the may not be able to deal
 *				with the change of state very well).
 *
 *				If the user clicked on Continue, then check to see
 *				if a Gremlin/Horde was running.  If so, then DoDialog
 *				simply logs the error message and returns kContinue.
 *				At this point, we need to set the machinery in motion
 *				to switch to a different Gremlin.
 *
 *				"Emulator" isn't really the right place for this
 *				function, but I'm not sure where the right place is...
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Emulator::HandleDlgButton (int button, uaecptr oldpc)
{
	if (button == Errors::kDebug)
	{
		// If the user clicked on Debug, simulate a bus error in order to
		// get us into the debugger.
		//
		// !!! Need to check the result of EnterDebugger to see if we could
		// connect to a debugger.

		Hordes::Stop ();
//		Debug::SendMessage ("Control transferred to the debugger.");
		Debug::EnterDebugger (kException_Trap0 + sysDbgBreakpointTrapNum, oldpc, NULL);
		Emulator::SetBreakReason (kException_Trap0 + sysDbgBreakpointTrapNum);
	}
	
	else if (button == Errors::kReset)
	{
		Emulator::Reset ();
		Errors::Scram ();
	}

	else if (button == Errors::kNextGremlin)
	{
		Hordes::ErrorEncountered ();
		Errors::Scram ();
	}

	else	// ...Errors::kContinue...
	{
	}
}


/***********************************************************************
 *
 * FUNCTION:	Emulator::GetInstructionCount
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

#if TIME_STARTUP
unsigned long Emulator::GetInstructionCount (void)
{
	return gInstructionCount;
}
#endif


/***********************************************************************
 *
 * FUNCTION:	Emulator::GetCycleCount
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

unsigned long Emulator::GetCycleCount (void)
{
	return gCycleCount;
}


#pragma mark -

// ===========================================================================
//		 Hardware
// ===========================================================================

static Bool 		gHavePenEvent;
static Bool 		gPenIsDown;
static PointType	gPenLocation;


/***********************************************************************
 *
 * FUNCTION:	Hardware::Initialize
 *
 * DESCRIPTION: Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Hardware::Initialize (void)
{
	// Initialize some CPU-related tables

	// All of the stuff in this function needs to be done only once;
	// it doesn't need to be executed every time we load in a new ROM.

	static int	initialized;

	if (initialized)
		return;

	initialized = true;

	// (This initialization code is taken from init_m68k in newcpu.c)

	int 	i, j;

	for (i = 0 ; i < 256 ; i++)
	{
		for (j = 0 ; j < 8 ; j++)
		{
			if (i & (1 << j))
			{
				break;
			}
		}

		movem_index1[i] = j;
		movem_index2[i] = 7-j;
		movem_next[i] = i & (~(1 << j));
	}

	for (i = 0 ; i < 256 ; i++)
	{
		for (j = 7 ; j >= 0 ; j--)
		{
			if (i & (1 << j))
			{
				break;
			}
		}

		fpp_movem_index1[i] = j;
		fpp_movem_index2[i] = 7-j;
		fpp_movem_next[i] = i & (~(1 << j));
	}

	// read_table68k is going to try to allocate a bunch of memory.
	// But it assumes that that attempt will succeed, and try using
	// that memory without checking for NULL.  This can crash you horribly
	// on a Mac, so preflight that allocation here.

	table68k = (struct instr*) Platform::AllocateMemory (65536 * sizeof (struct instr));

	Platform::DisposeMemory (table68k);

	read_table68k ();
	do_merges ();

	// The rest of this code is based on build_cpufunctbl in newcpu.c.

	unsigned long	opcode;
	struct cputbl*	tbl = op_smalltbl_3;

	for (opcode = 0; opcode < 65536; opcode++)
	{
		cpufunctbl[opcode] = op_illg;
	}

	for (i = 0; tbl[i].handler != NULL; i++)
	{
		if (!tbl[i].specific)
		{
			cpufunctbl[tbl[i].opcode] = tbl[i].handler;
#if HAS_PROFILING
			perftbl[tbl[i].opcode] = tbl[i].perf;
#endif
		}
	}

	for (opcode = 0; opcode < 65536; opcode++)
	{
		cpuop_func* f;

		if (table68k[opcode].mnemo == i_ILLG || table68k[opcode].clev > 0)
		{
			continue;
		}

		if (table68k[opcode].handler != -1)
		{
			f = cpufunctbl[table68k[opcode].handler];
			if (f == op_illg)
			{
				abort ();
			}

			cpufunctbl[opcode] = f;
#if HAS_PROFILING
			perftbl[opcode] = perftbl[table68k[opcode].handler];
#endif

		}
	}	

	for (i = 0; tbl[i].handler != NULL; i++)
	{
		if (tbl[i].specific)
		{
			cpufunctbl[tbl[i].opcode] = tbl[i].handler;
#if HAS_PROFILING
			perftbl[tbl[i].opcode] = tbl[i].perf;
#endif
		}
	}

	// (hey readcpu doesn't free this guy!)

	Platform::DisposeMemory (table68k);
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::Reset
 *
 * DESCRIPTION: Sets the registers to default values.  This occurs when
 *				the emulator boots for the first time or is reset. Also
 *				called directly from the RESET opcode handler (which
 *				would seem to bypass m68k_reset in UAE, but is not
 *				skipped here, since we merged m68K_reset and customreset
 *				into Hardware::Reset).
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Hardware::Reset (void)
{
	// Re-initialize some variables.

	gHavePenEvent = false;
	gPenIsDown = false;

	// (taken from m68k_reset in newcpu.c)

	uaecptr romStart = ROMBank::GetMemoryStart ();
	m68k_areg (regs, 7) = get_long (romStart);
	m68k_setpc (get_long (romStart + 4));

	// Note, on the 68K, the t0 and m flags must always be zero.

	regs.kick_mask	= 0xF80000;			// (not a 68K register)
	regs.s			= 1;				// supervisor/user state
	regs.m			= 0;				// master/interrupt state
	regs.stopped	= 0;				// (not a 68K register)
	regs.t1 		= 0;				// Trace enable: 00 = no trace, 10 = trace on any instruction
	regs.t0 		= 0;				//		01 = trace on change of flow, 11 = undefined
	ZFLG = CFLG = NFLG = VFLG = 0;
	regs.spcflags	= 0;				// (not a 68K register)
	regs.intmask	= 7;				// disable all interrupts
	regs.vbr = regs.sfc = regs.dfc = 0;
	regs.fpcr = regs.fpsr = regs.fpiar = 0;

	// UAE doesn't do this next step, but it seems like a good idea.

	Registers::UpdateSRFromRegisters ();
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::Save
 *
 * DESCRIPTION: Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Hardware::Save (SessionFile& f)
{
	// Write out the CPU Registers

	regstruct	tempRegs;
	Registers::GetRegisters (tempRegs);

	Canonical (tempRegs);
	f.WriteDBallRegs (tempRegs);
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::Load
 *
 * DESCRIPTION: Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Hardware::Load (SessionFile& f)
{
	// Read in the CPU Registers.

	regstruct	tempRegs;
	f.ReadDBallRegs (tempRegs);

	Canonical (tempRegs);
	Registers::SetRegisters (tempRegs);
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Hardware::Dispose (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::Cycle
 *
 * DESCRIPTION: Handles periodic events that need to occur when the
 *				processor cycles (like updating timer registers).  This
 *				function is called in two places from Emulator::Execute.
 *				Interestingly, the loop runs 3% FASTER if this function
 *				is in its own separate function instead of being inline.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Hardware::Cycle (Bool sleeping)
{
	// Don't do anything if we're in the middle of an ATrap call. We don't
	// need interrupts firing or tmr counters incrementing.

	if (ATrap::DoingCall ())
	{
		return;
	}

	// Perform CPU-specific idling.
	//
	// (Doing this inline makes a performance difference.)

//	HWRegisters::Cycle (sleeping);

	if (Emulator::EZMode ())
		MC68EZ328Bank::Cycle (sleeping);
	else
		MC68328Bank::Cycle (sleeping);

	// If we're not sleeping, and there's data in the queue,
	// make sure our UART registers reflect that.

	if (!sleeping)
	{
		if (UART::GetRxQueue().GetUsed () > 0)
		{
			HWRegisters::UpdateUARTRegisters (false);
		}
	}

	// Note, the way we do this is very critical!  Previously, I'd updated
	// the screen and checked for events separately by calling two different
	// functions.  Each of these functions would check to see if an
	// appropropriate amount of time had elapsed, and perform their function
	// if it had.
	//
	// Making 4 function calls (one to the "update screen" function, one to
	// the "handle events" function, and two to LMGetTicks()) was extremely
	// expensive.  Changing this to a single call to LMTicks (in the inline
	// Platform_TimeForCycle function) speeds up the main CPU loop by 20%!

	if (Platform::TimeForCycle (sleeping))
	{
		HWRegisters::CycleSlowly (sleeping);

		// Do some platform-specific stuff.

		Platform::Cycle ();
	}
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::PenEvent
 *
 * DESCRIPTION: Handles a touch-screen pen event by updating the
 *				appropriate registers.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Hardware::PenEvent (Bool	penIsDown,
						uae_s32 penX,
						uae_s32 penY)
{
	if (!CanBotherCPU())
		return;

	PointType	pen = { penX, penY };

	// If this is a pen-down event...

	if (penIsDown)
	{
		// If this pen-down event is the same as the last pen-down
		// event, do nothing.

		if (gPenIsDown && (pen.x == gPenLocation.x) && (pen.y == gPenLocation.y))
		{
			return;
		}

		gPenLocation = pen;
	}

	// Remember pen up/down state for next time.

	gPenIsDown = penIsDown;
	gHavePenEvent = true;

	WakeUpCPU (kStr_EnterPen);
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::ButtonEvent
 *
 * DESCRIPTION: Handles a Palm device button event by updating the
 *				appropriate registers.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Hardware::ButtonEvent (Bool			iButton_IsDown,
							SkinElementType iButton)
{
	// If Gremlins is running, do nothing. If we tried to post anything
	// now, we might be interrupting a call to KeyHandleInterrupt.

	if (Hordes::IsOn ())
		return;

//	if (!Patches::UIInitialized ())
//		return;

	int bitNumber = 0;
	switch (iButton)
	{
		case kElement_PowerButton:		bitNumber = keyBitPower;	break;
		case kElement_UpButton: 		bitNumber = keyBitPageUp;	break;
		case kElement_DownButton:		bitNumber = keyBitPageDown; break;
		case kElement_App1Button:		bitNumber = keyBitHard1;	break;
		case kElement_App2Button:		bitNumber = keyBitHard2;	break;
		case kElement_App3Button:		bitNumber = keyBitHard3;	break;
		case kElement_App4Button:		bitNumber = keyBitHard4;	break;
		case kElement_CradleButton: 	bitNumber = keyBitCradle;	break;
		case kElement_Antenna:			bitNumber = keyBitAntenna;	break;
		case kElement_ContrastButton:	bitNumber = keyBitContrast; break;
	}

	HWRegisters::ButtonEvent (iButton_IsDown, bitNumber);
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::HotSyncEvent
 *
 * DESCRIPTION: Handles a HotSync button event by updating the
 *				appropriate registers.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Hardware::HotSyncEvent (Bool iButton_IsDown)
{
	// If Gremlins is running, do nothing. If we tried to post anything
	// now, we might be interrupting a call to KeyHandleInterrupt.

	if (Hordes::IsOn ())
		return;

//	if (!Patches::UIInitialized ())
//		return;

	HWRegisters::HotSyncEvent (iButton_IsDown);
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::KeyboardEvent
 *
 * DESCRIPTION: Handles a key down event on the desktop keyboard by
 *				adding the key to the key queue.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Hardware::KeyboardEvent (uae_u8 iKey)
{
	if (!CanBotherCPU())
		return;

	// Add the key to the queue if it isn't full.

	if (gKeyQueue.GetFree () > 0)
	{
		gKeyQueue.Put (iKey);
	}

	WakeUpCPU (kStr_EnterKey);
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::CanBotherCPU
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Hardware::CanBotherCPU (void)
{
	// If Gremlins is running, do nothing. If we tried to post anything
	// now, we might be interrupting a call to EvtEnqueuePenPoint.

	if (Hordes::IsOn ())
		return false;

	if (!Patches::UIInitialized ())
		return false;

	if (regs.stopped)
		return false;

	return true;
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::WakeUpCPU
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Hardware::WakeUpCPU (long strID)
{
	// Make sure the app's awake.  Normally, we post events on a patch to
	// SysEvGroupWait.	However, if the Palm device is already waiting,
	// then that trap will never get called.  By calling EvtWakeup now,
	// we'll wake up the Palm device from its nap.

	// Only try waking up if the interrupt level is low-enough.  When the
	// kernel is mucking with sensitive data, it sets the level to 6.  If
	// we were to call EvtWakeup at this time, we could be munging data
	// that's already in use.

//	if (regs.intmask < 6)
	{
		Errors::ReportIfPalmError (strID, EvtWakeup ());
	}
}

/***********************************************************************
 *
 * FUNCTION:	Hardware::HavePenEvent
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Hardware::HavePenEvent (void)
{
	return gHavePenEvent;
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::SetHavePenEvent
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Hardware::SetHavePenEvent (Bool b)
{
	gHavePenEvent = b;
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::PenLocation
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

PointType Hardware::PenLocation (void)
{
	return gPenLocation;
}


/***********************************************************************
 *
 * FUNCTION:	Hardware::PenIsDown
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Hardware::PenIsDown (void)
{
	return gPenIsDown;
}


#pragma mark -

// ===========================================================================
//		 Screen
// ===========================================================================

static uaecptr				gScreenLow;
static uaecptr				gScreenHigh;

static uaecptr				gScreenBegin;
static uaecptr				gScreenEnd;

static Screen::BufferInfo	gBufferInfo;

const int	kBitsPerByte = 8;		// Once we figure out how many bits are on a
									// scanline, divide by this to get the number
									// of bytes.

/***********************************************************************
 *
 * FUNCTION:	Screen::Initialize
 *
 * DESCRIPTION: Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Screen::Initialize (void)
{
	gScreenLow		= 0xFFFFFFFF;
	gScreenHigh 	= UAE_NULL;

	gScreenBegin	= UAE_NULL;
	gScreenEnd		= UAE_NULL;

	memset (&gBufferInfo, 0, sizeof (gBufferInfo));
}


/***********************************************************************
 *
 * FUNCTION:	Screen::Reset
 *
 * DESCRIPTION: Standard reset function.  Sets the sub-system to a
 *				default state.	This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Screen::Reset (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	Screen::Save
 *
 * DESCRIPTION: Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Screen::Save (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	Screen::Load
 *
 * DESCRIPTION: Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Screen::Load (SessionFile&)
{
	// !!! Need to re-sync
}


/***********************************************************************
 *
 * FUNCTION:	Screen::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Screen::Dispose (void)
{
	Platform::DisposeMemory (gBufferInfo.myCLUT);
	Platform::DisposeMemory (gBufferInfo.myBuffer);
}


/***********************************************************************
 *
 * FUNCTION:	Screen::MarkDirty
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Screen::MarkDirty (uaecptr address, uae_u32 size)
{
	// SUBTLE BUG NOTE:  Both of these tests need to be performed.
	// Originally, I had this as an if (...) ... else if (...) ...
	// After all, how could a given address be above the high-
	// water mark if it was below the low-water mark?  "Obviously",
	// it couldn't, so I didn't see the need to test against the
	// high-water mark if we were below the low-water mark.
	//
	// Well, duh, that assumption was false.  After the LCD buffer
	// is copied to the screen, we set the low-water mark to the
	// end of the buffer and the high-water mark to the beginning
	// of the buffer.  If the screen is modified in such a way
	// that each pixel affected appears lower in memory than any
	// previous pixel (as happens when we scroll a document down),
	// then we always entered the low-water mark of the code below;
	// we never entered the high-water mark of the code.  Thus, the
	// high-water mark stayed set to the beginning of the buffer,
	// giving us a NULL update range.

	if (address < gScreenLow)
		gScreenLow = address;

	if (address + size > gScreenHigh)
		gScreenHigh = address + size;
}


/***********************************************************************
 *
 * FUNCTION:	Screen::InvalidateAll
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Screen::InvalidateAll (void)
{
	gScreenLow = 0;
	gScreenHigh = 0xFFFFFFFF;

	uaecptr newScreenBegin = MetaMemory::GetScreenBegin ();
	uaecptr newScreenEnd = MetaMemory::GetScreenEnd ();

	if (newScreenBegin != gScreenBegin || newScreenEnd != gScreenEnd)
	{
		MetaMemory::UnmarkScreen (gScreenBegin, gScreenEnd);
		gScreenBegin	= newScreenBegin;
		gScreenEnd		= newScreenEnd;
		MetaMemory::MarkScreen (gScreenBegin, gScreenEnd);
	}
}


/***********************************************************************
 *
 * FUNCTION:	Screen::GetBits
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Screen::GetBits (BufferInfo& info, Bool incremental)
{
	// Get the current state of things.

	// Fill out info on the LCD screen.

	info.lcdBuffer		= HWRegisters::GetLCDStartAddr ();
	info.lcdDepth		= HWRegisters::GetLCDDepth ();
	info.lcdRowBytes	= HWRegisters::GetLCDRowBytes ();

	// Fill out info on the buffer we'll be returning.

	info.myBuffer		= NULL;
	info.myCLUT 		= NULL;
	info.myDepth		= 8;
	info.myRowBytes 	= info.lcdRowBytes * info.myDepth / info.lcdDepth;

	if (gBufferInfo.myCLUT == NULL)
	{
		Platform::DisposeMemory (gBufferInfo.myCLUT);
		gBufferInfo.myCLUT = (RGBType*) Platform::AllocateMemory (sizeof(RGBType) * 256);
	}

	HWRegisters::GetLCDPalette (gBufferInfo.myCLUT);
	
	info.myCLUT = gBufferInfo.myCLUT;
	
	// Fill out some misc (common) info.

	info.firstLine		= 0;
	info.lastLine		= HWRegisters::GetLCDHeight ();
	info.height 		= info.lastLine;
	info.backlightOn	= HWRegisters::BacklightIsOn ();
	info.lcdOn			= HWRegisters::LCDIsOn ();

	info.visibleBounds.left 	= 0 + HWRegisters::GetLCDOffset ();
	info.visibleBounds.top		= 0;
	info.visibleBounds.right	= HWRegisters::GetLCDWidth () + HWRegisters::GetLCDOffset ();
	info.visibleBounds.bottom	= HWRegisters::GetLCDHeight ();

	// If the screen is off, we can just return now.  If the LCD on/off
	// state changed since the last time, return true so that the
	// screen is updated.

	if (!info.lcdOn)
	{
		Bool somethingChanged = info.lcdOn != gBufferInfo.lcdOn;
		gBufferInfo.lcdOn = info.lcdOn;
		return somethingChanged;
	}

	// Get the first and last scanlines to update.	This is a *big* win when
	// running Gremlins.  Without this check, a typical test ran 240 seconds.
	// With the check, the test ran 170 seconds.

	// Get the screen begin and end.  We'll be clipping against this range (just in
	// case screenLow and/or screenHigh got out of whack, which might happen in
	// a multi-threaded system and we aren't using synchronization objects).

	uaecptr screenBegin = MetaMemory::GetScreenBegin ();
	uaecptr screenEnd	= MetaMemory::GetScreenEnd ();

	// Get the range of bytes affected.  From this information, we can determine
	// the first and last affected scanlines.  After we have those values, reset
	// gScreenLow/High with sentinel values so that they can be munged again
	// by DRAMBank::SetFoo methods.

	uaecptr screenLow	= gScreenLow;	gScreenLow = screenEnd;
	uaecptr screenHigh	= gScreenHigh;	gScreenHigh = screenBegin;

	// Clip screenLow and screenHigh to the range of the screen.

	if (screenLow < screenBegin)
		screenLow = screenBegin;

	if (screenHigh > screenEnd)
		screenHigh = screenEnd;

	// If anything cataclysmic has changed, set screenLow and screenHigh
	// to cover the entire screen.

	if (info.lcdBuffer				!= gBufferInfo.lcdBuffer			||
		info.lcdDepth				!= gBufferInfo.lcdDepth 			||
		info.lcdRowBytes			!= gBufferInfo.lcdRowBytes			||
		info.height 				!= gBufferInfo.height				||
		info.backlightOn			!= gBufferInfo.backlightOn			||
		info.lcdOn					!= gBufferInfo.lcdOn				||
		info.visibleBounds.left 	!= gBufferInfo.visibleBounds.left	||
		info.visibleBounds.top		!= gBufferInfo.visibleBounds.top	||
		info.visibleBounds.right	!= gBufferInfo.visibleBounds.right	||
		info.visibleBounds.bottom	!= gBufferInfo.visibleBounds.bottom ||
		!incremental)
	{
		screenLow = screenBegin;
		screenHigh = screenEnd;
	}

	// If no lines need to be updated, we can return now.

	if (screenLow >= screenHigh)
	{
		return false;
	}

	// Determine the first and last changed scanlines.

	info.firstLine	= (screenLow - info.lcdBuffer) / info.lcdRowBytes;
	info.lastLine	= (screenHigh - info.lcdBuffer - 1) / info.lcdRowBytes + 1;

	// Get a buffer, in case we don't have one already, or
	// in case we need a bigger one.

	long	oldArea = gBufferInfo.height * gBufferInfo.myRowBytes;
	long	newArea = info.height * info.myRowBytes;

	if (gBufferInfo.myBuffer == NULL || newArea > oldArea)
	{
		Platform::DisposeMemory (gBufferInfo.myBuffer);
		gBufferInfo.myBuffer = (uae_u8*) Platform::AllocateMemory (newArea);
	}
	
	info.myBuffer = gBufferInfo.myBuffer;

	if (!gConvert1To8)
	{
		const int	kTableSize = 16;
		gConvert1To8 = (uae_u32*) Platform::AllocateMemory (sizeof (*gConvert1To8) * kTableSize);

		for (int ii = 0; ii < kTableSize; ++ii)
		{
			uae_u32 mask = 0x08;
			uae_u32 pixValue = 0x01000000;
			uae_u32 pixels = 0;

			while (mask)
			{
				if ((ii & mask) != 0)
				{
					pixels |= pixValue;
				}

				mask >>= 1;
				pixValue >>= 8;
			}

			Canonical (pixels);
			gConvert1To8[ii] = pixels;
		}
	}

	if (!gConvert2To8)
	{
		const int	kTableSize = 256;
		gConvert2To8 = (uae_u32*) Platform::AllocateMemory (sizeof (*gConvert2To8) * kTableSize);

		for (int ii = 0; ii < kTableSize; ++ii)
		{
			uae_u32 mask = 0xC0;
			uae_u32 shift = 24 - 6;
			uae_u32 pixels = 0;

			while (mask)
			{
				uae_u32 pixel = (ii & mask);
				pixel <<= shift;
				pixels |= pixel;

				mask >>= 2;
				shift -= 8 - 2;
			}

			Canonical (pixels);
			gConvert2To8[ii] = pixels;
		}
	}

	if (!gConvert4To8)
	{
		const int	kTableSize = 256;
		uae_u16 		pixels;
		
		gConvert4To8 = (uae_u16*) Platform::AllocateMemory (sizeof (*gConvert4To8) * kTableSize);

		for (int ii = 0; ii < kTableSize; ++ii)
		{
			pixels = ((ii & 0xf0) << 4) | (ii & 0x0f);
			Canonical(pixels);
			gConvert4To8[ii] = pixels;
		}
	}

	// Get the changed scanlines, converting to 8 bpp along the way.

	uae_u8* 	realBaseAddress = get_real_address (info.lcdBuffer);
	uae_u8* 	baseSrcP = realBaseAddress + info.firstLine * info.lcdRowBytes;
	uae_u8* 	baseDestP = info.myBuffer + info.firstLine * info.myRowBytes;
	long		srcBytesToCopy = info.lcdRowBytes * (info.lastLine - info.firstLine);

	if (info.lcdDepth == 1)
	{
		if (info.myDepth == 1)
		{
			uae_u8* srcP = (uae_u8*) baseSrcP;
			uae_u8* destP = (uae_u8*) baseDestP;

			while (srcBytesToCopy)
			{
				*destP++ = do_get_mem_byte (srcP++);

				srcBytesToCopy -= sizeof (*srcP);
			}
		}
		else if (info.myDepth == 8)
		{
			uae_u8* 	srcP = (uae_u8*) baseSrcP;
			uae_u32*	destP = (uae_u32*) baseDestP;

			while (srcBytesToCopy)
			{
				// Get 8 pixels.

				uae_u8	bits = do_get_mem_byte (srcP++);

				// Convert each nybble (which contains 4 1-bit pixels) to a 32-bit value
				// containing 4 8-bit pixels.

				*destP++ = gConvert1To8[(bits >>  4) & 0x0F];
				*destP++ = gConvert1To8[(bits >>  0) & 0x0F];

				srcBytesToCopy -= sizeof (*srcP);
			}
		}
		else
		{
			assert (!"Unsupported bit depth");
		}
	}
	else if (info.lcdDepth == 2)
	{
		if (info.myDepth == 2)
		{
			uae_u8* srcP = (uae_u8*) baseSrcP;
			uae_u8* destP = (uae_u8*) baseDestP;

			while (srcBytesToCopy)
			{
				*destP++ = do_get_mem_byte (srcP++);

				srcBytesToCopy -= sizeof (*srcP);
			}
		}
		else if (info.myDepth == 8)
		{
			uae_u8* 	srcP = (uae_u8*) baseSrcP;
			uae_u32*	destP = (uae_u32*) baseDestP;

			while (srcBytesToCopy)
			{
				// Get 4 pixels.

				uae_u8	bits = do_get_mem_byte (srcP++);

				// Convert each byte (which contains 4 2-bit pixels) to a 32-bit value
				// containing 4 8-bit pixels.

				*destP++ = gConvert2To8[bits];

				srcBytesToCopy -= sizeof (*srcP);
			}
		}
		else
		{
			assert (!"Unsupported bit depth");
		}
	}

	else if (info.lcdDepth == 4)
	{
		if (info.myDepth == 8)
		{
			uae_u8* 	srcP = (uae_u8*) baseSrcP;
			uae_u16*	destP = (uae_u16*) baseDestP;
			uae_u8		byte;

			while (srcBytesToCopy)
			{
				// Get 4 pixels.

				byte = do_get_mem_byte (srcP++);

				// Convert each word (which contains 4 4-bit pixels) to a 32-bit value
				// containing 4 8-bit pixels.

				*destP++ = gConvert4To8[byte];

				srcBytesToCopy -= sizeof (*srcP);
			}
		}
		else
		{
			assert (!"Unsupported bit depth");
		}
	}

	else if (info.lcdDepth == 8)
	{
		if (info.myDepth == 8)
		{
			uae_u8* srcP = (uae_u8*) baseSrcP;
			uae_u8* destP = (uae_u8*) baseDestP;

			while (srcBytesToCopy)
			{
				*destP++ = do_get_mem_byte (srcP++);

				srcBytesToCopy -= sizeof (*srcP);
			}
		}
		else
		{
			assert (!"Unsupported bit depth");
		}
	}

	else
	{
		assert (!"Unsupported bit depth");
	}

	gBufferInfo = info;

	return true;
}


#pragma mark -

// ===========================================================================
//		 Software
// ===========================================================================

static ExceptionHandlerProc gExceptionHandlers[kException_LastException];
static StackList			gStackList;


/***********************************************************************
 *
 * FUNCTION:	Software::Initialize
 *
 * DESCRIPTION: Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::Initialize (void)
{
	InstallExceptionHandler (kException_Trap0 + sysDispatchTrapNum,
							Software::ProcessTrap15);
}


/***********************************************************************
 *
 * FUNCTION:	Software::Reset
 *
 * DESCRIPTION: Standard reset function.  Sets the sub-system to a
 *				default state.	This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::Reset (void)
{
	gStackList.clear ();
}


/***********************************************************************
 *
 * FUNCTION:	Software::Save
 *
 * DESCRIPTION: Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::Save (SessionFile&)
{
	// !!! Need to write out the stack list.
}


/***********************************************************************
 *
 * FUNCTION:	Software::Load
 *
 * DESCRIPTION: Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::Load (SessionFile&)
{
	// !!! Need to restore the stack list
}


/***********************************************************************
 *
 * FUNCTION:	Software::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::Dispose (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	Software::BusError
 *
 * DESCRIPTION: Called by the FooBank_XXX routines when an illegal
 *				reference to memory is made.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void	Software::BusError(void)
{
	Software::ProcessException (kException_BusErr, gInstructionStart, m68k_getpc ());

	// Get outta here fast.  We don't try to execute the rest of the
	// instruction.

	Errors::Scram ();
}


/***********************************************************************
 *
 * FUNCTION:	Software::AddressError
 *
 * DESCRIPTION: Called by the FooBank_XXX routines when an illegal
 *				reference to memory is made.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void	Software::AddressError(void)
{
	Software::ProcessException (kException_AddressErr, gInstructionStart, m68k_getpc ());

	// Get outta here fast.  We don't try to execute the rest of the
	// instruction.

	Errors::Scram ();
}


/***********************************************************************
 *
 * FUNCTION:	Software::ProcessException
 *
 * DESCRIPTION: Handles an exception by updating the registers and the
 *				processor state.
 *
 *				Note that the original PC may or may not have been
 *				bumped by now, so use "oldPC" to get address of the
 *				instruction that caused the exception.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::ProcessException (uae_s32 iException, uaecptr oldpc, uaecptr curpc)
{
#if EXCEPTION_HISTORY
	// Save the exception for the post-mortem

	exception_history_index++;

	long	index = exception_history_index & (exception_history_size - 1);
	exception_history[index].name = kExceptionNames[iException];

	exception_history[index].pc = oldpc;
	exception_history[index].sp = m68k_areg (regs, 7);
#endif

	// See if we've been asked to break on this exception.

	if (gBreakOnException[iException])
	{
		// Back up the PC to point to where this whole mess started.

		m68k_setpc (oldpc);

		// Tell the CPU loop to stop.

		Emulator::SetBreakReason (iException);

		return;
	}

#if HAS_PROFILING
	// Don't count cycles spent in exception handlers against functions.

	if (gProfilingEnabled)
	{
		ProfileInterruptEnter (iException, curpc);
	}
#endif

	// Let any custom exception handler have a go at it.  If it returns
	// true, it handled it completely, and we don't have anything else to do.

	if (gExceptionHandlers[iException] &&
		gExceptionHandlers[iException] (iException, oldpc))
	{
#if HAS_PROFILING
		if (gProfilingEnabled)
		{
			ProfileInterruptExit (curpc);
		}
#endif
		return;
	}

	// The following is vaguely modelled after Exception() from newcpu.c

	// Make sure the Status Register is up-to-date.

	Registers::UpdateSRFromRegisters ();

	// If not in supervisor mode, set the usp, restore A7 from the isp,
	// and transfer to supervisor mode.

	if (!regs.s)
	{
		regs.usp = m68k_areg (regs, 7);
		m68k_areg (regs, 7) = regs.isp;
		regs.s = 1;
	}

	// Set up the stack frame.

	if (iException == kException_BusErr || iException == kException_AddressErr)
	{
		assert (sizeof (ExceptionStackFrame2) == 14);
		m68k_areg (regs, 7) -= sizeof (ExceptionStackFrame2);

		uaecptr frame = m68k_areg (regs, 7);

		put_word (frame + offsetof (ExceptionStackFrame2, functionCode), 0);	// Eh...Palm OS doesn't use these 3 anyway...
		put_long (frame + offsetof (ExceptionStackFrame2, accessAddress), 0);
		put_word (frame + offsetof (ExceptionStackFrame2, instructionRegister), 0);

		put_word (frame + offsetof (ExceptionStackFrame2, statusRegister), regs.sr);
		put_long (frame + offsetof (ExceptionStackFrame2, programCounter), curpc);
	}
	else
	{
		assert (sizeof (ExceptionStackFrame1) == 6);
		m68k_areg (regs, 7) -= sizeof (ExceptionStackFrame1);

		uaecptr frame = m68k_areg (regs, 7);

		put_word (frame + offsetof (ExceptionStackFrame1, statusRegister), regs.sr);
		put_long (frame + offsetof (ExceptionStackFrame1, programCounter), curpc);
	}

	{
		CEnableFullAccess	munge;	// Remove blocks on memory access.

		// Get the exception handler address and jam it in the PC.
		m68k_setpc (get_long (regs.vbr + 4 * iException));

	}

	// Turn tracing off.

	regs.t1 = regs.t0 = regs.m = 0;
	regs.spcflags &= ~(SPCFLAG_TRACE | SPCFLAG_DOTRACE);
}


/***********************************************************************
 *
 * FUNCTION:	Software::ProcessInterrupt
 *
 * DESCRIPTION: Handles an interrupt by updating the registers and the
 *				processor state.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::ProcessInterrupt (uae_s32 iInterrupt)
{
	Software::ProcessException (iInterrupt + HWRegisters::GetInterruptBase (),
								m68k_getpc (), m68k_getpc ());
	regs.intmask = iInterrupt;
	regs.spcflags |= SPCFLAG_INT;

	Registers::UpdateSRFromRegisters ();	// This isn't in UAE, but it seems like
											// a good idea to do after changing intmask.
}


/***********************************************************************
 *
 * FUNCTION:	Software::ProcessIllegalInstruction
 *
 * DESCRIPTION: Handles an illegal opcode by updating the registers and
 *				the processor state.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::ProcessIllegalInstruction (uae_u32 iOpcode)
{
	// This function is loosely based on op_illg in newcpu.c

	uaecptr oldpc = m68k_getpc ();
	uaecptr curpc = oldpc + 2;

	// Process an FTrap.

	if ((iOpcode & 0xF000) == 0xF000)
	{
		Software::ProcessException (kException_FTrap, oldpc, curpc);
		return;
	}

	// Process an ATrap.

	if ((iOpcode & 0xF000) == 0xA000)
	{
		Software::ProcessException (kException_ATrap, oldpc, curpc);
		return;
	}

	Software::ProcessException (kException_IllegalInstr, oldpc, curpc);
}


/***********************************************************************
 *
 * FUNCTION:	Software::InstallExceptionHandler
 *
 * DESCRIPTION: Specifies a function to be called when the given
 *				exception occurs.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::InstallExceptionHandler (uae_s32 exceptionNumber,
										ExceptionHandlerProc fn)
{
	gExceptionHandlers[exceptionNumber] = fn;
}


/***********************************************************************
 *
 * FUNCTION:	Software::RemoveExceptionHandler
 *
 * DESCRIPTION: Specifies a function to be called when the given
 *				exception occurs.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::RemoveExceptionHandler (uae_s32 exceptionNumber)
{
	gExceptionHandlers[exceptionNumber] = NULL;
}


/***********************************************************************
 *
 * FUNCTION:	Software::ProcessJSR
 *
 * DESCRIPTION: Specifies a function to be called when the given
 *				exception occurs.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Software::ProcessJSR (uaecptr oldpc, uaecptr dest)
{
#if HAS_PROFILING
	if (gProfilingEnabled)
	{
		StDisableMemoryCycleCounting	stopper;

		// In multi-segmented application generated by CodeWarrior,
		// inter-segment function calls are implemented by JSR'ing
		// to a jump table entry.  This jump table entry contains
		// a JMP to the final location.  Detect this case and emit
		// a function entry record for the final location, not the
		// jump table entry.
		uae_u8* realMem = get_real_address (dest);
		if (do_get_mem_word (realMem) == kOpcode_JMP_Abs32)
		{
			dest = do_get_mem_long (realMem + 2);
		}

		ProfileFnEnter (dest, oldpc);
		}
#endif

	return false;	// We didn't completely handle it; do default handling.
}


/***********************************************************************
 *
 * FUNCTION:	Software::ProcessJSR_Ind
 *
 * DESCRIPTION: Check for SYS_TRAP_FAST calls.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Software::ProcessJSR_Ind (uaecptr oldpc)
{
	Bool	handledIt = false;	// Default to calling ROM.

	//	inline asm SysTrapFast(Int trapNum)
	//	{
	//			MOVE.L	struct(LowMemType.fixed.globals.sysDispatchTableP), A1
	//			MOVE.L	((trapNum-sysTrapBase)*4)(A1), A1
	//			JSR 	(A1)																// call it
	//	}
	//
	//	#define SYS_TRAP_FAST(trapNum)
	//		FIVEWORD_INLINE(
	//			0x2278, 0x0122,
	//			0x2269, (trapNum-sysTrapBase)*4,
	//			0x4e91)

	uae_u8* realMem = get_real_address (oldpc);
	if (do_get_mem_word (realMem) == 0x4e91 &&
		do_get_mem_word (realMem - 4) == 0x2269 &&
		do_get_mem_word (realMem - 8) == 0x2278)
	{
		handledIt = Software::ProcessSystemCall (oldpc);
	}

	return handledIt;
}


/***********************************************************************
 *
 * FUNCTION:	Software::ProcessRTS
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Software::ProcessRTS (void)
{
#if HAS_PROFILING
	if (gProfilingEnabled)
	{
		StDisableMemoryCycleCounting	stopper;
		
		ProfileFnExit (get_long (m68k_areg (regs, 7)), m68k_getpc () + 2);
	}
#endif

	return false;	// We didn't completely handle it; do default handling.
}


/***********************************************************************
 *
 * FUNCTION:	Software::ProcessRTE
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Software::ProcessRTE (uaecptr newpc)
{
#if HAS_PROFILING
	if (gProfilingEnabled)
	{
		ProfileInterruptExit (newpc);
	}
#endif

	return false;	// We didn't completely handle it; do default handling.
}


/***********************************************************************
 *
 * FUNCTION:	Software::ProcessLINK
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Software::ProcessLINK (uae_s32 linkSize)
{
#if 0
	Preference<bool>	pref (kPrefKeyFillStack);
	if (*pref && linkSize < 0)
	{
		uae_memset (m68k_areg (regs, 7), 0xD5, -linkSize);
	}
#endif

	return false;	// We didn't completely handle it; do default handling.
}


/***********************************************************************
 *
 * FUNCTION:	Software::ProcessTrap15
 *
 * DESCRIPTION: Handle a trap. Traps are of the format TRAP $F / $Axxx.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Software::ProcessTrap15 (uae_s32 exceptionNumber, uaecptr oldpc)
{
	UNUSED_PARAM(exceptionNumber)

	return Software::ProcessSystemCall (oldpc);
}


/***********************************************************************
 *
 * FUNCTION:	Software::ProcessSystemCall
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Software::ProcessSystemCall (uaecptr startPC)
{
	// ======================================================================
	//	Determine what ROM function is about to be called, and determine
	//	the method by which it is being called.
	// ======================================================================

	SystemCallContext	context;
	GetSystemCallContext (startPC, context);


	// ======================================================================
	//	Log the fact that this ROM function was being called.  This is
	//	internal debugging information only.
	// ======================================================================

#if COUNT_TRAPS
	// Record this trap call, allocating the table used to hold that
	// information if necessary.

	if (LowMem::TrapExists (context.fTrapWord))
	{
		if (gTrapCounts == NULL)
		{
			gTrapCounts = new long[LowMem::gSysDispatchTableSize];
			memset (gTrapCounts, 0, sizeof(long) * LowMem::gSysDispatchTableSize);
		}

		gTrapCounts[context.fTrapIndex]++;
		gTotalTrapCount++;
	}
#endif

	// ======================================================================
	//	Validate the address for the ROM function we're about to call.
	// ======================================================================

	if (context.fDestPC == UAE_NULL)
	{
		Errors::ReportInvalidSystemCall (context);

		// *I* don't know what to do with this trap...maybe the ROM
		// does.  But I don't expect we'll ever get here anyway.

		return kExecuteROM;
	}

	// ======================================================================
	//	Record what function we're calling.
	// ======================================================================

	if (LogSystemCalls () && !ATrap::DoingCall ())
	{
		char	name [sysPktMaxNameLen];

		strcpy (name, ::GetTrapName (context, true));

		LogAppendMsg ("--- System Call 0x%04X: %s.", (long) context.fTrapWord, name);
	}

	// ======================================================================
	// Let the debugger have a crack at it.  It may want to do a "break
	// on ATrap" thingy.  If so, it will return true.  Otherwise,
	// if no action is necessary, it will return false.
	// ======================================================================

	if (Debug::HandleSystemCall (context))
	{
		return kSkipROM;
	}


	// ======================================================================
	//	Tell the profiler that we're entering a function.
	// ======================================================================
#if HAS_PROFILING
	// This ProfileFnEnter will show the function call within the trap handler
	// exception, which gives us a nice breakdown of which traps are being
	// called and frequency.

	if (gProfilingEnabled && context.fViaTrap)
	{
		StDisableMemoryCycleCounting	stopper;
		
		ProfileFnEnter (context.fTrapWord, context.fNextPC);
	}
#endif

	// ======================================================================
	// If this trap is patched, let the patch handler handle the patch.
	// ======================================================================

	CallROMType result = Patches::HandleSystemCall (context);


	// ======================================================================
	//	If we completely handled the function in head and tail patches, tell
	//	the profiler that we exited the function and get out of here.
	// ======================================================================
	
	if (result == kSkipROM)
	{
#if HAS_PROFILING
		if (gProfilingEnabled)
		{
			StDisableMemoryCycleCounting	stopper;
			ProfileFnExit (context.fNextPC, context.fTrapWord);
		}
#endif

		// Set the PC to point past the instructions that got us here.

		m68k_setpc (context.fNextPC);

		// Return true to say that everything has been handled.

		return true;
	}


	// ======================================================================
	//	If we're profiling, don't dispatch to the ROM function outselves.
	//	We want the ROM to do it so that we get accurate dispatch times.
	// ======================================================================

#if HAS_PROFILING
	// Don't do native dispatching if the profiler is on!

	if (gProfilingEnabled)
	{
		return false;	// Return false to do default exception handler processing.
	}
#endif

#if NATIVE_DISPATCHING
	// ======================================================================
	// Otherwise, let's run the trap dispatcher native.  This gives
	// us about a 10-12% speed-up.
	// ======================================================================

	// Push the return address onto the stack. Subtract 4 from the stack,
	// and then store the appropriate return address.

	uaecptr oldA7 = m68k_areg(regs, 7);

	m68k_areg (regs, 7) -= 4;
	put_long (m68k_areg (regs, 7), context.fNextPC);

	// Change to the new PC.

	assert (context.fDestPC != UAE_NULL);
	m68k_setpc (context.fDestPC);

//	Software_CheckStackPointer (0, oldA7);

	return true;	// Return true to say that everything has been handled.
#else
	return false;	// Return false to do default exception handler processing.
#endif
}


/***********************************************************************
 *
 * FUNCTION:	Software::CheckNewPC
 *
 * DESCRIPTION: The PC is about to be changed via a JSR, BSR, RTS, or
 *				RTE.  Check that the new address appears to be valid.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::CheckNewPC (uaecptr dest)
{
	// If the address is odd, then it's bad.

	if ((dest & 1) != 0)
		Errors::ReportInvalidPC (dest, kOddAddress);

//	if (!Patches::UIInitialized ())
//		return;

	// If the address is not owned by a bank, then it's bad.

	if (get_mem_bank(dest).lput == DummyBank::SetLong)
	{
		if (!DummyBank::ValidAddress (dest, 2))
		{
			Errors::ReportInvalidPC (dest, kUnmappedAddress);
		}
	}
	else if (dest < (256 + 693 * 4) /*MetaMemory::GetSysGlobalsEnd ()*/)
	{
		Errors::ReportInvalidPC (dest, kNotInCodeSegment);
	}

	return;
}


/***********************************************************************
 *
 * FUNCTION:	Software::CheckStackPointer
 *
 * DESCRIPTION: !!! This operation slows down Gremlins by about 10%; it
 *				should be sped up or made an option.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::CheckStackPointer (int opcodeType, uaecptr oldA7)
{
#if 0
	// See if the stack pointer changed. If not, just leave.

	if (oldA7 == m68k_areg (regs, 7))
		return;

	uaecptr curA7 = m68k_areg (regs, 7);

	// See what kind of operation just changed the stack pointer.

	if (opcodeType == kMoveAOpcode || opcodeType == kEXGOpcode || opcodeType == kLEAOpcode)
	{
		// If it was a "drastic" change, assume that we're *setting* the
		// stack pointer to a new stack.  Scarf up information about that
		// block of memory and treat that block as a stack.

		if (!Patches::HeapInitialized ())	// If heaps not set up yet, just leave.
			return;

		// See if we already know about this stack.

		StackList::iterator iter = FindStackChunk (curA7);

		// If not, get some information about it and save it off

		if (iter == gStackList.end ())
		{
			// !!! Should ensure that the memory semaphore is not acquired;
			// the heap may be in flux at this point...

			uaecptr heapHdr = PalmHeap::FindHeapHeader (curA7);

			PalmHeap::HeapInfo	heapInfo;
			PalmHeap::GetHeapInfo (heapHdr, &heapInfo);

			uaecptr chunkHdr = PalmHeap::FindChunkHeader (curA7, heapInfo);

			PalmHeap::ChunkInfo chunkInfo;
			PalmHeap::GetChunkInfo (chunkHdr, heapInfo, &chunkInfo);

			// Couldn't find the chunk? Just leave...

			if (chunkInfo.version == 0)
				return;

			// The chunk is free? Just leave...assume this is the SysAppExit bug.
			// If it isn't, we'll catch it soon enough when something tries
			// to use the stack.

			if (chunkInfo.free)
				return;

			RememberStackChunk (chunkInfo);
		}
	}
	else
	{
		// If it was an "incremental" change, make sure that A7 is not
		// now outside the stack it used to be in (as indicated by oldA7).

		// If we haven't yet recorded any memory chunks as stack, then assume
		// that we're still booting up and using a range of memory that was
		// "allocated" before the memory manager was started up.  Under that
		// circumstance, we don't really know the stack range, so allow
		// any sort of access.

		if (gStackList.empty ())
			return;

		// Find the stack indicated by oldA7

		StackList::iterator iter = FindStackChunk (oldA7);

		// If we couldn't find it, just exit (what else to do?)

		if (iter == gStackList.end ())
			return;

		// Make sure that the current A7 is in the stack, too.

		const int	kStackSlush = 100;	// Warn when we get within this many bytes
										// of the bottom of the stack.

		// Test for "close to the end" (in which case we display a warning), and
		// "past the end" (in which case we print a fatal error).

		Bool	warning = curA7 <= (*iter).fBottom + kStackSlush;
		Bool	fatal = curA7 <= (*iter).fBottom;

		if (warning)
		{
			int button = Errors::ReportStackOverflow (fatal);
			Emulator::HandleDlgButton (button, m68k_getpc ());
		}
	}
#endif
}


/***********************************************************************
 *
 * FUNCTION:	Software::RememberStackChunk
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Software::RememberStackChunk (const PalmHeap::ChunkInfo& chunkInfo)
{
	// Get the range of the chunk.	(!!! Remember just the body,
	// or the header and any trailer, too?)

	uaecptr chunkStart = chunkInfo.chunkHdrStart;	// + chunkInfo.chunkHdrSize;
	uaecptr chunkSize = chunkInfo.size; // - chunkInfo.chunkHdrSize - chunkInfo.sizeAdj;

	// See if this range is already in the list.

	StackList::iterator iter = FindStackChunk (chunkStart);

	// If so, do nothing and return true.

	if (iter != gStackList.end ())
		return true;	// Already there

	// Create a range object for the chunk.

	StackRange	range (chunkStart, chunkStart + chunkSize);

	// Add the range to the list and return false.

	gStackList.insert (iter, range);

	return false;	// Chunk wasn't already there.
}


/***********************************************************************
 *
 * FUNCTION:	Software::ForgetStackChunk
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Software::ForgetStackChunk (uaecptr chunkStart)
{
	// See if this chunk is already in the list.

	StackList::iterator iter = FindStackChunk (chunkStart);

	// If so, remove it.

	if (iter != gStackList.end ())
	{
		gStackList.erase (iter);
	}
}


/***********************************************************************
 *
 * FUNCTION:	Software::FindStackChunk
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

StackList::iterator Software::FindStackChunk (uaecptr testAddress)
{
	StackList::iterator iter = gStackList.begin ();

	for (; iter != gStackList.end (); ++iter)
	{
		StackRange	range = *iter;
		if ((testAddress >= range.fBottom) && (testAddress < range.fTop))
		{
			break;
		}
	}

	return iter;
}


#pragma mark -

// ===========================================================================
//		 LowMem
// ===========================================================================

static uae_u16		gSysDispatchTableSize;

/***********************************************************************
 *
 * FUNCTION:	LowMem::Initialize
 *
 * DESCRIPTION: Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void LowMem::Initialize (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	LowMem::Reset
 *
 * DESCRIPTION: Standard reset function.  Sets the sub-system to a
 *				default state.	This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void LowMem::Reset (void)
{
	gSysDispatchTableSize = 0;
}


/***********************************************************************
 *
 * FUNCTION:	LowMem::Save
 *
 * DESCRIPTION: Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void LowMem::Save (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	LowMem::Load
 *
 * DESCRIPTION: Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void LowMem::Load (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	LowMem::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void LowMem::Dispose (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	LowMem::GetEvtMgrIdle
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uae_u8 LowMem::GetEvtMgrIdle (void)
{
	CEnableFullAccess	munge;	// Remove blocks on memory access.

	// First, get sysEvtMgrGlobalsP, which is a pointer in low-memory.

	uae_u32 sysEvtMgrGlobalsP	= LowMem_GetGlobal (sysEvtMgrGlobalsP);

	/*
	**	Next look at the "idle" boolean in the SysEvtMgrGlobalsPtr
	**	contained in sysEvtMgrGlobalsP:
	**
	**	struct SysEvtMgrGlobalsType
	**	{
	**		...
	**		Boolean 		idle;
	**		...
	**	};
	*/

	size_t	idleOffset = offsetof (SysEvtMgrGlobalsType, idle);

	// Under Palm OS 1.0, there was an extra 4-byte value at the
	// location currently holding "gremlinsFlag", which is before
	// the "idle" field.  Under that version of the OS, we therefore
	// have to add 4 to get the right offset.

	if (Patches::OSMajorVersion () == 1)
	{
		idleOffset += sizeof (Long);
	}

	uae_u8	idle = get_byte (sysEvtMgrGlobalsP + idleOffset);

	return idle;
}


/***********************************************************************
 *
 * FUNCTION:	LowMem::TrapExists
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool LowMem::TrapExists (uae_u16 iATrap)
{
	// If we haven't already, get the number of traps in this system.

	if (gSysDispatchTableSize == 0)
	{
		CEnableFullAccess	munge;	// Remove blocks on memory access.

		gSysDispatchTableSize = LowMem_GetGlobal (sysDispatchTableSize);
	}

	return (SysTrapIndex (iATrap) == SysTrapIndex (sysTrapHostControl)) ||
			SysTrapIndex (iATrap) < gSysDispatchTableSize;
}


/***********************************************************************
 *
 * FUNCTION:	LowMem::GetTrapAddress
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uaecptr LowMem::GetTrapAddress (uae_u16 iATrap)
{
	uaecptr address = UAE_NULL;

	CEnableFullAccess	munge;	// Remove blocks on memory access.

	// If the trap number is not in range, just return NULL to signal
	// that normal process of the TRAP $F should occur.

	if (TrapExists (iATrap))
	{
		// Get sysDispatchTableP, which is a pointer in low-memory.

		uae_u32 sysDispatchTableP	= LowMem_GetGlobal (sysDispatchTableP);

		// If it's NULL, return NULL.

		if (sysDispatchTableP != UAE_NULL)
		{
			// Next, get the right guy from its contents.

			address = get_long (sysDispatchTableP + SysTrapIndex (iATrap) * 4);
		}
	}

	return address;
}


#pragma mark -

// ===========================================================================
//		 Registers
// ===========================================================================

/***********************************************************************
 *
 * FUNCTION:	Registers::GetRegisters
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Registers::GetRegisters (regstruct& oRegisters)
{
	Registers::UpdateSRFromRegisters ();

	oRegisters = regs;
	oRegisters.pc = m68k_getpc ();
}


/***********************************************************************
 *
 * FUNCTION:	Registers::SetRegisters
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Registers::SetRegisters (regstruct& iRegisters)
{
	regs = iRegisters;
	Registers::UpdateRegistersFromSR ();
	m68k_setpc (iRegisters.pc);
}


/***********************************************************************
 *
 * FUNCTION:	Registers::UpdateSRFromRegisters
 *
 * DESCRIPTION: Updates the status register to reflect the state of the
 *				other registers.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Registers::UpdateSRFromRegisters (void)
{
	// (taken from MakeSR in newcpu.c)

	regs.sr = ((regs.t1 << 15) | (regs.t0 << 14)
			| (regs.s << 13) | (regs.m << 12) | (regs.intmask << 8)
			| (XFLG << 4) | (NFLG << 3) | (ZFLG << 2) | (VFLG << 1) 
			|  CFLG);
}


/***********************************************************************
 *
 * FUNCTION:	Registers::UpdateRegistersFromSR
 *
 * DESCRIPTION: Updates the registers to reflect the state of the
 *				status register.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Registers::UpdateRegistersFromSR (void)
{
	// (taken from MakeFromSR in newcpu.c)
	
//	int oldm = regs.m;
	int olds = regs.s;
	
	regs.t1 = (regs.sr >> 15) & 1;
	regs.t0 = (regs.sr >> 14) & 1;
	regs.s = (regs.sr >> 13) & 1;
	regs.m = (regs.sr >> 12) & 1;
	regs.intmask = (regs.sr >> 8) & 7;

	XFLG = (regs.sr >> 4) & 1;
	NFLG = (regs.sr >> 3) & 1;
	ZFLG = (regs.sr >> 2) & 1;
	VFLG = (regs.sr >> 1) & 1;
	CFLG = regs.sr & 1;

	if (olds != regs.s)
	{
		if (olds)
		{
			regs.isp = m68k_areg(regs, 7);
			m68k_areg(regs, 7) = regs.usp;
		}
		else
		{
			regs.usp = m68k_areg(regs, 7);
			m68k_areg(regs, 7) = regs.isp;
		}
	}
	
	regs.spcflags |= SPCFLAG_INT;

	if (regs.t1 || regs.t0)
	{
		regs.spcflags |= SPCFLAG_TRACE;
	}
	else
	{
		regs.spcflags &= ~(SPCFLAG_TRACE | SPCFLAG_DOTRACE);
	}
}


#pragma mark -

// ===========================================================================
//		 HWRegisters
// ===========================================================================

static uae_u32	gCurrentMilliseconds;


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::Initialize
 *
 * DESCRIPTION: Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::Initialize (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::Reset
 *
 * DESCRIPTION: Standard reset function.  Sets the sub-system to a
 *				default state.	This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::Reset (void)
{
	if (Emulator::EZMode ())
		MC68EZ328Bank::Reset ();
	else
		MC68328Bank::Reset ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::Save
 *
 * DESCRIPTION: Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::Save (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::Load
 *
 * DESCRIPTION: Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::Load (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::Dispose (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::Cycle
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::Cycle (Bool sleeping)
{
	if (Emulator::EZMode ())
		MC68EZ328Bank::Cycle (sleeping);
	else
		MC68328Bank::Cycle (sleeping);
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::CycleSlowly
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::CycleSlowly (Bool sleeping)
{
	if (Emulator::EZMode ())
		MC68EZ328Bank::CycleSlowly (sleeping);
	else
		MC68328Bank::CycleSlowly (sleeping);
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::GetInterruptLevel
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uae_s32 HWRegisters::GetInterruptLevel (void)
{
	if (Emulator::EZMode ())
		return MC68EZ328Bank::GetInterruptLevel ();

	return MC68328Bank::GetInterruptLevel ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::GetInterruptBase
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uae_s32 HWRegisters::GetInterruptBase (void)
{
	if (Emulator::EZMode ())
		return MC68EZ328Bank::GetInterruptBase ();

	return MC68328Bank::GetInterruptBase ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::ButtonEvent
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::ButtonEvent (Bool iButton_IsDown, uae_s32 iButton)
{
	if (Emulator::EZMode ())
		MC68EZ328Bank::ButtonEvent (iButton_IsDown, iButton);
	else
		MC68328Bank::ButtonEvent (iButton_IsDown, iButton);
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::HotSyncEvent
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::HotSyncEvent (Bool iButton_IsDown)
{
	if (Emulator::EZMode ())
		MC68EZ328Bank::HotSyncEvent (iButton_IsDown);
	else
		MC68328Bank::HotSyncEvent (iButton_IsDown);
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::GetLCDDepth
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

int HWRegisters::GetLCDDepth (void)
{
	// Are we running a Color LCD?
	if (Emulator::HasSED())
		return SED1375Bank::GetLCDDepth();

	if (Emulator::EZMode ())
		return MC68EZ328Bank::GetLCDDepth ();

	return MC68328Bank::GetLCDDepth ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::GetLCDRowBytes
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

int HWRegisters::GetLCDRowBytes (void)
{
	// Are we running a Color LCD?
	if (Emulator::HasSED())
		return SED1375Bank::GetLCDRowBytes();

	if (Emulator::EZMode ())
		return MC68EZ328Bank::GetLCDRowBytes ();

	return MC68328Bank::GetLCDRowBytes ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::GetLCDWidth
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

int HWRegisters::GetLCDWidth (void)
{
	// Are we running a Color LCD?
	if (Emulator::HasSED())
		return SED1375Bank::GetLCDWidth();

	if (Emulator::EZMode ())
		return MC68EZ328Bank::GetLCDWidth ();

	return MC68328Bank::GetLCDWidth ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::GetLCDHeight
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

int HWRegisters::GetLCDHeight (void)
{
	// Are we running a Color LCD?
	if (Emulator::HasSED())
		return SED1375Bank::GetLCDHeight();

	if (Emulator::EZMode ())
		return MC68EZ328Bank::GetLCDHeight ();

	return MC68328Bank::GetLCDHeight ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::GetLCDStartAddr
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uaecptr HWRegisters::GetLCDStartAddr (void)
{
	// Are we running a Color LCD?
	if (Emulator::HasSED())
		return SED1375Bank::GetLCDStartAddr();

	if (Emulator::EZMode ())
		return MC68EZ328Bank::GetLCDStartAddr ();

	return MC68328Bank::GetLCDStartAddr ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::GetLCDPalette
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::GetLCDPalette (RGBType* thePalette)
{
	// Are we running a Color LCD?
	if (Emulator::HasSED())
	{
		SED1375Bank::GetLCDPalette (thePalette);
		return;
	}

#if 1
	Preference<RGBType> pref1 (kPrefKeyBackgroundColor);
	Preference<RGBType> pref2 (kPrefKeyHighlightColor);

	RGBType foreground (0, 0, 0);
	RGBType background;

	if (BacklightIsOn ())
	{
		if (pref2.Loaded ())
			background = *pref2;
		else
			background = ::SkinGetHighlightColor ();
	}
	else
	{
		if (pref1.Loaded ())
			background = *pref1;
		else
			background = ::SkinGetBackgroundColor ();
	}

	long	br = ((long) background.fRed);
	long	bg = ((long) background.fGreen);
	long	bb = ((long) background.fBlue);

	long	dr = ((long) foreground.fRed) - ((long) background.fRed);
	long	dg = ((long) foreground.fGreen) - ((long) background.fGreen);
	long	db = ((long) foreground.fBlue) - ((long) background.fBlue);

	int numColors = 1 << GetLCDDepth ();

	for (int color = 0; color < numColors; ++color)
	{
		thePalette[color].fRed		= (Byte) (br + dr * color / (numColors - 1));
		thePalette[color].fGreen	= (Byte) (bg + dg * color / (numColors - 1));
		thePalette[color].fBlue 	= (Byte) (bb + db * color / (numColors - 1));
	}
#else
	if (Emulator::EZMode ())
	{
		MC68EZ328Bank::GetLCDPalette (thePalette);
		return;
	}

	MC68328Bank::GetLCDPalette (thePalette);
#endif
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::GetLCDOffset
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

int HWRegisters::GetLCDOffset (void)
{
	// Are we running a Color LCD?
	if (Emulator::HasSED())
		return SED1375Bank::GetLCDOffset();

	if (Emulator::EZMode ())
		return MC68EZ328Bank::GetLCDOffset ();

	return MC68328Bank::GetLCDOffset ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::LCDIsOn
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool HWRegisters::LCDIsOn (void)
{
	// Are we running a Color LCD?
	if (Emulator::HasSED())
		return SED1375Bank::LCDIsOn();

	if (Emulator::EZMode ())
		return MC68EZ328Bank::LCDIsOn ();

	return MC68328Bank::LCDIsOn ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::BacklightIsOn
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool HWRegisters::BacklightIsOn (void)
{
	// Are we running a Color LCD?
	if (Emulator::HasSED())
		return SED1375Bank::BacklightIsOn();

	if (Emulator::EZMode ())
		return MC68EZ328Bank::BacklightIsOn ();

	return MC68328Bank::BacklightIsOn ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::TurnSoundOff
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::TurnSoundOff (void)
{
	if (Emulator::EZMode ())
		MC68EZ328Bank::TurnSoundOff ();
	else
		MC68328Bank::TurnSoundOff ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::UpdateUARTRegisters
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::UpdateUARTRegisters (Bool refreshRxData)
{
	if (Emulator::EZMode ())
		MC68EZ328Bank::UpdateUARTState (refreshRxData);
	else
		MC68328Bank::UpdateUARTState (refreshRxData);
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::GetDynamicHeapSize
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uae_u32 HWRegisters::GetDynamicHeapSize	(void)
{
	if (Emulator::EZMode ())
		return MC68EZ328Bank::GetDynamicHeapSize ();

	return MC68328Bank::GetDynamicHeapSize ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::GetROMSize
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uae_u32 HWRegisters::GetROMSize	(void)
{
	if (Emulator::EZMode ())
		return MC68EZ328Bank::GetROMSize ();

	return MC68328Bank::GetROMSize ();
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::SetCurrentMilliseconds
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void HWRegisters::SetCurrentMilliseconds (uae_u32 t)
{
	gCurrentMilliseconds = t;
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::GetCurrentMilliseconds
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uae_u32 HWRegisters::GetCurrentMilliseconds (void)
{
	return gCurrentMilliseconds;
}


/***********************************************************************
 *
 * FUNCTION:	HWRegisters::CanStop
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool HWRegisters::CanStop (void)
{
	if (Emulator::EZMode ())
		return MC68EZ328Bank::CanStop ();

	return MC68328Bank::CanStop ();
}


#pragma mark -

// ---------------------------------------------------------------------------
// Glue functions to bridge from UAE-generated code to the implemenations
// we define.
// ---------------------------------------------------------------------------

void customreset (void)
{
	Emulator::Reset ();
}


void Exception (int nr, uaecptr oldpc)
{
	uaecptr curpc = m68k_getpc ();

	// Some opcode handlers (like op_4e40 for the TRAP #xx functions)
	// call Exception with oldpc == 0. We really need the PC so that
	// we can look up dispatch numbers, so provide that here.

	if (oldpc == 0)
	{
		if (nr >= kException_Trap0 && nr <= kException_Trap15)
		{
			// pc was bumped by 2 already in op_4e40 handler.

			oldpc = curpc - 2;
		}
		else
		{
			assert (false);
		}
	}

	Software::ProcessException (nr, oldpc, curpc);
}


unsigned long op_illg (uae_u32 iOpcode)
{
	Software::ProcessIllegalInstruction (iOpcode);
	return 0;
}


void Software_ProcessLINK (uae_s32 linkSize)
{
	Software::ProcessLINK (linkSize);
}


int Software_ProcessRTS (uaecptr dest)
{
	Software::CheckNewPC (dest);
	return Software::ProcessRTS ();
}


int Software_ProcessRTE (uaecptr dest)
{
	Software::CheckNewPC (dest);
	return Software::ProcessRTE (dest);
}


int Software_ProcessJSR (uaecptr oldpc, uaecptr dest)
{
	Software::CheckNewPC (dest);
	return Software::ProcessJSR (oldpc, dest);
}


int Software_ProcessJSR_Ind (uaecptr oldpc)
{
	return Software::ProcessJSR_Ind (oldpc);
}


void Software_CheckNewPC (uaecptr dest)
{
	Software::CheckNewPC (dest);
}


void Software_CheckStackPointer (int opcodeType, uaecptr oldA7)
{
	UNUSED_PARAM(opcodeType)
	UNUSED_PARAM(oldA7)

	// Disabled for now.  There's a problem where interrupt code can attempt
	// to switch to a previously unknown stack, causing us to walk the heap
	// to learn stuff about the chunk.	If the heap is in flux because of a
	// pending memory manager call, then we're screwed, and end up reporting
	// that the heap is corrupted.
	//
	// The question remains as to how the interrupt code managed to get the
	// block of memory, allowed the non-interrupt code to enter the memory
	// manager, and then attempted to switch the stack to that block...

//	Software::CheckStackPointer (opcodeType, oldA7);
}


void MakeSR (void)
{
	Registers::UpdateSRFromRegisters ();
}


void MakeFromSR (void)
{
	Registers::UpdateRegistersFromSR ();
}


void* xmalloc (size_t size)
{
	return Platform::AllocateMemory (size);
}

// Include this because VC++ can't figure out that it can be dead-code stripped!
uae_u32 get_disp_ea_020 (uae_u32 base, uae_u32 dp)
{
    int reg = (dp >> 12) & 15;
    uae_s32 regd = regs.regs[reg];
    if ((dp & 0x800) == 0)
	regd = (uae_s32)(uae_s16)regd;
    regd <<= (dp >> 9) & 3;
    if (dp & 0x100) {
	uae_s32 outer = 0;
	if (dp & 0x80) base = 0;
	if (dp & 0x40) regd = 0;

	if ((dp & 0x30) == 0x20) base += (uae_s32)(uae_s16)next_iword();
	if ((dp & 0x30) == 0x30) base += next_ilong();

	if ((dp & 0x3) == 0x2) outer = (uae_s32)(uae_s16)next_iword();
	if ((dp & 0x3) == 0x3) outer = next_ilong();

	if ((dp & 0x4) == 0) base += regd;
	if (dp & 0x3) base = get_long (base);
	if (dp & 0x4) base += regd;

	return base + outer;
    } else {
	return base + (uae_s32)((uae_s8)dp) + regd;
    }
}

uae_u32 get_disp_ea_000 (uae_u32 base, uae_u32 dp)
{
    int reg = (dp >> 12) & 15;
    uae_s32 regd = regs.regs[reg];
#if 1
    if ((dp & 0x800) == 0)
	regd = (uae_s32)(uae_s16)regd;
    return base + (uae_s8)dp + regd;
#else
    /* Branch-free code... benchmark this again now that
     * things are no longer inline.  */
    uae_s32 regd16;
    uae_u32 mask;
    mask = ((dp & 0x800) >> 11) - 1;
    regd16 = (uae_s32)(uae_s16)regd;
    regd16 &= mask;
    mask = ~mask;
    base += (uae_s8)dp;
    regd &= mask;
    regd |= regd16;
    return base + regd;
#endif
}

// Dummy functions just so that we can link under VC++.

void m68k_move2c (int, uae_u32 *)
{
}


void m68k_movec2 (int, uae_u32 *)
{
}


void m68k_divl (uae_u32, uae_u32, uae_u16, uaecptr)
{
}


void m68k_mull (uae_u32, uae_u32, uae_u16)
{
}


void fpp_opp (uae_u32, uae_u16)
{
}


void fdbcc_opp (uae_u32, uae_u16)
{
}


void fscc_opp (uae_u32, uae_u16)
{
}


void ftrapcc_opp (uae_u32,uaecptr)
{
}


void fbcc_opp (uae_u32, uaecptr, uae_u32)
{
}


void fsave_opp (uae_u32)
{
}


void frestore_opp (uae_u32)
{
}

