/*
 * File: AlphaNumberFormat.java
 *
 * $Id: AlphaNumberFormat.java,v 1.2 1999/01/07 15:25:47 ttaylor Exp $
 */

package org.mitre.tjt.text;

import java.text.FieldPosition;
import java.text.NumberFormat;
import java.text.ParsePosition;

/**
 * This class implements a NumberFormat that generates an alphabetic sequence
 * that represents the integer value provided.
 *
 * @author Tim Taylor
 * @version 1.0
 */
public class AlphaNumberFormat extends NumberFormat {
  /** If true, the returned string will use uppercase letters. */
  private boolean upperCaseFormat;

  /** Number of symbols we have to express numbers. */
  private static final int RADIX = 26;

  /**
   * Creates a new AlphaNumberFormat object.  The formats generated by this
   * object will use uppercase letters if <code>upperCase</code> is true,
   * lower case otherwise.
   * 
   * @param upperCase True if the format methods should use upper case
   *  letters, (ie: ABC).  If false, lowercase letters will be used.
   */
    public AlphaNumberFormat(boolean upperCase) {
        upperCaseFormat = upperCase;
    }

  /**
   * Formats the provided double as an alphabetic sequence by converting to a
   * long. Loss of precision may result, however, these alphabetic sequence
   * doesn't have fractions.
   * 
   * @param number The number to format as an alphabetic sequence.
   * @param result The StringBuffer to write the formated number to.
   * @param fieldPosition The field position in the result.
   * @return The StringBuffer provided in <code>result</code> with the
   *  formated value of <code>number</code> appended.
   */
  public StringBuffer format(double number, StringBuffer result,
                             FieldPosition fieldPosition) {
    return format((long)number, result, fieldPosition);
  }

  /**
   * Formats the provided long as an alphabetic sequence.
   * 
   * @param number The number to format as an alphabetic sequence.
   * @param result The StringBuffer to write the formated number to.
   * @param fieldPosition The field position in the result.
   * @return The StringBuffer provided in <code>result</code> with the
   *  formated value of <code>number</code> appended.
   */
  public StringBuffer format(long number, StringBuffer result,
                             FieldPosition fieldPosition) {
    char buf[] = new char[33];
    int pos = 32;

    while(number > RADIX) {
      buf[pos--] = (char)('a' + ((number - 1) % RADIX));
      number = (number - 1) / RADIX;
    }
    buf[pos] = (char)('a' + number - 1);

    String formatted = new String(buf, pos, buf.length - pos);
    if(upperCaseFormat) result.append(formatted.toUpperCase());
    else result.append(formatted);
    return result;
  }

  /**
   * Parses a string containing a number expressed as an alphabetic sequence
   * and converts it to a <code>Number</code>.  The parsePosition will be set
   * to the character following the last character parsed.
   * 
   * @param text The string to parse the number from.
   * @param parsePosition On input, contains the position in text to start
   *  parse at.  On return, contains the position that caused parsing to stop.
   * @return The parsed number.  This will be an instance of java.lang.Long.
   */
  public Number parse(String text, ParsePosition parsePosition) {
    text = text.toLowerCase();
    int len = text.length();
    long result = 0;                    // Holds our parse value
    int pos;

    for(pos = parsePosition.getIndex(); pos < len; pos++) {
      char c = text.charAt(pos);
      if((c < 'a') || (c > 'z')) break;  // Invalid character
      result *= 26; 
      result += c - 'a' + 1;
    }
    parsePosition.setIndex(pos);
    return new Long(result);
  }
  
  /**
   * This method is for testing the class.
   * @param args The first argument should be either true or false.  It
   *  determines if the returned string is uppercase (true) or lowercase
   *  (false).  The second argument should be a number to format.
   */
  public static void main(String[] args) throws Exception {
    if(args.length < 2) {
      System.out.println("Usage: java AlphaNumberSequence true|false number");
      System.exit(0);
    }

    boolean caps = Boolean.valueOf(args[0]).booleanValue();
    long count = Long.parseLong(args[1]);
    AlphaNumberFormat fmt = new AlphaNumberFormat(caps);

    // Convert number to string
    System.out.print("Formated number: ");
    String str = fmt.format(count);
    System.out.println(str);

    // Convert string from last step to a number
    System.out.print("Parse number: ");
    long result = fmt.parse(str).longValue();
    System.out.println(result);

    // Compare the two numbers.  Should be the same
    System.out.println((count == result) ? "Success" : "Failure");
  }
	//{{DECLARE_CONTROLS
	//}}
}
