/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;

import org.w3c.dom.*;
import org.mitre.tjt.xsl.XslNumberFormat;
import com.kvisco.util.QuickStack;
import com.kvisco.xsl.functions.PositionFunctionCall;
import com.kvisco.xsl.functions.ErrorFunctionCall;

import java.util.Hashtable;
import java.util.StringTokenizer;

/**
 * Represents the xsl:number element. 
 * Handles numbering in the source tree<BR>
 * Section 2.7.10 of the W3C XSL Working Draft 1.0 (19981216)
 * <BR />
 * Section 9.7 of the W3C XSLT Working Draft 1.0 (1990421)
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class XSLNumber extends XSLObject {
    
      //----------------------/
     //- Instance Variables -/
    //----------------------/
    
    /**
     * The expr for this XSLNumber
    **/
    private Expr expr = null;
    
    /**
     * The count pattern of this XSLNumber
    **/
    private MatchExpr count = null;
    
    
    /**
     * The count pattern of this XSLNumber
    **/
    private MatchExpr  from = null;
    
    /**
     * int value representing the numbering level of this XSLNumber
    **/
    private String level = Names.SINGLE_VALUE;

    
      //---------------/
     //- Contructors -/
    //---------------/
    
    /**
     * Creates a new XSLNumber Object
    **/
    public XSLNumber(XSLStylesheet parentStylesheet) {
        super(parentStylesheet, XSLObject.NUMBER);
    } //-- XSLNumber
    
      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Performs the numbering of the given Element
     * @param element the Element to get the number of
     * @return an array of integers which represent the
     * multi-level number of the given element. Single level
     * numbering will return an array of 1.
    **/
    public int[] doNumbering(Element element, ProcessorState ps) 
        throws InvalidExprException
    {
        
        
        int[] counts = null;
        
        //-- check for expr
        if (expr != null) {
            counts = new int[1];
            ExprResult result = expr.evaluate(element, ps);
            double dbl = expr.evaluate(element, ps).numberValue();
            counts[0] = (int)dbl;
            return counts;
        }
        
        //-- check for null element
        if (element == null) return new int[0];
        
        MatchExpr local_count = this.count;
        
        if (local_count == null)
            local_count 
                = ExpressionParser.createMatchExpr(element.getNodeName());
        
        NodeSet nodes;
        
        if (Names.MULTI_VALUE.equals(level)) {
            nodes = getAncestorsOrSelf(local_count, element, ps, false);
            counts = new int[nodes.size()];
            int cnum = 0;
            for (int i = nodes.size()-1; i >= 0; i--) {
                counts[cnum++] =
                    countPreceedingSiblings(local_count, nodes.get(i), ps);
            }
        }
        else if (Names.ANY_VALUE.equals(level)) {
            nodes = getAnyPreviousNodes(local_count, element, ps);
            counts = new int[1];
            counts[0] = nodes.size();
        }
        else {
            nodes = getAncestorsOrSelf(local_count, element, ps, true);
            counts = new int[nodes.size()];
            if (nodes.size() > 0) {
                counts[0] =
                    countPreceedingSiblings(local_count, nodes.get(0), ps);
            }
        }
        
        return counts;
    } //-- doNumbering
    
    /**
     * Performs the numbering of the given Element and 
     * returns the number using the format of this XSLNumber.
     * @param element the Element to get the number of
     * @return the formatted number as a String
    **/
    public String getFormattedNumber(Element element, ProcessorState ps) {
        
        try {
            int[] counts = doNumbering(element, ps);        
            // Uses Tim Taylor's XslNumberFormat to format
            // the result
            String format = getAttribute(Names.FORMAT_ATTR);
            if (format == null) format = "1";
            return XslNumberFormat.format(counts, format);
        }
        catch(InvalidExprException iee) {};
        return "";
        
    } //-- getFormattedNumber
    
    /**
     * @see XSLObject
    **/
    public void setAttribute(String name, String value) 
        throws XSLException 
    {
        
        ErrorFunctionCall efc = null;
        //-- set expr
        
        if (Names.EXPR_ATTR.equals(name)) {
            if (value == null) expr = null;
            else {
                try {
                    expr = ExpressionParser.createExpr(value);
                }
                catch (InvalidExprException iee) {
                    efc = new ErrorFunctionCall();
                    efc.setError("invalid 'expr' attribute of xsl:number"+
                        iee.getMessage());
                    expr = efc;
                }
            }
        }
        //-- set count
        else if (Names.COUNT_ATTR.equals(name)) {
            if (value == null) count = null; //-- use default
            else {
                try {
                    count = ExpressionParser.createMatchExpr(value);
                }
                catch(InvalidExprException iee) {
                    efc = new ErrorFunctionCall();
                    efc.setError("invalid 'count' attribute of xsl:number"+
                        iee.getMessage());
                    expr = efc;
                }
            }
        }
        //-- set from
        else if (Names.FROM_ATTR.equals(name)) {
            if (value == null) from = null; //-- use default
            else {
                try {
                    from = ExpressionParser.createMatchExpr(value);
                }
                catch(InvalidExprException iee) {
                    efc = new ErrorFunctionCall();
                    efc.setError("invalid 'from' attribute of xsl:number"+
                        iee.getMessage());
                    expr = efc;
                }
            }
        }
        //-- set level
        else if (Names.LEVEL_ATTR.equals(name)) {
            level = value;
        } //--
        
        super.setAttribute(name,value);
    } //-- setAttribute
    
    /**
     * Sets the count expression of this XSLNumber
     * @param count the String value to use as the count expr
     * expression of this XSLNumber
    **/
    public void setCountAttr(String count) {
        try {
            setAttribute(Names.COUNT_ATTR, count);
        }
        catch(XSLException xslException) {};
    } //-- setCountAttr
    
    /**
     * Sets the format pattern of this XSLNumber
     * @param format the Number Format to use
    **/
    public void setFormatAttr(String format) {
        try {
            setAttribute(Names.FORMAT_ATTR, format);
        }
        catch(XSLException xslException) {};
    } //-- setFormatAttr
    
    public void setFromAttr(String from) {
        try {
            setAttribute(Names.FROM_ATTR, from);
        }
        catch(XSLException xslException) {};
    } //-- setFrom
    
    /**
     * Sets the level of numbering for this XSLNumber
     * @param level the desired level.
     * <PRE>
     *  Levels are "single", "multi", or "any"
     * </PRE>
    **/
    public void setLevel(String level) {
        try {
            setAttribute(Names.LEVEL_ATTR, level);
        }
        catch(XSLException xslException) {};
    } //-- setLevel
    
      //---------------------/
     //- Protected Methods -/
    //---------------------/
    
    private static int countPreceedingSiblings
        (MatchExpr matchExpr, Node context, ProcessorState ps)
        throws InvalidExprException
    {
        int count = 1;
        
        if (context == null) return 0;
        
        Node sibling = context;
        while ( (sibling = sibling.getPreviousSibling()) != null ) {
            if (sibling.getNodeType() != Node.ELEMENT_NODE) continue;
            if (matchExpr.matches(sibling, sibling, ps))
                ++count;
        }
        return count;
    } //-- countPreceedingSiblings
    
    private NodeSet getAncestorsOrSelf
        ( 
          MatchExpr matchExpr, 
          Element context, 
          ProcessorState ps,
          boolean findNearest
         ) 
        throws InvalidExprException
    {
        
        NodeSet nodeSet = new NodeSet();
        Node parent = context;
        while ((parent != null)  && 
               (parent.getNodeType() == Node.ELEMENT_NODE)) 
        {
            if ((from != null) && from.matches(parent, parent, ps)) break;
                
            if (matchExpr.matches(parent, parent, ps)) {
                nodeSet.add(parent);
                if (findNearest) break;
            }
                
            parent = parent.getParentNode();
        }
        return nodeSet;
        
    } //-- fromAncestorsOrSelf
    
    /**
     * Retrieves all nodes that come before the given element
     * at any level in the document that match the count pattern of 
     * this XSLNumber starting from the closest element that matches 
     * the from pattern
     * @param element the element to find the nearest ancestor of
     * @return a List of all matching nodes
    **/
    private NodeSet getAnyPreviousNodes
        ( MatchExpr matchExpr, 
          Element context, 
          ProcessorState ps
         ) 
        throws InvalidExprException
    {
        
        NodeSet nodes = new NodeSet();
        
        Element element = context;
        
        while (element != null) {
            // Check from MatchExpr
            if ((from != null) && from.matches(element,element,ps)) 
                return nodes;
                
            // Check count MatchExpr
            if (matchExpr.matches(element, element, ps)) 
                nodes.add(element);
            
            Node sibling = element;
            while ( (sibling = sibling.getPreviousSibling()) != null) {
                if (sibling.getNodeType() == Node.ELEMENT_NODE) break;
            }
            if (sibling == null) {
                Node parent = element.getParentNode();
                if (parent.getNodeType() != Node.ELEMENT_NODE) break;
                element = (Element)parent;
            }
            else element = (Element)sibling;
        }
        return nodes;
    } //-- getAnyPreviousNodes
        
} //-- XSLNumber