/*
 * (C) Copyright Keith Visco 1998, 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;

import com.kvisco.xml.parser.DOMPackage;
import com.kvisco.util.List;
import com.kvisco.util.QuickStack;
import com.kvisco.util.Iterator;
import com.kvisco.xml.XMLUtil;

import java.util.Hashtable;
import java.util.Enumeration;

import org.w3c.dom.*;

/**
 * The current RuleProcessor environment
 * @author Keith Visco (kvisco@ziplink.net)
**/
public class ProcessorState {

    private String DEFAULT_SCRIPT_HANDLER 
        = "com.kvisco.scripting.ECMAScriptHandler";
    private String DEFAULT_LANGUAGE = ECMASCRIPT;
    
    public static final String ECMASCRIPT = "ECMASCRIPT";
    public static final String JPYTHON    = "JPYTHON";
    
    /**
     * the name of the result tree document element,
     * if one is missing
    **/
    private static final String RESULT_NAME = "xslp:result";
    
    private static final String NO_DOC_ELEMENT = 
        "xslp:result has been added, because nodes were being "+
        "added to the result tree, but no document element " +
        "was present. XSLT result tree's must be well-formed. ";
        
    private static final String MULITPLE_DOC_ELEMENTS = 
        "xslp:result has been added, because an element was being "+
        "added to the result tree at the document level, but a " +
        "document element already existed. " +
        "XSLT result tree's must be well-formed. ";
        
    
      //----------------------/
     //- Instance Variables -/
    //----------------------/

    /**
     * A hastable for referencing the parents of
     * nodes, that do not have parents
    **/
    private Hashtable  parents            = null;
    /**
     * A pointer to the next node to check, 
     * when indexing attributes
    **/
    private Hashtable  indexingInfo            = null;
    
    /**
     * Stack for called Templates
    **/
    private QuickStack calledTemplates   = null;
    
    /**
     * defined constants
    **/
    private Hashtable constants          = null;
    
    /**
     * keeps track of the document order for a node
    **/
    private Hashtable documentOrders     = null;
    
    
    /**
     * The DOMPackage for reading XML Documents and creating
     * XML DOM nodes
    **/
    private DOMPackage domPackage        = null;
    
    /**
     * ID Reference table for Source tree
    **/
    private Hashtable idRefs             = null;
    
    /**
     * Default Script Handler
    **/
    private ScriptHandler dScriptHandler = null;
    
    /**
     * ProcessorCallback for calling back into the RuleProcessor
    **/
    private ProcessorCallback processorCallback = null;
    
    /**
     * Registered ScriptHandlers
    **/
    private Hashtable scriptHandlers    = null;

    private Hashtable avtCache          = null;
    
    private Document sourceDoc          = null;
    private Document resultDoc          = null;
    
    private XSLStylesheet stylesheet    = null;
    
    private QuickStack variableStack    = null;
    private VariableSet globalVars      = null;
    private List        cyclicVarCheck  = null;
    private QuickStack parameterStack   = null;
    
    private QuickStack modes            = null;
    private QuickStack xmlSpaceModes    = null;

    /**
     * Stack used for processing
     * This stack is intended to only contain the
     * ResultDocument and descendant Element children.
    **/
    private QuickStack nodeStack        = null;
    
    private QuickStack nodeSetStack     = null;
    
      //----------------/
     //- Constructors -/
    //----------------/
    
    /**
     * Creates a new ProcessorState
    **/
    protected ProcessorState 
        (RuleProcessor ruleProcessor, Document source,
         XSLStylesheet xslStylesheet, DOMPackage domPackage)
    {
        super();
        this.processorCallback = new ProcessorCallback(ruleProcessor, this);
        this.sourceDoc = source;
        this.stylesheet = xslStylesheet;
        this.domPackage = domPackage;
        
        parents           = new Hashtable();
        avtCache          = new Hashtable();
        calledTemplates   = new QuickStack();
        documentOrders    = new Hashtable();
        
        indexingInfo      = new Hashtable();
        variableStack     = new QuickStack();
        parameterStack    = new QuickStack();
        nodeSetStack      = new QuickStack();
        nodeStack         = new QuickStack();
        scriptHandlers    = new Hashtable();
        xmlSpaceModes     = new QuickStack();
        cyclicVarCheck    = new List();
        
        resultDoc        = domPackage.createDocument();
        
        //-- add global variables to variable scoping
        variableStack.push((globalVars = new VariableSet()));  
        
        //-- add global Parameter set
        parameterStack.push(new VariableSet());
        
        // add Result Document to the nodeStack
        nodeStack.push(resultDoc);
        
        //-- add default xml:space mode
        xmlSpaceModes.push(Names.DEFAULT_VALUE);
        
        // handle DOCTYPE PI
        
        String docType = stylesheet.getResultDocType();
        
        if (docType != null) {
            domPackage.setDocumentType(resultDoc, docType);
        }

    } //-- ProcessorState

      //------------------/
     //- Public Methods -/
    //------------------/
    
    
    /**
     * Creates a unique identifier for the given node
     * @return the String that is a unique identifier for the
     * given node
    **/
    public String generateId(Node node) {
        StringBuffer genId = new StringBuffer("id");
        genId.append(System.identityHashCode(node.getOwnerDocument()));
        
        if (node.getNodeType() == Node.DOCUMENT_NODE)
            return genId.toString();
            
        int[] order = getDocumentOrder(node);
        //-- skip first node, since it's the document node
        for (int i = 1; i < order.length; i++) {
            genId.append('_');
            genId.append(order[i]);
        }
        return genId.toString();
    } //-- generateId
    
    /**
     * @returns the DOMPackage currently being used by this ProcessorState
    **/
    public DOMPackage getDOMPackage() {
        return domPackage;
    } //-- getDOMPackage
    
    public String getStylesheetHref() {
        return stylesheet.getHref();
    } //-- getStylesheetHref()
    
    /**
     * Returns the current NodeSet stack being used by this ProcessorState
     * @return the current NodeSet stack being used by this ProcessorState
    **/
    public QuickStack getNodeSetStack() {
        return this.nodeSetStack;
    } //-- getNodeSetStack

    /**
     * Returns the ID references for the xml source document
     * @return the ID references for the xml source document
    **/
    public Hashtable getIDReferences() {
        return this.idRefs;
    } //-- getIDReferences

    
    /**
     * Returns the Property value associated with the given name.
     * All property names without a namespace are defaulted to the
     * System evironment scope.
     * @return the Property value associated with the given name.
    **/
    public String getProperty(String name) {
        if (name == null) return "";
        
        String ns = XMLUtil.getNameSpace(name);
        String lp = XMLUtil.getLocalPart(name);
        
        String value = null;
        
        
        //-- Environment Property
        if (ns.length() == 0) {
            value = System.getProperty(lp, "");
        }
        //-- XSL property
        else if (ns.equals(stylesheet.getXSLNSPrefix())) {
            value = processorCallback.getProperty(lp);
        }
        //-- user defined?
        else {
            value = "";
        }
        return value;
    } //-- getProperty
    
    /**
     * Returns the stack of XML space modes
    **/
    public QuickStack getXMLSpaceModes() {
        return xmlSpaceModes;
    } //-- getXMLSpaceModes
    
      //---------------------/
     //- Protected Methods -/
    //---------------------/

	/**
	 * adds the given node to the result tree
	 * @param node the Node to add to the result tree
	**/
	protected void addToResultTree(Node node) {
	    if (node == null) return;
	    
	    Node    current = (Node)nodeStack.peek();
	   
	    
	    switch(node.getNodeType()) {
	        case Node.ATTRIBUTE_NODE:
	            if (current.getNodeType() != Node.ELEMENT_NODE) break;
	            Element element = (Element)current;
	            Attr attr = (Attr)node;
	            if (element.getAttributeNode(attr.getName()) == null) {
	                element.setAttribute(attr.getName(),attr.getValue());
	            }
	            break;
	        case Node.ELEMENT_NODE:
	            if (current == resultDoc) {
	                Element docElement = resultDoc.getDocumentElement();
	                if (docElement != null) {
	                    resultDoc.removeChild(docElement);
	                    //-- create wrapper element
	                    Element wrapper = resultDoc.createElement(RESULT_NAME);
	                    resultDoc.appendChild(wrapper);
	                    nodeStack.push(wrapper);
	                    //-- add old document element to wrapper
	                    wrapper.appendChild(docElement);
	                    //-- set current element to be wrapper
	                    current = wrapper;
	                    //-- add comment to wrapper
	                    Comment comment = 
	                        resultDoc.createComment(MULITPLE_DOC_ELEMENTS);
	                    current.appendChild(comment);
	                }
	            }
	            current.appendChild(node);
	            break;
	        case Node.COMMENT_NODE:
	        case Node.PROCESSING_INSTRUCTION_NODE:
	            current.appendChild(node);
	            break;
	        case Node.TEXT_NODE:
	            if (current == resultDoc) {
	                //-- check for whitespace
	                if (XMLUtil.isWhitespace(((Text)node).getData())) {
	                    current.appendChild(node);
	                    break;
	                }
	                //-- create wrapper element
	                Element wrapper = resultDoc.createElement(RESULT_NAME);
	                resultDoc.appendChild(wrapper);
	                nodeStack.push(wrapper);
	                current = wrapper;
	                //-- add comment to wrapper
	                Comment comment = 
	                    resultDoc.createComment(NO_DOC_ELEMENT);
	                current.appendChild(comment);
	            }
	            current.appendChild(node);
	            break;
	        //-- only add if not adding to document Node
	        default:
	            if (current == resultDoc) {
	                //-- create wrapper element
	                Element wrapper = resultDoc.createElement(RESULT_NAME);
	                resultDoc.appendChild(wrapper);
	                nodeStack.push(wrapper);
	                current = wrapper;
	                //-- add comment to wrapper
	                Comment comment = 
	                    resultDoc.createComment(NO_DOC_ELEMENT);
	                current.appendChild(comment);
	                
	                
	            }
	            current.appendChild(node);
	            break;
	    }
	} //-- addToResultTree
	
    /**
     * Adds the given ScriptHandler to this ProcessorState
     * @param scriptHandler the ScriptHandler to add
     * @param language the script language to register the
     * given ScriptHandler for.
    **/
    protected void addScriptHandler(ScriptHandler scriptHandler, String language) {
        if (language != null) {
            if (scriptHandler != null) {
                scriptHandlers.put(language.toUpperCase(), scriptHandler);
            }
        }
    } //-- addScriptHandler
    
    /**
	 * Returns the value of the given String as an AttributeValueTemplate
	 * @returns the value of the given String as an AttributeValueTemplate
	 * @exception InvalidExprException when the String argument is not a valid 
	 * AttrubueValueTemplate
	**/
    public AttributeValueTemplate getAttributeValueTemplate(String avtString) 
        throws InvalidExprException
    {
        
        AttributeValueTemplate avt = null;
        
        if (avtString != null) {
            // look in cache first
            avt = (AttributeValueTemplate) avtCache.get(avtString);
            if (avt == null) {
                avt = new AttributeValueTemplate(avtString);
                // add to cache for performace
                avtCache.put(avtString, avt);
            }
        }
       return avt;
    } //-- getAttributeValueTemplate
    
    /**
     * Determines the document order of a given node.
     * Document Order is defined by
     * the document order of the parent of the given node and 
     * the childNumber of the given Node.
     * The document order for a Document node is 0.
    **/
    protected int[] getDocumentOrder(Node node) {
        
        int[] order = null;
        
        if (node == null) {
            order = new int[1];
            order[0] = -1;
            return order;
        }
        
        //-- check cache
        //-- * due to bugs in XML4J 1.1.x (2.x works fine) 
        //-- * we need to use the System.identityHash to
        //-- * create a unique key. The problem is Attr nodes
        //-- * with the same name, generate the same hash code.
        Object key = createKey(node);
        
        order = (int[]) documentOrders.get(key);
        if (order != null) return order;
        
        
        Node parent = null;
        //-- calculate document order
        if (node.getNodeType() == Node.ATTRIBUTE_NODE) {
            // Use parent's document order for attributes
            parent = findParent((Attr)node);
            if (parent == null) {
                order = new int[3];
                order[0] = 0;
                order[1] = 0;
                order[2] = childNumber(node);
            }
            else { 
                int[] porder = getDocumentOrder(parent);
                order = new int[porder.length+2];
                System.arraycopy(porder, 0, order,0, porder.length);
                order[order.length-2] = 0;
                order[order.length-1] = childNumber(node);
            }
        }
        else if (node.getNodeType() == Node.DOCUMENT_NODE) {
            order = new int[1];
            order[0] = 0;
        }
        else {
            
            //-- get parent's document order
            parent = getParentNode(node);
            int[] porder = getDocumentOrder(getParentNode(node));
            order = new int[porder.length+1];
            System.arraycopy(porder, 0, order,0, porder.length);
            order[order.length-1] = childNumber(node);
        }
        
        //-- add to cache
        documentOrders.put(key,order);
        return order;
    } //-- getDocumentOrder
    
    
    /**
     * Returns the top of the Parameter Stack
     * @return the top most VariableSet of the Parameter Stack
    **/
    protected QuickStack getParameterStack() {
        return parameterStack;
    } //-- getParameters
    
    protected Node getParentNode(Node node) {
        if (node == null) return null;
        switch (node.getNodeType()) {
            case Node.ATTRIBUTE_NODE:
                return findParent((Attr)node);
            default:
                return node.getParentNode();
        } 
    } //-- getParentNode
    
    
    /**
     * Retrieves the defined stylesheet Variable with the given name
     * @param name the name of the Variable to get
     * @return the Variablet with the specified name, or null if not found
    **/
    protected ExprResult getVariable(String name) {
        
        VariableSet varSet = getVariableSet(name);
        
        ExprResult result = null;
        if (varSet == null) {
            //-- check for unbound global definition
            Variable var = stylesheet.getVariableDecl(name);
            if (var != null) {
                if (!cyclicVarCheck.contains(name)) {
                    cyclicVarCheck.add(name);
                    result = processorCallback.processVariable
                            (var,sourceDoc,this);
                    cyclicVarCheck.remove(name);
                    globalVars.put(name, result);
                }
                else {
                    result = new StringResult("#cyclic variable definition");
                }
            }
            else return null;
        }
        
        if (result == null) result = varSet.get(name);
        switch (result.getResultType()) {
            case ExprResult.NODE_SET:
                //-- copy NodeSet
                result = ((NodeSet)result).copy();
                break;
            default:
                break;
        }
        return result;
    } //-- getVariable
    
    /**
     * Retrieves the set of defined stylesheet variables
     * @return the set of defined stylesheet variables
    **/
    protected QuickStack getVariableSets() {
        return variableStack;
    } //-- getVariableSets
        
    /**
     * Gets the nearest set of variables that contains the variable
     * with the given name. If no VariableSet is found null is
     * returned
    **/
    protected VariableSet getVariableSet(String name) {
        
        if (name == null) return null;    
        VariableSet variableSet = null;
        Iterator iter = variableStack.iterator();
        while (iter.hasNext()) {
            variableSet = (VariableSet) iter.next();
            if (variableSet.get(name) != null) return variableSet;
            variableSet = null;
        }
        return variableSet;
    } //-- getVariableSet
    
    /**
     * Returns the current called templates stack being used by 
     * this ProcessorState
     * @return the current called templates stack being used by 
     * this ProcessorState
    **/
    protected QuickStack getInvocationStack() {
        return this.calledTemplates;
    } //-- getInvocationStack    
    
    /**
     * Returns the current Node stack being used by this ProcessorState
     * @return the current Node stack being used by this ProcessorState
    **/
    protected QuickStack getNodeStack() {
        return this.nodeStack;
    } //-- getNodeStack
        
    /**
     * Returns the result document from this ProcessorState
     * @return the result document from this ProcessorState
    **/
    protected Document getResultDocument() {
        return resultDoc;
    } //-- getResultDocument
    
    /**
     * Retrieves the default ScriptHandler for xsl:script evaluations.
     * @return the ScriptHandler for evaluating xsl:script elements
    **/
    public ScriptHandler getScriptHandler() {
        
        // I set the default script handler here so that
        // people not using Scripting won't have to pay
        // the price of creating the scripting environment
        if (dScriptHandler == null) {
            dScriptHandler = (ScriptHandler)scriptHandlers.get(DEFAULT_LANGUAGE);
            if (dScriptHandler == null) {
                //-- This is bad practice, but it works for now
                try {
                    Class shClass = Class.forName(DEFAULT_SCRIPT_HANDLER);
                    dScriptHandler = (ScriptHandler)shClass.newInstance();
                    dScriptHandler.initialize(processorCallback);
                    scriptHandlers.put(DEFAULT_LANGUAGE, dScriptHandler);
                }
                catch(ClassNotFoundException cnfe) {}
                catch(IllegalAccessException iae) {}
                catch(InstantiationException ie) {};
            
            }
        }
        return this.dScriptHandler;
    } //-- getScriptHandler

    /**
     * Retrieves the ScriptHandler that has the given function defined
     * within the given namespace
     * @param functionName the name of the function to look for
     * @param namespace the name of the Namespace in which the function has
     * been defined
     * @return the ScriptHandler which defines the given function, otherwise
     * null is returned
    **/
    public ScriptHandler getScriptHandler
        (String functionName, String namespace)
    {
        Enumeration enum = scriptHandlers.elements();
        while (enum.hasMoreElements()) {
            ScriptHandler sh = (ScriptHandler)enum.nextElement();
            if (sh.hasDefinedFunction(functionName, namespace))
                return sh;
        }
        return null;
    } //-- getScriptHandler
    
    /**
     * Retrieves the ScriptHandler for xsl:script evaluations.
     * @param language the script language to get a handler for.
     * @return the ScriptHandler for evaluating xsl:script elements
     * or null if no handler is found
    **/
    public ScriptHandler getScriptHandler(String language) {
        
        // Check for empty string to solve changes from 
        // XML4J 1.1.9 to 1.1.14 as suggested by Domagoj Cosic.
        if ((language == null) || (language.length() == 0))
            return getScriptHandler();
        
        ScriptHandler sh = 
            (ScriptHandler)scriptHandlers.get(language.toUpperCase());
        
        //-- fix to load defualt Script Handler if it hasn't
        //-- been loaded (Aaron Metzger)
        //-- change to look in properties file later
        if ((sh == null) && (DEFAULT_LANGUAGE.equals(language)))
            return getScriptHandler();
            
        return sh;
    } //-- getScriptHandler
    
    /**
     * Sets the ID references for the xml source document
     * @param idRefs the ID references for the xml source document
    **/
    protected void setIDReferences(Hashtable idRefs) {
        this.idRefs = idRefs;
    } //-- setIDReferences

    /**
     * Sets the ScriptHandler to be used for xsl:script evaluations
     * @param scriptHandler the ScriptHandler to use for evaluating xsl:script
     *   elements
    **/
    protected void setDefaultScriptHandler(ScriptHandler scriptHandler) {
        this.dScriptHandler = scriptHandler;
    } //-- setScriptHandler
    
    
    /**
     * Returns the child number of a Node
     * @param node the Node to retrieve the child number of
     * @return the child number of the given node
    **/
    private int childNumber(Node node) {
        int c = 0;
        Node tmpNode = node;
        while ((tmpNode = tmpNode.getPreviousSibling())!=  null) ++c;
        return c;
    } //-- childNumber
    
    /**
     * This method is only here because XML4J 1.1.x has a nasty
     * bug with it's Attr Node implementation. All Attr nodes with
     * the same name, hash to the same value. This has been fixed
     * in their 2.x version.
    **/
    private Object createKey(Node node) {
        //if (node == null) return node;
        //if (node.getNodeType() == Node.ATTRIBUTE_NODE) {
            return new Integer(System.identityHashCode(node));
        //}
        //return node;
    } //-- createKey
    
    /**
     * Prints the variable table. This method is used for debugging
     * purposes
    **
    private void debugShowVars() {
        VariableSet variableSet = null;
        Iterator iter = variableStack.iterator();
        int c = variableStack.size();
        while (iter.hasNext()) {
            variableSet = (VariableSet) iter.next();
            
            System.out.print("---- Scope Level: ");
            System.out.print(c);
            if (!iter.hasNext()) System.out.print(" - global -");
            System.out.println(" ----");
            
            Enumeration keys = variableSet.keys();
            while (keys.hasMoreElements()) {
                Object key = keys.nextElement();
                System.out.print(key);
                System.out.print("->");
                System.out.println(variableSet.get(key));
            }
            System.out.println();
            --c;
        }
        
    } //-- debugShowVars
    /* */
    
    protected Node findParent(Attr attr) {
        
        if (attr == null) return null;
        
        //-- due to bugs in XML4J 1.1.x we need to
        //-- create a unique key for attr
        Node parent = (Node)parents.get(createKey(attr));
        
        if (parent != null) return parent;
        
        //-- getIndexingInfo
        Document doc = attr.getOwnerDocument();
        IndexState idxState = (IndexState)indexingInfo.get(doc);
        if (idxState == null) {
            indexingInfo.put(doc, (idxState = new IndexState()));
            idxState.next = doc.getDocumentElement();
            if (idxState.next == null) idxState.done = true;
        }
        if (idxState.done) return null; //-- yikes no parent found!
        
        
        boolean found = false;
        boolean alreadyProcessed  = false;
        while (!found) {
            
            if (idxState.next == null) {
                idxState.done = true;
                break;
            }
            if (!alreadyProcessed) {
                //-- index attributes
                if (idxState.next.getNodeType() == Node.ELEMENT_NODE) {
                    Element element = (Element)idxState.next;
                    NamedNodeMap atts = element.getAttributes();
                    if (atts != null) {
                        for (int i = 0; i < atts.getLength(); i++) {
                            Node tmpNode = atts.item(i);
                            //-- due to bugs in XML4J 1.1.x we need to
                            //-- create a unique key for attr
                            parents.put(createKey(tmpNode), element);
                            if (attr == tmpNode) {
                                found = true;
                                parent = element;
                            }
                        }
                    }
                }
            }
            
            //-- set next node to check
            if ((!alreadyProcessed) && (idxState.next.hasChildNodes())) {
                Node child = idxState.next.getFirstChild();
                idxState.next = child;
            }
            else if (idxState.next.getNextSibling() != null) {
                idxState.next = idxState.next.getNextSibling();
                alreadyProcessed = false;
            }
            else {
                idxState.next = getParentNode(idxState.next);
                //-- already checked parent, now check sibling
                alreadyProcessed = true;
            }
        }
        return parent;
    } //-- findParent;
    
    class IndexState {
        Node next = null;
        boolean done = false;
        protected IndexState() {
            super();
        }
    }
        
} //-- ProcessorState