/*
 * (C) Copyright Keith Visco 1998, 1999.  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */
package com.kvisco.xml;

import org.w3c.dom.*;
import java.io.PrintWriter;
import java.util.StringTokenizer;

/**
 * A class for printing XML nodes
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class XMLPrinter {

    /**
     * The default indent size
    **/
    public static final int DEFAULT_INDENT = 2;

    private int width  = 80;

    private String entityTokens = "&<>";

    //-- predefined entities
    private final String AMP_ENTITY    = "&amp;";
    private final String GT_ENTITY     = "&gt;";
    private final String LT_ENTITY     = "&lt;";
    private final String HEX_ENTITY    = "&#";
    private final String QUOTE_ENTITY  = "&quot;";

    //-- special characters
    private final String LINEFEED_ENTITY = "&#xA;";

    protected final static String CDATA_END        = "]]>";
    protected final static String CDATA_START      = "<![CDATA[";
    protected final static String COMMENT_START    = "<!-- ";
    protected final static String COMMENT_END      = " -->";
    protected final static String DOCTYPE          = "DOCTYPE";
    protected final static String PI_START         = "<?";
    protected final static String PI_END           = "?>";
    protected final static String PUBLIC           = "PUBLIC";
    protected final static String SYSTEM           = "SYSTEM";

    // chars
    protected final char   AMPERSAND        = '&';
    protected final char   GT               = '>';
    protected final char   LT               = '<';
    protected final char   DASH             = '-';
    protected final char   NULL             = '\0';

    private final char CR  = '\r';
    private final char LF  = '\n';

    private String version = "1.0";

    /**
     * The a string comprised of indentSize number of indentChar's
    **/
    private String indent = null;

    /**
     * The character used for indentation
    **/
    private final char indentChar = ' ';

    /**
     * The size of the indentation
    **/
    private int indentSize = DEFAULT_INDENT;


    /**
     * The PrintWriter to print results to
    **/
    private PrintWriter pw = null;

    /**
     * A flag indicating whether or not to unescape CDATA sections
    **/
    private boolean unescapeCDATA = false;

    private boolean useEmptyElementShorthand = true;

    /**
     * A flag indicating whether or not to add whitespace
     * such as line breaks while printing
    **/
    private boolean useFormat = false;

      //---------------/
     //- Contructors -/
    //---------------/

    /**
     * Creates a new XML Printer using the given PrintWriter
     * for output
     * @param writer the PrintWriter to use for output
    **/
    public XMLPrinter(PrintWriter writer) {
        this(writer, XMLPrinter.DEFAULT_INDENT);
    } //-- XMLPrinter

    /**
     * Creates a new XML Printer using the given PrintWriter
     * for output, and nodes are indenting using the specified
     * indent size
     * @param writer the PrintWriter to use for output
     * @param indent the number of spaces to indent
    **/
    public XMLPrinter (PrintWriter writer, int indent) {
        super();
        this.pw = writer;
        setIndentSize(indent);
    } //-- XMLPrinter

    /**
     * Prints the given Node
     * @param node the Node to print
    **/
    public void print(Node node) {
        print(node,"");
        pw.flush();
    } //-- print

    /**
     * Prints the given Data as a CDATA section
    **/
    public void printCDATASection(String data) {
    	pw.print(CDATA_START);
    	printUTF8Chars(data.toCharArray());
    	pw.print(CDATA_END);
    } //-- printCDATASection

    /**
     * prints the closing tag using the given name
     * @param name the name of the element to print a closing
     * tag for
    **/
    public void printClosingTag(String name) {
        pw.print("</");
        pw.print(name);
        pw.print('>');
    } //-- printCloseElement

    /**
     * Prints the given data as a comment
     * @param data the character data to print as a comment
    **/
    public void printComment(String data) {
		pw.print(COMMENT_START);
		printCommentData(data);
		pw.print(COMMENT_END);
		if (useFormat) pw.println();
    } //-- printComment

    public void printDoctype(DocumentType docType) {
		//-- needs work
		if (docType == null) return;
		pw.print("<!");
		pw.print(DOCTYPE);
		pw.print(' ');
		pw.print(docType.getName());

		// print notations
		int size = 0;
		NamedNodeMap notations = docType.getNotations();
		if (notations != null) size = notations.getLength();
        String str;
		for (int i = 0; i < size; i++) {
		    if (i > 0) pw.println();
		    Notation ntn = (Notation)notations.item(i);
		    if ((str = ntn.getSystemId()) != null) {
		        pw.print(SYSTEM);
		        pw.print(' ');
		        pw.print(str);
		    }
		    if ((str = ntn.getPublicId()) != null) {
		        pw.print(PUBLIC);
		        pw.print(' ');
		        pw.print(str);
		    }
		}
		// print entities
		pw.println('>');
    } //-- printDoctype


    /**
     * Prints an empty element with the given name
     * and attributes
     * @param name the element name
     * @param attributes the element's attlist
    **/
    public void printEmptyElement
        (String name, NamedNodeMap attributes) {

        printOpenTag(name, attributes);
        pw.print("/>");

        if (useFormat) pw.println();

    } //-- printStartElement

    /**
     * prints the entity reference with the given name
     * @param the name of the entity reference
    **/
    public void printEntityReference(String entityName) {
		pw.print('&');
		pw.print(entityName);
		pw.print(';');
    } //-- printEntityReference

    /**
     * Prints the processing instruction
     * @param target, the pi target
     * @param data the contents of the pi
    **/
    public void printProcessingInstruction(String target, String data) {
	    pw.print(PI_START);
	    pw.print(target);
	    pw.print(' ');
	    pw.print(data);
	    pw.print(PI_END);
	    if (useFormat) pw.println();
    } //-- printProcessingInstruction

    /**
     * Prints the Start of an element
     * @param name the element name
     * @param attributes the element's attlist
    **/
    public void printStartTag
        (String name, NamedNodeMap attributes) {
        printOpenTag(name, attributes);
        pw.print('>');

    } //-- printStartElement

    public void printText(String data) {
        printWithXMLEntities(data);
    } //-- printText

    /**
     * prints the XML declaration
     * @param version the xml version
     * @param encoding, the character encoding. If null, UTF-8 is
     * the default encoding.
     * <BR/>
    **/
    public void printXMLDeclaration(String version, String encoding) {
        pw.print("<?xml version=\"");
        pw.print(version);
        pw.print("\"");
        if (encoding != null) {
            pw.print(" encoding=\"");
            pw.print(encoding);
            pw.print("\"");
        }
        pw.println("?>");
    } //-- printXMLDeclaration

    /**
     * Sets the indent size
     * @param indent the number of spaces to indent
    **/
    public void setIndentSize(int indent) {
        this.indentSize = indent;
        StringBuffer sb = new StringBuffer(indent);
        for (int i = 0; i < indent; i++) {
            sb.append(indentChar);
        }
        this.indent = sb.toString();
    } //-- setIndentSize

    /**
     * Sets whether or not to "unwrap" CDATA Sections
     * when printing. By Default CDATA Sections are left as is.
     * @param unescape the boolean indicating whether or not
     * to unescape CDATA Sections
    **/
    public void setUnescapeCDATA(boolean unescape) {
        this.unescapeCDATA = unescape;
    } //-- setUnescapeCDATA


    public void setUseEmptyElementShorthand(boolean useShorthand) {
        this.useEmptyElementShorthand = useShorthand;
    } //-- setUseEmptyElementShorthand

    /**
     * Sets whether or not this XMLPrinter should add whitespace
     * to pretty print the XML tree
     * @param useFormat a boolean to indicate whether to allow the
     * XMLPrinter to add whitespace to the XML tree. (false by default)
    **/
    public void setUseFormat(boolean useFormat) {
        this.useFormat = useFormat;
    } //-- setUseFormat

      //---------------------/
     //- Protected Methods -/
    //---------------------/

    /**
     * Prints the normalized attribute value,
     * <BR />replaces line breaks with &#xA
     * <BR />replaces '"' with &quote;
    **/
    protected void printNormalizedAttrValue(String value) {
        char[] chars = value.toCharArray();

        int startIdx = 0;
        int i = 0;
        for (; i < chars.length; i++) {
            char ch = chars[i];
            switch (ch) {
                
                case ''':
                    break;
                case '"':
                    //-- print current buffer
                    pw.write(chars,startIdx,i-startIdx);
                    startIdx = i+1;
                    //-- print entity
                    pw.print(QUOTE_ENTITY);
                    break;
                case '\n':
                    //-- print current buffer
                    pw.write(chars, startIdx, i-startIdx);
                    startIdx = i+1;
                    //-- print entity
                    pw.print(LINEFEED_ENTITY);
                    break;
                default:
                    break;
            }
        }
        //-- print remaining buffer
        pw.write(chars,startIdx,i-startIdx);

    } //-- printNormalizedAttrValue

    /**
     * prints the given node to this XMLPrinter's Writer. If the
     * useFormat flag has been set, the node will be printed with
     * indentation equal to currentIndent + indentSize
     * @param node the Node to print
     * @param currentIndent the current indent String
     * @return true, if and only if a new line was printed at
     * the end of printing the given node
    **/
    protected boolean print(Node node, String currentIndent) {

        if (node == null) return false;
        String data;
        NodeList nl = null;
        switch(node.getNodeType()) {
            case Node.DOCUMENT_NODE:
                printXMLDeclaration(version, "UTF-8");
                //printDoctype(doc.getDoctype());
                nl = node.getChildNodes();
                for (int i = 0; i < nl.getLength(); i++) {
                    print(nl.item(i),currentIndent);
                }
                break;
            case Node.ATTRIBUTE_NODE:
                Attr attr = (Attr)node;
                pw.print(attr.getName());
                data = attr.getValue();
                if (data != null) {
                    pw.print("=\"");
                    printNormalizedAttrValue(data);
                    pw.print('"');
                }
               break;
            case Node.ELEMENT_NODE:

                Element element = (Element) node;
                String elementName = element.getNodeName();
                //-- check for children
                NodeList childList = element.getChildNodes();
                int size = childList.getLength();

		        if ((size == 0) && (useEmptyElementShorthand)) {
		            printEmptyElement(elementName, element.getAttributes());
		            break;
		        }
		        else {
                    printStartTag(elementName, element.getAttributes());
		            boolean newLine = false;
		            if (useFormat) {
		                // Fix formatting of PCDATA elements by Peter Marks and
		                // David King Lassman
		                // -- add if statement to check for text node before
		                //    adding line break
		                if (size > 0) {
					        if (childList.item(0).getNodeType() != Node.TEXT_NODE) {
						        pw.println();
						        newLine = true;
						    }
						}
		            }

                    Node child = null;
                    String newIndent = indent+currentIndent;
	                for (int i = 0; i < size; i++) {
	                    child = childList.item(i);
	                    if ((useFormat) && newLine)
	                    {
	                        pw.print(newIndent);
	                    }
	                    newLine = print(child,newIndent);
	                }
	                if (useFormat) {
		                // Fix formatting of PCDATA elements by Peter Marks and
		                // David King Lassman
		                // -- add if statement to check for text node before
		                //    adding line break
		                if (child != null) {
    					    if (child.getNodeType() != Node.TEXT_NODE) {
						     //   pw.println();
    						    pw.print(currentIndent);
						    }
						}
						else {
						   // pw.println();
						   // pw.print(currentIndent);
						}
		            }
		            printClosingTag(element.getNodeName());
		            if (useFormat) {
		                Node sibling = node.getNextSibling();
		                if ((sibling == null) ||
		                    (sibling.getNodeType() != Node.TEXT_NODE))
		                {
		                    pw.println();
		                    return true;
		                }
		            }
		        } //-- end if
		        break;

		    case Node.TEXT_NODE:
		        data = ((Text)node).getData();
		        printWithXMLEntities(data);
		        break;
		    case Node.CDATA_SECTION_NODE:
		        data = ((CharacterData)node).getData();
		        if (unescapeCDATA) printWithXMLEntities(data);
		        else printCDATASection(data);
		        break;
		    case Node.COMMENT_NODE:
		        printComment( ((CharacterData)node).getData() );
		        if (useFormat) return true;
		        break;
		    case Node.ENTITY_REFERENCE_NODE:
		        printEntityReference(node.getNodeName());
		        break;
		    case Node.PROCESSING_INSTRUCTION_NODE:
		        ProcessingInstruction pi = (ProcessingInstruction)node;
		        printProcessingInstruction(pi.getTarget(), pi.getData());
		        if (useFormat) return true;
		        break;
		    case Node.DOCUMENT_TYPE_NODE:
		        printDoctype((DocumentType)node);
		        break;
		    default:
		        break;
        } //-- switch

        //-- no new line, so return false;
        return false;
    } //-- print

    /**
     * Print the proper UTF8 character
     * based on code submitted by Majkel Kretschmar
    **/
    protected void printUTF8Char(char ch) {
        if (ch >= '\u0080') {
            pw.print(HEX_ENTITY);
            pw.print((int)ch);
            pw.print(';');
        }
        else pw.print(ch);
    } //-- printUTF8Char

    /**
     * Print the proper UTF8 characters
     * based on code submitted by Majkel Kretschmar
    **/
    protected void printUTF8Chars(char[] chars) {

        int offset = 0;
        int count = 0;
        for (int i = 0; i < chars.length; i++) {
            //handle characters in the range (0x80-0xff) for
            // UTF-8 encoding
            //provided by: majkel kretschmar
            if (chars[i] >= '\u0080') {
                // clear buffer
                pw.write(chars, offset, count);
                count = 0;
                offset = i+1;
                pw.print(HEX_ENTITY);
                pw.print((int) chars[i]);
                pw.print(';');
            }
            // increase buffer count
            else {
                count++;
            }
        }
        // clear buffer
        if (offset < chars.length) pw.write(chars,offset,count);
    } //-- printUTF8Chars

      //-------------------/
     //- Private Methods -/
    //-------------------/


    private void printWithXMLEntities(String data) {
        if (data == null) return;
        char[] chars = data.toCharArray();
        int count = 0;
        int offset = 0;
        for (int i = 0; i < chars.length; i++) {
            switch (chars[i]) {
                case AMPERSAND:
                    // clear buffer
                    pw.write(chars, offset,count);
                    count = 0;
                    offset = i+1;
                    pw.print(AMP_ENTITY);
                    break;
                case LT:
                    // clear buffer
                    pw.write(chars, offset,count);
                    count = 0;
                    offset = i+1;
                    pw.print(LT_ENTITY);
                    break;
                case GT:
                    // clear buffer
                    pw.write(chars, offset,count);
                    count = 0;
                    offset = i+1;
                    pw.print(GT_ENTITY);
                    break;
                default:
                    //handle characters in the range (0x80-0xff) for
                    // UTF-8 encoding
                    //provided by: majkel kretschmar
                    if (chars[i] >= '\u0080') {
                        // clear buffer
                        pw.write(chars, offset, count);
                        count = 0;
                        offset = i+1;
                        pw.print(HEX_ENTITY);
                        pw.print((int) chars[i]);
                        pw.print(';');
                    }
                    // increase buffer count
                    else {
                      count++;
                    }
            }
        }
        // clear buffer
        if (offset < chars.length) pw.write(chars,offset,count);
    } // -- printWithXMLEntities

    /**
     * Replaces any occurances of -- inside comment data with - -
     * @param data the comment data (does not include start and end tags)
    **/
    private void printCommentData(String data) {
        if (data == null) return;
        char[] chars = data.toCharArray();
        int count = 0;
        int offset = 0;
        char lastChar = NULL;
        for (int i = 0; i < chars.length; i++) {
            if ((chars[i] == DASH) && (lastChar == DASH)) {
                // clear buffer
                pw.write(chars, offset,count);
                offset = i;
                pw.print(' ');
                count=1;
            }
            else {
                lastChar = chars[i];
                ++count;
            }
        }
        // clear buffer
        if (offset < chars.length) pw.write(chars,offset,count);
    } //-- printCommentData


    /**
     * Prints the Start of an element
     * @param name the element name
     * @param attributes the element's attlist
    **/
    private void printOpenTag(String name, NamedNodeMap attributes) {
        pw.print('<');
		pw.print(name);
		if (attributes != null) {
		    //-- print attribute nodes
		    Attr attr;
		    for (int i = 0; i < attributes.getLength(); i++) {
		        attr = (Attr) attributes.item(i);
		        String data = attr.getValue();
			    pw.print(' ');
			    pw.print(attr.getName());
			    if (data != null) {
			        pw.print('=');
			        pw.print('"');
			        printNormalizedAttrValue(data);
			        pw.print('"');
			    }
		    }
		}
    } //-- printOpenElement

} //-- XMLPrinter

