
package jde.debugger;

import jde.debugger.spec.*;
import jde.debugger.expr.*;

import com.sun.jdi.*;
import com.sun.jdi.request.*;

import java.util.*;

/**
 * GeneralCommands.java
 * <p>
 * Handles general commands. Jump to
 * '{@link #doRun run}', '{@link #doFinish finish}',
 * '{@link #doEvaluate evaluate}'
 * '{@link #doGetObject get_object}', '{@link #doGetArray get_array}',
 * '{@link #doGetLocals get_locals}',
 * '{@link #doGetLoadedClasses get_loaded_classes}',
 * '{@link #doGetPathInformation get_path_information}',
 * '{@link #doTraceClasses trace_classes}',
 * '{@link #doCancelTraceClasses cancel_trace_classes}',
 * '{@link #doTraceMethods trace_methods}',
 * '{@link #doCancelTraceMethods cancel_trace_methods}'
 *
 * <p>
 * Created: Fri Jul 30 16:29:00 1999
 * 
 * @author Amit Kumar
 * @since 0.1
 */

public class GeneralCommands extends ApplicationCommands {

    /**
     * 'run' command.
     * <p>
     *
     * <b>Syntax:</b>
     * <pre>
     * run
     * </pre>
     */
    public void doRun(Integer cmd_id, List args)
	throws JDEException {
	try {
	    app.getVM().resume();
	} catch (Exception ex) {
	    throw new JDEException("Unspecified Error occured: "+ex.toString());
	}
	app.signalCommandResult(cmd_id);
    }

    
    /**
     * 'finish' command.
     * <p>
     *
     * <b>Syntax:</b>
     * <pre>
     * finish
     * </pre>
     *
     * <b>Comments:</b>
     * <ul>
     * <li> if multiple VMs are being debugged, this command will
     * kill the one corresponding to app_id, retaining others.
     * </ul>
     */
    public void doFinish(Integer cmd_id, List args) {
	app.shutdown();
	app.signalCommandResult(cmd_id);
    }


    /**
     * 'trace_classes' command.
     * <p>
     *
     * <b>Syntax:</b>
     * <pre>
     * trace_classes <u>type</u>
     *      [{@link Etc#getSuspendPolicyFromArgs(List) suspend-policy}]
     *      [{@link Etc#getClassFiltersFromArgs(List) class-filters}]
     *      [{@link Etc#getClassExFiltersFromArgs(List) class-exclusion-filters}]
     * </pre>
     *
     * <b>Returns:</b>
     * <pre>
     * (jde-dbo-command-result cmd_id <u>requestID</u>)
     * </pre>
     *
     * <b>Comments:</b>
     * <ul>
     * <li> <u>type</u> is either "preparation" or "unloading"
     * <li> use <u>requestID</u> to cancel the trace request.
     * </ul>
     *
     * <p>
     * @see EventHandler#classPrepareEvent(ClassPrepareEvent)
     * @see EventHandler#classUnloadEvent(ClassUnloadEvent)
     */
    public void doTraceClasses(Integer cmd_id, List args)
	throws JDEException {

	if (args.size() < 1)
	    throw new JDEException("Insufficient arguments");

	String type = args.remove(0).toString().toLowerCase();

	if (!(type.equals("preparation") || type.equals("unloading")))
	    throw new JDEException("Invalid type");

	Long requestID = null;
	
	List classFilters = Etc.getClassFiltersFromArgs(args);
	List classExFilters = Etc.getClassExFiltersFromArgs(args);

	EventRequestManager em = app.getVM().eventRequestManager();

	if (type.equals("preparation")) {

	    ClassPrepareRequest cpr = em.createClassPrepareRequest();

	    cpr.setSuspendPolicy(Etc.getSuspendPolicyFromArgs(args));

	    if (classFilters != null) {
		Iterator it = classFilters.iterator();
		while (it.hasNext())
		    cpr.addClassFilter(it.next().toString());
	    }
	    if (classExFilters != null) {
		Iterator it = classExFilters.iterator();
		while (it.hasNext())
		    cpr.addClassExclusionFilter(it.next().toString());
	    }
	    requestID = addIdentifiableRequest(cpr);

	} else if (type.equals("unloading")) {

	    ClassUnloadRequest cur = em.createClassUnloadRequest();

	    cur.setSuspendPolicy(Etc.getSuspendPolicyFromArgs(args));

	    if (classFilters != null) {
		Iterator it = classFilters.iterator();
		while (it.hasNext())
		    cur.addClassFilter(it.next().toString());
	    }
	    if (classExFilters != null) {
		Iterator it = classExFilters.iterator();
		while (it.hasNext())
		    cur.addClassExclusionFilter(it.next().toString());
	    }
	    requestID = addIdentifiableRequest(cur);
	}
	app.signalCommandResult(cmd_id, requestID);
    }

    
    /**
     * 'cancel_trace_classes' command.
     * <p>
     *
     * <b>Syntax: </b>
     * <pre>
     * cancel_trace_classes <u>requestID</u>
     * </pre>
     *
     * <b>Comments:</b>
     * <ul>
     * <li> <u>requestID</u> is returned in the trace classes reply
     * </ul>
     */
    public void doCancelTraceClasses(Integer cmd_id, List args)
	throws JDEException {

	if (args.size() < 1)
	    throw new JDEException("Insufficient arguments");

	deleteIdentifiableRequest(Etc.safeGetLong
				  (args.remove(0), "request ID"));
	
	app.signalCommandResult(cmd_id);
    }


    /**
     * 'trace_methods' command.
     * <p>
     *
     * <b>Syntax:</b>
     * <pre>
     * trace_methods <u>type</u>
     *      [{@link Etc#getThreadFromArgs(List) thread-restriction}]
     *      [{@link Etc#getSuspendPolicyFromArgs(List) suspend-policy}]
     *      [{@link Etc#getClassFiltersFromArgs(List) class-filters}]
     *      [{@link Etc#getClassExFiltersFromArgs(List) class-exclusion-filters}]
     * </pre>
     *
     * <b>Returns:</b>
     * <pre>
     * (jde-dbo-command-result cmd_id <u>requestID</u>)
     * </pre>
     *
     * <b>Comments:</b>
     * <ul>
     * <li> <u>type</u> is either "entry" or "exit"
     * <li> Use <u>requestID</u> to cancel the trace request.
     * </ul>
     *
     * <p>
     * @see EventHandler#methodEntryEvent(MethodEntryEvent)
     * @see EventHandler#methodExitEvent(MethodExitEvent)
     */
    public void doTraceMethods(Integer cmd_id, List args)
	throws JDEException {

	if (args.size() < 2)
	    throw new JDEException("Insufficient arguments");

	String type = args.remove(0).toString().toLowerCase();
	if (!(type.equals("entry") || type.equals("exit")))
	    throw new JDEException("Invalid type");

	Object thread = Etc.getThreadFromArgs(args);
	ObjectReference tRef = null;
	if (thread == null) {
	    tRef = null;
	} else if (thread instanceof Long) {
	    tRef = (ObjectReference)store.get(thread);
	    if (tRef == null) {
		throw new JDEException("No such thread exists");
	    } else if (!(tRef instanceof ThreadReference)) {
		throw new JDEException("No such thread exists (anymore?)");
	    }
	} else if (thread instanceof String) {
	    tRef = ThreadCommands.getThread(app.getVM(),
					    thread.toString());
	}

	List classFilters = Etc.getClassFiltersFromArgs(args);
	List classExFilters = Etc.getClassExFiltersFromArgs(args);

	Long requestID = null;
	
	EventRequestManager em = app.getVM().eventRequestManager();

	if (type.equals("entry")) {

	    MethodEntryRequest mer = em.createMethodEntryRequest();

	    if (tRef != null) 
		mer.addThreadFilter((ThreadReference)tRef);

	    mer.setSuspendPolicy(Etc.getSuspendPolicyFromArgs(args));

	    if (classFilters != null) {
		Iterator it = classFilters.iterator();
		while (it.hasNext())
		    mer.addClassFilter(it.next().toString());
	    }
	    if (classExFilters != null) {
		Iterator it = classExFilters.iterator();
		while (it.hasNext())
		    mer.addClassExclusionFilter(it.next().toString());
	    }
	    requestID = addIdentifiableRequest(mer);
	    
	} else if (type.equals("exit")) {

	    MethodExitRequest mer = em.createMethodExitRequest();

	    if (tRef != null) 
		mer.addThreadFilter((ThreadReference)tRef);
	    
	    mer.setSuspendPolicy(Etc.getSuspendPolicyFromArgs(args));
	    
	    if (classFilters != null) {
		Iterator it = classFilters.iterator();
		while (it.hasNext())
		    mer.addClassFilter(it.next().toString());
	    }
	    if (classExFilters != null) {
		Iterator it = classExFilters.iterator();
		while (it.hasNext())
		    mer.addClassExclusionFilter(it.next().toString());
	    }
	    requestID = addIdentifiableRequest(mer);
	}
	app.signalCommandResult(cmd_id, requestID);
    }

    /**
     * 'cancel_trace_methods' command.
     * <p>
     *
     * <b>Syntax: </b>
     * <pre>
     * cancel_trace_methods <u>requestID</u>
     * </pre>
     *
     * <b>Comments:</b>
     * <ul>
     * <li> <u>requestID</u> is returned in the trace methods reply
     * </ul>
     */
    public void doCancelTraceMethods(Integer cmd_id, List args)
	throws JDEException {

	if (args.size() < 1)
	    throw new JDEException("Insufficient arguments");

	deleteIdentifiableRequest(Etc.safeGetLong
				  (args.remove(0), "request ID"));
	
	app.signalCommandResult(cmd_id);
    }

    
    /**
     * 'get_locals' command. Returns all locals in the stack frame
     * <p>
     *
     * <b>Syntax:</b>
     * <pre>
     * get_locals threadID stackFrameIndex
     * </pre>
     *
     * <b>Returns:</b>
     * <pre>
     * (jde-dbo-command-result cmd_id {@link Rep#getLocalVariableValueMapRep(Map, ObjectStore) local-variables-values})
     * </pre>
     *
     * <b>Comments:</b>
     * <ul>
     * <li> Note that stackFrameIndex = 0 corresponds to the
     * current stackframe.
     * <li> The threadID and stackFrameIndex can be got from the
     *	'get_threads' command. Note that many evaluations
     *  might not be possible depending on the state of the thread
     * </ul>
     */
    public void doGetLocals(Integer cmd_id, List args)
	throws JDEException {
    
	// see doEvaluate() to explain some of the black magic below
	
	boolean weSuspendedThread = false;
	ThreadReference tRef = null;

	if (args.size() != 2) 
	    throw new JDEException("Insufficient arguments");

	try {
	    Long uniqueID = Etc.safeGetLong(args.remove(0), "thread ID");
	    int frameIndex = Etc.safeGetint(args.remove(0), "frame index");
	    
	    Object oRef = store.get(uniqueID);
	    if (oRef == null) {
		throw new JDEException("No such thread exists");
	    } else if (!(oRef instanceof ThreadReference)) {
		throw new JDEException("Object is not a thread");
	    }
	    
	    tRef = (ThreadReference)oRef;

	    tRef.suspend();
	    weSuspendedThread = true;

	    StackFrame frame = null;
	    try {
		frame = tRef.frame(frameIndex);
	    } catch (IncompatibleThreadStateException ex) {
		throw new JDEException("Thread is not suspended");
	    } catch (IndexOutOfBoundsException ex) {
		throw new JDEException("Invalid frame");
	    } catch (ObjectCollectedException ex) {
		throw new JDEException("The frame has already been garbage collected");
	    }

	    if (frame == null) {
		throw new JDEException("Error ascertaining frame");
	    }

	    LispForm localVariableValues = null;
	    try {
		localVariableValues = Rep.getLocalVariableValueMapRep(frame.getValues(frame.visibleVariables()), store);
	    } catch (AbsentInformationException ex) {
		throw new JDEException("Local variable information not available: compile with -g");
	    } catch (NativeMethodException ex) {
		throw new JDEException("Can't access local variables in native methods");
	    }

	    app.signalCommandResult(cmd_id, localVariableValues);
				    
	} finally {
	    if (weSuspendedThread && (tRef != null)) tRef.resume();
	}
    }
	    
    
    /**
     * 'evaluate' command.
     * <p>
     *
     * <b>Syntax:</b>
     * <pre>
     * evaluate threadID stackFrameIndex "expression"
     * </pre>
     *
     * <b>Returns:</b>
     * <pre>
     * (jde-dbo-command-result cmd_id {@link Rep#getValueRep(Value, ObjectStore) value})
     * </pre>
     *
     * <b>Comments:</b>
     * <ul>
     * <li> Note that stackFrameIndex = 0 corresponds to the
     * current stackframe.
     * <li> The threadID and stackFrameIndex can be got from the
     *	'get_threads' command. Note that many evaluations
     *  might not be possible depending on the state of the thread
     * </ul>
     */
    public void doEvaluate(Integer cmd_id, List args)
	throws JDEException {

	// we'll suspend the thread later on, but wanna resume it later
	// so keep track.
	boolean weSuspendedThread = false;
	ThreadReference tRef = null;
	try {
	    
	    if (args.size() < 3) 
		throw new JDEException("Insufficient arguments");

	    // need a valid thread to work in...
	    Long uniqueID = Etc.safeGetLong(args.remove(0), "thread ID");
	    int frameIndex = Etc.safeGetint(args.remove(0), "frame index");
		    
	    Object oRef = store.get(uniqueID);
	    if (oRef == null) {
		throw new JDEException("No such thread exists");
	    } else if (!(oRef instanceof ThreadReference)) {
		throw new JDEException("Object is not a thread");
	    }
	    
	    tRef = (ThreadReference)oRef;

	    // suspend it on our own so that nothing funny happens during
	    // the time we're trying to do expression evaluation
	    tRef.suspend();
	    weSuspendedThread = true;
	    
	    StackFrame frame = null;
	    try {
		frame = tRef.frame(frameIndex);
	    } catch (IncompatibleThreadStateException ex) {
		throw new JDEException("Thread is not suspended");
	    } catch (IndexOutOfBoundsException ex) {
		throw new JDEException("Invalid frame");
	    } catch (ObjectCollectedException ex) {
		throw new JDEException("The frame has already been garbage collected");
	    }
	    if (frame == null) {
		throw new JDEException("Error ascertaining frame");
	    }
	    String expr = args.remove(0).toString();
	    Value val = Etc.evaluate(expr, frame);
	    
	    app.signalCommandResult(cmd_id, Rep.getValueRep(val, store));
	    
	} finally {
	    if (weSuspendedThread && (tRef != null)) tRef.resume();
	}
    }


    /**
     * 'get_object' command. Information about a particular object.
     * <p>
     *
     * <b>Syntax:</b>
     * <pre>
     * get_object objectID
     * </pre>
     *
     * <b>Returns:</b>
     * <pre>
     * (jde-dbo-command-result cmd_id {@link Rep#getObjectRep(ObjectReference, ObjectStore) detailed-object-info})
     * </pre>
     */
    public void doGetObject(Integer cmd_id, List args)
	throws JDEException {

	if (args.size() < 1)
	    throw new JDEException("Insufficient arguments");

	Long uniqueID = Etc.safeGetLong(args.remove(0), "object ID");
	ObjectReference oRef = store.get(uniqueID);

	if (oRef == null) 
	    throw new JDEException("No such object exists");

	app.signalCommandResult(cmd_id, Rep.getObjectRep(oRef, store, true));
    }

    
    /**
     * 'get_array' command. Information about a given array, and,
     * optionally, values of a range of indices
     * <p>
     *
     * <b>Syntax:</b>
     * <pre>
     * get_array objectID [index, length]
     * </pre>
     *
     * <b>Returns:</b>
     * <pre>
     * (jde-dbo-command-result cmd_id {@link Rep#getArrayRep(ArrayReference, ObjectStore, int, int) array})
     * </pre>
     */
    public void doGetArray(Integer cmd_id, List args)
	throws JDEException {

	if (args.size() < 1)
	    throw new JDEException("Insufficient arguments");

	Long uniqueID = Etc.safeGetLong(args.remove(0), "object ID");
	ObjectReference oRef = store.get(uniqueID);
	    
	if (oRef == null) {
	    throw new JDEException("No such object exists");
	} else if (!(oRef instanceof ArrayReference)) {
	    throw new JDEException("Object is not an array");
	}

	if (args.size() == 0) {
	    app.signalCommandResult(cmd_id, Rep.getArrayRep((ArrayReference)oRef, store, -1, -1));
	} else if (args.size() == 2) {
	    int index = Etc.safeGetint(args.remove(0), "index");
	    int length = Etc.safeGetint(args.remove(0), "length");
	    app.signalCommandResult(cmd_id, Rep.getArrayRep((ArrayReference)oRef, store, index, length));
	} else {
	    throw new JDEException("Syntax error: Wrong number of arguments");
	}
    }

    
    /**
     * 'get_string' command. Returns the value of a string
     * <p>
     *
     * <b>Syntax:</b>
     * <pre>
     * get_string objectID 
     * </pre>
     *
     * <b>Returns:</b>
     * <pre>
     * (jde-dbo-command-result cmd_id {@link Rep#getStringRep(StringReference, ObjectStore) string-representation})
     * </pre>
     */
    public void doGetString(Integer cmd_id, List args)
	throws JDEException {

	if (args.size() < 1)
	    throw new JDEException("Insufficient arguments");

	Long uniqueID = Etc.safeGetLong(args.remove(0), "object ID");
	ObjectReference oRef = store.get(uniqueID);
	    
	if (oRef == null) {
	    throw new JDEException("No such object exists");
	} else if (!(oRef instanceof StringReference)) {
	    throw new JDEException("Object is not a string");
	}

	app.signalCommandResult(cmd_id, Rep.getStringRep((StringReference)oRef, store));
    }


    /**
     * 'get_loaded_classes' command. Returns a list of all loaded classes
     * <p>
     *
     * <b>Syntax:</b>
     * <pre>
     * get_loaded_classes
     * </pre>
     *
     * <b>Returns:</b>
     * <pre>
     * (jde-dbo-command-result cmd_id (list ["type-name"]*))
     * </pre>
     */
    public void doGetLoadedClasses(Integer cmd_id, List args)
	throws JDEException {

	String typeNames = "(list";
	Iterator it = app.getVM().allClasses().iterator();
	while (it.hasNext()) {
	    typeNames += " \""+((ReferenceType)it.next()).name()+"\"";
	}
	typeNames += ")";

	app.signalCommandResult(cmd_id, new LispForm(typeNames));
    }

    /**
     * 'get_path_information' command. Returns all the vm knows about
     * paths.
     * <p>
     *
     * <b>Syntax:</b>
     * <pre>
     * get_path_information
     * </pre>
     *
     * <b>Returns:</b>
     * <pre>
     * (jde-dbo-command-result cmd_id "base-directory" (list [boot-class-path component]*) (list [class-path component]*))
     * </pre>
     */
    public void doGetPathInformation(Integer cmd_id, List args)
	throws JDEException {

	if (!(app.getVM() instanceof PathSearchingVirtualMachine))
	    throw new JDEException("VM doesn't search paths");

	PathSearchingVirtualMachine vm =
	    (PathSearchingVirtualMachine)app.getVM();

	String bootClassPathString = "(list";
	Iterator it = vm.bootClassPath().iterator();
	while (it.hasNext()) {
	    bootClassPathString += " \""+it.next()+"\"";
	}
	bootClassPathString += ")";

	bootClassPathString = bootClassPathString.replace('\\', '/');

	String classPathString = "(list";
	it = vm.classPath().iterator();
	while (it.hasNext()) {
	    classPathString += " \""+it.next()+"\"";
	}
	classPathString += ")";
	
	classPathString = classPathString.replace('\\', '/');

	app.signalCommandResult(cmd_id,
				new LispForm("\""+vm.baseDirectory().replace('\\', '/')+"\""
					     + BR +bootClassPathString
					     + BR +classPathString));
    }

    public GeneralCommands(Application a, ObjectStore s) {
	super(a, s);
    }

} // GeneralCommands
