/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                        Author :  Paul Taylor                          */
/*                        Date   :  August 1995                          */
/*-----------------------------------------------------------------------*/
/*                   File I/O functions for EST_Track class              */
/*                                                                       */
/*=======================================================================*/

#include <fstream.h>
#include <iostream.h>
#include <stdlib.h>
#include <math.h>
#include "EST_unix.h"
#include "EST_types.h"
#include "EST_Track.h"
#include "EST_track_aux.h"
#include "EST_TrackMap.h"
#include "EST_cutils.h"
#include "track_io.h"
#include "EST_Token.h"
#include "EST_TList.h"
#include "EST_string_aux.h"

// size of locally allocated buffer. If more channels than this we have to
// call new

#define REASONABLE_FRAME_SIZE (20)

EST_read_status load_esps(const EST_String filename, EST_Track &tr)
{
    float **tt;
    float fsize;
    char **fields;
    int num_points, num_fields;
    int i,j;
    EST_read_status  r_val;

    tr.clear();
    
    r_val = get_track_esps(filename, &fields, &tt, &fsize, &num_points, 
			   &num_fields);
    if (r_val == misc_read_error)
    {
	cerr << "Error reading ESPS file " << filename << endl;
	return misc_read_error;
    }
    else if (r_val == wrong_format)
	return wrong_format;

    tr.resize(num_points,num_fields);

    for (i = 0; i < num_points; ++i)
    {
      for (j = 0; j < num_fields; ++j)
	tr.a(i, j) = tt[i][j];
      tr.set_value(i);
    }

    for (i = 0; i < num_fields; ++i)
	tr.set_field_name(fields[i], i);

    tr.fill_time(fsize);

    // REORG not sure what these should be -- awb
    tr.set_break_type("MANY");
    tr.set_space_type("FIXED");

    // get_track_esps allocs all the memory, we therefore need to release it
    for (i = 0; i < num_fields; ++i)
	wfree(fields[i]);
    wfree(fields);
    for (i = 0; i < num_points; ++i)
	wfree(tt[i]);
    wfree(tt);

    tr.set_file_type("esps");
    tr.set_name(filename);
 
   if (tr.field_name(0) == "F0")
       espsf0_to_track(tr);

    return format_ok;
}

EST_read_status load_ascii(const EST_String filename, EST_Track &tr, float ishift)
{
    EST_TokenStream ts, tt;
    EST_StrList sl;
    const float NEARLY_ZERO = 0.001;
    int i, j, n_rows, n_cols;
    EST_String t;
    EST_TBI *p;
    
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "Can't open track file " << filename << endl;
	return misc_read_error;
    }
    // set up the character constant values for this stream
    ts.set_SingleCharSymbols(";");
    
    if (ishift < NEARLY_ZERO)
    {
	cerr<<
	    "Error: Frame spacing must be specified for ascii track input\n";
	return misc_read_error;
    }
    // first read in as list

    for (n_rows = 0; !ts.eof(); ++n_rows)
	sl.append(ts.get_upto_eoln().string());

    tt.open_string(sl.first());
    for (n_cols = 0; !tt.eof(); ++n_cols)
    	tt.get().string();

    // resize track and copy values in
    tr.resize(n_rows, n_cols);

    for (p = sl.head(), i = 0; p != 0; ++i, p = next(p))
    {
	tt.open_string(sl(p));
	for (j = 0; !tt.eof(); ++j)
	    tr.a(i, j) = atof(tt.get().string());
	if (j != n_cols)
	{
	    cerr << "Wrong number of points in row " << i << endl;
	    cerr << "Expected " << n_cols << " got " << j << endl;
	    return misc_read_error;
	}
    }

    tr.fill_time(ishift);
    tr.set_break_type("MANY");
    tr.set_space_type("FIXED");
    tr.set_file_type("ascii");
    tr.set_name(filename);

    return format_ok;
}

EST_read_status load_xmg(const EST_String filename, EST_Track &tr)
{
    EST_TokenStream ts;
    EST_StrList sl;
    int i, n, sr;
    EST_String t, k, v;
    EST_TBI *p;
    
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "Can't open track file " << filename << endl;
	return misc_read_error;
    }
    // set up the character constant values for this stream
    ts.set_SingleCharSymbols(";");

    if (ts.peek().string() != "XAO1")
	return wrong_format;

    ts.get().string();

    while ((!ts.eof()) && (ts.peek().string() != "\014"))
    {
	k = ts.get().string();
	v = ts.get().string();
	if (k == "Freq")
	    sr = atoi(v) * 1000;
	else if (k == "YMin")
	    tr.amin = atof(v);
	else if (k == "YMax")
	    tr.amax = atof(v);
    }

    if (ts.eof())
    {
	cerr << "Unexpected end of file in reading xmg header\n";
	return misc_read_error;
    }
    ts.get().string(); // read control L
    ts.get_upto_eoln().string(); // read until end of header

    // read in lines to a list
    for (n = 0; !ts.eof(); ++n)
	sl.append(ts.get_upto_eoln().string());
    
    // note the track size is total number of points *and* breaks
    tr.resize(n, 1 );  // REORG - fix this for multi channel work

    for (p = sl.head(), i = 0; p != 0; ++i, p = next(p))
    {
	ts.open_string(sl(p));
	if (ts.peek().string() != "=")
	{
	    tr.t(i) = atof(ts.get().string()) / 1000.0;
	    tr.a(i) = atof(ts.get().string());
	}
	else
	{
	    ts.get().string();
	    tr.set_break(i);
	}
    }

    tr.set_break_type("SINGLE");
    tr.set_space_type("VARI");
    tr.set_file_type("xmg");
    tr.set_name(filename);

    return format_ok;
}

EST_write_status save_esps(const EST_String filename, EST_Track tr)
{
    EST_write_status rc;
    EST_Track track_tosave;
    int i, j;

    // extra special hack of extra hacky waves
//    if (tr.num_channels() == 1 || tr.field_name(0) == "F0")
//      track_to_espsf0(tr, track_tosave);
//    else
    track_tosave = tr;
    
    if (filename == "-")
    {
	cerr << "Output to stdout not available for ESPS file types:";
	cerr << "no output written\n";
	return write_fail;
    }

    /*
      if (track_tosave.space_type() != "FIXED")
	track_tosave.change_type(TRACK_DEF_FS, "MANY");
    else 
	track_tosave.change_type(0.0,"MANY");
	*/

    float **a = new float*[track_tosave.num_frames()];
    // pity we need to copy it
    for (i=0; i < track_tosave.num_frames(); i++)
    {
	a[i] = new float[track_tosave.num_channels()];
	for (j=0; j < track_tosave.num_channels(); j++)
	    a[i][j] = track_tosave.a(i,j);
    }

    char **f_names = new char*[track_tosave.num_channels()];
    for (i=0; i < track_tosave.num_channels(); i++)
      {
	// cout << "field " << i << "is '" << track_tosave.field_name(i) << "'\n";
	f_names[i] = wstrdup(track_tosave.field_name(i, esps_channel_names, 0));
      }

    if (track_tosave.space_type() == "FIXED")
      {
	rc = put_track_esps(filename, f_names,
			    a, track_tosave.shift(), 1.0/track_tosave.shift(), 
			    track_tosave.num_channels(), 
			    track_tosave.num_frames());
      }
    else
      {
	rc = put_track_esps(filename, f_names,
			    a, TRACK_DEF_FS/1000.0, 1.0/(TRACK_DEF_FS/1000.0), 
			    track_tosave.num_channels(), 
			    track_tosave.num_frames());
      }

    for (i=0; i < track_tosave.num_frames(); i ++)
	delete [] a[i];
    delete [] a;
    for (i=0; i < track_tosave.num_channels(); i++)
	delete [] f_names[i];
    delete [] f_names;

    return rc;
}

EST_write_status save_ascii(const EST_String filename, EST_Track tr)
{
    if (tr.space_type() == "FIXED")
	tr.change_type(0.0, "MANY");
    
    ostream *outf;
    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);

    if (outf == 0)
	return misc_write_error;
    
    for (int i = 0; i < tr.num_frames(); ++i)
    {
	for (int j = 0; j < tr.num_channels(); ++j)
	    *outf << tr.a(i, j) << " ";
	*outf << endl;
    }
    
    if (outf != &cout)
	delete outf;
    
    return write_ok;
}

EST_write_status save_xgraph(const EST_String filename, EST_Track tr)
{
    ostream *outf;
    
    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);
    
    if (outf == 0)
	return misc_write_error;

    tr.change_type(0.0, "SINGLE");

    for  (int j = 0; j < tr.num_channels(); ++j)
    {
	*outf << "\""<< tr.field_name(j) << "\"\n";    
	for (int i = 0; i < tr.num_frames(); ++i)
	    if (tr.val(i))
		*outf << tr.t(i) << "\t" << tr.a(i, j) << endl;
	    else
		*outf << "move  ";
    }
    if (outf != &cout)
	delete outf;

    return write_ok;
}

EST_write_status save_xmg(const EST_String filename, EST_Track tr)
{
    ostream *outf;
    int i;
    float min, max;
    int sr = 16000; // REORG - fixed sample rate until xmg is fixed

    // this takes care of rescaling
    tr.change_type(0.0, "SINGLE");

    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);
    if (outf == 0)
	return misc_write_error;
    
    min = max = tr.a(0);
    for (i = 0; i < tr.num_frames(); ++i)
    {
	if (tr.a(i) > max) max = tr.a(i);
	if (tr.a(i) < min) min = tr.a(i);
    }
    *outf << "XAO1\n\n";	// xmg header identifier.
    *outf << "LineType        segments \n";
    *outf << "LineStyle       solid \n";
    *outf << "LineWidth       0 \n";
    *outf << "Freq " << sr / 1000 << endl; // a REAL pain!
    *outf << "Format  Binary \n";
    *outf << "YMin    " << ((tr.amin != 0.0) ? tr.amin : min) << endl;
    *outf << "YMax    " << ((tr.amax != 0.0) ? tr.amax : max) << endl;
    if (tr.color != "")
	*outf << "LineColor  " << tr.color << endl;
    *outf << char(12) << "\n";	// control L character
    
    //    rm_excess_breaks();
    //    rm_trailing_breaks();
    for (i = 0; i < tr.num_frames(); ++i)
	if (tr.val(i))
	    *outf << tr.ms_t(i) << "\t" <<tr.a(i) << endl;
	else
	    *outf << "=\n";
    if (outf != &cout)
	delete outf;

    return write_ok;
}

#define HTK_ENERGY   0x0040
#define HTK_NO_E     0x0080
#define HTK_DELTA    0x0100
#define HTK_AC       0x0200
#define HTK_MASK     0x003f

#define HTK_WAVE     0x0000
#define HTK_LPC      0x0001
#define HTK_LPCREFC  0x0002
#define HTK_LPCCEP   0x0003
#define HTK_LPDELCEP 0x0004
#define HTK_IREFC    0x0005
#define HTK_MFCC     0x0006
#define HTK_FBANK    0x0007
#define HTK_MELSPEC  0x0008

/* 100ns = 10 per us = 10 000 per ms = 10 000 000 per second*/
#define HTK_UNITS_PER_SECOND 10000000

struct htk_header 
{
    int num_samps; 
    int samp_period;   /* in 100ns units -design choice of the centuary*/
    /* I don't think this is right -- awb, is this an old HTK format ? */
    short samp_size;    /* number of bytes per sample */
    short samp_type;    /* code from 0 to 8 indicating sample kind */
                        /* - see HTK reference manual */
};

EST_write_status save_htk(const EST_String filename, EST_Track orig)
{
    // file format is a 12 byte header
    // followed by data
    float s;

    EST_Track track;
    int type;

    if (orig.contour_type() == EST_ContourType::LPC)
      type = track_to_htk_lpc(orig, track);
    else
      {
	track = orig;
	type = HTK_FBANK;
      }

    if (track.space_type() != "FIXED")
	track.change_type(TRACK_DEF_FS, "MANY");
    
    // this is rounded to the nearest micro-second.
    s = rint((HTK_UNITS_PER_SECOND * track.shift())/10.0) * 10.0;

    struct htk_header header;
    header.num_samps = track.num_frames();
    header.samp_period = (long) s;
    header.samp_size = sizeof(float) * track.num_channels();
    header.samp_type = type;	
    
    int i, j;
    FILE *outf;
    if (filename == "-")
	outf = stdout;
    else if ((outf = fopen(filename,"wb")) == NULL)
    {
	cerr << "save_htk: cannot open file \"" << filename << 
	    "\" for writing." << endl;
	return misc_write_error;
    }
    
    // write the header
    fwrite((char*)&(header.num_samps), 1, sizeof(header.num_samps), outf);
    fwrite((char*)&(header.samp_period), 1, sizeof(header.samp_period), outf);
    fwrite((char*)&(header.samp_size), 1, sizeof(header.samp_size), outf);
    fwrite((char*)&(header.samp_type), 1, sizeof(header.samp_type), outf);
    
    // write the data
    for (i = 0; i < track.num_frames(); ++i)	
	for (j = 0; j < track.num_channels(); ++j)
	    fwrite((unsigned char*) &(track.a(i, j)), 1, sizeof(float), outf);
    
    if (outf != stdout)
	fclose(outf);

    return write_ok;
}

static int htk_swapped_header(htk_header *header)
{
    //  Tries to guess if the header is swapped.  If so it
    //  swaps the contents and returns TRUE, other returns FALSE
    //  HTK doesn't have a magic number so we need heuristics to
    //  guess when its byte swapped
    int rval = FALSE;

    if ((header->num_samps < 0) ||
	(header->samp_period < 0) ||
	(header->samp_size < 0))
	rval = TRUE;  // header would be invalid otherwise
    // But this might not accurate enough to be generally useful

    if (rval)
    {
	header->num_samps = SWAPINT(header->num_samps);
	header->samp_period = SWAPINT(header->samp_period);
	header->samp_size = SWAPSHORT(header->samp_size);
	header->samp_type = SWAPSHORT(header->samp_type);
    }
    
    return rval;

}

EST_read_status load_htk(const EST_String filename, EST_Track &tmp)
{
    int i,j, order, new_frames, new_order;
    EST_ContourType pname;
    int swap;
    
    FILE *fp;
    struct htk_header header;
    int header_sz = sizeof(header);
    
    unsigned short samp_type, base_samp_type;
    
    if ((fp = fopen(filename, "rb")) == NULL)
    {
	cerr << "EST_Track load: couldn't open EST_Track input file" << endl;
	return misc_read_error;
    }
    
    // try and read the header
    if (fread(&header, header_sz, 1, fp) != 1)
    {
	fclose(fp);
	return wrong_format;
    }

    swap = htk_swapped_header(&header);
    // see if the header looks reasonable, i.e. can we work out the sample type
    
    samp_type = header.samp_type;
    base_samp_type = samp_type & HTK_MASK;
    
    switch(base_samp_type)
    {
	
      case HTK_WAVE:
	cerr << "Can't read HTK WAVEFORM format file into track" << endl;
	return misc_read_error;
	break;
	
      case HTK_LPC:
	pname = EST_ContourType::LPC;
	break;
	
      case HTK_LPCREFC:
	pname = EST_ContourType::make("lpcrefc");
	break;
	
      case HTK_LPCCEP:
	pname = EST_ContourType::CEPSTRUM;
	break;
	
      case HTK_LPDELCEP:
	// equivalent to HTK_LPCCEP + DELTA
	base_samp_type = HTK_LPCCEP;
	samp_type = HTK_LPCCEP | HTK_DELTA; // set delta bit 
	pname = EST_ContourType::CEPSTRUM;
	break;
	
      case HTK_IREFC:
	pname = EST_ContourType::make("irefc");
	break;
	
      case HTK_MFCC:
	pname = EST_ContourType::make("mfcc");
	break;
	
      case HTK_FBANK:
	pname = EST_ContourType::OTHER;
	break;
	
      case HTK_MELSPEC:
	pname = EST_ContourType::make("melspec");
	break;
	
      default:
	fclose(fp);
	return wrong_format;
	break;
    }
    
    // if we get this far we have decided this is a HTK format file
    
    // new_order is total number of fields in track
    // order is order of LPC etc. analysis
    // e.g. if order is 12 and we have energy and delta then new_order = (12 + 1) * 2 = 26
    
    new_order = header.samp_size / sizeof(float); // assumes data always stored as floats !

    float shift = ((float)header.samp_period/ (float)HTK_UNITS_PER_SECOND);

    new_frames = header.num_samps;
    tmp.resize(new_frames, new_order);
    tmp.fill_time(shift);
    tmp.set_space_type("FIXED");
    
    // check length of file is as expected from header info
    long file_length;
    if (fseek(fp,0,SEEK_END)){
	fclose(fp);
	return wrong_format;
    }
    
    if ((file_length = ftell(fp)) == -1)
    {
	fclose(fp);
	return wrong_format;
    }
    
    file_length = file_length + 1 - header_sz;
    
    long expected_floats = file_length / sizeof(float);
    
    if(expected_floats != (new_order*new_frames) )
    {   // it probably isn't HTK format after all
	fclose(fp);
	return wrong_format;
    }
    
    // work out the order of the analysis
    // Reorg -- surely you can't increase order
    order = new_order;
    if (samp_type & HTK_NO_E) 
	order++;
    
    if (samp_type & HTK_AC)
	order /= 3;
    else if (samp_type & HTK_DELTA)
	order /= 2;
    
    if (samp_type & HTK_ENERGY)
	order--;

    // go to start of data
    if (fseek(fp, header_sz, SEEK_SET) == -1)
    {
	cerr << "Couldn't position htk file at start of data" << endl;
	fclose(fp);
	return misc_read_error;
    }

    float *frame, frame_buffer[REASONABLE_FRAME_SIZE];
    
    if (new_order > REASONABLE_FRAME_SIZE)
      frame = new float[new_order];
    else
      frame = frame_buffer;
    
    // actually read in the data
    for (i=0; i<new_frames; i++)
    {
	if (fread(frame, sizeof(float), new_order, fp) != 
	    (size_t) new_order )
	{
	    cerr << "Could not read data from htk track file" << endl;
	    fclose(fp);
	    if (frame != frame_buffer)
	      delete[] frame;
	    return misc_read_error;
	}
	if (swap)
	    swap_bytes_float(frame,new_order);
	for (j = 0; j < new_order; ++j)
	    tmp.a(i, j) = frame[j];
    }

    if (frame != frame_buffer)
      delete[] frame;

    // name the fields
    EST_String t;
    // the first 'order' fields are always called c1,c2...
    // AWB bug -- the following corrupts memory
    for (i=0;i<order;i++)
    {
	EST_String t2;
	t2 = EST_String("c") + itoString(i+1);
	tmp.set_field_name(t2, i);
    }
    i=order;

    // energy present and not suppressed
    if ( (samp_type & HTK_ENERGY) && !(samp_type & HTK_NO_E) )
	tmp.set_field_name("E", i++);
    
    // delta coeffs ?
    if (samp_type & HTK_DELTA)
    {
	for (j = 0; j < order; j++)
	{
	    t = EST_String("c") + itoString(j+1) + "_d";
	    tmp.set_field_name(t, i++);
	}
	
	// energy ?
	if (samp_type & HTK_ENERGY)
	    tmp.set_field_name("E_d", i++);
    }
    
    // 'acceleration'  coeffs ?
    if (samp_type & HTK_AC)
    {
	for(j=0;j<order;j++)
	{
	    t = EST_String("ac")+ itoString(j+1)+ "_d_d";
	    tmp.set_field_name(t, j + i++);
	}
	// energy ?
	if (samp_type & HTK_ENERGY)
	    tmp.set_field_name("E_d_d", i++);
    }
    
    // sanity check
    if (i != new_order)
    {
	cerr << "Something went horribly wrong - wanted " << new_order
	    << " fields in track but got " << i << endl;
	fclose(fp);
	return wrong_format;
    }
    tmp.set_contour_type(pname);
    tmp.set_name(filename);
    tmp.set_file_type("htk");

    fclose(fp);
    return format_ok;
}

 /************************************************************************/
 /*                                                                      */
 /* Convert single f0 channel tracks into arbitrarilly chosen esps FEA   */
 /* subtype, reputedly to make waves happy. This is evil beyond all      */
 /* understanding.                                                       */
 /*                                                                      */
 /************************************************************************/

// format of the desired track.
static EST_ChannelMapping espsf0_mapping =
{
  { channel_f0, 0 },
  { channel_voiced,  1 },
  { channel_power, 2},
  { channel_peak, 3},
  { channel_unknown},	
};
static EST_TrackMap ESPSF0TrackMap(espsf0_mapping);

// It seems that the vital thing is to call the track "F0", and  so
// we only need 1 field instead of the normal 5. This saves  *lots* of
// space. For the time being we use 2 fields as the prob_voicnug filed is
// used by our input routine.

int track_to_espsf0(EST_Track &track, EST_Track &f0_track)
{
    f0_track.resize(track.num_frames(), 2);

    f0_track.assign_map(ESPSF0TrackMap);

    // k1 is ratio of the first two cross-correlation values
//    f0_track.set_field_name("k1", 4);

    // copy data. Remaining fields zeroed by resize. This is of course stupid
    // as if k1 iz zero mathematics is in deep problems.
    for (int i = 0; i < track.num_frames(); ++i)
    {
	f0_track.a(i, channel_voiced) = track.track_break(i) ? 0.1 : 1.2;
	f0_track.a(i, channel_f0) = track.track_break(i) ? 0.0: track.a(i,0);
    }

    f0_track.set_contour_type(EST_ContourType::F0);
    f0_track.set_file_type("esps");
    f0_track.fill_time(track.shift());
    track.set_name(track.name());

/*    f0_track.resize(track.num_frames(), 5);

    f0_track.assign_map(ESPSF0TrackMap);

    // k1 is ratio of the first two cross-correlation values
    f0_track.set_field_name("k1", 4);

    // copy data. Remaining fields zeroed by resize. This is of course stupid
    // as if k1 iz zero mathematics is in deep problems.
    for (int i = 0; i < track.num_frames(); ++i)
    {
	f0_track.a(i, channel_voiced) = track.track_break(i) ? 0.1 : 1.2;
	f0_track.a(i, channel_f0) = track.a(i,0);
    }

    f0_track.set_contour_type(EST_ContourType::F0);
    f0_track.set_file_type("esps");
    f0_track.fill_time(track.shift());
    track.set_name(track.name());
*/

    return 0;
}

int espsf0_to_track(EST_Track &fz)
{
    int f, p, i;
    f = p = -1;

    // check to see if prob of voicing field exists
    for (i = 0; i < fz.num_channels(); ++i)
    {
	if (fz.field_name(i) == "prob_voice")
	    p = i;
    }
    for (i = 0; i < fz.num_channels(); ++i)
    {
	if (fz.field_name(i) == "F0")
	    f = i;
    }

    for (i = 0; i < fz.num_frames(); ++i)
    {
	if (p == -1) // if f0 val is < 1 make this a break
	{
	    if (fz.a(i, f) < 1.0)
		fz.set_break(i);
	    else
		fz.set_value(i);
	}
	else // use prob voicing
	{
	    if (fz.a(i, p) < 0.5)
	    {
		fz.a(i, f) = 0.0;
		fz.set_break(i);
	    }
	    else
		fz.set_value(i);
	}
    }

    fz.set_contour_type(EST_ContourType::F0);

    return 0;
}

int track_to_htk_lpc(EST_Track &track, EST_Track &lpc)
{
    int type = HTK_LPC;
    int ncoefs, nchannels;

    if (track.has_channel(channel_coefN))
	ncoefs = track.channel_position(channel_coefN) - track.channel_position(channel_coef0)+1;
    else
	ncoefs = track.num_channels()-track.channel_position(channel_coef0);

    nchannels = ncoefs;

    if (track.has_channel(channel_power))
    {
	nchannels++;
	type |= HTK_ENERGY;
    }

    lpc.resize(track.num_frames(), nchannels);

    for(int i = 0; i< track.num_frames(); i++)
	for (int c = 0; c < ncoefs; c++)
	{
	    lpc.a(i, c) = track.a(i, channel_coef0, c);
	    lpc.t(i) = track.t(i);
	}

  
    if (track.has_channel(channel_power))
    {
	for(int ii = 0; ii< track.num_frames(); ii++)
	    lpc.a(ii, ncoefs) = track.a(ii, channel_power);
    }

    return type;

}

EST_write_status save_ind_TrackList(EST_TrackList &tlist, EST_String &otype)
{
    for (EST_TBI *p = tlist.head(); p ; p = next(p))
	tlist(p).save(tlist(p).name(), otype);
    
    return write_ok;
}    

EST_read_status read_TrackList(EST_TrackList &tlist, EST_StrList &files, 
			       EST_Option &al)
{
    EST_Track s;
    EST_TBI *p, *plp;

    for (p = files.head(); p; p = next(p))
    {
	s.clear();
	tlist.append(s);
	plp = tlist.tail();
	if (read_track(tlist(plp), files(p), al) != format_ok)
	    exit (-1);

	tlist(plp).set_name(files(p));
    }

    return format_ok;
}
