/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                 Authors:  Alan W Black                                */
/*                 Date   :  July 1996                                   */
/*-----------------------------------------------------------------------*/
/*  A viterbi decoder                                                    */
/*                                                                       */
/*  User provides the candidates, target and combine score function      */
/*  and it searches for the best path through the candidates             */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "EST_viterbi.h"

static void init_paths_array(EST_VTPoint *n,int num_states);

EST_VTPoint::~EST_VTPoint()
{
    int i;

    if (paths != 0) delete paths;
    if (num_states != 0)
    {
	for (i=0; i<num_states; i++)
	    if (st_paths[i] != 0)
		delete st_paths[i];
	delete st_paths;
    }
    if (cands != 0) delete cands;
    if (next != 0) delete next;
}

EST_Viterbi_Decoder::EST_Viterbi_Decoder(uclist_f_t a, unpath_f_t b)
{
    beam_width = 0;
    cand_width = 0;
    user_clist = a;
    user_npath = b;
    num_states = 0;
    debug = FALSE;
    big_is_good = TRUE;  // for probabilities it is
}

EST_Viterbi_Decoder::EST_Viterbi_Decoder(uclist_f_t a, unpath_f_t b, int s)
{
    beam_width = 0;
    cand_width = 0;
    user_clist = a;
    user_npath = b;
    // if s == -1 number of states is determined by number of cands
    // this is specific to a particular search but very efficient
    num_states = s;  // can do the lattice search more directly 
    debug = FALSE;
    big_is_good = TRUE;  // for probabilities it is
}

EST_Viterbi_Decoder::~EST_Viterbi_Decoder()
{
    delete timeline;
}

void EST_Viterbi_Decoder::initialise(EST_Stream &p)
{
    // Creates a time line with each point pointing at each item in p
    EST_Stream_Item *i;
    EST_VTPoint *t = 0,*n;

    for (i=p.head(); i != 0; i=next(i))
    {
	n = new EST_VTPoint;
	n->s = i;
	if (num_states > 0)  
	    init_paths_array(n,num_states);
	if (t == 0)
	    timeline = n;
	else
	    t->next = n;
	t=n;
    }
    // Extra one at the end for final paths
    n = new EST_VTPoint;
    if (num_states > 0)
	init_paths_array(n,num_states);

    // Need init path on first point so a search can start
    if (num_states == 0)   // general search case 
	timeline->paths = new EST_VTPath;
    if (num_states == -1)
	init_paths_array(timeline,1);  // a start point

    if (t == 0)
	timeline = n;   // its an empty stream
    else
	t->next = n;
}

static void init_paths_array(EST_VTPoint *n,int num_states)
{
    // Create the states array and initialize it
    int j;
    
    n->num_states = num_states;
    n->st_paths = new EST_VTPath*[num_states];
    for (j=0;j<num_states;j++)
	n->st_paths[j] = 0;
}

const int EST_Viterbi_Decoder::betterthan(const float a,const float b) const
{
    // Some thing big ig better, others want it to be as small as possible
    // this just tells you if a is better than b by checking the variable
    // in the decoder itself.

    // For probabilities big_is_good == TRUE (or log probabilities)
    
    if (big_is_good)
	return (a > b);
    else
	return (a < b);
}

static void init_dynamic_states(EST_VTPoint *p, EST_VTCandidate *cands)
{
    // In a special (hmm maybe not so special), the number of "states"
    // is the number of candidates
    EST_VTCandidate *c;
    int i;

    for (i=0, c=cands; c != 0; c=c->next,i++)
	c->pos = i;
    init_paths_array(p,i);
}

void EST_Viterbi_Decoder::search(void)
{
    // Searches for the best path 
    // Haven't done the state bit yet
    EST_VTPoint *p;
    EST_VTPath *t,*np;
    EST_VTCandidate *c;
    int i;

    for (p=timeline; p->next != 0; p=p->next)
    {   // For each point in time 
	p->cands  = (*user_clist)(*p->s);  // P(S|B)
	if (num_states != 0)  // true viterbi -- optimized for states
	{
	    if (num_states == -1)  // special case, dynamic state size
		init_dynamic_states(p->next,p->cands);
	    for (i=0; i<p->num_states; i++)
	    {   // for each path up to here
		if ((p == timeline) || (p->st_paths[i] != 0))
		for (c=p->cands; c != 0; c=c->next)
		{   // for each new candidate
		    np = (*user_npath)(p->st_paths[i],c);
		    if (debug)
                    {
			fprintf(stdout,"%s: ", (const char *)p->s->name());
			cout << c->name;
			fprintf(stdout," %1.3f B %1.3f (%1.3f) st %d s %1.3f ",
				np->c->score,
				(np->c->score==0 ? 0 : 
				 ((float)np->feature("lscore"))/np->c->score),
				(float)np->feature("lscore"),np->state,
				np->score);
			if (p->st_paths[i] == 0)
			    cout << "(I)" << endl;
			else
			    cout << p->st_paths[i]->c->name << endl;
		    }
		    vit_add_path(p->next,np);   // index best by state
		}
	    }
	}
	else                  // general beam search
	    for (t=p->paths; t != 0; t=t->next)
	    {   // for each path up to here
		for (c=p->cands; c != 0; c=c->next)
		{   // for each new candidate
		    np = (*user_npath)(t,c);
		    add_path(p->next,np);      // be a little cleverer
		}
	    }
	if (debug) fprintf(stdout,"\n");
    }
}

void EST_Viterbi_Decoder::vit_add_path(EST_VTPoint *p, EST_VTPath *np)
{
    // We are doing true viterbi so we need only keep the best
    // path for each state.  This means we can index into the
    // array of paths ending at P and only keep np if its score
    // is better than any other path of that state

    if (np->state > p->num_states)
    {
	cerr << "State too big\n";
    }
    if ((p->st_paths[np->state] == 0) ||
	(betterthan(np->score,p->st_paths[np->state]->score)))
//	(p->st_paths[np->state]->score < np->score))
    {
	// This new one is better so replace it
	if (p->st_paths[np->state] != 0)
	    delete p->st_paths[np->state];
	p->st_paths[np->state] = np;
    }
    else
	delete np;
}

void EST_Viterbi_Decoder::add_path(EST_VTPoint *p, EST_VTPath *np)
{
    // Add new path to point,  Prunes as appropriate to strategy
    EST_VTPath *pp;

    if (p == 0)
	cerr << "Viterbi: tried to add path to NULL point\n";
    else 
    {
	if ((beam_width == 0) ||            // Add if no beam restritions or
	    (p->num_paths < beam_width) ||  //        beam not filled  or
	    (betterthan(np->score,p->paths->score)))//this is better than best
//	    (np->score > p->paths->score))  //        this is better than best
	{
	    EST_VTPath **l = &p->paths;
	    EST_VTPath *a;
	    
	    for (a=p->paths; /* scary */ ; a=a->next)
	    {
		if ((a == 0) || (betterthan(a->score,np->score)))
		{   // Add new path here 
		    np->next = a;
		    *l = np;
		    p->num_paths++;
		    break;
		}
		l = &a->next;
	    }
	    // Prune the first one of the list 
	    if ((beam_width > 0) &&
		(p->num_paths > beam_width))
	    {
		pp = p->paths;
		p->paths = pp->next;
		pp->next = 0;
		p->num_paths--;
		delete pp;
	    }
	}
	else
	{
	    delete np;  // failed to make it
	}
    }

}

EST_VTCandidate *EST_Viterbi_Decoder::add_cand_prune(EST_VTCandidate *newcand,
					     EST_VTCandidate *allcands)
{
    // Add newcand to allcand, in score order and prune it to 
    // cand_width, delete newcand if its not good enough
    EST_VTCandidate *newlist = allcands;
    EST_VTCandidate *pp;
    int numcands;
    
    if (allcands == 0)
	numcands = 0;
    else
	numcands = allcands->pos;
    
    if ((cand_width == 0) ||	// Add if no candbeam restritions or
	(numcands < cand_width) || //        candbeam not filled  or
	(betterthan(newcand->score,allcands->score))) //this better than best
    {
	EST_VTCandidate **l = &newlist;
	EST_VTCandidate *a;
	
	for (a=newlist;		/* scary */ ; a=a->next)
	{
	    if ((a == 0) || (betterthan(a->score,newcand->score)))
	    {			// Add new path here 
		newcand->next = a;
		*l = newcand;
		numcands++;
		break;
	    }
	    l = &a->next;
	}
	// Prune the first one off the list 
	if ((cand_width > 0) &&
	    (numcands > cand_width))
	{
	    pp = newlist;
	    newlist = pp->next;
	    pp->next = 0;
	    numcands--;
	    delete pp;
	}
    }
    else
	delete newcand;		// failed to make it
    
    newlist->pos = numcands;
    return newlist;
}

void EST_Viterbi_Decoder::result(const EST_String &n)
{
    // Finds best path through the search space (previously searched)
    // Adds field to the EST_Stream, named n, with chosen value 
    EST_VTPath *p;
    
    for (p = find_best_end(); p != 0; p=p->from)
    {
	// Hmm need the original EST_Stream_Item
	if (p->c != 0)
	{
	    p->c->s->set_feature_val(n,p->c->name);
	    p->c->s->set_feature(n+"_score",(float)p->feature("lscore"));
	}
    }
}

void EST_Viterbi_Decoder::copy_feature(const EST_String &n)
{
    // Copy feature from path to related stream
    EST_VTPath *p;
    
    for (p = find_best_end(); p != 0; p=p->from)
    {
	// Hmm need the original EST_Stream_Item
	if (p->c != 0)
	{
	    p->c->s->set_feature_val(n,p->feature(n));
	}
    }
}

EST_VTPath *EST_Viterbi_Decoder::find_best_end() const
{
    EST_VTPoint *t;
    double best,worst;
    EST_VTPath *p, *best_p=0;
    int i;
    // I'd like to use HUGE_VAL or FLT_MAX for this but its not portable 
    // (on alphas)
    double a_big_number = 1.0e10;
    
    if (big_is_good) 
	worst = -a_big_number;	// worst possible;
    else
	worst = a_big_number;	// worst possible;
    best = worst;		// hopefully we can find something better;
    
    for (i=0,t=timeline; t->next != 0; t=t->next,i++)
    {
	if ((t->num_states == 0) && (t->num_paths == 0))
	    cerr << "No paths " << i << " " << t->s->name() << endl;
    }
    
    if (num_states != 0)
	for (i=0; i<t->num_states; i++)
	{
	    if ((t->st_paths[i] != 0) &&
		(betterthan(t->st_paths[i]->score,best)))
	    {
		best = t->st_paths[i]->score;
		best_p = t->st_paths[i];
	    }
	}
    else
	for (p=t->paths; p!=0; p=p->next)
	{
	    if (betterthan(p->score,best))
	    {
		best = p->score;
		best_p = p;
	    }
	}
    if (debug)
    {
	if (best == worst)
	    cerr << "Failed to find path" << endl;
	cout << "Best score is " << best << endl;
    }
    return best_p;
}

