/********************************************************************************
 * Copyright (c) Des Herriott 1993, 1994
 *               Erik Kunze   1995, 1996, 1997
 *
 * Permission to use, distribute, and sell this software and its documentation
 * for any purpose is hereby granted without fee, provided that the above
 * copyright notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that the name
 * of the copyright holder not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.  The
 * copyright holder makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or implied
 * warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Martin Smith (msmith@lssec.bt.co.uk)
 *         Erik Kunze
 *
 * changed by EKU
 *******************************************************************************/
#ifdef XZX_IF1
#ifndef lint
static char rcsid[] = "$Id: if1.c,v 3.12 1997/09/13 16:13:58 erik Rel $";
#endif

#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <assert.h>
#include "z80.h"
#include "debug.h"
#include "resource.h"
#include "mem.h"
#include "util.h"
#include "dialog.h"
#include "if1.h"
#include "main.h"


static void if1Quit(void);
static int setupCartridgeFile(int, char *);
static void unloadCartridgeFiles(void);
static void unloadCartridgeFile(int d);
static int loadCartridgeFile(int);
static void flushCache(void);
static void startDrive(int);
static void stopDrive(int);


static long mdriveBytesOut = 0L;
static long mdriveBytesIn = 0L;
static int gapClock = 0;
static int gapBit = 0;
static mdriveData drives[MAX_DRIVES];
static long driveSeq = 0L;
static int preamble = 0;
static int driveRunning = -1;
static int driveCount = 0;


void
If1Init(void)
{
	int i;
	OnQuit(if1Quit);

	for (i = 0; i < MAX_DRIVES; i++)
	{
		if (setupCartridgeFile(i, GETCFG(cartfiles[i])) == -1)
		{
			SETCFG(cartfiles[i], NULL);
		}
	}
}

static void
if1Quit(void)
{
	if (GETCFG(if1_active))
	{
		unloadCartridgeFiles();
	}
#ifdef DEBUG
	if (GETCFG(debug) & D_IF1)
	{

		Msg(M_DEBUG, "%d bytes out to microdrive, %d bytes in", mdriveBytesOut,
			mdriveBytesIn);
	}
#endif
}

void
If1OnOff(int status)
{
	int i;
	if (status != GETCFG(if1_active))
	{
		SETCFG(if1_active, status);
		if (status)
		{
			assert(GETCFG(machine) != 3);
			(void)ReadROMs();

			if (GETCFG(if1_active))
			{
				for (i = 0; i < MAX_DRIVES; i++)
				{
					drives[i].tapepos = 0L;
					drives[i].sequence = 0L;
				}
			}
		}
		else
		{
			unloadCartridgeFiles();
		}
	}
}

void
SetRS232Traps(void)
{
	if (RealMemory[IF1ROM][RS232_IP_ADDR_V1] == RS232_IP_BPT_BYTE &&
		RealMemory[IF1ROM][RS232_OP_ADDR_V1] == RS232_OP_BPT_BYTE)
	{
		RealMemory[IF1ROM][RS232_IP_ADDR_V1] = 0xed;
		RealMemory[IF1ROM][RS232_IP_ADDR_V1 + 1] = RS232_IP_BPT;
		RealMemory[IF1ROM][RS232_OP_ADDR_V1] = 0xed;
		RealMemory[IF1ROM][RS232_OP_ADDR_V1 + 1] = RS232_OP_BPT;
	}
	else if (RealMemory[IF1ROM][RS232_IP_ADDR_V2] == RS232_IP_BPT_BYTE &&
			 RealMemory[IF1ROM][RS232_OP_ADDR_V2] == RS232_OP_BPT_BYTE)
	{
		RealMemory[IF1ROM][RS232_IP_ADDR_V2] = 0xed;
		RealMemory[IF1ROM][RS232_IP_ADDR_V2 + 1] = RS232_IP_BPT;
		RealMemory[IF1ROM][RS232_OP_ADDR_V2] = 0xed;
		RealMemory[IF1ROM][RS232_OP_ADDR_V2 + 1] = RS232_OP_BPT;
	}
	else
	{
		Msg(M_WARN, "unknown IF1 ROM, RS232 to stdio is disabled");
	}
}

int
InsertCartridge(int d)
{
	char *filename;
	if ((filename = FileSelector("Insert Cartridge", True, "mdr", NULL, NULL)))
	{
		unloadCartridgeFile(d);
		return (setupCartridgeFile(d, filename));
	}
	return -1;
}

void
EjectCartridge(int d)
{
	if (drives[d].filename)
	{
		unloadCartridgeFile(d);
		free(drives[d].filename);
		drives[d].filename = NULL;
		if (drives[d].buffer)
		{
			free(drives[d].buffer);
			drives[d].buffer = NULL;
		}
	}
}

char *
GetCartName(int d)
{
	return (drives[d].filename);
}

static int
setupCartridgeFile(int d, char *fname)
{
	struct stat buf;
	int i;

	for (i = 0; i < MAX_DRIVES; i++)
	{
		if (i != d
			&& drives[i].filename
			&& !strcmp(fname, drives[i].filename))
		{
			Msg(M_WARN, "cartridge image <%s> allready assigned to drive %d",
				fname,  1 + i);
			free(fname);
			return -1;
		}
	}

	if (stat(fname, &buf)
		|| !S_ISREG(buf.st_mode)
		|| buf.st_size != CART_FILE_SIZE)
	{
		Msg(M_WARN, "invalid cartridge image <%s>", fname);
		free(fname);
		return -1;
	}

	if (drives[d].filename)
	{
		free(drives[d].filename);
	}
	drives[d].filename = fname;

	if (drives[d].buffer)
	{
		free(drives[d].buffer);
		drives[d].buffer = NULL;
	}
	drives[d].tapepos = 0L;
	drives[d].protect = 1;
	drives[d].sequence = 0L;
	return 0;
}

static void
unloadCartridgeFiles(void)
{
	int f;
	for (f = 0; f < MAX_DRIVES; ++f)
	{
		if (drives[f].buffer)
		{
			unloadCartridgeFile(f);
		}
	}
}

static void
unloadCartridgeFile(int d)
{
	FILE *fp;
	if (drives[d].dirty && !drives[d].protect)
	{
		if ((fp = Fopen(drives[d].filename, "r+b")))
		{
			if (fwrite(drives[d].buffer, 1, CART_FILE_SIZE, fp)	!=
				CART_FILE_SIZE)
			{
				Msg(M_PERR, "couldn't write cartridge file <%s>",
					drives[d].filename);
			}
#ifdef DEBUG
			else
			{
				if (GETCFG(debug) & D_IF1)
				{
					Msg(M_DEBUG, "wrote cartridge file %d <%s>", d,
						drives[d].filename);
				}
			}
#endif
			(void)fclose(fp);
		}
		else
		{
			Msg(M_PERR, "couldn't open cartridge file <%s> for writing",
				drives[d].filename);
		}
	}
#ifdef DEBUG
	else
	{
		if (GETCFG(debug) & D_IF1)
		{
			Msg(M_DEBUG, "unloaded cartridge %d", d);
		}
	}
#endif
	drives[d].dirty = 0;
}

static int
loadCartridgeFile(int d)
{
	FILE *fp;
	int res = -1;
	if (!drives[d].filename && InsertCartridge(d) == -1)
	{
		return res;
	}
	if (!(fp = Fopen(drives[d].filename, "rb")))
	{
		Msg(M_PERR, "couldn't open cartridge file <%s> for reading",
			drives[d].filename);
		return res;
	}
	drives[d].tapepos /= (long)CART_FILE_SECTOR_SIZE;
	drives[d].tapepos *= (long)CART_FILE_SECTOR_SIZE;
	drives[d].dirty = 0;
	drives[d].protect = 0;
	drives[d].buffer = (uns8 *)Malloc(CART_FILE_SIZE, "loadCartridgeFile");
	if (fread(drives[d].buffer, 1, CART_FILE_SIZE, fp) == CART_FILE_SIZE)
	{

		drives[d].protect = drives[d].buffer[CART_FILE_SIZE - 1] ? 1 : 0;

		if (access(drives[d].filename, W_OK) != 0)
		{
			drives[d].protect = 1;
		}
#ifdef DEBUG
		if (GETCFG(debug) & D_IF1)
		{
			Msg(M_DEBUG, "read %s cart file %d [%s]",
				drives[d].protect ? "R/O" : "R/W", d, drives[d].filename);
		}
#endif
		res = 0;
	}
	else
	{
		Msg(M_PERR, "couldn't read cartridge file <%s>", drives[d].filename);
		free(drives[d].buffer);
		drives[d].buffer = NULL;
	}
	(void)fclose(fp);
	return res;
}

static void
flushCache(void)
{
	int count = 0;
	int picked = 0;
	long sequence = 0L;
	int f;
	for (f = 0; f < MAX_DRIVES; ++f)
	{
		if (drives[f].buffer)
		{
			++count;

			if (!sequence || drives[f].sequence < sequence)
			{
				sequence = drives[f].sequence;
				picked = f;
			}
		}
	}
	if (picked && count >= KEEP_DRIVES_IN_RAM)
	{
		free(drives[picked].buffer);
		drives[picked].buffer = NULL;
#ifdef DEBUG
		if (GETCFG(debug) & D_IF1)
		{
			Msg(M_DEBUG, "threw drive %d out of RAM", picked);
		}
#endif
	}
}

static void
startDrive(int d)
{
#ifdef DEBUG
	if (GETCFG(debug) & D_IF1)
	{
		Msg(M_DEBUG, "drive %d on", d);
	}
#endif
	if (d < MAX_DRIVES)
	{
		gapClock = 0;
		gapBit = 0;
		drives[d].tapepos /= (long) CART_FILE_SECTOR_SIZE;
		drives[d].tapepos *= (long) CART_FILE_SECTOR_SIZE;
		if (driveRunning != d)
		{
			if (driveRunning >= 0)
			{
				stopDrive(driveRunning);
			}
			if (drives[d].buffer)
			{
				driveRunning = d;
				drives[driveRunning].sequence = ++driveSeq;
			}
			else
			{
				flushCache();
				if (loadCartridgeFile(d) != -1)
				{
					drives[driveRunning].sequence = ++driveSeq;
					driveRunning = d;
				}
#ifdef DEBUG
				else
				{
					if (GETCFG(debug) & D_IF1)
					{
						Msg(M_DEBUG, "but no cartridge");
					}
				}
#endif
			}
		}
	}
}

static void
stopDrive(int d)
{
#ifdef DEBUG
	if (GETCFG(debug) & D_IF1)
	{
		Msg(M_DEBUG, "drive %d stopped", d);
	}
#endif

	if (d < MAX_DRIVES && d == driveRunning)
	{
		unloadCartridgeFile(d);
		driveRunning = -1;
	}
}

void
OutPortEF(uns8 byte)
{
	static int lastByte = 0;
	if (driveCount > 7 || driveCount < 0)
	{
		driveCount = 0;
	}

	if (byte == 0xe2 && !preamble)
	{
		preamble = 12;
	}

	if (byte == 0xee && lastByte == 0xee)
	{
		driveCount = 0;
		byte = 0;
#ifdef DEBUG
		if (GETCFG(debug) & D_IF1)
		{
			Msg(M_DEBUG, "interface 1 drive_count reset");
		}
#endif
	}
	else if (byte == 0xec && lastByte == 0xee)
	{
		++driveCount;
		startDrive(8 - driveCount);
	}
	else if (byte == 0xed && lastByte == 0xef)
	{
		++driveCount;
		stopDrive(8 - driveCount);
	}
	lastByte = byte;
}

void
OutPortE7(uns8 byte)
{
	++mdriveBytesOut;
	if (driveRunning >= 0 && !preamble)
	{

		int pos = drives[driveRunning].tapepos % CART_FILE_SECTOR_SIZE;

		if ((!pos && (byte & 1) != 1) || (pos == 15 && (byte & 1) != 0))
		{
			Msg(M_ERR, "microdrive write sync lost %d",
				drives[driveRunning].tapepos);
			drives[driveRunning].protect = 1;
		}

		drives[driveRunning].buffer[drives[driveRunning].tapepos] = byte;
		drives[driveRunning].dirty = 1;

		if (!preamble && (pos == (CART_FILE_SECTOR_SIZE - 1) || pos == 14))
		{
			preamble = 3;
		}

		if (++(drives[driveRunning].tapepos) == CART_FILE_SIZE - 1)
		{
			drives[driveRunning].tapepos = 0L;
#ifdef DEBUG
			if (GETCFG(debug) & D_IF1)
			{
				Msg(M_DEBUG, "drive %d revolution (write)", driveRunning);
			}
#endif
		}
	}
	else if (driveRunning >= 0 && preamble > 0)
	{
		--preamble;
	}
}

uns8
InPortE7(void)
{
	int res = 0xff;
	if (driveRunning >= 0)
	{
		++mdriveBytesIn;
		res = drives[driveRunning].buffer[drives[driveRunning].tapepos];
		if (++drives[driveRunning].tapepos == CART_FILE_SIZE - 1)
		{
			drives[driveRunning].tapepos = 0L;
#ifdef DEBUG
			if (GETCFG(debug) & D_IF1)
			{
				Msg(M_DEBUG, "drive %d revolution (read)", driveRunning);
			}
#endif
		}
	}
	return res;
}

uns8
InPortEF(void)
{
	uns8 res = 0xe0;
	if (driveRunning >= 0)
	{

		int pos = drives[driveRunning].tapepos % CART_FILE_SECTOR_SIZE;
		int syncBit = SYNC_BIT;
		if (pos == 0 || pos == 15)
		{
			syncBit = 0;
		}
		else
		{

			if ((++drives[driveRunning].tapepos) == CART_FILE_SIZE - 1)
			{
				drives[driveRunning].tapepos = 0L;
#ifdef DEBUG
				if (GETCFG(debug) & D_IF1)
				{
					Msg(M_DEBUG, "drive %d revolution (sync)", driveRunning);
				}
#endif
			}
		}
		if (!drives[driveRunning].protect)
		{
			res |= WRITE_PROT_BIT;
		}

		if (!(++gapClock % GAP_CLOCK_RATE))
		{
			gapBit = !gapBit;
		}
		if (gapBit)
		{
			res |= GAP_BIT;
		}
		res |= syncBit;
	}
	else
	{
		res |= GAP_BIT | SYNC_BIT | WRITE_PROT_BIT;
	}
	return res;
}

void
RS232IpTrap(void)
{
	char c;
	int ret;
	if ((ret = read(fileno(stdin), (void *)&c, 1)) == 1)
	{
#ifdef DEBUG
		if (GETCFG(debug) & D_IF1)
		{
			Msg(M_DEBUG, "read code <%02x> from stdin", (unsigned int)c);
		}
#endif
		A = (c == 0x0a && GETCFG(translate_nl)) ? 0x0d : c;
	}
	else
	{
		switch (StdinFileType)
		{
			case FD_PIPE:
			case FD_TTY:
				if (errno == 0 || errno == EAGAIN)
				{
					CLR(C_FLAG);
					SET(Z_FLAG);
					goto end;
				}
				break;
			case FD_FILE:
				if (!ret)
				{
					CLR(C_FLAG);
					CLR(Z_FLAG);
					goto end;
				}
			case FD_UNKNOWN:
			default:
				break;
		}
		Msg(M_PERR, "read error on stdin - got %d bytes", ret);
	}
  end:
	RET();
}

void
RS232OpTrap(void)
{
	char c;
	if (A == 13 && GETCFG(strip_nl))
	{
		return;
	}
	c = (A == 0x0d && GETCFG(translate_nl)) ? 0x0a : A;
	if (write(fileno(stdout), (void *)&c, 1) == 1)
	{
		(void)fflush(stdout);
#ifdef DEBUG
		if (GETCFG(debug) & D_IF1)
		{
			Msg(M_DEBUG, "wrote code <%02x> to stdout", (unsigned int)A);
		}
#endif
	}
	else
	{
		Msg(M_PERR, "write error on stdout");
	}
	RET();
}
#endif

