//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  This is the cartridge class for Arcadia (aka Starpath) Supercharger
  games.  Christopher Salomon provided most of the details and code
  used for this class.

  The Supercharger has four 2K banks.  There are three banks of RAM
  and one bank of ROM.  All 6K of the RAM can be read and written.

  @author  Bradford W. Mott
  @version $Id: CartAR.cxx,v 1.1 1997/05/17 19:04:12 bwmott Exp $
*/

#include "machine.hxx"
#include "CartAR.hxx"
#include "Error.hxx"
#include "System.hxx"
#include "M6507Hi.hxx"

//============================================================================
// Constructor
//============================================================================
CartridgeAR::CartridgeAR(System& system, uByte* image, uLong size)
    : Cartridge(system)
{
  uWord i;

  // Create a load image buffer and copy the given image
  myLoadImages = new uByte[size];
  myNumberOfLoadImages = size / 8448;

  for(i = 0; i < size; ++i)
  {
    myLoadImages[i] = image[i];
  } 

  // Map all of my addresses in the system
  for(uWord addr = 0; addr < 8192; ++addr)
  {
    if(addr & 0x1000)
    {
      mySystem.mapPeek(addr, *this);
    }
  }

  // Set the current image to load 0
  setupCurrentImage(0);
}
 
//============================================================================
// Destructor
//============================================================================
CartridgeAR::~CartridgeAR()
{
  delete[] myLoadImages;
}

//============================================================================
// Answer the byte at the given address
//============================================================================
uByte CartridgeAR::peek(uWord addr)
{
  // Are they trying to access the tape rewind routine?
  if((myImageOffset[1] == 3 * 2048) && (addr == 0x180A))
  {
    uLong cycles = mySystem.m6507().cycles();

    // Yes, so let's do something...
    if((myPower && ((cycles - myPowerRomCycle) < 1000)) || !myPower)
    {
      Error err;
      err.message() << "Supercharger: The ROM has not powered up...";
      Throw(err);
    }
    else
    {
      Error err;
      err.message() << "Supercharger: Can't handle rewinding tape...";
      Throw(err);
    }
  }

  // Are they trying to access the multiload routine?
  if((myImageOffset[1] == 3 * 2048) && ((addr & 0x0FFF) == 0x0800))
  {
    uLong cycles = mySystem.m6507().cycles();

    if((myPower && ((cycles - myPowerRomCycle) < 1000)) || !myPower)
    {
      Error err;
      err.message() << "Supercharger: The ROM has not been powered up...";
      Throw(err);
    }
    else
    {
      // Get the load they're trying to access
      uByte load = mySystem.peek(0x00FA);
 
      // Setup specified load as the current image
      setupCurrentImage(load);

      return myImage[(addr & 0x07FF) + myImageOffset[1]];
    }
  } 

  uWord operandAddress = LastOperandAddress;

  // Handle bank configuration
  if(((addr & 0x0FFF) == 0x0FF8) && operandAddress)
  {
    uWord operandOffset = LastOperandOffset;
    bankConfiguration((uByte)operandAddress + operandOffset);

    PokePossible = false;
    return myImage[(addr & 0x07FF) + myImageOffset[1]];
  }

  // Handle poke if writing enabled
  if(myWriteEnabled && PokePossible)
  {
    if(((operandAddress & 0x1fff) >= 0x1000) &&
        ((operandAddress & 0x1fff) <= 0x10ff))
    {
      uLong operandCycle = LastOperandCycle;
      uLong cycles = mySystem.m6507().cycles();

      if(((cycles-operandCycle) > 3) && ((cycles-operandCycle) < 7))
      {
        uWord operandOffset = LastOperandOffset;
        uByte value = operandAddress + operandOffset;

        if((addr & 0x0fff)>=0x800)
        {
          myImage[(addr & 0x7ff) + myImageOffset[1]] = value;
        }
        else
        {
          myImage[(addr & 0x7ff) + myImageOffset[0]] = value;
        }

        PokePossible = false;
      } 
    }
  }

  if(addr < 0x1800)
    return myImage[(addr & 0x07FF) + myImageOffset[0]];
  else
    return myImage[(addr & 0x07FF) + myImageOffset[1]];
}

//============================================================================
// Reset to power on state
//============================================================================
void CartridgeAR::reset()
{
  // Set the current image to load 0
  setupCurrentImage(0);

  // Set bank configuration upon reset so ROM is selected
  myPower = true;
  myPowerRomCycle = 0;
  myWriteEnabled = false;

  myImageOffset[0] = 0 * 2048;
  myImageOffset[1] = 3 * 2048;
}

//============================================================================
// Handle a change to the bank configuration
//============================================================================
void CartridgeAR::bankConfiguration(uByte configuration)
{
  // D7-D5 of this byte: Write Pulse Delay (n/a for emulator)
  //
  // D4-D0: RAM/ROM configuration:
  //       $F000-F7FF    $F800-FFFF Address range that banks map into
  //  000wp     2            ROM
  //  001wp     0            ROM
  //  010wp     2            0      as used in Commie Mutants and many others
  //  011wp     0            2      as used in Suicide Mission
  //  100wp     2            ROM
  //  101wp     1            ROM
  //  110wp     2            1      as used in Killer Satellites
  //  111wp     1            2      as we use for 2k/4k ROM cloning
  // 
  //  w = Write Enable (1 = enabled; accesses to $F000-$F0FF cause writes
  //    to happen.  0 = disabled, and the cart acts like ROM.)
  //  p = ROM Power (0 = enabled, 1 = off.)  Only power the ROM if you're
  //    wanting to access the ROM for multiloads.  Otherwise set to 1.

  myPower = !(configuration & 0x01);

  if(myPower)
  {
    myPowerRomCycle = mySystem.m6507().cycles();
  }

  myWriteEnabled = configuration & 0x02;

  switch((configuration >> 2) & 0x07)
  {
    case 0:
    {
      myImageOffset[0] = 2 * 2048;
      myImageOffset[1] = 3 * 2048;
      break;
    }

    case 1:
    {
      myImageOffset[0] = 0 * 2048;
      myImageOffset[1] = 3 * 2048;
      break;
    }

    case 2:
    {
      myImageOffset[0] = 2 * 2048;
      myImageOffset[1] = 0 * 2048;
      break;
    }

    case 3:
    {
      myImageOffset[0] = 0 * 2048;
      myImageOffset[1] = 2 * 2048;
      break;
    }

    case 4:
    {
      myImageOffset[0] = 2 * 2048;
      myImageOffset[1] = 3 * 2048;
      break;
    }

    case 5:
    {
      myImageOffset[0] = 1 * 2048;
      myImageOffset[1] = 3 * 2048;
      break;
    }

    case 6:
    {
      myImageOffset[0] = 2 * 2048;
      myImageOffset[1] = 1 * 2048;
      break;
    }

    case 7:
    {
      myImageOffset[0] = 1 * 2048;
      myImageOffset[1] = 2 * 2048;
      break;
    }
  }
}

//============================================================================
// Sets up a "dummy" bootstrap ROM in the ROM bank of the cartridge
//============================================================================
void CartridgeAR::setupROM()
{
  static uByte dummyROMCode[] = {
0xa9, 0x0, 0xa2, 0x0, 0x95, 0x80, 0xe8, 0xe0, 
0x80, 0xd0, 0xf9, 0x4c, 0x2b, 0xfa, 0xff, 0xff, 
0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
0xa9, 0x0, 0xa2, 0x0, 0x95, 0x80, 0xe8, 0xe0, 
0x1e, 0xd0, 0xf9, 0xa2, 0x0, 0xbd, 0x44, 0xfa, 
0x95, 0xf7, 0xe8, 0xe0, 0x9, 0xd0, 0xf6, 0xa2, 
0xff, 0xa0, 0x0, 0xa9, 0x0, 0x85, 0x80, 0x85, 
0xf8, 0x4c, 0xf7, 0x0, 0xcd, 0x0, 0xf0, 0xad, 
0xf8, 0xff, 0x4c, 0x0, 0x0
  };

  int size = sizeof(dummyROMCode);

  // Copy the "dummy" ROM code into the ROM area
  for(int i = 0; i < size; ++i)
  {
    myImage[0x1A00 + i] = dummyROMCode[i];
  }

  // Put a JMP $FA20 at multiload entry point ($F800)
  myImage[0x1800] = 0x4C;
  myImage[0x1801] = 0x20;
  myImage[0x1802] = 0xFA;

  // Update ROM code to have the correct reset address and bank configuration
  myImage[0x1A00 + size - 2] = myHeader[0];
  myImage[0x1A00 + size - 1] = myHeader[1];
  myImage[0x1A00 + size - 17] = myHeader[2];

  // Finally set vectors to point to this "dummy" code at 0xFA00
  myImage[3 * 2048 + 2044] = 0x00;
  myImage[3 * 2048 + 2045] = 0xFA;
  myImage[3 * 2048 + 2046] = 0x00;
  myImage[3 * 2048 + 2047] = 0xFA;
}

//============================================================================
// Handle setting the current image from the load images
//============================================================================
void CartridgeAR::setupCurrentImage(uByte load)
{
  uByte image;
  uWord j;

  // Scan through the ROM looking for the given load
  for(image = 0; image < myNumberOfLoadImages; ++image)
  {
    // Copy the ROM image into my buffer
    for(j = 0; j < 8192; ++j)
      myImage[j] = myLoadImages[(image * 8448) + j];
 
    // Copy the Supercharger "header" from the image
    for(j = 0; j < 256; ++j)
      myHeader[j] = myLoadImages[(image * 8448) + 8192 + j];

    if(myHeader[5] == load)
    {
      setupROM();
      return;
    }
  }

  // Could find the requested load :-(
  Error err;
  err.message() << "Supercharger: Multiload image is missing in ROM image...";
  err.description() << "ROM image is corrupt or not complete." << endl;
  Throw(err);
}

