/*;-*-C-*-;
** Copyright (c) Massachusetts Institute of Technology 1994-1997.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. Works derived from this code are not distributed for
**        commercial gain without explicit permission from MIT 
**        (for permission contact lclint-request@larch.lcs.mit.edu).
*/
/*
**	 Copyright (c) Massachusetts Institute of Technology, 1993
**	    All Rights Reserved.  Unpublished rights reserved
**	    under the copyright laws of the United States.
**++
**  FACILITY:  LSLC
**
**  MODULE DESCRIPTION:
**
**      FILENAME: signature.y
**
**	PURPOSE:  bison grammar for LSL signatures.
** 
**  AUTHOR:
**	Yang Meng Tan, Massachusetts Institute of Technology
*/

%{

# include "bison.reset"

# include <stdio.h>
# include "lclintMacros.nf"
# include "llbasic.h"
# include "lslparse.h"
# include "signature.h"

void lslerror (char *s);
extern /*@dependent@*/ /*@null@*/ lslOp importedlslOp;

# include "bison.head"

extern unsigned int lsllex ();
static void yyprint (/* FILE *file, int type, YYSTYPE value */);
# define YYDEBUG 1
# define YYPRINT(file, type, value) yyprint (file, type, value)

%}

%pure_parser 

/* CONVENTIONS:  Reserved words are in ALL CAPS (plus markerSym)
		Characters appearing in the grammar are reserved:
						% ( ) , : \ */

/* Changes made
   - added LBRACKET stuff in OpForm (missing in LSL)
   - dif from LCL: open vs openSym, close vs closeSym
   - yylval vs yylval.tok
*/

%union {
  ltoken ltok;  /* a leaf is also an ltoken */
  unsigned int count;
  /*@only@*/  ltokenList ltokenList;
  /*@only@*/  opFormNode opform;
  /*@owned@*/ sigNode signature;
  /*@only@*/  nameNode name;
  /*@owned@*/ lslOp operator;
  /*@only@*/  lslOpList operators;
}

%token <ltok> SIMPLEID
%token <ltok> LOGICALOP         /* \implies, \and, \not, \or */
%token <ltok> LST_EQOP              /* \eq, \neq, ==, != */
%token <ltok> SIMPLEOP          /* opSym - reserved */
%token <ltok> MAPSYM            /* LSL's -> */
  /* FIELDMAPSYM = "\\field_arrow", only for LCL symtable import */
%token <ltok> FIELDMAPSYM       /* LCL's struct field operator -> */
%token <ltok> MARKERSYM         /* \marker, __ */
%token <ltok> ifTOKEN thenTOKEN elseTOKEN
  /* different from LSL: separate LBRACKET from openSym and
     RBRACKET from closeSym */
%token <ltok> LBRACKET RBRACKET
%token <ltok> SELECTSYM         /* \select or . */
%token <ltok> SEPSYM OPENSYM CLOSESYM
%token <ltok> COLON             /* : */
%token <ltok> COMMA             /* , */
%token <ltok> LST_EOL COMMENTSYM WHITESPACE 

%token WHITESPACE
%token QUANTIFIERSYM
%token LST_EQUATIONSYM
%token LST_EQSEPSYM
%token COMPOSESYM
%token LPAR
%token RPAR
%token assertsTOKEN
%token assumesTOKEN
%token byTOKEN
%token convertsTOKEN
%token enumerationTOKEN
%token equationsTOKEN
%token exemptingTOKEN
%token forTOKEN
%token generatedTOKEN 
%token impliesTOKEN 
%token includesTOKEN
%token introducesTOKEN
%token ofTOKEN
%token partitionedTOKEN
%token traitTOKEN
%token tupleTOKEN
%token unionTOKEN
%token BADTOKEN

%type <ltok> anyOp separator sortId opId
%type <count> middle placeList
%type <ltokenList> domain sortList
%type <opform> opForm
%type <signature> signature
%type <name> name 
%type <operator> operator
%type <operators> operatorList top

%%

top: operatorList { lslOpList_free ($1); } 
 
operatorList: operator 
              { lslOpList x = lslOpList_new ();
		g_importedlslOp = $1;
		lslOpList_add (x, $1);
 		$$ = x; }
            | operatorList operator
              { lslOpList_add ($1, $2);
		$$ = $1; } 

operator: name COLON signature 
          { $$ = makelslOpNode ($1, $3); }
/* The next production is never used in the output of lsl -syms 
          |  name
          { $$ = makelslOpNode ($1, (sigNode)0); } */
 
name: opId /* check for the case of if_then_else */
      { $$ = makeNameNodeId ($1); } 
    | opForm
      { $$ = makeNameNodeForm ($1); }
 
opForm 
     : ifTOKEN MARKERSYM thenTOKEN MARKERSYM elseTOKEN MARKERSYM
       { $$ = makeOpFormNode ($1, OPF_IF, opFormUnion_createMiddle (0), ltoken_undefined); }
     | anyOp
       { $$ = makeOpFormNode ($1, OPF_ANYOP, opFormUnion_createAnyOp ($1), ltoken_undefined); }
     | MARKERSYM anyOp
       { $$ = makeOpFormNode ($1, OPF_MANYOP, opFormUnion_createAnyOp ($2), ltoken_undefined); }
     | anyOp MARKERSYM
       { $$ = makeOpFormNode ($1, OPF_ANYOPM, opFormUnion_createAnyOp ($1), ltoken_undefined); }
     | MARKERSYM anyOp MARKERSYM
       { $$ = makeOpFormNode ($1, OPF_MANYOPM, opFormUnion_createAnyOp ($2), ltoken_undefined); }
     | OPENSYM middle CLOSESYM
       { $$ = makeOpFormNode ($1, OPF_MIDDLE, opFormUnion_createMiddle ($2), $3); }
     | MARKERSYM OPENSYM middle CLOSESYM
       { $$ = makeOpFormNode ($2, OPF_MMIDDLE, opFormUnion_createMiddle ($3), $4); }
     | OPENSYM middle CLOSESYM MARKERSYM
       { $$ = makeOpFormNode ($1, OPF_MIDDLEM, opFormUnion_createMiddle ($2), $3); }
     | MARKERSYM OPENSYM middle CLOSESYM MARKERSYM
       { $$ = makeOpFormNode ($2, OPF_MMIDDLEM, 
			      opFormUnion_createMiddle ($3), $4); }
     | LBRACKET middle RBRACKET
       { $$ = makeOpFormNode ($1, OPF_BMIDDLE, 
			      opFormUnion_createMiddle ($2), $3); }
     | MARKERSYM LBRACKET middle RBRACKET 
       { $$ = makeOpFormNode ($2, OPF_BMMIDDLE, 
			      opFormUnion_createMiddle ($3), $4); }
     | LBRACKET middle RBRACKET MARKERSYM
     { $$ = makeOpFormNode ($1, OPF_BMIDDLEM, 
			    opFormUnion_createMiddle ($2), $3); }
     | MARKERSYM LBRACKET middle RBRACKET MARKERSYM
     { $$ = makeOpFormNode ($2, OPF_BMMIDDLEM, 
			    opFormUnion_createMiddle ($3), $4); }
     | SELECTSYM SIMPLEID
     { $$ = makeOpFormNode ($1, OPF_SELECT, 
			    opFormUnion_createAnyOp ($2), ltoken_undefined); }
     | MARKERSYM SELECTSYM SIMPLEID
     { $$ = makeOpFormNode ($1, OPF_MSELECT, 
			    opFormUnion_createAnyOp ($3), ltoken_undefined); }
        /* not in LSL, need FILEDMAPSYM to distinguish it from MAPSYM */ 
        /* FIELDMAPSYM = "\\field_arrow", only for LCL symtable import */
     | MARKERSYM FIELDMAPSYM SIMPLEID
     { $$ = makeOpFormNode ($1, OPF_MMAP, 
			    opFormUnion_createAnyOp ($3), ltoken_undefined); }
			  

anyOp: SIMPLEOP
       { $$ = $1; }
     | LOGICALOP
       { $$ = $1; }
     | LST_EQOP
       { $$ = $1; }

middle: /* empty */
        { $$ = 0; }      
      | placeList
        { $$ = $1; }      
 
placeList: MARKERSYM
           { $$ = 1; }      
         | placeList separator MARKERSYM
           { $$ = $1 + 1; }      
 
separator: COMMA
           { $$ = $1; }      
         | SEPSYM
           { $$ = $1; }      

signature: domain MAPSYM sortId
           { $$ = makesigNode ($2, $1, $3); }

domain: /* empty */
        { $$ = ltokenList_new (); } 
      | sortList
        { $$ = $1; }
 
sortList: sortId
          { $$ = ltokenList_singleton ($1); } 
        | sortList COMMA sortId
          { $$ = ltokenList_push ($1, $3); }  

sortId: SIMPLEID 
        { 
	  $$ = $1; 
	  ltoken_setText ($$, processTraitSortId (ltoken_getText ($1))); 
	} 

opId: SIMPLEID
      { $$ = $1; }
%%

# include "bison.reset"

extern char *yytext;

void lslerror (char *s) 
{
  lclplainerror (message ("An error has occurred in parsing LSL signature: %s", 
			  cstring_fromChars (s)));
}

static void yyprint (FILE *file, int type, YYSTYPE value)
{
  fprintf (file, " (%u:%u type: %d; text: %s) ", 
	   ltoken_getLine (value.ltok), 
	   ltoken_getCol (value.ltok), 
	   type, 
	   ltoken_getRawTextChars (value.ltok));
}

extern void PrintToken (ltoken tok) {
  char *codStr;
  
  switch (ltoken_getCode (tok)) 
    {
    case NOTTOKEN:          codStr = "*** NOTTOKEN ***"; break;
    case QUANTIFIERSYM:     codStr = "QUANTIFIERSYM"; break;
    case LOGICALOP:         codStr = "LOGICALOP: "; break; 
    case SELECTSYM:         codStr = "SELECTSYM"; break;
    case OPENSYM:	    codStr = "OPENSYM"; break;
    case SEPSYM:	    codStr = "SEPSYM"; break;
    case CLOSESYM:	    codStr = "CLOSESYM"; break;
    case SIMPLEID:	    codStr = "SIMPLEID"; break;
    case MAPSYM:	    codStr = "MAPSYM"; break;
    case MARKERSYM:	    codStr = "MARKERSYM"; break;
    case COMMENTSYM:	    codStr = "COMMENTSYM"; break;
    case SIMPLEOP:	    codStr = "SIMPLEOP"; break;
    case COLON:             codStr = "COLON"; break;
    case COMMA:	            codStr = "COMMA"; break;
    case LBRACKET:	    codStr = "LBRACKET"; break;
    case LPAR:	            codStr = "LPAR"; break;
    case RBRACKET:	    codStr = "RBRACKET"; break;
    case RPAR:              codStr = "RPAR"; break;
    case LST_EQOP:          codStr = "LST_EQOP"; break;
    case WHITESPACE:	    codStr = "WHITESPACE,"; break;
    case LST_EOL:           codStr = "LST_EOL"; break;
    case elseTOKEN:         codStr = "elseTOKEN"; break;
    case ifTOKEN:           codStr = "ifTOKEN"; break;
    case thenTOKEN:         codStr = "thenTOKEN"; break;
    case BADTOKEN:          codStr = "*** BADTOKEN ***"; break;
    case LEOFTOKEN: /* can't reach LEOFTOKEN easily */
      codStr = "LEOFTOKEN"; break;
    default:
      codStr = "*** invalid token code ***";
      break;
    } /* end switch */
  
  /* only used for debugging */
  printf ("%u:%u: Token Code (%u): %s",
	  ltoken_getLine (tok), ltoken_getCol (tok), 
	  ltoken_getCode (tok), codStr);
  if (ltoken_getRawText (tok) != 0) 
    {
      printf (", Token String (%lu): %s\n", 
	      ltoken_getRawText (tok), ltoken_getRawTextChars (tok));
    }
  else printf ("\n");
}




