package biss.jde;

import biss.FileLib;
import java.io.File;

/**
 * class to parse the sources of a single method and to build a list if
 * FoldedStatement objects from it (as some kind of an AST)
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class MthdParser
  extends Parser
{

public MthdParser ( String src ) {
	super( src);
}

void block ( FoldedStmt stmt ) throws ParseException {
	CurNt = "block"; CurLn = Scan.Line;
	// logEnter( "block");

	// matchValChar( '{' )

	while ( Scan.Tok != 0 && Scan.Tok != '}' )
		simple_or_compound( stmt);

	Scan.nextToken();

	// logExit( "block");
}

void case_part ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "case"; CurLn = Scan.Line;
	// logEnter( "case_part");

	Scan.skipTo( ':');
	String cExpr = Scan.string();
	Scan.nextToken();

	/* semantic action */		FoldedStmt stmt = new FoldedStmt( JavaScanner.CASE, "case", cExpr);
	/* semantic action */		enclStmt.addFoldedStmt( stmt); 

	while ( Scan.Tok != 0 && Scan.Tok != '}' &&
	        Scan.Tok != JavaScanner.CASE && Scan.Tok != JavaScanner.DEFAULT) {
		simple_or_compound( stmt);
	}

	// logExit( "case_part");
}

void catch_stmt ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "catch"; CurLn = Scan.Line;
	// logEnter( "catch_stmt");

	// match( JavaScanner.CATCH);
	String cond = matchParenExpr();

	/* semantic action */		FoldedStmt stmt = new FoldedStmt( JavaScanner.CATCH, "catch", cond);
	/* semantic action */		enclStmt.addFoldedStmt( stmt); 

	if ( isMatchValChar( '{' ) )
		block( stmt);
	else
		simple_stmt( stmt);

	// logExit( "catch_stmt");
}

void default_part ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "default"; CurLn = Scan.Line;
	// logEnter( "default_part");

	Scan.nextToken(); Scan.nextToken();

	/* semantic action */		FoldedStmt stmt = new FoldedStmt( JavaScanner.DEFAULT, "default", null);
	/* semantic action */		enclStmt.addFoldedStmt( stmt); 

	while ( Scan.Tok != 0 && Scan.Tok != '}' )
		simple_or_compound( stmt);

	// logExit( "default_part");
}

void do_stmt ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "do"; CurLn = Scan.Line;
	// logEnter( "do_stmt");

	String expr;
	match( JavaScanner.DO);

	/* semantic action */		FoldedStmt stmt = new FoldedStmt( JavaScanner.DO, "do", null);
	/* semantic action */		enclStmt.addFoldedStmt( stmt); 

	if ( isMatchValChar( '{') )
		block( stmt);
	else
		simple_or_compound( stmt);

	match( JavaScanner.WHILE);
	String cond = matchParenExpr();
	matchValChar( ';');

	/* semantic action */		stmt.Paren = cond;

	// logExit( "do_stmt");
}

void else_stmt ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "else"; CurLn = Scan.Line;
	// logEnter( "else_stmt");

	/* semantic action */		FoldedStmt stmt = new FoldedStmt( JavaScanner.ELSE, "else", null);
	/* semantic action */		enclStmt.addFoldedStmt( stmt); 

	if ( isMatchValChar( '{') )
		block( stmt);
	else
		simple_stmt( stmt);

	// logExit( "else_stmt");
}

void elseif_stmt ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "elseif"; CurLn = Scan.Line;
	// logEnter( "elseif");

	String cond = matchParenExpr();

	/* semantic action */		FoldedStmt stmt = new FoldedStmt( JavaScanner.ELSE, "else if", cond);
	/* semantic action */		enclStmt.addFoldedStmt( stmt); 

	if ( isMatchValChar( '{') )
		block( stmt );
	else
		simple_stmt( stmt );

	// logExit( "elseif");
}

void finally_stmt ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "finally"; CurLn = Scan.Line;
	// logEnter( "finally_stmt");

	/* semantic action */		FoldedStmt stmt = new FoldedStmt( JavaScanner.FINALLY, "finally", null);
	/* semantic action */		enclStmt.addFoldedStmt( stmt); 

	if ( isMatchValChar( '{') )
		block( stmt);
	else
		simple_stmt( stmt);

	// logExit( "finally_stmt");
}

void for_stmt ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "for"; CurLn = Scan.Line;
	// logEnter( "for_stmt");

	String cond;

	match( JavaScanner.FOR);
	cond = matchParenExpr();

	/* semantic action */		FoldedStmt stmt = new FoldedStmt( JavaScanner.FOR, "for", cond);
	/* semantic action */		enclStmt.addFoldedStmt( stmt);

	if ( isMatchValChar( '{' ) )
		block( stmt);
	else
		simple_stmt( stmt);

	// logExit( "for_stmt");
}

void if_stmt ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "if"; CurLn = Scan.Line;
	// logEnter( "if_stmt");

	String cond;

	match( JavaScanner.IF);
	cond = matchParenExpr();

	/* semantic action */		FoldedStmt stmt = new FoldedStmt( JavaScanner.IF, "if", cond);
	/* semantic action */		enclStmt.addFoldedStmt( stmt); 

	if ( isMatchValChar( '{' ) )
		block( stmt);
	else
		simple_stmt( stmt);

	while ( isMatch( JavaScanner.ELSE) ) {
		if ( isMatch( JavaScanner.IF) )
			elseif_stmt( enclStmt);
		else
			else_stmt( enclStmt);
	}

	// logExit( "if_stmt");
}

public static void main ( String[] args ) {
	String src = FileLib.getFileDataAsString( new File(args[0]) );
	MthdParser p = new MthdParser( src);

	try {
		p.parseMethod();
	}
	catch ( ParseException x ) {
		p.reportException( x);
	}	
}

String matchParenExpr () throws ParseException {

	String s;
	if ( Scan.Tok != '(' )
		throw new ParseException( "(..) expected");

	Scan.skipBlock( '(', ')');
	s = Scan.string();
	Scan.nextToken();

	return s;
}

FoldedStmt  parseMethod () throws ParseException {
	CurNt = "method"; CurLn = Scan.Line;
	// logEnter( "method");

	String type, name, parm;

	Scan.nextToken();
	while ( isMatchSet( MorD_Modifier)) Scan.nextToken();

	type = matchString( "return type expected");
	if ( Scan.Tok == '(' ){		// ctor
		name = type;
		type = null;
	}
	else {
		name = matchString( "method name expected");
	}

	if ( Scan.Tok == '(' ) {
		Scan.skipBlock( '(', ')');
		parm = Scan.string();
		Scan.nextToken();
	}
	else
		throw new ParseException ( "(..) expected", Scan);

	/* semantic action */		FoldedStmt stmt = new FoldedStmt();
	/* semantic action */		stmt.Name = name; stmt.Paren = parm;

	if ( isMatch( JavaScanner.THROWS) )
		id_list();		

	if ( isMatchValChar( '{') )
		block( stmt);

	// logExit( "method");

	return stmt;
}

void simple_or_compound ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "stmt"; CurLn = Scan.Line;
	// logEnter( "simple_or_compound");

	switch ( Scan.Tok ) {
	case JavaScanner.IF:			if_stmt( enclStmt); break;
	case JavaScanner.FOR:			for_stmt( enclStmt); break;
	case JavaScanner.WHILE:		while_stmt( enclStmt); break;
	case JavaScanner.DO:			do_stmt( enclStmt); break;
	case JavaScanner.TRY:			try_stmt( enclStmt); break;
	case JavaScanner.SWITCH:  switch_stmt( enclStmt); break;
	default:
		simple_stmt( enclStmt);
	}

	// logExit( "simple_or_compound");
}

void simple_stmt ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "simple_stmt"; CurLn = Scan.Line;
	// logEnter( "simple_stmt");

	String txt;

	if ( Scan.Tok != ';' && Scan.skipTo( ';') )
		Scan.I++;

	txt = Scan.string();
	Scan.nextToken();

	/* semantic action */		enclStmt.addSimpleStmt( txt);

	// logExit( "simple_stmt");
}

void switch_stmt ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "switch"; CurLn = Scan.Line;
	// logEnter( "switch_stmt");

	match( JavaScanner.SWITCH);
	String cond = matchParenExpr();

	/* semantic action */		FoldedStmt stmt = new FoldedStmt( JavaScanner.SWITCH, "switch", cond);
	/* semantic action */		enclStmt.addFoldedStmt( stmt); 

	matchValChar( '{');

	while ( isMatch( JavaScanner.CASE) )
		case_part( stmt);

	if ( isMatch( JavaScanner.DEFAULT) )
		default_part( stmt);

	matchValChar( '}');

	// logExit( "switch_stmt");
}

void try_stmt ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "try"; CurLn = Scan.Line;
	// logEnter( "try_stmt");

	match( JavaScanner.TRY);

	/* semantic action */		FoldedStmt stmt = new FoldedStmt( JavaScanner.TRY, "try", null);
	/* semantic action */		enclStmt.addFoldedStmt( stmt); 

	if ( isMatchValChar( '{' ) )
		block( stmt);
	else
		simple_stmt( stmt);

	while ( isMatch( JavaScanner.CATCH) )
		catch_stmt( enclStmt);

	if ( isMatch( JavaScanner.FINALLY) )
		finally_stmt( enclStmt);

	// logExit( "try_stmt");
}

void while_stmt ( FoldedStmt enclStmt ) throws ParseException {
	CurNt = "while"; CurLn = Scan.Line;
	// logEnter( "while_stmt");

	String cond;

	match( JavaScanner.WHILE);
	cond = matchParenExpr();

	/* semantic action */		FoldedStmt stmt = new FoldedStmt( JavaScanner.WHILE, "while", cond);
	/* semantic action */		enclStmt.addFoldedStmt( stmt); 

	if ( isMatchValChar( '{' ) )
		block( stmt);
	else
		simple_stmt( stmt);

	// logExit( "while_stmt");
}
}
