/*****************************************************************************/
/*                                                                           */
/*  wmmail - WindowMaker Mail                                                */
/*                                                                           */
/*  Copyright (c) 1996, 1997, 1998  Per Liden                                */
/*  Copyright (c) 1997, 1998  Bryan Chan                                     */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                */
/*                                                                           */
/*    mh.c: functions to handle MH-style mail folders                        */
/*                                                                           */
/*****************************************************************************/

#include "wmmail.h"
#include <dirent.h>

/* 
 * This module contains routines to check an MH-style mail folder
 * during the periodic update of wmmail_status. We can classify a
 * given mail folder as being in one of the states listed below.
 *
 * The states and their corresponding effects on wmmail_status are:
 *
 *   - no folder or empty folder                NO_MAIL
 *   - no new mails                             OLD_MAIL
 *   - same number of new mails or fewer        no change
 *   - more new mails than last time            NEW_MAIL
 *
 * This module assumes that overriding automatically changes wmmail_status
 * to OLD_MAIL. Furthermore, the TimeStampMode setting does not affect 
 * MH-style mail folders.
 *
 */

/* internal function */
static int count_mail(char *, int *, int *);


void check_mh_folder(mbox, should_beep, should_redraw, should_run)
  Mailbox  *mbox;
  int      *should_beep,
           *should_redraw,
           *should_run;
{
  int prev_status, 
      prev_new_mail_count;

  prev_status = mbox->status;
  prev_new_mail_count = mbox->new_mail_count;

  if (count_mail(mbox->path, &mbox->mail_count, &mbox->new_mail_count))
  {
    if (mbox->mail_count == 0)
    {                               /* there is no mail in the mailbox */
      mbox->status = NO_MAIL;
    }
    else if (mbox->new_mail_count == 0)
    {                                        /* there are no new mails */
      mbox->status = OLD_MAIL;
    }
    else if (mbox->new_mail_count > prev_new_mail_count)
    {                                       /* new mails have arrived! */
      if (mbox->status == NEW_MAIL && always_new_mail_exec)
        *should_run |= True;
      else if (mbox->status == NEW_MAIL)
        *should_run |= False;
      else
        *should_run |= True;

      *should_beep = True;

      mbox->status = NEW_MAIL;
    }

    /* else no change */
  }
  else
    mbox->status = NO_MAIL;                         /* no such mailbox */

  *should_redraw |= (prev_status != mbox->status);
}


static int count_mail(char *path, int *mail_count, int *new_mail_count)
{
  FILE *seq;
  char buf[512], *tmpchar;
  int new_mail = 0, all_mail = 0, ismsg, first_unseen = 0, last_unseen = 0;
  DIR *folder;
  struct dirent *entry;

  /* This is a lie - For simplicity, we assume that there are no unseen
   * messages between seen messages (for now)
   */
  sprintf(buf, "%s/.mh_sequences", path);
  if (seq = fopen(buf, "r"))
  {
    while(fgets(buf, 511, seq))
    {
      char *end;
      end = strrchr(buf, '\0');
      while(!isdigit(*(--end)))
	*end = '\0';
      if(!strncmp(buf, "unseen:", 7))
      {
	tmpchar = buf+7;
	while(*tmpchar)
	{
          while(*tmpchar && !isdigit(*tmpchar))
	    tmpchar++;
	  if(!first_unseen || (atoi(tmpchar) < first_unseen))
	    first_unseen = atoi(tmpchar);
	  if(!last_unseen || (atoi(tmpchar) > last_unseen))
	    last_unseen = atoi(tmpchar);
          while(*tmpchar && isdigit(*tmpchar))
	    tmpchar++;
	}
      }
    }
    fclose(seq);
  }
  folder = opendir(path);
  while((entry = readdir(folder)) != NULL) {
    tmpchar = entry->d_name;
    ismsg = 1;
    while(*tmpchar)
    {
      isdigit(*tmpchar++) || (ismsg = 0);
    }

    if(ismsg)
    {
      all_mail++;
      if((atoi(entry->d_name) >= first_unseen) &&
	 (atoi(entry->d_name) <= last_unseen))
      {
	new_mail++;
      }
    }
  }
  closedir(folder);

  *mail_count = all_mail;
  *new_mail_count = new_mail;

  return True;
}
