/*
 * IceWM
 *
 * Copyright (C) 1997,98 Marko Macek
 *
 * TaskBar
 */

#include "icewm.h"

TaskBarApp::TaskBarApp(YFrameWindow *frame, YWindow *aParent, Window win): YWindow(aParent, win) {
    fFrame = frame;
    fPrev = fNext = 0;
    selected = 0;
    fShown = 1;
    setToolTip((char *)frame->client()->windowTitle());
}

TaskBarApp::~TaskBarApp() {
}

bool TaskBarApp::isFocusTraversable() {
    return true;
}

void TaskBarApp::setShown(int ashow) {
    if (ashow != fShown) {
        fShown = ashow;
    }
}

void TaskBarApp::paint(Graphics &g, int /*x*/, int /*y*/, unsigned int /*width*/, unsigned int /*height*/) {
    YColor *bg;
    YColor *fg;
    int p;

    if (frame()->isMinimized()) {
        bg = minimizedTaskBarAppBg;
        fg = minimizedTaskBarAppFg;
    } else {
        bg = normalTaskBarAppBg;
        fg = normalTaskBarAppFg;
    }

    if (frame()->focused() || selected == 2) {
        p = 2;
        if (frame()->focused()) {
            bg = activeTaskBarAppBg;
            fg = activeTaskBarAppFg;
        }
        g.setColor(bg);
        g.drawBorderW(0, 0, width() - 1, height() - 1, false);
    } else {
        p = 1;
        g.setColor(bg);
        g.drawBorderW(0, 0, width() - 1, height() - 1, true);
    }
    g.fillRect(p, p, width() - 3, height() - 3);
    if (frame()->clientIcon() && frame()->clientIcon()->small()) {
        int y = (height() - 3 - frame()->clientIcon()->small()->height()) / 2;
        g.drawMaskPixmap(frame()->clientIcon()->small(), p + 1, p + 1 + y);
    }

    char *str = (char *)frame()->client()->iconTitle();
    if (!str)
        str = (char *)frame()->client()->windowTitle();
    if (str) {
        g.setColor(fg);
        if (frame()->focused())
            g.setFont(activeTaskBarFont);
        else
            g.setFont(normalTaskBarFont);
        int ty = (height() - 1 + titleFont->height()) / 2 - titleFont->descent();
        if (ty < 2)
            ty = 2;
        g.drawChars((char *)str, 0, strlen(str),
                    p + 3 + 16,
                    p + ty);
        //(yheight() - font->height()) / 2 - titleFont->descent() - 4);
    }
}

void TaskBarApp::handleButton(const XButtonEvent &button) {
    YWindow::handleButton(button);
    if (button.button == 1 || button.button == 2) {
        if (button.type == ButtonPress) {
            selected = 2;
            repaint();
        } else if (button.type == ButtonRelease) {
            if (selected == 2) {
                if (button.button == 1) {
                    if (frame()->focused() || (button.state & ControlMask))
                        frame()->wmMinimize();
                    else {
                        if (button.state & ShiftMask)
                            frame()->wmOccupyOnlyWorkspace(app->activeWorkspace());
                        frame()->activate();
                        frame()->wmRaise();
                    }
                } else if (button.button == 2) {
                    if (frame()->focused() || (button.state & ControlMask))
                        frame()->wmLower();
                    else {
                        if (button.state & ShiftMask)
                            frame()->client()->setWorkspaces(frame()->client()->workspaces() |
                                                             (1 << app->activeWorkspace()));
                        frame()->activate();
                        frame()->activate();
                        frame()->wmRaise();
                    }
                }
            }
            selected = 0;
            repaint();
        }
    }
}

void TaskBarApp::handleCrossing(const XCrossingEvent &crossing) {
    if (selected > 0) {
        if (crossing.type == EnterNotify) {
            selected = 2;
            repaint();
        } else if (crossing.type == LeaveNotify) {
            selected = 1;
            repaint();
        }
    }
    YWindow::handleCrossing(crossing);
}

void TaskBarApp::handleClick(const XButtonEvent &/*down*/, const XButtonEvent &up, int /*count*/) {
    //if (up.button == 2)
    //    frame()->wmMinimize();
    //else
    if (up.button == 3) {
        frame()->updateMenu();
        frame()->windowMenu()->popup(0, frame(),
                                     up.x_root, up.y_root, -1, -1,
                                     YPopupWindow::pfCanFlipVertical |
                                     YPopupWindow::pfCanFlipHorizontal |
                                     YPopupWindow::pfPopupMenu);
    }
}

TaskBar::TaskBar(YWindow *aParent, Window win): YWindow(aParent, win) {
    unsigned int ht = 26;

#define BASE1 2
#define ADD1 3
#define BASE2 3
#define ADD2 5

    fWorkspaceButton = 0;
    if (taskBarShowWorkspaces) {
        fWorkspaceButton = (YButton **)malloc(sizeof(YButton *) * workspaceCount);
        assert(fWorkspaceButton != 0);
    }
    
    fFirst = fLast = 0;
    fCount = 0;
    setStyle(wsOverrideRedirect);

    fClock = new YClock(this);
    if (fClock->height() + ADD1 > ht) ht = fClock->height() + ADD1;
    if (showMailBoxStatus)
        mailBoxStatus = new MailBoxStatus(mailBoxPath, mailCommand, this);
    fApplications = new YButton(this, rootMenu);
    fApplications->setPixmap(startPixmap);
    if (fApplications->height() + ADD1 > ht) ht = fApplications->height() + ADD1;
    fWinList = new YButton(this, windowListMenu);
    fWinList->setPixmap(windowsPixmap);
    if (fWinList->height() + ADD1 > ht) ht = fWinList->height() + ADD1;

    if (taskBarShowWorkspaces) {
        for (int w = 0; w < workspaceCount; w++) {
            YButton *wk = new YButton(this, cmdActivateWorkspace, (void *)w);
            wk->setText(workspaceNames[w]);
            if (wk->height() + ADD2 > ht) ht = wk->height() + ADD2;
            fWorkspaceButton[w] = wk;
        }
    }
    setGeometry(-1,
                taskBarAtTop ? -1 : int(app->root()->height() - ht),
                app->root()->width() + 2, ht + 1);

    leftX = 2;
    rightX = width() - 4;
    if (showTaskBarClock) {
        fClock->setPosition(rightX - fClock->width(),
                            BASE1 + (ht - ADD1 - fClock->height()) / 2);
        fClock->show();
        rightX -= fClock->width() + 2;
    }
    if (mailBoxStatus) {
        mailBoxStatus->setPosition(rightX - mailBoxStatus->width() - 1,
                                   BASE2 + (ht - ADD2 - mailBoxStatus->height()) / 2);
	mailBoxStatus->show();
        rightX -= mailBoxStatus->width() + 3;
    }
    leftX += 2;
    fApplications->setPosition(leftX,
                               BASE1 + (ht - ADD1 - fApplications->height()) / 2);
    fApplications->show();
    leftX += fApplications->width();
    //leftX += 2;
    fWinList->setPosition(leftX,
                         BASE1 + (ht - ADD1 - fWinList->height()) / 2);
    fWinList->show();
    leftX += fWinList->width() + 2;

    if (taskBarShowWorkspaces) {
        leftX += 2;
        for (int w = 0; w < workspaceCount; w++) {
            YButton *wk = fWorkspaceButton[w];
            //leftX += 2;
            wk->setSize(wk->width(), ht - ADD2);
            wk->setPosition(leftX, BASE2); // + (ht - ADD - wk->height()) / 2);
            wk->show();
            leftX += wk->width();
        }
        leftX += 4;
    }
}

TaskBar::~TaskBar() {
    delete fClock; fClock = 0;
    delete mailBoxStatus; mailBoxStatus = 0;
    if (fWorkspaceButton) {
        for (int w = 0; w < workspaceCount; w++)
            delete fWorkspaceButton[w];
        delete fWorkspaceButton;
    }
}

void TaskBar::paint(Graphics &g, int /*x*/, int /*y*/, unsigned int /*width*/, unsigned int /*height*/) {
    g.setColor(taskBarBg);
    g.draw3DRect(0, 0, width() - 1, height() - 1, true);
    g.fillRect(1, 1, width() - 2, height() - 2);
}

bool TaskBar::handleKey(const XKeyEvent &key) {
    return YWindow::handleKey(key);
}

void TaskBar::handleButton(const XButtonEvent &button) {
    if ((button.type == ButtonRelease) &&
        (button.button == 1 || button.button == 3) &&
        (button.state == Button1Mask + Button3Mask))
    {
        if (windowList)
            windowList->showFocused();
    } else if (button.type == ButtonPress) {
        if (button.button == 1) {
            if (button.state & Mod1Mask)
                lower();
            else if (!(button.state & ControlMask))
                raise();
        }
    }
    YWindow::handleButton(button);
}

void TaskBar::handleClick(const XButtonEvent &down, const XButtonEvent &up, int count) {
    if (up.button == 1) {
    } else if (up.button == 2) {
    } else if (up.button == 3 && count == 1) {
        rootMenu->popup(0, 0, up.x_root, up.y_root, -1, -1,
                        YPopupWindow::pfCanFlipVertical |
                        YPopupWindow::pfCanFlipHorizontal);
    }
    YWindow::handleClick(down, up, count);
}

void TaskBar::handleDrag(const XButtonEvent &/*down*/, const XMotionEvent &motion) {
#ifndef NO_CONFIGURE
    int newPosition = 0;
    
    if (motion.y_root < int(app->root()->height() / 2))
        newPosition = 1;
    
    if (taskBarAtTop != newPosition) {
        taskBarAtTop = newPosition;
        setPosition(x(), taskBarAtTop ? -1 : int(app->root()->height() - height() + 1));

        app->relocateWindows(0, taskBarAtTop ? height() : - height());
    }
#endif
}

void TaskBar::insert(TaskBarApp *tapp) {
    fCount++;
    tapp->setNext(0);
    tapp->setPrev(fLast);
    if (fLast)
        fLast->setNext(tapp);
    else
        fFirst = tapp;
    fLast = tapp;
}

void TaskBar::remove(TaskBarApp *tapp) {
    fCount--;

    if (tapp->getPrev())
        tapp->getPrev()->setNext(tapp->getNext());
    else
        fFirst = tapp->getNext();

    if (tapp->getNext())
        tapp->getNext()->setPrev(tapp->getPrev());
    else
        fLast = tapp->getPrev();
}

TaskBarApp *TaskBar::addApp(YFrameWindow *frame) {
    if (frame->client() == windowList)
        return 0;

    TaskBarApp *tapp = new TaskBarApp(frame, this);

    if (tapp != 0) {
        insert(tapp);
        tapp->show();
        if (!frame->visibleOn(app->activeWorkspace()) &&
            !taskBarShowAllWindows)
            tapp->setShown(0);
        relayout();
    }
    return tapp;
}

void TaskBar::removeApp(YFrameWindow *frame) {
    TaskBarApp *f = fFirst, *next;

    while (f) {
        next = f->getNext();
        if (f->frame() == frame) {
            f->hide();
            remove(f);
            delete f;
            relayout();
            return ;
        }
        f = next;
    }
}

void TaskBar::relayout() {
    int x, y, w, h;
    int tc = 0;

    TaskBarApp *a = fFirst;
    
    while (a) {
        if (a->getShown())
            tc++;
        a = a->getNext();
    }

    if (tc < 3) tc = 3;

    h = height() - ADD2 - 1;
    w = (rightX - leftX - 2) / tc - 2;
    x = leftX;
    y = BASE2 + (height() - ADD2 - 1 - h) / 2;

    TaskBarApp *f = fFirst;
        
    while (f) {
        if (f->getShown()) {
            f->setGeometry(x, y, w, h);
            f->show();
            x += w;
            x += 2;
        } else
            f->hide();
        f = f->getNext();
    }
}

void TaskBar::popupStartMenu() {
    if (fApplications) {
        /*requestFocus();
        fApplications->requestFocus();
        fApplications->setFocus();*/
        fApplications->popupMenu();
    }
}

void TaskBar::popupWindowListMenu() {
    if (fWinList) {
        fWinList->popupMenu();
    }
}

