/*
 * IceWM
 *
 * Copyright (C) 1997 Marko Macek
 */

#include "icewm.h"

int configurationLoaded = 0;

#define CFGDEF
#include "default.h"

char *menuFile = 0;
char *programFile = 0;

char **progNames = 0;
char **progCmds = 0;
char ***progArgs = 0;
int progCount = 0;

int workspaceCount = 0;
char *workspaceNames[MAXWORKSPACES];

const char *clrBlack                  = "rgb:00/00/00";
const char *clrWhite                  = "rgb:FF/FF/FF";

int is_reg(const char *path) {
    struct stat sb;

    if (stat(path, &sb) != 0)
        return 0;
    if (!S_ISREG(sb.st_mode))
        return 0;
    return 1;
}

int findPath(const char *path, int mode, const char *name, char **fullname) {
#ifdef __EMX__
    char tmp[1024];
    strcpy(tmp, name);
    if (mode & X_OK)
        strcat(tmp, ".exe");
    name = tmp;
#endif
    
    if (strchr(name, '/') != 0
#ifdef __EMX__
        || strchr(name, '\\') != 0
#endif
       )
    {
#ifdef __EMX__
        if (access(name, 0) == 0) {
            *fullname = strdup(name);
            return 1;
        }
#else
        if (access(name, mode) == 0 && is_reg(name)) {
            *fullname = strdup(name);
            return 1;
        }
#endif
    } else {
        if (path == 0)
            return 0; // ??? assume some default path?

        char prog[1024];
        const char *p, *q;
        unsigned int len, nameLen = strlen(name);

        if (nameLen > sizeof(prog))
            return 0;

        p = path;
        while (*p) {
            q = p;
            while (*p && *p != PATHSEP)
                p++;

            len = p - q;

            if (len > 0 && len < sizeof(prog) - nameLen - 2) {
                strncpy(prog, q, len);
                if (!ISSLASH(prog[len - 1]))
                    prog[len++] = SLASH;
                strcpy(prog + len, name);

#ifdef DEBUG_P
                if (debug)
                    msg("checking program path %s\n", prog);
#endif
#ifdef __EMX__
                if (access(prog, 0) == 0) {
                    *fullname = strdup(prog);
                    return 1;
                }
#else
                if (access(prog, mode) == 0 && is_reg(prog)) {
                    *fullname = strdup(prog);
                    return 1;
                }
#endif
            }
            if (*p == PATHSEP)
                p++;
        }
    }
    return 0;
}

int addProgram(const char *name, const char *command, char **args) {
    char *fullname = 0;

    if (command && command[0] && findPath(getenv("PATH"), X_OK, command, &fullname) == 0) { // updates command with full path
        char **p = args;
        while (p && *p) {
            free(*p);
            p++;
        }
        free(args);

        fprintf(stderr, "Program %s (%s) not found.", name, command);
        return 0;
    }

    progNames = (char **)realloc((void *)progNames, sizeof(char *) * ((progCount) + 1));
    progCmds = (char **)realloc((void *)progCmds, sizeof(char *) * ((progCount) + 1));
    progArgs = (char ***)realloc((void *)progArgs, sizeof(char **) * ((progCount) + 1));

    assert(progNames != NULL);
    assert(progCmds != NULL);
    assert(progArgs != NULL);

    progNames[progCount] = name ? strdup(name) : 0;
    progCmds[progCount] = fullname ? fullname : command ? strdup(command) : 0;
    progArgs[progCount] = args;

    /*assert(progNames[progCount] != NULL);
    assert(progCmds[progCount] != NULL);*/
    
    progCount++;
    return 1;
}

void addWorkspace(const char *name) {
    if (workspaceCount >= MAXWORKSPACES)
        return;
    workspaceNames[workspaceCount] = strdup(name);
    assert(workspaceNames[workspaceCount] != NULL);
    workspaceCount++;
}

char *getArgument(char *dest, int maxLen, char *p) {
    char *d;
    int len = 0;
    int in_str = 0;
    
    while (*p && (*p == ' ' || *p == '\t'))
        p++;

    d = dest;
    len = 0;
    while (*p && len < maxLen - 1 &&
           (in_str || (*p != ' ' && *p != '\t' && *p != '\n')))
    {
        if (in_str && *p == '\\' && p[1]) {
            p++;
            *d++ = *p++;
            len++;
        } else if (*p == '"') {
            in_str = !in_str;
            p++;
        } else {
            *d++ = *p++;
            len++;
        }
    }
    *d = 0;

    return p;
}

#ifndef NO_CONFIGURE


char *setOption(char *name, char *arg, char *rest) {
    unsigned int a;

#ifdef DEBUG_O
    if (debug) printf("SET %s := %s ;\n", name, arg);
#endif
    
    for (a = 0; a < ACOUNT(bool_options); a++)
        if (strcmp(name, bool_options[a].option) == 0) {
            if ((arg[0] == '1' || arg[0] == '0') && arg[1] == 0)
                *(bool_options[a].value) = (arg[0] == '1') ? 1 : 0;
            else {
                fprintf(stderr, "bad argument: %s for %s\n", arg, name);
                return 0;
            }
            return rest;
        }

    for (a = 0; a < ACOUNT(uint_options); a++)
        if (strcmp(name, uint_options[a].option) == 0) {
            int v = atoi(arg);
            
            if (v >= uint_options[a].min && v <= uint_options[a].max)
                *(uint_options[a].value) = v;
            else {
                fprintf(stderr, "bad argument: %s for %s\n", arg, name);
                return 0;
            }
            return rest;
        }
            
    for (a = 0; a < ACOUNT(string_options); a++)
        if (strcmp(name, string_options[a].option) == 0) {
            if (!string_options[a].initial) {
                free((char *)*string_options[a].value);
                string_options[a].initial = 0;
            }
            *string_options[a].value = strdup(arg);
            return rest;
        }
    if (strcmp(name, "AddWorkspace") == 0) {
        addWorkspace(arg);
        return rest;
    } else if (strcmp(name, "Look") == 0) {
#ifdef CONFIG_LOOK_WARP4
        if (strcmp(arg, "warp4") == 0)
            wmLook = lookWarp4;
        else
#endif
#ifdef CONFIG_LOOK_WARP3
        if (strcmp(arg, "warp3") == 0)
            wmLook = lookWarp3;
        else
#endif
#ifdef CONFIG_LOOK_WIN95
        if (strcmp(arg, "win95") == 0)
            wmLook = lookWin95;
        else
#endif
#ifdef CONFIG_LOOK_MOTIF
        if (strcmp(arg, "motif") == 0)
            wmLook = lookMotif;
        else
#endif
#ifdef CONFIG_LOOK_NICE
        if (strcmp(arg, "nice") == 0)
            wmLook = lookNice;
        else
#endif
#ifdef CONFIG_LOOK_PIXMAP
        if (strcmp(arg, "pixmap") == 0)
            wmLook = lookPixmap;
        else
#endif
        {
            fprintf(stderr, "bad Look name\n");
            return 0;
        }
        return rest;
    }
    fprintf(stderr, "bad option: %s\n", name);
    return 0;
}

// parse option name and argument
// name is string without spaces up to =
// option is a " quoted string or characters up to next space
char *parseOption(char *str) {
    char name[64];
    char argument[256];
    char *p = str;
    unsigned int len = 0;

    while (*p && *p != '=' && *p != ' ' && len < sizeof(name) - 1)
        p++, len++;

    strncpy(name, str, len);
    name[len] = 0;

    while (*p && *p != '=')
        p++;
    if (*p != '=')
        return 0;
    p++;

    p = getArgument(argument, sizeof(argument), p);

    return setOption(name, argument, p);
}

void parseConfiguration(char *data) {
    char *p = data;

    while (p && *p) {
        while (*p == ' ' || *p == '\t' || *p == '\n' || (*p == '\\' && p[1] == '\n'))
            p++;
        
        if (*p != '#')
            p = parseOption(p);
        else {
            while (*p && *p != '\n') {
                if (*p == '\\' && p[1] != 0)
                    p++;
                p++;
            }
        }
    }
}

void loadConfiguration(const char *fileName) {
    int fd = open(fileName, O_RDONLY | O_TEXT);

    if (fd == -1)
        return ;

    struct stat sb;

    if (fstat(fd, &sb) == -1)
        return ;

    char *buf = (char *)malloc(sb.st_size + 1);
    if (buf == 0)
        return ;

    if (read(fd, buf, sb.st_size) != sb.st_size)
        return;

    buf[sb.st_size] = 0;
    close(fd);
    parseConfiguration(buf);
    free(buf);
    
    configurationLoaded = 1;
}
#endif

char *getWord(char *word, int maxlen, char *p) {
    while (*p && (*p == ' ' || *p == '\t' || *p == '\r' || *p == '\n'))
        p++;
    while (*p && isalnum(*p) && maxlen > 1) {
        *word++ = *p++;
        maxlen--;
    }
    *word++ = 0;
    return p;
}

char *parseMenus(char *data, YMenu *menu) {
    char *p = data;
    char word[32];

    if (menu == 0)
        return 0;
    
    while (p && *p) {
        p = getWord(word, sizeof(word), p);

        if (strcmp(word, "separator") == 0) {
            menu->addSeparator();
        } else if (strcmp(word, "prog") == 0 || strcmp(word, "restart") == 0) {
            char name[64];
            char icons[128];
            int cmd = (strcmp(word, "restart") == 0) ? cmdRestart : cmdExec;

            p = getArgument(name, sizeof(name), p);
            if (p == 0)
                return p;

            p = getArgument(icons, sizeof(icons), p);
            if (p == 0)
                return p;

            char command[256];
            char **args = 0;
            int argCount = 0;
            
            p = getArgument(command, sizeof(command), p);
            if (p == 0) {
                fprintf(stderr, "missing 2nd argument for %s %s\n", word, name);
                return p;
            }
            
            while (p) {
                char argx[256];
                
                while (*p && (*p == ' ' || *p == '\t'))
                    p++;
                
                if (*p == '\n')
                    break;
                
                p = getArgument(argx, sizeof(argx), p);
                if (p == 0) {
                    fprintf(stderr, "missing bad argument %d for %s %s\n", argCount + 1, word, name);
                    return p;
                }
                
                if (args == 0) {
                    args = (char **)realloc((void *)args, ((argCount) + 2) * sizeof(char *));
                    assert(args != NULL);
                    
                    args[argCount] = strdup(command);
                    assert(args[argCount] != NULL);
                    args[argCount + 1] = NULL;
                    
                    argCount++;
                }
                
                args = (char **)realloc((void *)args, ((argCount) + 2) * sizeof(char *));
                assert(args != NULL);
                
                args[argCount] = strdup(argx);
                assert(args[argCount] != NULL);
                args[argCount + 1] = NULL;
                
                argCount++;
            }
            if (!p)
                fprintf(stderr, "missing 2nd argument for prog %s\n", name);
            else {
                if (addProgram(name, command, args)) {
                    
                    YIcon *icon = 0;
                    
                    if (icons[0] != '-')
                        icon = getIcon(icons);
                    
                    YMenu::YMenuItem *item =
                        new YMenu::YMenuItem(name, -1, "",
                                             (WMCommand)cmd, 0, (void *)(progCount - 1));
                    
                    if (icon)
                        item->setPixmap(icon->small());
                    
                    menu->add(item);
                }
            }
        } else if (strcmp(word, "menu") == 0) {
            char name[64];
            char icons[128];
            
            p = getArgument(name, sizeof(name), p);

            p = getArgument(icons, sizeof(icons), p);
            if (p == 0)
                return p;

            p = getWord(word, sizeof(word), p);
            if (*p != '{')
                return 0;
            p++;

            YMenu *newMenu = new YMenu(app->root());
            if (newMenu) {
                p = parseMenus(p, newMenu);
                if (newMenu->itemCount() == 0)
                    delete newMenu;
                else {
                    YIcon *icon = 0;
                    
                    if (icons[0] != '-')
                        icon = getIcon(icons);

                    //menu->addSubmenu(name, -1, newMenu);

                    YMenu::YMenuItem *item =
                        new YMenu::YMenuItem(name, -1, 0,
                                             cmdSubmenu, newMenu, 0);
                    
                    if (icon)
                        item->setPixmap(icon->small());

                    menu->add(item);
                }
            } else
                return p;
        } else if (*p == '}') {
            p++;
            return p;
        } else {
            return 0;
        }
    }
    return p;
}


void loadMenus(const char *menuFile, bool programs) {
    if (menuFile == 0)
        return ;
    
    int fd = open(menuFile, O_RDONLY | O_TEXT);

    if (fd == -1)
        return ;

    struct stat sb;

    if (fstat(fd, &sb) == -1)
        return ;

    char *buf = (char *)malloc(sb.st_size + 1);
    if (buf == 0)
        return ;

    if (read(fd, buf, sb.st_size) != sb.st_size)
        return;

    buf[sb.st_size] = 0;
    close(fd);

    if (programs) {
        YMenu *programs = new YMenu(app->root());

        parseMenus(buf, programs);
        rootMenu->addSeparator();
        YMenu::YMenuItem *item =
            new YMenu::YMenuItem("Programs", 0, 0, cmdSubmenu, programs, 0);

        rootMenu->add(item);
    } else
        parseMenus(buf, rootMenu);
    
    free(buf);

    /*for (int p = 0; p < progCount; p++)
        if (progNames[p] && progNames[p][0])
            rootMenu->addItem(progNames[p], 0, "", cmdExec, (void *)p);
        else
            rootMenu->addSeparator();*/
}

#ifndef NO_WINDOW_OPTIONS
WindowOption *winOptions = 0;
int winOptionsCount = 0;
char *winOptFile = 0;

WindowOption *getWindowOption(char *name, int create) {
    int L = 0, R = winOptionsCount, M, cmp;

    while (L < R) {
        M = (L + R) / 2;
        if (name == 0 && winOptions[M].name == 0)
            cmp = 0;
        else if (name == 0)
            cmp = -1;
        else if (winOptions[M].name == 0)
            cmp = 1;
        else
            cmp = strcmp(name, winOptions[M].name);
        if (cmp == 0)
            return winOptions + M;
        else if (cmp > 0)
            L = M + 1;
        else
            R = M;
    }
    if (!create)
        return 0;

    WindowOption *newOptions =
        (WindowOption *)realloc(winOptions,
                                sizeof(winOptions[0]) * (winOptionsCount + 1));
    if (newOptions == 0)
        return 0;
    winOptions = newOptions;

    memmove(winOptions + L + 1, winOptions + L, sizeof(winOptions[0]) * (winOptionsCount - L));
    winOptionsCount++;

    memset(winOptions + L, 0, sizeof(WindowOption));
    winOptions[L].workspace = -1;
    winOptions[L].name = name;
    return winOptions + L;
}

void setWinOption(char *class_instance, char *opt, char *arg) {
    WindowOption *op = getWindowOption(class_instance, 1);

    if (debug)
        fprintf(stderr, "%s-%s-%s\\", class_instance, opt, arg);
    if (strcmp(opt, "icon") == 0) {
        op->icon = strdup(arg);
    } else {
        int t = 0;
        if (atoi(arg) == 1)
            t = 1;

        {
            static struct {
                int what;
                const char *name;
                unsigned long flag;
            } options[] = {
                { 0, "fMove", YFrameWindow::ffMove },
                { 0, "fResize", YFrameWindow::ffResize },
                { 0, "fClose", YFrameWindow::ffClose },
                { 0, "fMinimize", YFrameWindow::ffMinimize },
                { 0, "fMaximize", YFrameWindow::ffMaximize },
                { 0, "fHide", YFrameWindow::ffHide },
                { 0, "fShade", YFrameWindow::ffRollup}, // !!! remove soon
                { 0, "fRollup", YFrameWindow::ffRollup },
                { 1, "dTitleBar", YFrameWindow::fdTitleBar },
                { 1, "dSysMenu", YFrameWindow::fdSysMenu },
                { 1, "dBorder", YFrameWindow::fdBorder },
                { 1, "dResize", YFrameWindow::fdResize },
                { 1, "dClose", YFrameWindow::fdClose },
                { 1, "dMinimize", YFrameWindow::fdMinimize },
                { 1, "dMaximize", YFrameWindow::fdMaximize },
                { 1, "dHide", YFrameWindow::fdHide },
                { 2, "onTop", YFrameWindow::foOnTop },
                { 2, "allWorkspaces", YFrameWindow::foAllWorkspaces },
                { 2, "ignoreTaskBar", YFrameWindow::foIgnoreTaskBar },
                { 2, "ignoreWinList", YFrameWindow::foIgnoreWinList },
                { 2, "fullKeys", YFrameWindow::foFullKeys },
                { 3, "workspace", 0 },
            };
            for (unsigned int a = 0; a < ACOUNT(options); a++) {
                unsigned long *what, *what_mask;
            
                if (options[a].what == 0) {
                    what = &op->functions;
                    what_mask  = &op->function_mask;
                } else if (options[a].what == 1) {
                    what = &op->decors;
                    what_mask = &op->decor_mask;
                } else if (options[a].what == 2) {
                    what = &op->options;
                    what_mask = &op->option_mask;
                } else if (options[a].what == 3) {
                    op->workspace = atoi(arg);
                    break;
                } else
                    abort();

                if (*what < 3 && strcmp(opt, options[a].name) == 0) {
                    if (t)
                        *what |= options[a].flag;
                    else
                        *what &= ~options[a].flag;
                    *what_mask |= options[a].flag;
                }
            }
        }
    }
}

char *parseWinOptions(char *data) {
    char *p = data;
    char *w, *e, *c;
    char *class_instance;

    char *opt;

    while (*p) {
        while (*p == ' ' || *p == '\t' || *p == '\n')
            p++;
        w = p;
        c = 0;
        while (*p && *p != ':') {
            if (*p == '.')
                c = p;
            p++;
        }
        e = p;

        if (e == w || *p == 0)
            break;
        if (c == 0) {
            fprintf(stderr, "icewm: syntax error in window options\n");
            break;
        }
        
        if (c - w + 1 == 0)
            class_instance = 0;
        else {
            class_instance = (char *)malloc(c - w + 1);
            if (class_instance == 0)
                goto nomem;
            memcpy(class_instance, w, c - w);
            class_instance[c - w] = 0;
        }

        *e = 0;
        c++;
        opt = c;
        e++;

        p = e;
        while (*p == ' ' || *p == '\t')
            p++;

        w = p;
        while (*p && (*p != '\n' && *p != ' ' && *p != '\t'))
            p++;

        if (*p != 0) {
            *p = 0;
            setWinOption(class_instance, opt, w);
        } else {
            setWinOption(class_instance, opt, w);
            break;
        }
        p++;
    }
    return p;
nomem:
    fprintf(stderr, "icewm: out of memory for window options\n");
    return 0;
}

void loadWinOptions(const char *optFile) {
    if (optFile == 0)
        return ;
    
    int fd = open(optFile, O_RDONLY | O_TEXT);

    if (fd == -1)
        return ;

    struct stat sb;

    if (fstat(fd, &sb) == -1)
        return ;

    char *buf = (char *)malloc(sb.st_size + 1);
    if (buf == 0)
        return ;

    if (read(fd, buf, sb.st_size) != sb.st_size)
        return;

    buf[sb.st_size] = 0;
    close(fd);

    parseWinOptions(buf);
    
    free(buf);
}
#endif
