//<copyright>
//
// Copyright (c) 1995-96
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
//</copyright>


//<file>
//
//
// Name :       jpegimage.C
//
// Purpose :    Implementation of class JpegImage
//
// Created :    04 Jan 95    Bernhard Marschall
//
// $Id: jpegimage.C,v 1.7 1997/02/19 09:20:14 bmarsch Exp $
//
// $Log: jpegimage.C,v $
// Revision 1.7  1997/02/19 09:20:14  bmarsch
// Changed return value of HgRasterCallback::getData()
//
// Revision 1.6  1996/10/02 13:28:35  bmarsch
// verbose.h was moved from hyperg to utils
//
// Revision 1.5  1996/03/05 09:28:38  bmarsch
// Set progress to 100% before calling last drawing callback
//
// Revision 1.4  1996/02/29 12:42:32  bmarsch
// Always call decodedInterlaced() callback after a pass (also
// in finishDecode())
//
// Revision 1.3  1996/01/18 16:28:18  bmarsch
// Removed warnings
//
//</file>

#include <stdlib.h>
#include <setjmp.h>

#include "hgrastercb.h"
#include "jpegimage.h"

// otherwise InterViews's boolean and libjpeg's boolean collide
#include <InterViews/enter-scope.h>
#define HAVE_BOOLEAN

#include "../jpeg/jpeglib.h"
#include "../jpeg/jinclude.h"
#include "../jpeg/jerror.h"

//#define VERBOSE
#include <hyperg/utils/verbose.h>


// ************************************************************************
// bmarsch: these static variables are needed for the C (JPEG library) to
// C++ (callback) interface

static HgRasterCallback* the_callback = nil;
static char jpegError[256];


// ************************************************************************
// JpegImpl:

struct jpeg_error_manager {
  struct jpeg_error_mgr pub;
  jmp_buf jmpState;
};

typedef struct jpeg_error_manager* jpeg_error_ptr;

class JpegImpl {
public:
  struct jpeg_decompress_struct cinfo_;   // info struct for decoding
  struct jpeg_error_manager jerr_;        // error manager
  JSAMPARRAY buffer_;                     // buffer to decode into
};


// ************************************************************************
// bmarsch: the source manager is taken from jdatasrc.c and modified,
// so that it waits for new data, when the input file is empty

/* Expanded data source object for stdio input */

typedef struct {
  struct jpeg_source_mgr pub;	/* public fields */

  FILE * infile;		/* source stream */
  JOCTET * buffer;		/* start of buffer */
  boolean start_of_file;	/* have we gotten any data yet? */
} my_source_mgr;

typedef my_source_mgr * my_src_ptr;

#define INPUT_BUF_SIZE  4096	/* choose an efficiently fread'able size */


/*
 * Initialize source --- called by jpeg_read_header
 * before any data is actually read.
 */

METHODDEF void
init_source (j_decompress_ptr cinfo)
{
  my_src_ptr src = (my_src_ptr) cinfo->src;

  /* We reset the empty-input-file flag for each image,
   * but we don't clear the input buffer.
   * This is correct behavior for reading a series of images from one source.
   */
  src->start_of_file = TRUE;
}


/*
 * Fill the input buffer --- called whenever buffer is emptied.
 *
 * In typical applications, this should read fresh data into the buffer
 * (ignoring the current state of next_input_byte & bytes_in_buffer),
 * reset the pointer & count to the start of the buffer, and return TRUE
 * indicating that the buffer has been reloaded.  It is not necessary to
 * fill the buffer entirely, only to obtain at least one more byte.
 *
 * There is no such thing as an EOF return.  If the end of the file has been
 * reached, the routine has a choice of ERREXIT() or inserting fake data into
 * the buffer.  In most cases, generating a warning message and inserting a
 * fake EOI marker is the best course of action --- this will allow the
 * decompressor to output however much of the image is there.  However,
 * the resulting error message is misleading if the real problem is an empty
 * input file, so we handle that case specially.
 *
 * In applications that need to be able to suspend compression due to input
 * not being available yet, a FALSE return indicates that no more data can be
 * obtained right now, but more may be forthcoming later.  In this situation,
 * the decompressor will return to its caller (with an indication of the
 * number of scanlines it has read, if any).  The application should resume
 * decompression after it has loaded more data into the input buffer.  Note
 * that there are substantial restrictions on the use of suspension --- see
 * the documentation.
 *
 * When suspending, the decompressor will back up to a convenient restart point
 * (typically the start of the current MCU). next_input_byte & bytes_in_buffer
 * indicate where the restart point will be if the current call returns FALSE.
 * Data beyond this point must be rescanned after resumption, so move it to
 * the front of the buffer rather than discarding it.
 */

METHODDEF boolean
fill_input_buffer (j_decompress_ptr cinfo)
{
  my_src_ptr src = (my_src_ptr) cinfo->src;
  size_t nbytes;

// bmarsch: here is the actual modification !!!
tryagain:
  nbytes = JFREAD(src->infile, src->buffer, INPUT_BUF_SIZE);
  if (nbytes <= 0) {
    if (the_callback && the_callback->loading()) {
      if (!the_callback->getData()) {
        if (src->start_of_file) {   /* Treat empty input file as fatal error */
          ERREXIT(cinfo, JERR_INPUT_EMPTY);
        }
        WARNMS(cinfo, JWRN_JPEG_EOF);
        /* Insert a fake EOI marker */
        src->buffer[0] = (JOCTET) 0xFF;
        src->buffer[1] = (JOCTET) JPEG_EOI;
        nbytes = 2;
      }
      goto tryagain;
    }
    else {
      if (src->start_of_file) {   /* Treat empty input file as fatal error */
        ERREXIT(cinfo, JERR_INPUT_EMPTY);
      }
      WARNMS(cinfo, JWRN_JPEG_EOF);
      /* Insert a fake EOI marker */
      src->buffer[0] = (JOCTET) 0xFF;
      src->buffer[1] = (JOCTET) JPEG_EOI;
      nbytes = 2;
    }
  }

  src->pub.next_input_byte = src->buffer;
  src->pub.bytes_in_buffer = nbytes;
  src->start_of_file = FALSE;

  return TRUE;
}


/*
 * Skip data --- used to skip over a potentially large amount of
 * uninteresting data (such as an APPn marker).
 *
 * Writers of suspendable-input applications must note that skip_input_data
 * is not granted the right to give a suspension return.  If the skip extends
 * beyond the data currently in the buffer, the buffer can be marked empty so
 * that the next read will cause a fill_input_buffer call that can suspend.
 * Arranging for additional bytes to be discarded before reloading the input
 * buffer is the application writer's problem.
 */

METHODDEF void
skip_input_data (j_decompress_ptr cinfo, long num_bytes)
{
  my_src_ptr src = (my_src_ptr) cinfo->src;

  /* Just a dumb implementation for now.  Could use fseek() except
   * it doesn't work on pipes.  Not clear that being smart is worth
   * any trouble anyway --- large skips are infrequent.
   */
  if (num_bytes > 0) {
    while (num_bytes > (long) src->pub.bytes_in_buffer) {
      num_bytes -= (long) src->pub.bytes_in_buffer;
      (void) fill_input_buffer(cinfo);
    }
    src->pub.next_input_byte += (size_t) num_bytes;
    src->pub.bytes_in_buffer -= (size_t) num_bytes;
  }
}


/*
 * An additional method that can be provided by data source modules is the
 * resync_to_restart method for error recovery in the presence of RST markers.
 * For the moment, this source module just uses the default resync method
 * provided by the JPEG library.  That method assumes that no backtracking
 * is possible.
 */


/*
 * Terminate source --- called by jpeg_finish_decompress
 * after all data has been read.  Often a no-op.
 *
 * NB: *not* called by jpeg_abort or jpeg_destroy; surrounding
 * application must deal with any cleanup that should happen even
 * for error exit.
 */

METHODDEF void
term_source (j_decompress_ptr /*cinfo*/)
{
  /* no work necessary here */
}


/*
 * Prepare for input from a stdio stream.
 * The caller must have already opened the stream, and is responsible
 * for closing it after finishing decompression.
 */

GLOBAL void
jpeg_waitio_src (j_decompress_ptr cinfo, FILE * infile)
{
  my_src_ptr src;

  /* The source object and input buffer are made permanent so that a series
   * of JPEG images can be read from the same file by calling jpeg_stdio_src
   * only before the first one.  (If we discarded the buffer at the end of
   * one image, we'd likely lose the start of the next one.)
   * This makes it unsafe to use this manager and a different source
   * manager serially with the same JPEG object.  Caveat programmer.
   */
  if (cinfo->src == NULL) {	/* first time for this JPEG object? */
    cinfo->src = (struct jpeg_source_mgr *)
      (*cinfo->mem->alloc_small) ((j_common_ptr) cinfo, JPOOL_PERMANENT,
				  SIZEOF(my_source_mgr));
    src = (my_src_ptr) cinfo->src;
    src->buffer = (JOCTET *)
      (*cinfo->mem->alloc_small) ((j_common_ptr) cinfo, JPOOL_PERMANENT,
				  INPUT_BUF_SIZE * SIZEOF(JOCTET));
  }

  src = (my_src_ptr) cinfo->src;
  src->pub.init_source = init_source;
  src->pub.fill_input_buffer = fill_input_buffer;
  src->pub.skip_input_data = skip_input_data;
  src->pub.resync_to_restart = jpeg_resync_to_restart; /* use default method */
  src->pub.term_source = term_source;
  src->infile = infile;
  src->pub.bytes_in_buffer = 0; /* forces fill_input_buffer on first read */
  src->pub.next_input_byte = NULL; /* until buffer loaded */
}

// ************************************************************************
// bmarsch: here come my error handlers: they just aborts the decoder
// and tells the viewer which error has occured

METHODDEF void my_error_exit(j_common_ptr cinfo)
{
  jpeg_error_ptr err = (jpeg_error_ptr) cinfo->err; // I know, it's a jpeg_error_ptr
  err->pub.output_message(cinfo);
  longjmp(err->jmpState, 1);
}

METHODDEF void my_error_output(j_common_ptr cinfo)
{
  jpeg_error_ptr err = (jpeg_error_ptr) cinfo->err; // I know, it's a jpeg_error_ptr
  err->pub.format_message(cinfo, jpegError);
  if (the_callback)
    the_callback->error(jpegError);
}

// ************************************************************************

//#define NUMROWS 16

JpegImage::JpegImage(FILE* fp, const char* fname, Raster*& raster,
                     HgRasterCallback* cb, boolean dither)
: RasterImage(fp, fname, raster, cb, dither)
{
  DEBUGNL("JI::JI");
  the_callback = cb;

  impl_ = new JpegImpl();
  struct jpeg_decompress_struct& cinfo = impl_->cinfo_;
  struct jpeg_error_manager& jerr = impl_->jerr_;

  // set up error handler & create decompressor
  cinfo.err = jpeg_std_error(&jerr.pub);
  jerr.pub.error_exit = my_error_exit;
  jerr.pub.output_message = my_error_output;
  jpeg_create_decompress(&cinfo);

  // set up the input file
  jpeg_waitio_src(&cinfo, file_);

  // read header, set paramters, and start compression
  /*(void)*/ jpeg_read_header(&cinfo, 1);
  cinfo.buffered_image = true;   // want to display progressive JPEG's "live"
  jpeg_start_decompress(&cinfo);

  // alloc output buffer
  int row_stride = cinfo.output_width * cinfo.output_components;
  impl_->buffer_ = (*cinfo.mem->alloc_sarray) ((j_common_ptr) &cinfo,
                                               JPOOL_IMAGE,
                                               row_stride,
                                               1);

  setSize(cinfo.image_width, cinfo.image_height);

  // start reading first component
  jpeg_start_output(&cinfo, cinfo.input_scan_number);
}

// ************************************************************************

JpegImage::~JpegImage()
{
  delete impl_;
}

// ************************************************************************

void JpegImage::decode(int numrows)
{
  DEBUGNL("JI::decode(" << numrows << "); decoded_ = " << decoded_);

  // do nothing if an error has occured
  if (error_) return;

  struct jpeg_decompress_struct& cinfo = impl_->cinfo_;
  JSAMPARRAY& buffer = impl_->buffer_;

  // set entry point for recovery from errors
  if (setjmp(impl_->jerr_.jmpState)) {
    DEBUGNL("**** long jumped");
    jpeg_destroy_decompress(&cinfo);
    error_ = 1;
    return;
  }

  int maxrow = decoded_ + numrows;
  if (maxrow > cinfo.output_height) maxrow = cinfo.output_height;
  boolean interlaced = cinfo.progressive_mode;
  int num_passes = interlaced ? 10 : 1; // todo: determine number of
                                        // passes (I'm not sure if it can be done?)
  float incp = 1.0 / (cinfo.output_height * num_passes);

  while (cinfo.output_scanline < maxrow) {
    if (callback_)
      callback_->increaseProgress(incp);

    jpeg_read_scanlines(&cinfo, buffer, 1);
    put_pixel_rows(1);

    // live display of sequential JPEG
    decoded_++;
    if (callback_ && !interlaced && decoded_ % 20 == 0)
      callback_->decodedSequential(20);
  }

  // one pass finished
  if (decoded_ == cinfo.output_height) {
    jpeg_finish_output(&cinfo);

    int complete = jpeg_input_complete(&cinfo);
    // when finish set progress to 100%
    if (complete && callback_)
      callback_->increaseProgress(1.0);
    // display finished pass
    if (interlaced) {
      if (callback_) callback_->decodedInterlaced();
    }
    else {
      if (callback_) callback_->decodedSequential(decoded_ % 20);
    }

    // start next pass
    if (!complete) {
      jpeg_start_output(&cinfo, cinfo.input_scan_number);
      decoded_ = 0;
    }
  }
}

// ************************************************************************

void JpegImage::finishDecode()
{
  DEBUGNL("JI::finishDecode()");

  struct jpeg_decompress_struct& cinfo = impl_->cinfo_;

  int h = height();
  int rows_left = h - decoded_;
  if (rows_left > 0) decode(rows_left);
  while (!jpeg_input_complete(&cinfo))
    decode(h);

  // do nothing if an error has occured
  if (error_) return;

  jpeg_finish_decompress(&cinfo);
  jpeg_destroy_decompress(&cinfo);
}

// ************************************************************************

void JpegImage::abortDecode()
{
  DEBUGNL("JI::abortDecode");

  struct jpeg_decompress_struct& cinfo = impl_->cinfo_;
  jpeg_abort_decompress(&cinfo);
  jpeg_destroy_decompress(&cinfo);
}

// ************************************************************************
// bmarsch: this function copies the decompress data from the decoding
// buffer into RasterImage's data array

void JpegImage::put_pixel_rows(int num_rows)
{
  JSAMPROW ptr0;
  Byte* pixptr;

  struct jpeg_decompress_struct& cinfo = impl_->cinfo_;
  JSAMPARRAY& buffer = impl_->buffer_;

  if (cinfo.out_color_space == JCS_GRAYSCALE) {
    for (int row = 0; row < num_rows; row++) {
      ptr0 = buffer[row];
      pixptr = (Byte*) data() + (decoded_+row) * width() * 3;
      for (int col = width(); col > 0; col--) {
        unsigned char gray = GETJSAMPLE(*ptr0++);
        *pixptr++ = gray;
        *pixptr++ = gray;
        *pixptr++ = gray;
      }
    }
  }

  else {
    for (int row = 0; row < num_rows; row++) {
      ptr0 = buffer[row];
      pixptr = (Byte*) data() + (decoded_+row) * width() * 3;
      for (int col = width(); col > 0; col--) {
        *pixptr++ = GETJSAMPLE(*ptr0++);
        *pixptr++ = GETJSAMPLE(*ptr0++);
        *pixptr++ = GETJSAMPLE(*ptr0++);
      }
    }
  }
}
