/*
 *      Copyright (C) 1997 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 *     This program is a small utility to manipulate certain fields of
 *     the volume table of a floppy tape cartridge.  To be used with
 *     the QIC-40/80/3010/3020 floppy-tape driver "ftape" for Linux.
 */

char src[] = "$Source: /homes/cvs/ftape-stacked/contrib/vtblc/vtblc.c,v $";
char rev[] = "$Revision: 1.8 $";
char dat[] = "$Date: 1997/10/31 00:51:56 $";

#include <sys/types.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <getopt.h>
#include <ctype.h>
#include <stdarg.h>
#include <time.h>

#include <linux/ftape.h>
#include <linux/ftape-vendors.h>
#include <linux/zftape.h>
#include <linux/ftape-header-segment.h>

#include "vtblc.h"

static char *tape_dev = FTAPE_DEF_TAPE;
static char *progname = NULL;

static vtbl volumes[MAX_VOLUMES];
static u_int8_t vtbl_buffer[FT_SEGMENT_SIZE];
static int num_volumes;

static const char *short_options = "f:Vdvh?pa:t::m:#:";
static const struct option long_options[] =
{
	{"file",               1, 0, 'f'}, /* full path to device */
	{"version",            0, 0, 'V'}, /* version information */
	{"debug",              0, 0, 'd'}, /* switch on debugging, unused */
	{"verbose",            0, 0, 'v'}, /* verbosity level, unused */ 
	{"usage",              0, 0, 'h'},
	{"help",               0, 0, 'h'},
	{"print",              0, 0, 'p'}, /* pretty print the vtbl */
	{"append",             1, 0, 'a'}, /* append an entry at the end */
	{"truncate",           2, 0, 't'}, /* truncate to given size */
	{"modify",             1, 0, 'm'}, /* modify given vtbl */
	{"vtbl-entry",         1, 0, '#'}, /* vtbl to print or modify */ 
	{0, 0, 0, 0}
};

const char *vtbl_str_tags[] =
{
	"label",
	"date",
	"start",
	"end",
	NULL
};

static enum {
	vtbl_none  = -1,
	vtbl_label = 0,
	vtbl_date  = 1,
	vtbl_start = 2,
	vtbl_end   = 3
} vtbl_tag = vtbl_none;

static enum {
	none     = 0,
	append   = 1,
	modify   = 2,
	trunc    = 3
} op_mode = none;

static int vtbl_no      = -1;
static char *label      = NULL;
static char *datestr    = NULL;
static int setdate      = 0;
static int start_seg    = -1;
static int end_seg      = -1;
static int vtbl_print   = 0;
static int vtbl_maxsize = -1;
static int verbose      = 0;
static int debug        = 0;

static void usage(FILE *file, int exit_val, char *message, ...);
static int tape_open(const char *name, int rd_only);
static int tape_close(int tape_fd);
static int read_vtbl(int tape_fd, vtbl *volumes, u_int8_t *buffer,
					 int *first_data_segment, int *last_data_segment);
static int write_vtbl(int tape_fd, const vtbl *volumes, u_int8_t *buffer,
					  int vtbl_cnt, int first_data_segment, int write);
static int read_header_segment(const int tape_fd,
							   int *first_data_segment,
							   int *last_data_segment);
static void print_vtbl(const vtbl *volumes, int maxnum,
					   int first_seg, int last_seg);
static void print_one_vtbl(const vtbl *volumes,
						   int vtbl_no, int first_seg, int last_seg);
static char *decode_date(u_int32_t timestamp);
static int set_date(/* const */ char *date,
					vtbl *volumes, int maxnum, int vtbl_no);
static int set_label(const char *desc, vtbl *volumes, int maxnum, int vtbl_no);
static int set_start(int end, vtbl *volumes, int maxnum, int vtbl_no);
static int set_end(int end, vtbl *volumes, int maxnum, int vtbl_no);
static int set_id(vtbl *volumes, int maxnum, int vtbl_no);

int main(int argc, char *argv[])
{
	char *subopts, *value;
	int c;
	int tapefd;
	int first_seg, last_seg;
	progname = argv[0];

	while (1) {
		int option_index = 0;
		
		c = getopt_long(argc, argv, short_options, long_options,&option_index);

		if (c == -1) {
            break;
		}

		switch(c) {
		case 0:
			break;
		case '#':
			vtbl_no = strtol(optarg, NULL, 0);
			break;
		case 'a':
		case 'm':
			subopts = optarg;
			while (subopts && *subopts != '\0') {
				vtbl_tag = getsubopt(&subopts, vtbl_str_tags, &value);
				if (value == NULL && vtbl_tag != vtbl_date && vtbl_tag != -1) {
					usage(stderr, 1, "The option \"%s\" needs a value!\n",
						  vtbl_str_tags[vtbl_tag]);
				}
				switch(vtbl_tag) {
				case vtbl_label:
					label = value;
					break;
				case vtbl_date:
					datestr = value;
					setdate = 1;
					break;
				case vtbl_start:
					start_seg = strtol(value, NULL, 0);
					break;
				case vtbl_end:
					end_seg = strtol(value, NULL, 0);
					break;
				default:
					usage(stderr, 1, "Unknown option: %s\n", value);
					break;
				}
			}
			if (c == 'a') {
				op_mode = append;
			} else {
				op_mode = modify;
			}
			break;
		case 'p':
			vtbl_print = 1;
			break;
		case 't':
			if (optarg) vtbl_maxsize = strtol(optarg, NULL, 0);
			op_mode = trunc;
			break;
		case 'f':
			tape_dev = optarg;
			break;
		case 'V':
			printf("vtblc " VTBLC_VERSION "\n");
			exit(0);
			break;
		case 'd':
			debug ++;
			break;
		case 'v':
			verbose ++;
			break;
		case 'h':
		case '?':
			usage(stdout, 0, NULL);
			break;
		default:
			fprintf (stderr, "?? getopt returned character code 0%o ??\n", c);
			exit(1);
			break;
		}
	}
	if (op_mode == none) {
		vtbl_print ++;
	}
	if ((tapefd = tape_open(tape_dev, op_mode == none)) == -1) {
		exit(1);
	}
	if ((num_volumes = read_vtbl(tapefd, volumes, vtbl_buffer,
								 &first_seg, &last_seg)) == -1 ) {
		(void)tape_close(tapefd);
		exit(1);
	}
	if (op_mode == append) {
		if (++num_volumes > MAX_VOLUMES) {
			fprintf(stderr, "Too many volumes: %d (%d max)\n", 
					num_volumes, MAX_VOLUMES);
			(void)tape_close(tapefd);
			exit(1);
		}
		if (end_seg == -1 || start_seg == -1) {
			fprintf(stderr, "Missing start- or ending segment\n");
		}
		op_mode = modify;
		vtbl_no = -1;
	}
	if (op_mode == modify) {
		if (setdate && set_date(datestr, volumes, num_volumes, vtbl_no)) {
			(void)tape_close(tapefd);
			exit(1);
		}
		if (label && set_label(label, volumes, num_volumes, vtbl_no)) {
			(void)tape_close(tapefd);
			exit(1);
		}
		if (start_seg != -1 &&
			set_start(start_seg, volumes, num_volumes, vtbl_no)) {
			(void)tape_close(tapefd);
			exit(1);
		}
		if (end_seg != -1 &&
			set_end(end_seg, volumes, num_volumes, vtbl_no)) {
			(void)tape_close(tapefd);
			exit(1);
		}
		if (set_id(volumes, num_volumes, vtbl_no)) {
			(void)tape_close(tapefd);
			exit(1);
		}
	} else if (op_mode == trunc) {
		if (vtbl_maxsize == -1) {
			vtbl_maxsize = num_volumes - 1;
		}
		if (vtbl_maxsize < 0 || vtbl_maxsize > num_volumes) {
			fprintf(stderr, "Volume number too big or negative: %d\n", 
					vtbl_no);
			(void)tape_close(tapefd);
			exit(1);
		}
		num_volumes = vtbl_maxsize;
	}
	if (vtbl_print) {
		print_vtbl(volumes, num_volumes, first_seg, last_seg);
	}
	if (write_vtbl(tapefd, volumes, vtbl_buffer, num_volumes, first_seg,
				   op_mode == trunc)) {
		(void)tape_close(tapefd);
		exit(1);
	}
	if (tape_close(tapefd)) {
		exit(1);
	} else {
		exit(0);
	}
	exit(0);
}

#define VTBLC_HELP_TEXT														  \
VTBLC_VERSION "\n"															  \
"\n"																		  \
"Usage: %s [OPTIONS]\n"														  \
"Manipulate the volume table of a floppy tape cartridge, for use with\n"	  \
FTAPE_VERSION", "ZFTAPE_VERSION"\n"											  \
"\n"																		  \
"Mandatory or optional arguments to long options are mandatory or optional\n" \
"for short options too. Unique abbreviations for long options are "			  \
"accepted.\n"																  \
"\n"																		  \
"  -f, --file=FILE       Tape device to use. Default is "					  \
"\""FTAPE_DEF_TAPE"\".\n"													  \
"  -h, --help            Print this help.\n"								  \
"  -?                    Same as \'-h\'.\n"									  \
"      --usage           Same as \'-h\'.\n"									  \
"  -V, --version         Print version information.\n"						  \
"  -d, --debug           Unused yet.\n"										  \
"  -v, --verbose         Unused yet.\n"										  \
"  -#, --vtbl-entry=NR   Specify the volume number for \"--print\" and\n"	  \
"                        \"--modify\".\n"									  \
"  -p, --print           Print the volume table entry given by\n"			  \
"                        \"--vtbl-entry\" or the entire table if\n"			  \
"                        \"--vtbl-entry\" has been omitted.\n"				  \
"                        If no other action is specified then \"--print\"\n"  \
"                        is the default.\n"									  \
"  -t, --truncate[=SIZE] Truncate the volume table to SIZE entries by\n"	  \
"                        filling the remainder with zero bytes. If SIZE\n"	  \
"                        is omitted then the last entry is deleted.\n"		  \
"  -a, --append          Append an entry to the end of the volume table.\n"	  \
"  -m, --modify          Modify the volume table entry given by\n"			  \
"                        \"--vtbl-entry\". If \"--vtbl-entry\" has been\n"	  \
"                        omitted then the last entry is modified.\n"		  \
"\n"																		  \
"\"--apend\" and \"--modify\" understand the following sub-options:\n"		  \
"\n"																		  \
"      label=LABEL       The description for this volume table entry.\n"	  \
"                        LABEL will be truncated to 44 bytes.\n"			  \
"      date[=DATE]       The time stamp for this entry. DATE is parsed\n"     \
"                        by strptime(3) using the \"%%T %%D\" format (same\n" \
"                        as \"%%H:%%M:%%S %%m/%%d/%%y\").\n"	              \
"                        If DATE is omitted the current local time is used\n" \
"                        instead.\n"				                       	  \
"      start=SEG_ID      The starting segment for this volume table entry.\n" \
"      end=SEG_ID        The final segment for this volume table entry.\n"	  \
"\n"																		  \
"WARNING:\n"																  \
"\"start=SEG_ID\" and \"end=SEG_ID\" "										  \
"should be used with extreme prejudice.\n"									  \
"They are mainly intended for testing purposes to quickly create some\n"	  \
"dummy volumes to test the file mark support of ftape.\n"

static void usage(FILE *file, int exit_val, char *message, ...)
{
	va_list ap;
	if (message) {
		va_start(ap, message);
		vfprintf(file, message, ap);
		va_end(ap);
	}
	fprintf(file, VTBLC_HELP_TEXT, progname);
	exit(exit_val);
}

/*
 *  open the tape device, try to determine whether is is a floppy tape,
 *  and store the hardware status in drive_config, resp. tape_config.
 *  mmap() the dma buffers if op_mode != PROBING
 *  
 *  Returns the tape fd on success, -1 otherwise
 *
 */

static int tape_open(const char *name, int rd_only)
{
	int tape_fd;
	struct mtget mtget;
	static vendor_struct vendors[] = QIC117_VENDORS;
	vendor_struct *vendor;

	/* open the tape device
	 */
	if ((tape_fd = open(name, rd_only ? O_RDONLY : O_RDWR)) == -1) {
		perror("Error opening tape dev");
		return -1;
	}
	/* get its status
	 */
	if (ioctl(tape_fd, MTIOCGET, &mtget) == -1) {
		perror("Error getting tape drive status");
		(void)close(tape_fd);
		return -1;
	}
	if (GMT_DR_OPEN(mtget.mt_gstat)) {
		fprintf(stderr, "Error: No tape cartridge present!\n");
		(void)close(tape_fd);
		return -1;
	}
	if (GMT_WR_PROT(mtget.mt_gstat)) {
		fprintf(stderr, "Warning: Write protected cartridge!\n");
	}
	if (!GMT_ONLINE(mtget.mt_gstat)) {
		fprintf(stderr, "Error: Tape drive is offline!\n");
		(void)close(tape_fd);
		return -1;
	}
	if ((mtget.mt_type & MT_ISFTAPE_FLAG) == 0) {
		fprintf(stderr, "Error: This is not a floppy tape drive!\n");
		(void)close(tape_fd);
		return -1;
	}
	mtget.mt_type &= ~MT_ISFTAPE_FLAG; /* clear the flag bit */
	if (verbose > 0) {
		vendor = &vendors[0];
		while (vendor->vendor_id != UNKNOWN_VENDOR &&
			   vendor->vendor_id != mtget.mt_type) {
			vendor++;
		}
		printf("Tape drive type: %s (0x%04lx)\n", vendor->name, mtget.mt_type);
	}
	return tape_fd;
}

static int tape_close(int tape_fd)
{
	const struct mtop rewind = { MTREW, 1 };
	int result = 0;

	if (ioctl(tape_fd, MTIOCTOP, &rewind) == -1) {
		fprintf(stderr, "Ioctl error rewinding tape: %s\n", strerror(errno));
		result = -1;
	}
	if (close(tape_fd) == -1) {
		fprintf(stderr, "Error closing tape device: %s\n", strerror(errno));
		result = -1;
	}
	return result;
}

static int read_vtbl(int tape_fd, vtbl *volumes, u_int8_t *buffer,
					 int *first_data_segment, int *last_data_segment)
{
	const struct mtop rewind = { MTREW, 1 };
	struct mtftseg ft_seg;
	int vtbl_cnt; 
	int end_seg = 0;
	int fmt_code;
	const char *ids[] = VTBL_IDS;

	if (ioctl(tape_fd, MTIOCTOP, &rewind) == -1) {
		fprintf(stderr, "Ioctl error rewinding tape: %s\n", strerror(errno));
		return -1;
	}
	if ((fmt_code = read_header_segment(tape_fd,
											first_data_segment,
											last_data_segment)) == -1) {
		return -1;
	}
	if (verbose) {
		printf("Reading volume table segment ... ");
		fflush(stdout);
	}
	memset(&ft_seg, 0, sizeof(ft_seg));
	ft_seg.mt_data  = buffer;
	ft_seg.mt_segno = *first_data_segment;
	ft_seg.mt_mode  = MT_FT_RD_SINGLE;
	if (ioctl(tape_fd, MTIOCRDFTSEG, &ft_seg) == -1) {
		fprintf(stderr, "Ioctl error reading volume table: %s\n",
				strerror(errno));
		return -1;
	}
	if (ft_seg.mt_result != FT_SEGMENT_SIZE) {
		fprintf(stderr, "Short read() reading volume table: %d\n",
				ft_seg.mt_result);
		return -1;
	}
	if (verbose) {
		printf("done.\n");
	}
	vtbl_cnt = 0;
	if (fmt_code == fmt_big) {
		end_seg = *first_data_segment;
	}
	while (!memcmp(buffer, ids[0], 4) ||
		   !memcmp(buffer, ids[1], 4) ||
		   !memcmp(buffer, ids[2], 4) ||
		   !memcmp(buffer, ids[3], 4)) {
		memcpy(volumes[vtbl_cnt].id, buffer, 4);
		if (!memcmp(buffer, "VTBL", 4)) {
			memcpy(&volumes[vtbl_cnt].date, buffer + VTBL_DATE, 4);
			memcpy(volumes[vtbl_cnt].label, buffer + VTBL_DESC, 44);
			volumes[vtbl_cnt].label[44] = '\0';
			if (fmt_code == fmt_big) {
				volumes[vtbl_cnt].start = end_seg + 1;
				memcpy(&volumes[vtbl_cnt].end, buffer + VTBL_SCSI_SEGS, 4);
				volumes[vtbl_cnt].end += end_seg;
				end_seg = volumes[vtbl_cnt].end;
			} else {
				memcpy(&volumes[vtbl_cnt].start, buffer + VTBL_START, 2);
				memcpy(&volumes[vtbl_cnt].end,   buffer + VTBL_END,   2);
				volumes[vtbl_cnt].start &= 0x0000ffff;
				volumes[vtbl_cnt].end   &= 0x0000ffff;
			}
			volumes[vtbl_cnt].fmt_code = fmt_code;
		}
		vtbl_cnt ++;
		buffer += VTBL_SIZE;
	}
	return vtbl_cnt;
}

/*  returns the max. number of segments
 */
static int read_header_segment(int tape_fd,
							   int *first_data_segment, int *last_data_segment)
{
		struct mtftseg ft_seg;
		int i;
		const unsigned long hseg_magic = FT_HSEG_MAGIC;
		u_int8_t hseg[FT_SEGMENT_SIZE];
		
		if (verbose) {
			printf("Reading header segment ... ");
			fflush(stdout);
		}
		for (i = 0; i < 64; i++) {
			memset(&ft_seg, 0, sizeof(ft_seg));
			ft_seg.mt_data  = hseg;
			ft_seg.mt_segno = i;
			ft_seg.mt_mode  = MT_FT_RD_AHEAD;
			if (ioctl(tape_fd, MTIOCRDFTSEG, &ft_seg) == -1) {
				fprintf(stderr, "Ioctl error reading header segment: %s\n",
						strerror(errno));
				return -1;
			}
			if (ft_seg.mt_result == FT_SEGMENT_SIZE) {
				break;
			}
		}
		if (memcmp(&hseg_magic, hseg, 4)) {
			fprintf(stderr, "Unable to read the header segment\n");
			fprintf(stderr, "Consider using \"--header-discard\"\n");
			return -1;
		}
		if (verbose) {
			printf("done.\n");
		}
		if (hseg[FT_FMT_CODE] == fmt_big) {
			memcpy(first_data_segment, &hseg[FT_6_FRST_SEG], 4);
			memcpy(last_data_segment,  &hseg[FT_6_LAST_SEG], 4);
		} else {
			memcpy(first_data_segment, &hseg[FT_FRST_SEG], 2);
			memcpy(last_data_segment,  &hseg[FT_LAST_SEG], 2);
			*first_data_segment &= 0x0000ffff;
			*last_data_segment &= 0x0000ffff;
		}
		return (int)hseg[FT_FMT_CODE];
}

static void print_vtbl(const vtbl *volumes, int maxnum,
					   int first_seg, int last_seg)
{
	int i;

	printf("%3s %3s %*s %*s %*s %8s %8s\n",
		   "Nr", "Id", 14, "Label", 22, "Date", 15, "Start", "End", "Space");
	for (i = 0; i < 80; i++) {
		printf("-");
	}
	printf("\n");
	for (i = 0; i < maxnum; i++) {
		print_one_vtbl(volumes, i, first_seg, last_seg);
	}
}

static void print_one_vtbl(const vtbl *volumes,
						   int vtbl_no, int first_seg, int last_seg)
{
	char label[45];
	char sig[5];
	double usage;

	memcpy(label, volumes[vtbl_no].label, 44);
	label[22] = '\0';
	memcpy(sig, volumes[vtbl_no].id, 4);
	sig[4] = '\0';
	if (!strcmp(sig, "VTBL")) {
		usage = (double)(volumes[vtbl_no].end - volumes[vtbl_no].start + 1);
		usage = usage/(double)(last_seg - first_seg)*100.0;
		printf("%3d %4s \"%-*s\" %*s %8d %8d    %2.2f%%\n",
			   vtbl_no, sig, 22, label, 18, decode_date(volumes[vtbl_no].date),
			   volumes[vtbl_no].start, volumes[vtbl_no].end, usage);
	} else {
		printf("%4d %4s", vtbl_no, sig);
	}
}

static char *decode_date(u_int32_t timestamp)
{
	struct tm tapetm;
	time_t tapetime;
	static char date[18];

	memset(&tapetm, 0, sizeof(tapetm));
	
	tapetm.tm_year  = timestamp >> FT_YEAR_SHIFT;
	tapetm.tm_year += FT_YEAR_0 - 1900;
	timestamp      &= FT_TIME_MASK;

	tapetm.tm_sec   = timestamp % 60;
	timestamp      /= 60;
	tapetm.tm_min   = timestamp % 60;
	timestamp      /= 60;
	tapetm.tm_hour  = timestamp % 24;
	timestamp      /= 24;
	tapetm.tm_mday  = timestamp % 31 + 1;
	timestamp      /= 31;
	tapetm.tm_mon   = timestamp % 12;
	tapetm.tm_isdst = -1;
	if ((tapetime = mktime(&tapetm)) == (time_t)(-1)) {
		return "invalid";
	} else {
		(void)strftime(date, 18, "%T %D", &tapetm);
		return date;
	}
}

static int write_vtbl(int tape_fd, const vtbl *volumes, u_int8_t *buffer,
					  int vtbl_cnt, int first_data_segment, int modified)
{
	const struct mtop rewind = { MTREW, 1 };
	struct mtftseg ft_seg;
	int i;
	u_int8_t *p = buffer;

	for (i = 0; i < vtbl_cnt; i++) {
		if (volumes[i].modified) {
			memcpy(p + VTBL_SIG, volumes[i].id, 4);
			if (volumes[i].fmt_code == fmt_big) {
				u_int32_t segs = volumes[i].end - volumes[i].start + 1;
				memcpy(p + VTBL_SCSI_SEGS, &segs, 4);
			} else {
				memcpy(p + VTBL_START, &volumes[i].start, 2);
				memcpy(p + VTBL_END, &volumes[i].end, 2);
			}
			memcpy(p + VTBL_DESC, volumes[i].label, 44);
			memcpy(p + VTBL_DATE, &volumes[i].date, 4);
			modified ++;
		}
		p += VTBL_SIZE;
	}
	memset(p, 0, VTBL_SIZE * (MAX_VOLUMES - vtbl_cnt));
	if (modified) {
		if (verbose) {
			printf("Writing volume table segment ... ");
			fflush(stdout);
		}
		memset(&ft_seg, 0, sizeof(ft_seg));
		ft_seg.mt_data  = buffer;
		ft_seg.mt_segno = first_data_segment;
		ft_seg.mt_mode  = MT_FT_WR_SINGLE;
		if (ioctl(tape_fd, MTIOCWRFTSEG, &ft_seg) == -1) {
			fprintf(stderr, "Ioctl error writing volume table: %s\n",
					strerror(errno));
			return -1;
		}
		if (ft_seg.mt_result != FT_SEGMENT_SIZE) {
			fprintf(stderr, "Short write() reading volume table: %d\n",
					ft_seg.mt_result);
			return -1;
		}
		if (verbose) {
			printf("done.\n");
		}
	}
	if (ioctl(tape_fd, MTIOCTOP, &rewind) == -1) {
		fprintf(stderr, "Ioctl error rewinding tape: %s\n", strerror(errno));
		return -1;
	}
	return 0;
}

static int set_date(/*const*/ char *date,
					vtbl *volumes, int maxnum, int vtbl_no)
{
	time_t raw_time;
	struct tm tapetime;
	struct tm *tapetimep = &tapetime;

	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		fprintf(stderr, "Volume number too big or negative: %d\n", 
				vtbl_no);
		return -1;
	}
	if (date == NULL) {
		time(&raw_time);
		tapetimep = localtime(&raw_time);
	} else {
		(void)strptime(date, "%T %D", tapetimep);
	}
	volumes[vtbl_no].date = FT_TIME_STAMP(tapetimep->tm_year + 1900, 
										  tapetimep->tm_mon, 
										  tapetimep->tm_mday-1,
										  tapetimep->tm_hour,
										  tapetimep->tm_min,
										  tapetimep->tm_sec);
	volumes[vtbl_no].modified = 1;
	return 0;
}

static int set_label(const char *desc, vtbl *volumes, int maxnum, int vtbl_no)
{
	int len;

	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		fprintf(stderr, "Volume number too big or negative: %d\n", 
				vtbl_no);
		return -1;
	}
	strncpy(volumes[vtbl_no].label, desc, 44);
	if ((len = strlen(desc)) < 44) {
		memset(volumes[vtbl_no].label + len, ' ', 44 - len);
	}
	volumes[vtbl_no].modified = 1;
	return 0;
}

static int set_start(int start, vtbl *volumes, int maxnum, int vtbl_no)
{
	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		fprintf(stderr, "Volume number too big or negative: %d\n", 
				vtbl_no);
		return -1;
	}
	volumes[vtbl_no].start = start;
	volumes[vtbl_no].modified = 1;
	return 0;
}

static int set_end(int end, vtbl *volumes, int maxnum, int vtbl_no)
{
	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		fprintf(stderr, "Volume number too big or negative: %d\n", 
				vtbl_no);
		return -1;
	}
	volumes[vtbl_no].end = end;
	volumes[vtbl_no].modified = 1;
	return 0;
}

static int set_id(vtbl *volumes, int maxnum, int vtbl_no)
{
	if (vtbl_no == -1) {
		vtbl_no = maxnum - 1;
	}
	if (vtbl_no < 0 || vtbl_no >= maxnum) {
		fprintf(stderr, "Volume number too big or negative: %d\n", 
				vtbl_no);
		return -1;
	}
	memcpy(volumes[vtbl_no].id, "VTBL", 4);
	volumes[vtbl_no].modified = 1;
	return 0;
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
