/*
 * jobs.c
 *
 * Recording information about the print jobs
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97, 98 Akim Demaille, Miguel Santana
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * $Id: jobs.c,v 1.113 1998/03/04 16:20:32 demaille Exp $
 */
#include "a2ps.h"
#include "jobs.h"
#include "routines.h"
#include "xstrrpl.h"
#include "hashtab.h"
#include "useropt.h"
#include "caret.h"
#include "options.h"
#include "metaseq.h"
#include "msg.h"
#include "fonts.h"
#include "faces.h"
#include "prange.h"
#include "stream.h"
#include "fjobs.h"

/* Temporary file names should start with a2_
 * Don't make it too big (think of MSDOS), though by definition
 * it must be less than 5 chars.
 */
#define a2ps_file_prefix	"a2_"

/*
 * From isdir.c
 */
int isdir PARAMS ((char * path));
/************************************************************************
 * Information about the user						*
 ************************************************************************/
static inline struct user_info *
new_user_info (void)
{
  struct passwd *passwd = NULL;
  struct user_info * res = ALLOC (struct user_info, 1);
  char * cp;

  /* Information related to the user */
  passwd = getpwuid (getuid ());

  /* Home dir */
  if (!IS_EMPTY (cp = getenv ("HOME")))
    res->home = xstrdup (cp);
  else if (passwd && !IS_EMPTY(passwd->pw_dir))
    res->home = xstrdup (passwd->pw_dir);
  else
    res->home = xstrdup ("");

  /* Login */
  if (!IS_EMPTY (cp = getenv ("LOGNAME")))
    res->pw_name = xstrdup (cp);
  else if (!IS_EMPTY (cp = getenv ("USERNAME")))
    res->pw_name = xstrdup (cp);
  else if (passwd)
    res->pw_name = xstrdup (passwd->pw_name);
  else
    res->pw_name = xstrdup (_("user"));

  /* User name (always malloc'd, so that free is easy) */
  res->pw_gecos = NULL;
#ifdef HAVE_PW_GECOS
  if (passwd)
    res->pw_gecos = xstrdup (passwd->pw_gecos);
#endif
  if (!res->pw_gecos && !IS_EMPTY (res->pw_name)) {
    res->pw_gecos = xstrdup (res->pw_name);
    res->pw_gecos[0] = toupper (res->pw_gecos[0]);
  }
  if (!res->pw_gecos)
    res->pw_gecos = xstrdup (_("Unknown User"));

  /* As the special feature of finger(1), whenever `&'
   * appears, substitute name with capitalized first letter */
  cp = xstrdup (res->pw_name);
  cp [0] = toupper (cp [0]);
  vstrrpl (&(res->pw_gecos), "&", cp, NULL);
  XFREE (cp);
  
  return res;
}

/*
 * Release
 */ 
static inline void
user_info_free (struct user_info * user)
{
  FREE (user->home);
  FREE (user->pw_name);
  FREE (user->pw_gecos);

  FREE (user);
}

/*
 * Set the array to NULL values
 */
static void
tmpfiles_reset (struct a2ps_job * job)
{
  size_t i;

  for (i = 0 ; i < cardinalityof (job->tmp_filenames) ; i ++)
    job->tmp_filenames [i] = NULL;
}

/*
 * Create an array of temporary file names
 */
static void
tmpfiles_new (struct a2ps_job * job)
{
  size_t i;

  for (i = 0 ; i < cardinalityof (job->tmp_filenames) ; i ++)
    job->tmp_filenames [i] = xstrdup (tempnam (job->common.tmpdir,
					       a2ps_file_prefix));
}

/*
 * release both name and file of the array of temporary file names
 */
static void
tmpfiles_free (struct a2ps_job * job)
{
  size_t i;

  for (i = 0 ; i < cardinalityof (job->tmp_filenames) ; i ++)
    FREE (job->tmp_filenames[i]);
}

/*
 * Unlink all the (may be existing) temp files
 */
void
a2ps_job_unlink_tmpfiles (struct a2ps_job * job)
{
  unsigned int i;
  /* That of the structure */
  for (i = 0 ; i < cardinalityof (job->tmp_filenames) ; i ++)
    unlink (job->tmp_filenames[i]);
  
  /* That of the files */
  da_map (job->jobs, (da_map_func_t) file_job_unlink_tmpfile);
}

/*
 * Create a new a2ps_job (corresponding to a single file produced)
 * and return it initialized (but default values: config files are
 * not read)
 */
a2ps_job *
a2ps_job_new (void)
{
  time_t tim;
  struct tm *tm;
  a2ps_job * res;
  char * cp;

  res = ALLOC (a2ps_job, 1);

  /* Set the NLS on */
  setlocale (LC_TIME, "");
  setlocale (LC_MESSAGES, "");
  setlocale (LC_COLLATE, "");
  
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  /* The arguments */
  res->argv = NULL;
  res->argc = 0;

  a2ps_common_reset (&res->common);
  res->file_command = NULL;	/* Command to use to call file(1)	*/

  res->user = new_user_info ();	/* Information about the user		*/

  /*
   * Data that library needs (mostly read from config files)
   */
  res->media = new_medium_table ();	/* Media defined by the user 	*/

  /* Short cuts defined by the user */
  res->user_options = user_options_table_new ();

  /* These are declared as global in backupfile.c	*/
  if ((cp = getenv ("SIMPLE_BACKUP_SUFFIX")))
    simple_backup_suffix = cp;
  backup_type = a2ps_get_backup_type (getenv ("VERSION_CONTROL"));
	  		/* Honor what the user said in its environment	*/
  
  /* Get current time information */
  tim = time (NULL);
  tm = localtime (&tim);
  memcpy (&(res->run_tm), tm, sizeof (*tm));

  res->sheets = 0;
  res->pages = 0;
  res->total_files = 0;
  res->orientation = portrait;
  res->duplex = false;
  res->columns = 1;
  res->rows = 1;
  res->Major = major_rows;	/* by default, write horizontally	*/
  res->virtual = 0;
  res->copies = 1;
  res->margin = 0;

  /* Map to know where is the information related to the encodings */
  res->encodings_map = encodings_map_new ();

  /* Chunk in which PS def of the fonts is stored */
  res->ps_encodings = output_new ("PS encodings");

  res->page_prefeed = FALSE;	/* No page prefeed			*/

  /* Make sure not to be happy to use a not initialized array */
  init_face_eo_font (res);

  /* virtual file name given to stdin */
  res->stdin_filename = (uchar *) xstrdup ("stdin");

  /*
   * Related to the output
   */
  res->output_format = ps;	/* By default, make PostScript 		*/
  /* Reset the printers modules	*/
  res->printers = a2ps_printers_new (&res->common);
  res->output_stream = NULL;

  res->folding = TRUE;		/* Line folding option 			*/
  res->numbering = FALSE;	/* Line numbering option 		*/
  res->unprintable_format = caret;/* Replace non printable char by ^M etc. */
  res->interpret = TRUE;	/* Interpret TAB, FF and BS chars option */
  res->print_binaries = FALSE;	/* Force printing for binary files 	*/
  res->compact_mode = FALSE;	/* Allow 2 files per sheet 		*/
  res->border = TRUE;		/* print the surrounding border ?	*/
  res->debug = FALSE;		/* No debugging				*/
  res->prolog = xstrdup ("bw");	/* default ps header file		*/
  res->medium = NULL;		/* default paper is core dump		*/
  res->tabsize = 8;		/* length of tabulations		*/
  res->lines_requested = 0;	/* lines per page			*/
  res->columns_requested = 0;	/* columns per page			*/
  res->fontsize = 0.0;		/* Size of a char for body font 	*/
  res->encoding = NULL; 	/* What is the current char set ?	*/
  res->requested_encoding_name = NULL;	/* Encoding requested by -X 	*/
  res->requested_encoding = NULL;/* Encoding requested by -X		*/
  res->saved_encoding = NULL;/* Encoding requested by -X		*/
  res->encodings = encodings_table_new ();


  /* Map to know the name of the files describing fonts	*/
  res->fonts_map = fonts_map_new ();
  res->font_infos = font_info_table_new ();
  
  /* Title of the job */
  res->title = xustrdup ("a2ps output");	/* Default title */
  
  /* Headers and footers */
  res->header = UNULL;
  res->center_title = UNULL;
  res->left_title = UNULL;
  res->right_title = UNULL;
  res->left_footer = UNULL;
  res->footer = UNULL;
  res->right_footer = UNULL;
  res->water = UNULL;
  * res->tag1 = NUL;
  * res->tag2 = NUL;
  * res->tag3 = NUL;
  * res->tag4 = NUL;

  /* Definition of the macro meta sequences	*/
  res->macro_meta_sequences = macro_meta_sequence_table_new ();

  /* Private info for PS generation */
  res->status = new_ps_status();

  /* Where the diverted output is stored */
  res->divertion = output_new ("Main trunk");

  /* temporary file names cannot be initialized now, cause
   * job->tmpdir is not known yet.  See in a2ps_job_finalize. */
  tmpfiles_reset (res);

  /* List of the pages to print */
  res->page_range = page_range_new ();

  /* List of the jobs */
  res->jobs = da_new ("List of the jobs", 10,
		      da_linear, 10, 
		      (da_print_func_t) file_job_self_print, NULL);
  return res;
}

/*
 * Finish the initialization
 * Typically must be called after that the a2ps.cfg was read
 * so that the lib_path is known, so that the files to be
 * read at initialization time get read now
 */
void
a2ps_job_finalize (struct a2ps_job * job)
{
  /* Finalize the shared mem */
  a2ps_common_finalize (&job->common, job->user->home);

  /* Finalize the printers module */
  a2ps_printers_finalize (job->printers);

  /* Map to know where is the information related to the encodings */
  load_main_encodings_map (job);
  /* Now that the encoding.map is read, make sure to update
   * the encoding.  It must be correct */
  job->requested_encoding = get_encoding_by_alias 
    (job, job->requested_encoding_name);
  if (!job->requested_encoding)
    error (EXIT_UKNCODE, 0, _("encoding %s unknown"), 
	   UNNULL (job->requested_encoding_name));
  
  /* Map to know the name of the files describing fonts	*/
  load_main_fonts_map (job);

  /* Initialize the tmp_filenames */
  tmpfiles_new (job);
}

/*
 * Free the a2ps_job struture and its descendants
 */
void
a2ps_job_free (struct a2ps_job * job)
{
  if (message_test (msg_file))
    da_self_print (job->jobs, stderr);

  user_info_free (job->user);

  free_medium_table (job->media);
  user_options_table_free (job->user_options);

  /* Free the common mem.  Only a2ps_job is responsible for
   * this, the other only had pointers to this master.  They should
   * *not* free it */
  a2ps_common_free (&job->common);

  encodings_map_free (job->encodings_map);
  face_eo_font_free (job->face_eo_font);

  output_free (job->ps_encodings);

  a2ps_printers_free (job->printers);

  XFREE (job->stdin_filename);

  /* Encoding handling */
  encodings_table_free (job->encodings);

  /* Fonts handling	*/
  fonts_map_free (job->fonts_map);
  font_info_table_free (job->font_infos);

  /* Headers */
  XFREE (job->title);
  XFREE (job->header);
  XFREE (job->center_title);
  XFREE (job->left_title);
  XFREE (job->right_title);
  XFREE (job->left_footer);
  XFREE (job->footer);
  XFREE (job->right_footer);
  XFREE (job->water);

  FREE (job->prolog);

  /* Definition of the macro meta sequences	*/
  macro_meta_sequence_table_free (job->macro_meta_sequences);

  ps_status_free (job->status);

  if (message_test (msg_file))
    output_self_print (job->divertion, stderr);
  output_free (job->divertion);

  /* Unlink and free the temporary files */
  a2ps_job_unlink_tmpfiles (job);
  tmpfiles_free (job);

  page_range_free (job->page_range);

  /* Free the file jobs.  Note: their temp files have been freed
   * in a2ps_job_unlink_tmpfiles (job) */
  da_free (job->jobs, (da_map_func_t) file_job_free);

  FREE (job);
}
