#!/usr/bin/perl

#
# Self extracting help2man script.
# Suppress extraction message with: -s, --silent, -q, or --quiet.
#

require 5.003;
use Config;

my $target = $0;

$target =~ s/\.PL$// or die "$0: can't determine target name\n";
unlink $target       or die "$0: can't unlink $target ($!)\n" if -e $target;
open OUT, ">$target" or die "$0: can't create $target ($!)\n";

print "Extracting $target (with variable substitutions)\n"
    unless ($ARGV[0] || '') =~ /^--?(q(?:uiet)?|s(?:ilent)?)$/;

# Add startup header:
print OUT "$Config{startperl} -w\n";

# For systems without the crash-bang hack also add:
print OUT <<"!GROK!THIS!" if $Config{sharpbang} !~ /^#!/;
eval 'exec $Config{perlpath} -wS \$0 \${1+"\$@"}'
    if \$running_under_some_shell;

\$running_under_some_shell = 0; # for -w
!GROK!THIS!

# The rest of the script is dumped verbatim:
print OUT <<'!NO!SUBS!';

# Generate a short man page from --help and --version output.
# Copyright  1997, 98 Free Software Foundation, Inc.

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

# Written by Brendan O'Dea <bod@compusol.com.au>
#        and Franois Pinard <pinard@IRO.UMontreal.CA>

require 5.003;

use strict;
use Getopt::Long;
use POSIX 'strftime';

my $RCS_Id = '$Id: help2man.PL,v 1.2 1998/03/02 08:56:32 demaille Exp $';
my $this_program = 'help2man';
my $this_version = '0.0';

if ($RCS_Id =~ /\$Id:\s+(\S+)\s+(\S+)/)
{
    $this_version = $2;
   ($this_program = $1) =~ s/(\.\w+)?,v$//;
}

my $version_info = <<EOT;
$this_program $this_version

Copyright (C) 1997, 98 Free Software Foundation, Inc.
This is free software; see the source for copying conditions.  There is NO
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
EOT

my $help_info = <<EOT;
`$this_program' generates a man page out of `--help' and `--version' output.

Usage: $this_program [OPTION]... EXECUTABLE

  --name=STRING  use `STRING' as the description for the NAME paragraph
  --output=FILE  send output to `FILE'
  --help         print this help, then exit
  --version      print $this_program program version number, then exit

EXECUTABLE should accept `--help' and `version' options.
EOT

my ($name, $opt_output, $opt_help, $opt_version);

# Parse options.
GetOptions (
    'name=s'   => \$name,
    'output=s' => \$opt_output,
    help       => sub { print $help_info; exit },
    version    => sub { print $version_info; exit },
) or die $help_info;

die $help_info unless @ARGV == 1;

# Turn off localisation of executable's ouput.
@ENV{qw(LANGUAGE LANG LC_ALL)} = ('C') x 3;

# Grab help and version paragraphs from executable
my @help = split /\n\n+/, `$ARGV[0] --help 2>/dev/null`
    or die "$this_program: can't get `--help' info from $ARGV[0]\n";

my @version = split /\n\n+/, `$ARGV[0] --version 2>/dev/null`
    or die "$this_program: can't get `--version' info from $ARGV[0]\n";

my $date = strftime "%B %Y", localtime;
my $program = $ARGV[0]; $program =~ s!.*/!!;
my $package = $program;
my $version;

if ($opt_output)
{
    unlink $opt_output
	or die "$this_program: can't unlink $opt_output ($!)\n"
	if -e $opt_output;

    open STDOUT, ">$opt_output"
	or die "$this_program: can't create $opt_output ($!)\n";
}

# The first line of the --version information is assumed to be in one
# of the following formats:
#
#   <version>
#   <program> <version>
#   GNU <program> <version>
#   <program> (GNU <package>) <version>
#   <program> - GNU <package> <version>
#

$_ = shift @version;

if (/^(\S+)\s+\((GNU\s+[^)]+)\)\s+(.*)/ ||
    /^(\S+)\s+-\s*(GNU\s+\S+)\s+(.*)/)
{
    $program = $1;
    $package = $2;
    $version = $3;
}
elsif (/^(GNU\s+)?(\S+)\s+(.*)/)
{
    $program = $2;
    $package = $1 ? "$1$2" : $2;
    $version = $3;
}
else
{
    $version = $_;
}

# Default description for NAME paragraph
$name ||= "short documentation for $program $version";

# Man pages traditionally have the page title in caps.
my $PROGRAM = uc $program;

# Header.
print <<EOT;
.\" DO NOT MODIFY THIS FILE!  It was generated by $this_program $this_version.
.TH $PROGRAM 1 "$date" "$package $version" "GNU User's Manual"
.SH NAME
$program \\- $name
EOT

my $accumulate = 1;
my @description = ();

sub convert_option;

# Output converted --help information.
for (@help)
{
    chomp;

    if (s/^Usage:\s+\S+\s+(.*)\n?//)
    {
	# Turn the usage clause into a synopsis.
	print ".SH SYNOPSIS\n";

	do {
	    my $syn = $1;
	    $syn =~ s/(([][]|\.\.+)+)/\\fR$1\\fI/g;
	    $syn =~ s/^/\\fI/ unless $syn =~ s/^\\fR//;
	    $syn .= '\fR';
	    $syn =~ s/\\fI(\s*)\\fR/$1/g;

	    print ".br\n" unless $accumulate;
	    print ".B $program\n";
	    print "$syn\n";
	    $accumulate = 0;
	} while s/^(?:Usage|\s*or):\s+\S+\s+(.*)\n?//;

	# Dump any accumulated description text.
	print ".SH DESCRIPTION\n";
	print @description;

	next unless $_;
    }

    # Accumulate text if the synopsis has not been produced yet.
    if ($accumulate)
    {
	push @description, ".PP\n" if @description;
	push @description, "$_\n";
	next;
    }

    # Catch bug report text.
    if (/^Report bugs /)
    {
	print ".SH BUGS\n$_\n";
	next;
    }

    # Special case for tar 1.12: --label=NAME\nPATTERN.
    s{(\n[ \t]*)(-V,[ \t]+--label=NAME.*)\n[ \t]+PATTERN[ \t]+}
     {$1$2$1\\&...=PATTERN };

    # Convert options.
    s/(\s)(-[][\w=-]+|\\&\S+)/$1.convert_option $2/ge;

    # Option subsections have second line indented.
    print qq(.SS "$1"\n) if s/^(\S.*)\n(\s)/$2/;

    # Lines indented more than about 10 spaces may be assumed to be
    # continuations of the previous line.
    s/\n {10,}/ /g;

    # Lines following dotted (*) or numbered points may also be
    # continued if indented to the same level as the text following
    # the point.
    1 while s{((?:^|\n)(\s+)(?:[1-9][.)]|\*)(\s+)(?:[^\n]+))\n\2 \3(\S)}
	     {$1 $4}g;

    # Indented paragraph.
    if (/^\s/)
    {
	for (split /\n/)
	{
	    s/^\s+//;
	    s/([^,])\s+/$1\n/;
	    print ".TP\n$_\n";
	}
    }
    # Anything else.
    else
    {
	print ".PP\n$_\n";
    }
}

# Refer to the real documentation.

print <<EOT;
.SH SEE ALSO
The full documentation for
.B $program
is maintained as a Texinfo manual.  If the
.B info
and
.B $program
programs are properly installed at your site, the command
.IP
.B info $program
.PP
should allow you to access the manual as an hypertext.
EOT

# Output converted --version information.
for (@version)
{
    chomp;

    # Convert copyright symbol or (c) to nroff character
    s/Copyright\s+(?:\xa9|\([Cc]\))/Copyright \\(co/g;

    # Insert appropriate headings for copyright and author
    if    (/^Copyright\s\\/) { print ".SH COPYRIGHT\n" }
    elsif (/^Written\s+by/)  { print ".SH AUTHOR\n" }
    else		     { print ".PP\n"; }

    # Insert line breaks before additional copyright messages and the
    # disclaimer
    s/(.)\n(Copyright\s|This is free software)/$1\n.br\n$2/g;

    print "$_\n";
}

exit;

# Convert option dashes to \- to stop nroff from hyphenating 'em, and
# embolden.  Option arguments get italicised.
sub convert_option
{
    my $option = '\fB' . shift;

    $option =~ s/-/\\-/g;
    unless ($option =~ s/\[=(.*)\]$/\\fR[=\\fI$1\\fR]/)
    {
	$option =~ s/=(.)/\\fR=\\fI$1/;
	$option .= '\fR';
    }

    $option;
}
!NO!SUBS!

close OUT;
chmod 0555, $target or warn "$0: can't change ${target}'s mode ($!)\n";
exit 0;
