/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  PCM (Pulse Code Modulation) support
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "libgus.h"
#include "libgus_local.h"

/*
 *  defines
 */

#define FILE_PCM		"/dev/guspcm%i%i"
#define FILE_CONTROL		"/dev/gusctl%i"
 
/*
 *  structures
 */

typedef struct {
  int card;
  int handle;
  int direction;
  struct GUS_PCM_INFO info;
} CARD;

/*
 *  variables
 */

/*
 *  local functions
 */

/*
 *  EXPORTED FUNCTIONS
 */

extern int __gus_decode_id( const char *id );

int gus_pcm_cards( void )
{
  return gus_cards();
}

int gus_pcm_devices( const char *id )
{
  int card, fd, idx;
  char filename[ 16 ];
  struct GUS_PCM_INFO info;

  card = __gus_decode_id( id );
  if ( card < 0 || card >= GUS_CARDS )
    return -ENODEV;
  sprintf( filename, FILE_CONTROL, card );
  if ( ( fd = open( filename, O_RDONLY ) ) < 0 )
    return -ENODEV;
  for ( idx = 0; idx < GUS_PCM_DEVICES; idx++ ) {
    info.device = ( card << 16 ) | idx;
    if ( ioctl( fd, GUS_IOCTL_CTL_PCM_INFO, &info ) < 0 ) break;
  }
  close( fd );
  return idx;
}

int gus_pcm_look_for_card( const char *id )
{
  return __gus_decode_id( id );
}

/*
 *  ----------------- open / close
 */

int gus_pcm_open( void **rhandle, int card, int device, int direction, int flags )
{
  CARD *acard;
  int handle;
  char filename[ 16 ];

  if ( !rhandle ) return -ENODEV;
  *rhandle = NULL;
  if ( card < 0 || card >= GUS_CARDS ) return -ENODEV;
  if ( direction != GUS_PCM_DIRECTION_PLAYBACK &&
       direction != GUS_PCM_DIRECTION_RECORD &&
       direction != GUS_PCM_DIRECTION_DUPLEX ) return -EINVAL;

  sprintf( filename, FILE_PCM, card, device );
  if ( ( handle = open( filename, direction | ( flags & GUS_PCM_OF_NONBLOCK ? O_NONBLOCK : 0 ) ) ) < 0 )
    {
      gus_dprintf( "gus_pcm_open: %s", strerror( errno ) );
      return -1;
    }
  
  acard = (CARD *)malloc( sizeof( CARD ) );  
  if ( !acard )
    {
      gus_dprintf( "gus_pcm_open: malloc problem" );
      close( handle );
      return -1;
    }
  acard -> handle = handle;
  acard -> card = card;
  acard -> direction = direction;
  if ( ioctl( handle, SOUND_PCM_GUSINFO, &acard -> info ) < 0 )
    {
      gus_dprintf( "gus_pcm_open: info ioctl error" );
      free( acard );
      close( handle );
      return -1;
    }
 
  *rhandle = acard;
  return 0;
}

int gus_pcm_close( void *handle )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  close( acard -> handle );
  free( acard );
  return 0;
}

int gus_pcm_get_file_descriptor( void *handle )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  return acard -> handle;
}

static int gus_set_direction( CARD *acard, int direction )
{
  int mask;

  if ( acard -> direction == GUS_PCM_DIRECTION_DUPLEX ) {
    mask = 0;
    switch ( direction ) {
      case GUS_PCM_DIRECTION_PLAYBACK:
        mask = GUS_PCM_MASK_PLAYBACK;
        break;
      case GUS_PCM_DIRECTION_RECORD:
        mask = GUS_PCM_MASK_RECORD;
        break;
      case GUS_PCM_DIRECTION_DUPLEX:
        mask = GUS_PCM_MASK_DUPLEX;
        break;
      default:
        return -1;
    }
    if ( ioctl( acard -> handle, SOUND_PCM_GUSMASK, &mask ) < 0 )
      return -1;
  }
  return 0;
}

static int gus_ioctl( CARD *acard, int direction, int command )
{
  if ( gus_set_direction( acard, direction ) < 0 )
    return -1;
  if ( ioctl( acard -> handle, command ) < 0 )
    return -1;
  if ( gus_set_direction( acard, GUS_PCM_DIRECTION_DUPLEX ) < 0 )
    return -1;
  return 0;
}

static int gus_ioctl_int( CARD *acard, int direction, int command, int value )
{
  if ( gus_set_direction( acard, direction ) < 0 )
    return -1;
  if ( ioctl( acard -> handle, command, &value ) < 0 )
    return -1;
  if ( gus_set_direction( acard, GUS_PCM_DIRECTION_DUPLEX ) < 0 )
    return -1;
  return value < 0 ? 0 : value;
}

int gus_pcm_mode_set( void *handle, int direction, unsigned int mode )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  return gus_ioctl_int( acard, direction, SOUND_PCM_SETFMT, mode );
}

unsigned int gus_pcm_mode_get( void *handle, int direction )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  return gus_ioctl_int( acard, direction, SOUND_PCM_READ_BITS, 0 );
}

unsigned int gus_pcm_mode_supported( void *handle, int direction )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  switch ( direction ) {
    case GUS_PCM_DIRECTION_PLAYBACK:
      return acard -> info.formats_play;
    case GUS_PCM_DIRECTION_RECORD:
      return acard -> info.formats_record;
    default:
      return -1;
  } 
}

int gus_pcm_channels_set( void *handle, int direction, int channels )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  return gus_ioctl_int( acard, direction, SOUND_PCM_WRITE_CHANNELS, channels );
}

int gus_pcm_channels_get( void *handle, int direction )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  return gus_ioctl_int( acard, direction, SOUND_PCM_READ_CHANNELS, 0 );
}

int gus_pcm_channels_supported( void *handle, int direction )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  switch ( direction ) {
    case GUS_PCM_DIRECTION_PLAYBACK:
      return acard -> info.max_channels_play;
    case GUS_PCM_DIRECTION_RECORD:
      return acard -> info.max_channels_record;
    default:
      return -1;
  } 
}

int gus_pcm_rate_set( void *handle, int direction, int rate )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  return gus_ioctl_int( acard, direction, SOUND_PCM_WRITE_RATE, rate );
}

int gus_pcm_rate_get( void *handle, int direction )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;  
  return gus_ioctl_int( acard, direction, SOUND_PCM_READ_RATE, 0 );
}

int gus_pcm_rate_supported( void *handle, int direction )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  switch ( direction ) {
    case GUS_PCM_DIRECTION_PLAYBACK:
      return acard -> info.max_rate_play;
    case GUS_PCM_DIRECTION_RECORD:
      return acard -> info.max_rate_record;
    default:
      return -1;
  }
}

int gus_pcm_dma_size_set( void *handle, int direction, int fragments, int fragment_size )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  fragments <<= 16;
  fragments |= fragment_size > 0xffff ? 0xffff : fragment_size;
  return gus_ioctl_int( acard, direction, SOUND_PCM_SETFRAGMENT, fragments );
}

int gus_pcm_dma_size_get( void *handle, int direction, int current )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  if ( !current )
    {
      switch ( direction ) {
        case GUS_PCM_DIRECTION_PLAYBACK:
          return acard -> info.dma_size_play;
        case GUS_PCM_DIRECTION_RECORD:
          return acard -> info.dma_size_record;
        default:
          return -1;
      }
    }
   else
    {
      return gus_ioctl_int( acard, direction, SNDCTL_DSP_GETBLKSIZE, 0 );
    }
}

int gus_pcm_sync( void *handle, int direction )
{
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  return gus_ioctl( acard, direction, SOUND_PCM_SYNC );
}

int gus_pcm_read( void *handle, unsigned char *buffer, int count )
{
  int res;
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  res = read( acard -> handle, buffer, count );
  return res < 0 ? -1 : res;
}

int gus_pcm_write( void *handle, unsigned char *buffer, int count )
{
  int res;
  CARD *acard;

  acard = handle;
  if ( !acard ) return -EINVAL;
  res = write( acard -> handle, buffer, count );
  return res < 0 ? -1 : res;
}
