/*
 *  @(#) obuffer_sparc.cc 1.6, last edit: 02 Mar 1995 18:38:54
 *  @(#) Copyright (C) 1993, 1994 Tobias Bading (bading@cs.tu-berlin.de)
 *  @(#) Berlin University of Technology
 *
 *  Idea and first implementation for u-law output with fast downsampling by
 *  Jim Boucher (jboucher@flash.bu.edu)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *  Changes from version 1.2 to 1.3:
 *    - SparcObuffer uses asynchronous audio output now.
 *	Calculated PCM samples are stored in a ringbuffer and written to
 *	the audio device by a signal handler. The handler works with either
 *	SIGIO or SIGALRM signals.
 *    - should work with SPARC 5 machines, if the audio patch is installed
 *	correctly. But don't ask me what's correct in this matter,
 *	please ask Sun.
 *
 *  Changes from version 1.1 to 1.2:
 *    - If ULAW is defined:
 *      SparcObuffer feeds mono u-law output to an amd device (u-law format).
 *      The required downsampling to 8 kHz in done in the synthesis filter.
 *    - O_NDELAY flag is cleared now after a successful open syscall
 *      on /dev/audio (caused problems under Solaris)
 *    - options -us, -uh and -ul should work now
 */


#include <stropts.h>
extern "C" {
int ioctl (int, int ...);
#ifdef USE_TIMER
int setitimer (int, itimerval *, itimerval *);
#endif
}


// statics of class SparcObuffer:
uint32		SparcObuffer::ringbuffersize;
#ifdef ULAW
ulawsample     *SparcObuffer::buffer;
char	       *SparcObuffer::head;
ulawsample     *SparcObuffer::tail;
#else
int16	       *SparcObuffer::buffer;
char	       *SparcObuffer::head;
int16	       *SparcObuffer::tail[2];
uint32		SparcObuffer::channels;
#endif
int		SparcObuffer::audio_fd = -1;
boil		SparcObuffer::handler_activated;
boil		SparcObuffer::buffer_empty;
boil		SparcObuffer::drain_buffer;
#ifdef USE_TIMER
itimerval	SparcObuffer::timerval;
#endif


#ifdef ULAW
SparcObuffer::SparcObuffer (Header *header, boil use_speaker, boil use_headphone,
			    boil use_line_out, float volume)
#else
SparcObuffer::SparcObuffer (uint32 number_of_channels, Header *header, boil use_speaker,
			    boil use_headphone, boil use_line_out, float volume)
#endif
{
#ifndef ULAW
#ifdef DEBUG
  if (!number_of_channels || number_of_channels > MAXCHANNELS)
  {
    cerr << "SparcObuffer: 0 < number of channels < " << MAXCHANNELS << "!\n";
    exit (1);
  }
#endif
#endif	// !ULAW
  if (audio_fd < 0)
  {
    cerr << "Internal error: SparcObuffer::audio_fd has to be initialized\n"
	    "by SparcObuffer::class_suitable()!\n";
    exit (1);
  }

  // configure the audio device:
  audio_info info;
  AUDIO_INITINFO (&info);
#ifdef SunOS4_1_1
  if (use_speaker || use_headphone)
#else
  info.output_muted = False;
  if (use_speaker || use_headphone || use_line_out)
#endif
  {
    info.play.port = 0;
    if (use_speaker)
      info.play.port |= AUDIO_SPEAKER;
    if (use_headphone)
      info.play.port |= AUDIO_HEADPHONE;
#ifndef SunOS4_1_1
    if (use_line_out)
      info.play.port |= AUDIO_LINE_OUT;
#endif
  }
  if (volume >= 0.0 && volume <= 1.0)
    info.play.gain = (unsigned int)(volume * 255.0 + 0.5);

#ifdef ULAW
  // configure the amd device:
  info.play.encoding = AUDIO_ENCODING_ULAW;
  info.play.precision = 8;
  info.play.channels = 1;
  info.play.sample_rate = 8000;
#else
  // configure the dbri or audiocs device:
  info.play.encoding = AUDIO_ENCODING_LINEAR;
  info.play.precision = 16;
  info.play.channels = number_of_channels;
  info.play.sample_rate = header->frequency ();
#endif
  if (ioctl (audio_fd, AUDIO_SETINFO, &info) < 0)
  {
    perror ("ioctl AUDIO_SETINFO on /dev/audio");
    exit (1);
  }

  // initialize ringbuffer:
  buffer_empty = True;
  drain_buffer = False;
#ifdef ULAW
  ringbuffersize = 64 * 1024;		// 64 KB ringbuffer
  buffer = (ulawsample *)malloc (ringbuffersize);
  tail = buffer;
#else
  channels = number_of_channels;
  if (channels > 1)
    ringbuffersize = 512 * 1024;	// 512 KB ringbuffer
  else
    ringbuffersize = 256 * 1024;	// 256 KB ringbuffer
  buffer = (int16 *)malloc (ringbuffersize);
  tail[0] = buffer;
  tail[1] = buffer + 1;
#endif	// !ULAW
  head = (char *)buffer;

  // prepare to receive signals:
#ifdef USE_TIMER
  // prepare for SIGALRM timeouts:
  timerval.it_interval.tv_sec = 0;
  timerval.it_value.tv_sec = 0;
#ifdef ULAW
  // call signal_handler() three times a second:
  timerval.it_interval.tv_usec = 1000000 / 3;
  timerval.it_value.tv_usec =    1000000 / 3;
#else
  if (number_of_channels == 1)
  {
    // call signal_handler() six times a second:
    timerval.it_interval.tv_usec = 1000000 / 6;
    timerval.it_value.tv_usec =    1000000 / 6;
  }
  else
  {
    // call signal_handler() twelve times a second:
    timerval.it_interval.tv_usec = 1000000 / 12;
    timerval.it_value.tv_usec =    1000000 / 12;
  }
#endif
#ifdef Solaris
  if ((int)sigset (SIGALRM, (void (*)(int))signal_handler) == -1)
#else
  if ((int)signal (SIGALRM, (void (*)(int))signal_handler) == -1)
#endif
  {
    perror ("signal (SIGARLM, handler)");
    exit (1);
  }
#else
  // prepare for SIGIO signals from /dev/audio stream:
#ifdef Solaris
  if ((int)sigset (SIGIO, (void (*)(int))signal_handler) == -1)
#else
  if ((int)signal (SIGIO, (void (*)(int))signal_handler) == -1)
#endif
  {
    perror ("signal (SIGIO, SIG_IGN)");
    exit (1);
  }
#endif
  handler_activated = False;
}


SparcObuffer::~SparcObuffer (void)
{
#ifdef USE_TIMER
  // disable timer:
  static itimerval no_timer = { { 0, 0 }, { 0, 0 } };
  setitimer (ITIMER_REAL, &no_timer, NULL);
#else
  // disable SIGIO signals:
  if ((int)signal (SIGIO, SIG_IGN) == -1)
  {
    perror ("signal (SIGIO, SIG_IGN)");
    exit (1);
  }
#endif

  // wait until ringbuffer is empty:
  drain_buffer = True;
  while (!buffer_empty)
    signal_handler ();

  ioctl (audio_fd, AUDIO_DRAIN, NULL);
  close (audio_fd);
  free (buffer);
}


void SparcObuffer::append (uint32 channel, int16 value)
/* Remark:
   Because signal_handler() can interrupt the normal program execution at
   (nearly) any time in any statement and synchronization mechanisms like
   sigblock() create too much overload, some values are precomputed first
   and assigned to the target variable later. */
{
#ifdef ULAW
  // convert 16-bit PCM sample to 8-bit ulaw:
  *tail = linear2ulaw[value >> 3];
  if ((char *)(tail + 1) >= (char *)buffer + ringbuffersize)
    tail = buffer;
  else
    ++tail;
#else
#ifdef DEBUG
  if (channel >= channels)
  {
    cerr << "illegal channelnumber in SparcObuffer::append()!\n";
    exit (1);
  }
#endif
  *tail[channel] = value;
  if ((char *)(tail[channel] + channels) >= (char *)buffer + ringbuffersize)
  {
    register int16 *tmp = buffer + channel;		// fold into ringbuffer
    tail[channel] = tmp;
  }
  else
    tail[channel] += channels;
#endif
  buffer_empty = False;
#ifdef ULAW
  if ((char *)tail == head)
#else
  if ((char *)*tail == head)
#endif
  {
    // tail bites into head (buffer is full)
    if (!handler_activated)
    {
      // call handler by hand to (re)start playback:
      do
	signal_handler ();
#ifdef ULAW
      while (!buffer_empty && (char *)tail == head);
#else
      while (!buffer_empty && (char *)*tail == head);
#endif

#ifdef USE_TIMER
      // activate timer:
      if (setitimer (ITIMER_REAL, &timerval, NULL))
      {
	perror ("setitimer");
	exit (1);
      }
#else
      // give me SIGIO (== SIGPOLL) signals each time
      // there's empty room in the output queue:
      if (ioctl (audio_fd, I_SETSIG, S_OUTPUT) < 0)
      {
	perror ("ioctl I_SETSIG on /dev/audio");
	exit (1);
      }
#endif
      handler_activated = True;
    }
    else
    {
      // wait for a SIGALRM or SIGIO signal:
      sigset_t sigset;
      sigfillset (&sigset);
#ifdef USE_TIMER
      sigdelset (&sigset, SIGALRM);
#else
      sigdelset (&sigset, SIGIO);
#endif
      sigdelset (&sigset, SIGINT);
      sigdelset (&sigset, SIGQUIT);
      sigdelset (&sigset, SIGSTOP);
      do
	if (sigsuspend (&sigset) < 0 && errno != EINTR)
	{
	  perror ("sigsuspend");
	  exit (1);
	}
#ifdef ULAW
      while (!buffer_empty && (char *)tail == head);
#else
      while (!buffer_empty && (char *)*tail == head);
#endif
    }
  }
}


void SparcObuffer::signal_handler (void)
{
  extern boil verbose_mode;
  int length, length2, written_bytes;
  char *write_end;

  if (buffer_empty)
  {
pause:
#ifdef USE_TIMER
    // disable timer:
    static itimerval no_timer = { { 0, 0 }, { 0, 0 } };
    setitimer (ITIMER_REAL, &no_timer, NULL);
#else
    // no more SIGIO signals please:
    if (ioctl (audio_fd, I_SETSIG, 0) < 0)
    {
      perror ("ioctl I_SETSIG on /dev/audio");
      exit (1);
    }
#endif
    handler_activated = False;
    if (verbose_mode)
      write (2, "pausing playback...\n", 20);
    return;
  }

#ifdef ULAW
  write_end = (char *)tail;
#else
  write_end = (char *)(tail[channels - 1] - channels + 1);
#endif
  if (write_end == (char *)buffer)
    write_end = (char *)buffer + ringbuffersize;

  if (write_end > head)
  {
    length = write_end - head;
#ifdef ULAW
    if (!drain_buffer && length < 4 * 1024)
#else
    if (!drain_buffer && length < 32 * 1024)
#endif
      goto pause;
    if ((written_bytes = write (audio_fd, head, length)) < 0)
    {
      if (errno == EWOULDBLOCK)
	return;
      perror ("write to /dev/audio");
      exit (1);
    }

#ifdef AUDIO_DEBUG
    timeval actual_time;
    gettimeofday (&actual_time, NULL);
    char string[100];
    sprintf (string, "Written bytes (1):  %6d at time %09d.%06d\n",
	     written_bytes, actual_time.tv_sec, actual_time.tv_usec);
    write (2, string, strlen (string));
#endif

    if ((head += written_bytes) == write_end)
      buffer_empty = True;
    if (head == (char *)buffer + ringbuffersize)
      head = (char *)buffer;
  }
  else
  {
    length  = (char *)buffer + ringbuffersize - head;
    length2 = write_end - (char *)buffer;
#ifdef ULAW
    if (!drain_buffer && length + length2 < 4 * 1024)
#else
    if (!drain_buffer && length + length2 < 32 * 1024)
#endif
      goto pause;
    if ((written_bytes = write (audio_fd, head, length)) < 0)
    {
      if (errno == EWOULDBLOCK)
	return;
      perror ("write to /dev/audio");
      exit (1);
    }

#ifdef AUDIO_DEBUG
    timeval actual_time;
    gettimeofday (&actual_time, NULL);
    char string[100];
    sprintf (string, "Written bytes (2):  %6d at time %09d.%06d\n",
	     written_bytes, actual_time.tv_sec, actual_time.tv_usec);
    write (2, string, strlen (string));
#endif

    if (written_bytes == length && length2)
    {
      if ((written_bytes = write (audio_fd, (char *)buffer, length2)) < 0)
      {
	if (errno == EWOULDBLOCK)
	  written_bytes = 0;
	else
	{
	  perror ("write to /dev/audio");
	  exit (1);
	}
      }

#ifdef AUDIO_DEBUG
      timeval actual_time;
      gettimeofday (&actual_time, NULL);
      char string[100];
      sprintf (string, "Written bytes (2+): %6d at time %09d.%06d\n",
	       written_bytes, actual_time.tv_sec, actual_time.tv_usec);
      write (2, string, strlen (string));
#endif

      head = (char *)buffer + written_bytes;
      if (head == write_end)
	buffer_empty = True;
    }
    else
      if ((head += written_bytes) == (char *)buffer + ringbuffersize)
      {
	if (head == write_end)
	  buffer_empty = True;
	head = (char *)buffer;
      }
  }
}


int SparcObuffer::open_audio_device (void)
{
  int fd;

  if ((fd = open ("/dev/audio", O_WRONLY | O_NDELAY, 0)) < 0)
    if (errno == EBUSY)
    {
      extern boil verbose_mode, wait_if_busy;

      if (wait_if_busy)
      {
	if (verbose_mode)
	  cerr << "Audio device is busy, waiting...\n";
	if ((fd = open ("/dev/audio", O_WRONLY, 0)) < 0)
	{
	  perror ("Can't open /dev/audio for writing");
	  exit (1);
	}
	if (verbose_mode)
	  cerr << "Starting playback...\n";
      }
      else
      {
	cerr << "Sorry, the audio device is busy!\n";
	exit (1);
      }
    }
    else
    {
      perror ("Can't open /dev/audio for writing");
      exit (1);
    }

  // turn NDELAY mode on, because write() must never block maplay:
  int flags;
  if ((flags = fcntl (fd, F_GETFL, 0)) < 0)
  {
    perror ("fcntl F_GETFL on /dev/audio failed");
    exit (1);
  }
  flags |= O_NDELAY;
  if (fcntl (fd, F_SETFL, flags) < 0)
  {
    perror ("fcntl F_SETFL on /dev/audio failed");
    exit (1);
  }

  return fd;
}


#ifdef Solaris
void SparcObuffer::get_device_type (int fd, audio_device *devtype)
{
  if (ioctl (fd, AUDIO_GETDEV, devtype))
  {
    perror ("ioctl AUDIO_GETDEV on /dev/audio");
    exit (1);
  }
}

#else
// !Solaris

int SparcObuffer::get_device_type (int fd)
{
#ifdef AUDIO_GETDEV
  int devtype;
  if (ioctl (fd, AUDIO_GETDEV, &devtype))
  {
    perror ("ioctl AUDIO_GETDEV on /dev/audio");
    exit (1);
  }
  return devtype;
#else
  cerr << "SparcObuffer::get_device_type(): AUDIO_GETDEV ioctl not available!\n";
  return -1;
#endif
}
#endif	// !Solaris


#ifdef ULAW
boil SparcObuffer::class_suitable (uint32 number_of_channels, boil force_amd)
#else
boil SparcObuffer::class_suitable (void)
#endif
{
#ifdef ULAW
  if (number_of_channels > 1)
  {
    cerr << "Your audio hardware cannot handle more than one audio channel.\n"
	    "Please use the option -l or -r for stereo streams.\n";
    return False;
  }
#endif

  audio_fd = open_audio_device ();

#ifdef ULAW
  if (force_amd)
    return True;
#endif

#ifdef Solaris
  audio_device devtype;
  get_device_type (audio_fd, &devtype);
# ifdef ULAW
    if (!strcmp (devtype.name, "SUNW,am79c30"))
      return True;
    else if (!strcmp (devtype.name, "SUNW,dbri") ||
	     !strcmp (devtype.name, "SUNW,CS4231"))	// Sparc 5 audiocs device
    {
      cerr << "Your machine can produce CD-quality audio output,\n"
	      "but this binary was compiled for 8 kHz u-law ouput. (telephone quality)\n"
	      "Please recompile it without the ULAW define in COMPILERFLAGS.\n"
	      "(or use the -amd option to use this binary with low-quality output)\n";
      close (audio_fd);
      return False;
    }
# else
    if (!strcmp (devtype.name, "SUNW,dbri"))
      return True;
    else if (!strcmp (devtype.name, "SUNW,CS4231"))	// Sparc 5 audiocs device
      return True;
    else if (!strcmp (devtype.name, "SUNW,am79c30"))
    {
      cerr << "Your machine can produce 8 kHz u-law audio output only,\n"
	      "but this binary was compiled for CD-quality output.\n"
	      "Please recompile it with ULAW defined in COMPILERFLAGS\n"
	      "or use it in stdout mode as an decoder only.\n";
      close (audio_fd);
      return False;
    }
# endif
#else
  // !Solaris
# ifdef SunOS4_1_1
    // no AUDIO_GETDEV under SunOS 4.1.1, so we have to assume that there is
    // an amd device attached to /dev/audio
#   ifdef ULAW
      return True;
#   else
      cerr << "Your machine can produce 8 kHz u-law audio output only,\n"
	      "but this binary was compiled for CD-quality output.\n"
	      "Please recompile it with ULAW defined in COMPILERFLAGS\n"
	      "or use it in stdout mode as an decoder only.\n";
      close (audio_fd);
      return False;
#   endif	// !ULAW
# else
    // SunOS 4.1.3
    int device_type = get_device_type (audio_fd);
#   ifdef ULAW
      if (device_type == AUDIO_DEV_AMD)
	return True;
      else if (device_type == AUDIO_DEV_SPEAKERBOX || device_type == AUDIO_DEV_CODEC)
      {
	cerr << "Your machine can produce CD-quality audio output,\n"
		"but this binary was compiled for 8 kHz u-law ouput. (telephone quality)\n"
		"Please recompile it without the ULAW define in COMPILERFLAGS.\n"
		"(or use the -amd option to use this binary with low-quality output)\n";
	close (audio_fd);
	return False;
      }
#   else
      if (device_type == AUDIO_DEV_SPEAKERBOX || device_type == AUDIO_DEV_CODEC)
	return True;
      else if (device_type == AUDIO_DEV_AMD)
      {
	cerr << "Your machine can produce 8 kHz u-law audio output only,\n"
		"but this binary was compiled for CD-quality output.\n"
		"Please recompile it with ULAW defined in COMPILERFLAGS\n"
		"or use it in stdout mode as an decoder only.\n";
	close (audio_fd);
	return False;
      }
#   endif	// !ULAW
# endif	// !SunOS4_1_1
#endif	// !Solaris

#ifndef SunOS4_1_1
  close (audio_fd);
  cerr << "Sorry, I don't recognize your audio device.\n"
# ifdef ULAW
	  "Please try the -amd option or use the stdout mode.\n";
# else
	  "Please use the stdout mode.\n";
# endif
  return False;
#endif
}
