/*----------------------------------------------------------------
 * read midi file and parse events
 *----------------------------------------------------------------
 * based on the midi parser routine in TiMidity by Tuukka Toivonen.
 *----------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include "midievent.h"
#include "util.h"
#include "controls.h"
#include "channel.h"

/*----------------------------------------------------------------*/

/* midi event as linked list */
typedef struct {
	MidiEvent event;
	void *next;
} MidiEventList;

static MidiEventList *evlist, *curev;	/* event linked list */
static int event_count;		/* number of events allocated */
static int at;			/* current tick */
static int default_tempo;	/* default tempo (for CTMF) */
static int channel_shifted;	/* channel offset for multi midi part */
static MidiInfo *minfop;	/* midi header information */
static int text_printed;	/* first text event is found? */

#define MID_OK		0
#define MID_ERR		-1	/* recoverable error */
#define MID_END		-2	/* end of track */
#define MID_EOF		-3	/* reach at EOF */
#define MID_FATAL	-4	/* fatal error */

/*----------------------------------------------------------------
 * prototypes
 *----------------------------------------------------------------*/

static int skip(FILE *fp, size_t len);
static int readc(unsigned char *p, FILE *fp);
static int reads(unsigned char *s, int len, FILE *fp);
static int READLONG(FILE *fp);
static short READSHORT(FILE *fp);
static short READSHORT_LE(FILE *fp);

static int getvl(FILE *fp);

static MidiEventList *alloc_midi_event(int type, int ch);
static void init_midi_event(int append_mode);
static MidiEventList *append_to_list(MidiEventList *new);
static MidiEventList *midi_dbl_event(int type, int ch, int pa, int pb);
static MidiEventList *midi_val_event(int type, int ch, int val);
static MidiEventList *midi_text_event(int type, int ch, char *str);

static int print_message(int type, int len, FILE *fp);
static int nrpn_event(FILE *fp);
static int rpn_event(FILE *fp);
static int system_exclusive(FILE *fp);
static int meta_event(FILE *fp);
static int control_change(FILE *fp, unsigned char type, unsigned char val);

static int read_track(FILE *fp, int append);
static void free_midi_list(void);
static void convert_midi_list(MidiInfo *header);
static int read_header(FILE *fp, MidiInfo *header);

#ifdef CONVERT_JAPANESE
#include "kanji.h"
#endif

/*----------------------------------------------------------------
 * file handling functions
 *----------------------------------------------------------------*/

/* current file position in bytes */
static int filepos;

/* not use fseek due to piped input */
static int skip(FILE *fp, size_t len)
{
	size_t c;
	char tmp[1024];
	while (len > 0) {
		c = len;
		if (c > 1024) c = 1024;
		len -= c;
		filepos += c;
		if (c != fread(tmp, 1, c, fp)) {
			if (ctl)
			ctl->cmsg(CMSG_WARNING, 0,
				  "Corrupted midi file (pos:%d)", filepos);
			return MID_EOF;
		}
	}
	return MID_OK;
}

/* read one character */
static int readc(unsigned char *p, FILE *fp)
{
	filepos++;
	if (fread(p, 1, 1, fp) == 1)
		return 0;
	else {
		if (ctl)
			ctl->cmsg(CMSG_WARNING, 0,
				  "Corrupted midi file (pos:%d)", filepos);
		return MID_EOF;
	}
}

/* read some bytes */
static int reads(unsigned char *s, int len, FILE *fp)
{
	filepos += len;
	if (fread(s, 1, len, fp) == len)
		return 0;
	else {
		if (ctl)
			ctl->cmsg(CMSG_WARNING, 0,
				  "Corrupted midi file (pos:%d)", filepos);
		return -1;
	}
}


/* read 4bytes */
#define READID(str,fp)	reads(str,4,fp)

/* Instrument files are little-endian, MIDI files big-endian, so we
   need to do some conversions. */

#define XCHG_SHORT(x) ((((x)&0xFF)<<8) | (((x)>>8)&0xFF))
#ifdef __i486__
# define XCHG_LONG(x) \
     ({ int __value; \
        asm ("bswap %1; movl %1,%0" : "=g" (__value) : "r" (x)); \
       __value; })
#else
# define XCHG_LONG(x) ((((x)&0xFF)<<24) | \
		      (((x)&0xFF00)<<8) | \
		      (((x)&0xFF0000)>>8) | \
		      (((x)>>24)&0xFF))
#endif

/*#if ENDIAN==LITTLE_ENDIAN*/
#define BE_LONG(x)	XCHG_LONG(x)
#define BE_SHORT(x)	XCHG_SHORT(x)
#define LE_LONG(x)	(x)
#define LE_SHORT(x)	(x)
/*
#else
#define BE_LONG(x)	(x)
#define BE_SHORT(x)	(x)
#define LE_LONG(x)	XCHG_LONG(x)
#define LE_SHORT(x)	XCHG_SHORT(x)
#endif
*/

/* big endian 4bytes integer */
static int READLONG(FILE *fp)
{
	int v;
	reads((char*)&v, 4, fp);
	return BE_LONG(v);
}

/* big endian 2bytes integer */
static short READSHORT(FILE *fp)
{
	short v;
	reads((char*)&v, 2, fp);
	return BE_SHORT(v);
}		

/* little endian 2bytes integer */
static short READSHORT_LE(FILE *fp)
{
	short v;
	reads((char*)&v, 2, fp);
	return LE_SHORT(v);
}

/* Read variable-length number (7 bits per byte, MSB first) */
static int getvl(FILE *fp)
{
	int i = 0;
	int l = 0;
	unsigned char c;
	for (;;) {
		if (readc(&c, fp) != MID_OK)
			return MID_EOF;
		l += (c & 0x7f);
		if (!(c & 0x80)) {
			if (l < 0) {
				if (ctl) ctl->cmsg(CMSG_WARNING, -1, "#too deep getvl (pos:%d)", filepos);
				return MID_ERR;
			}
			return l;
		}
		l <<= 7;
		i++;
	}
}


/*----------------------------------------------------------------
 * generate a event list record
 *----------------------------------------------------------------*/

/* allocate midi event record */
static MidiEventList *alloc_midi_event(int type, int ch)
{
	MidiEventList *new=safe_malloc(sizeof(MidiEventList));
	new->event.time = at;
	new->event.type = type;
	new->event.channel = (unsigned char)ch;
	new->next = 0;
	return new;
}

/* initialize start time and midi record pointer */
static void init_midi_event(int append_mode)
{
	curev = evlist;
	if (append_mode && curev) {
		/* find the last event in the list */
		for (; curev->next; curev = curev->next)
			;
		at = curev->event.time;
	} else
		at = 0;
}

/* append midi event record */
static MidiEventList *append_to_list(MidiEventList *new)
{
	MidiEventList *next;
	if (channel_shifted)
		new->event.channel += channel_shifted;
			
	while ((next = curev->next) != NULL &&
	       (next->event.time <= new->event.time))
		curev = next;
	  
	new->next = next;
	curev->next = new;

	event_count++; /* increment the event counter */
	curev = new;

	return new;
}

/* get a midi event record with two parameters */
static MidiEventList *midi_dbl_event(int type, int ch, int pa, int pb)
{
	MidiEventList *p = append_to_list(alloc_midi_event(type, ch));
	p->event.p.par[0] = pa;
	p->event.p.par[1] = pb;
	return p;
}

/* get a midi event record with one integer parameter */
static MidiEventList *midi_val_event(int type, int ch, int val)
{
	MidiEventList *p = append_to_list(alloc_midi_event(type, ch));
	p->event.p.val = val;
	return p;
}

/* get a midi event record with a text string */
static MidiEventList *midi_text_event(int type, int ch, char *str)
{
	MidiEventList *p = append_to_list(alloc_midi_event(type, ch));
	p->event.p.text = str;
	return p;
}


/*----------------------------------------------------------------
 * midi status variables
 *----------------------------------------------------------------*/

/* currently processing status and channel */
static unsigned char laststatus, lastchan;

/* RPN & NRPN */
static unsigned char nrpn[MAX_MIDI_CHANNELS];  /* current event is NRPN? */
static int msb_bit;  /* current event is msb for RPN/NRPN */
/* RPN & NRPN indeces */
static unsigned char rpn_msb[MAX_MIDI_CHANNELS], rpn_lsb[MAX_MIDI_CHANNELS];
/* RPN & NRPN values */
static int rpn_val[MAX_MIDI_CHANNELS];

static void clear_rpn(void)
{
	int i;
	for (i = 0; i < MAX_MIDI_CHANNELS; i++) {
		nrpn[i] = 0;
		rpn_msb[i] = 127;
		rpn_lsb[i] = 127;
		rpn_val[i] = 0;
	}
	msb_bit = 0;
}

/*----------------------------------------------------------------*/

/*
 * output message
 */
static int print_message(int type, int len, FILE *fp)
{
	static char *label[]={
		"Text event: ", "Text: ", "Copyright: ", "Track name: ",
		"Instrument: ", "Lyric: ", "Marker: ", "Cue point: ",
	};
	char *s, *p;
	int slen;

	if (type < 0 || type > 7) type = 0;
	slen = strlen(label[type]);
	s = (char*)safe_malloc(len + slen + 1);
	strcpy(s, label[type]);
	for (p = s + slen; len > 0; len--) {
		readc(p, fp);
		/* convert to UNIX text format: CR+LF -> LF; just ignore CR */
		if (*p != 0x0d) p++;
	}
	*p = 0;
	midi_text_event(ME_LYRIC, lastchan, s);
	if (minfop->parse_title && !text_printed &&
	    (type == 1 || type == 3 || type == 5 || type == 6)) {
		text_printed = 1;
		if (ctl && ctl->title)
			ctl->title(s + slen); /* skip "Text: " header */
	}
	return MID_OK;
}

/*
 * NRPN events: accept as AWE32/SC88 specific controls
 */
static int nrpn_event(FILE *fp)
{
	if (rpn_msb[lastchan] == 127 && rpn_lsb[lastchan] <= 26) {
		if (! msb_bit) /* both MSB/LSB necessary */
			midi_dbl_event(ME_AWE_FX, lastchan,
				       rpn_lsb[lastchan], rpn_val[lastchan]);
	} else if (rpn_msb[lastchan] == 1) {
		if (msb_bit) /* only MSB is valid */
			midi_dbl_event(ME_GS_FX, lastchan,
				       rpn_lsb[lastchan], rpn_val[lastchan]/128);
	}
	return MID_OK;
}

/*
 * RPN events
 */
static int rpn_event(FILE *fp)
{
	int type;
	type = (rpn_msb[lastchan]<<8) | rpn_lsb[lastchan];
	switch (type) {
	case 0x0000: /* Pitch bend sensitivity */
		/* MSB only / 1 semitone per 128 */
		if (msb_bit)
			midi_val_event(ME_PITCH_SENS, lastchan, rpn_val[lastchan]);
		break;
					
	case 0x0001: /* fine tuning: */
		/* MSB/LSB, 8192=center, 100/8192 cent step */
		midi_val_event(ME_FINETUNE, lastchan, rpn_val[lastchan]);
		break;

	case 0x0002: /* coarse tuning */
		/* MSB only / 8192=center, 1 semitone per 128 */
		if (msb_bit)
			midi_val_event(ME_COARSETUNE, lastchan, rpn_val[lastchan]);
		break;

	case 0x7F7F: /* "lock-in" RPN */
		break;

	default:
		if (ctl)
			ctl->cmsg(CMSG_INFO, 2, "unsupported RPN %xh (%d) %d\n",
				  type, lastchan, rpn_val[lastchan]);
		break;
	}
	return MID_OK;
}

/*
 * parse system exclusive message
 * GM/GS/XG macros are accepted
 */
static int system_exclusive(FILE *fp)
{
	/* GM on */
	static unsigned char gm_on_macro[] = {
		0x7e,0x7f,0x09,0x01,
	};
	/* XG on */
	static unsigned char xg_on_macro[] = {
		0x43,0x10,0x4c,0x00,0x00,0x7e,0x00,
	};
	/* GS prefix
	 * drum channel: XX=0x1?(channel), YY=0x15, ZZ=on/off
	 * reverb mode: XX=0x01, YY=0x30, ZZ=0-7
	 * chorus mode: XX=0x01, YY=0x38, ZZ=0-7
	 */
	static unsigned char gs_pfx_macro[] = {
		0x41,0x10,0x42,0x12,0x40,/*XX,YY,ZZ*/
	};
	/* SC88 system mode set
	 * single module mode: XX=1
	 * double module mode: XX=0
	 */
	static unsigned char gs_mode_macro[] = {
		0x41,0x10,0x42,0x12,0x00,0x00,0x7F,/*ZZ*/
	};
	/* SC88 display macro: XX=01:bitmap, 00:text
	 */
	static unsigned char gs_disp_macro[] = {
		0x41,0x10,0x45,0x12,0x10,/*XX,00*/
	};


	unsigned char buf[1024];
	int len;
	int i; char tmp[256];

	if ((len = getvl(fp)) < 0) return len;
	if (len > sizeof(buf)) {
		if (ctl)
			ctl->cmsg(CMSG_WARNING, 0, "too long sysex!");
		return skip(fp, len);
	}

	reads(buf, len, fp);

	/* GM on */
	if (memcmp(buf, gm_on_macro, sizeof(gm_on_macro)) == 0) {
		if (ctl) ctl->cmsg(CMSG_INFO, 1, "SYSEX: GM on");
		if (minfop->midi_mode != MODE_GS &&
		    minfop->midi_mode != MODE_XG)
			minfop->midi_mode = MODE_GM;
	}

	/* GS macros */
	else if (minfop->check_gs_macro &&
		 memcmp(buf, gs_pfx_macro, sizeof(gs_pfx_macro)) == 0) {
		if (minfop->midi_mode != MODE_GS &&
		    minfop->midi_mode != MODE_XG) {
			minfop->midi_mode = MODE_GS;
			if (ctl) ctl->cmsg(CMSG_INFO, 1, "SYSEX: GS on");
		}

		if (buf[5] == 0x00 && buf[6] == 0x7f && buf[7] == 0x00) {
			/* GS reset */
			if (ctl) ctl->cmsg(CMSG_INFO, 1, "SYSEX: GS reset");

		} else if ((buf[5] & 0xf0) == 0x10 && buf[6] == 0x15) {
			/* drum pattern */
			int p = buf[5] & 0x0f;
			if (p == 0) p = 9;
			else if (p < 10) p--;
			if (buf[7] == 0)
				BITOFF(minfop->drumflag, DRUMBIT(p));
			else
				BITON(minfop->drumflag, DRUMBIT(p));
			if (ctl) ctl->cmsg(CMSG_INFO, 1, "SYSEX: drum part %d %s",
					   p, (buf[7] ? "on" : "off"));

		} else if ((buf[5] & 0xf0) == 0x10 && buf[6] == 0x21) {
			/* program */
			int p = buf[5] & 0x0f;
			if (p == 0) p = 9;
			else if (p < 10) p--;
			if (!(minfop->drumflag & DRUMBIT(lastchan)))
				midi_val_event(ME_PROGRAM, p, buf[7]);

		} else if (buf[5] == 0x01 && buf[6] == 0x30) {
			/* reverb mode */
			if (ctl) ctl->cmsg(CMSG_INFO, 1, "SYSEX: reverb_mode = %d", buf[7]);
			minfop->reverb = buf[7];

		} else if (buf[5] == 0x01 && buf[6] == 0x38) {
			/* chorus mode */
			if (ctl) ctl->cmsg(CMSG_INFO, 1, "SYSEX: chorus_mode = %d", buf[7]);
			minfop->chorus = buf[7];

		} else if (buf[5] == 0x00 && buf[6] == 0x04) {
			/* master volume */
			midi_val_event(ME_MASTER_VOLUME, 0, buf[7]);

		}
	}

	/* GS multi mode */
	else if (minfop->check_gs_macro &&
		 memcmp(buf, gs_mode_macro, sizeof(gs_mode_macro)) == 0) {
		/* single module mode */
		if (buf[7]) {
			if (ctl) ctl->cmsg(CMSG_INFO, 1, "SYSEX: single module mode");
			minfop->multi_part = FALSE;
		} else {
			if (ctl) ctl->cmsg(CMSG_INFO, 1, "SYSEX: double module mode");
			minfop->multi_part = TRUE;
		}
	}

	/* LCD macro */
	else if (minfop->check_gs_macro &&
		 memcmp(buf, gs_disp_macro, sizeof(gs_disp_macro)) == 0) {
		if (buf[5] == 0x00) {
			char *p = (char*)safe_malloc(len - 8);
			memcpy(p, buf + 7, len - 9);
			buf[len-9] = 0;
			midi_text_event(ME_LYRIC, lastchan, p);
		}
	}

	/* XG on */
	else if (minfop->check_xg_macro &&
		 memcmp(buf, xg_on_macro, sizeof(xg_on_macro)) == 0) {
		if (ctl) ctl->cmsg(CMSG_INFO, 1, "SYSEX: XG on");
		minfop->midi_mode = MODE_XG;
	}
		
	/* print sysex bytes on control interface */
	if (ctl) {
		strcpy(tmp, "SYSEX:");
		for (i = 0; i < len; i++) {
			sprintf(tmp+i*3+6," %02x", buf[i]);
			if (i >= 50) {
				sprintf(tmp+(i+1)*3+6,"...");
				break;
			}
		}
		ctl->cmsg(CMSG_INFO, 2, tmp);
	}
	
	return MID_OK;
}

/*
 * parse meta event
 * support only "end of track" and "tempo"
 */
static int meta_event(FILE *fp)
{
	unsigned char type;
	unsigned char a, b, c;
	int rc, len;

	if ((rc = readc(&type, fp)) != MID_OK) return rc;
	if ((len = getvl(fp)) < 0) return len;
	if (type > 0 && type < 16) {
		return print_message(type, len, fp);
	} else {
		switch(type) {
		case 0x2F: /* End of Track */
			return MID_END;
			
		case 0x51: /* Tempo */
			if ((rc = readc(&a, fp)) != MID_OK) return rc;
			if ((rc = readc(&b, fp)) != MID_OK) return rc;
			if ((rc = readc(&c, fp)) != MID_OK) return rc;
			midi_val_event(ME_TEMPO, 0, MidiTempo(a,b,c));
			break;
		
		default:
		/*Warning("unknown meta event: type%d, len%d", type, len);*/
			return skip(fp, len);
		}
	}
	return MID_OK;
}


/*
 * parse control change message
 */
static int control_change(FILE *fp, unsigned char type, unsigned char val)
{
	unsigned char control;

	/* controls #31 - #64 are LSB of #0 - #31 */
	msb_bit = 1;
	if (type >= 0x20 && type < 0x40) {
		msb_bit = 0;
		type -= 0x20;
	}

	control = ME_NONE;

	switch (type) {
	/* bank change */
	case CTL_BANK_SELECT:
		midi_dbl_event(ME_TONE_BANK, lastchan, val, msb_bit);
		return MID_OK;

	/* special controls */
	case 120: control = ME_ALL_SOUNDS_OFF; break;
	case 121: control = ME_RESET_CONTROLLERS; break;
	case 123: control = ME_ALL_NOTES_OFF; break;

	/* set RPN/NRPN parameter */
	case CTL_REGIST_PARM_NUM_MSB: nrpn[lastchan]=0; rpn_msb[lastchan]=val;
		return MID_OK;
	case CTL_REGIST_PARM_NUM_LSB: nrpn[lastchan]=0; rpn_lsb[lastchan]=val;
		return MID_OK;
	case CTL_NONREG_PARM_NUM_MSB: nrpn[lastchan]=1; rpn_msb[lastchan]=val;
		return MID_OK;
	case CTL_NONREG_PARM_NUM_LSB: nrpn[lastchan]=1; rpn_lsb[lastchan]=val;
		return MID_OK;

	/* send RPN/NRPN entry */
	case CTL_DATA_ENTRY:
		if (msb_bit)
			rpn_val[lastchan] = val * 128;
		else
			rpn_val[lastchan] |= val;
		if (nrpn[lastchan])
			return nrpn_event(fp);
		else
			return rpn_event(fp);
		break;
	/* increase/decrease data entry */
	case CTL_DATA_INCREMENT:
		rpn_val[lastchan]++; return rpn_event(fp);
	case CTL_DATA_DECREMENT:
		rpn_val[lastchan]--; return rpn_event(fp);

	/* CTMF specific controls */
	case 102: break; /* marker */
	case 103: /* rythm mode on/off */
		if (minfop->midi_mode == MODE_CMF && val) {
			/* rythm mode; 12-16 are drums */
			minfop->drumflag = 0xf800;
			minfop->midi_mode = MODE_CMF2;
			return MID_OK;
		}
		break;
	case 104: /* pitch increase */
		if (minfop->midi_mode == MODE_CMF ||
		    minfop->midi_mode == MODE_CMF2) {
			int shift = 0x2000 + val * 32;
			midi_val_event(ME_PITCHWHEEL, lastchan, shift);
			return MID_OK;
		}
		break;
	case 105: /* pitch decrease */
		if (minfop->midi_mode == MODE_CMF ||
		    minfop->midi_mode == MODE_CMF2) {
			int shift = 0x2000 - val * 32;
			midi_val_event(ME_PITCHWHEEL, lastchan, shift);
			return MID_OK;
		}
		break;

	/* other normal control messages */
	default:
		control = ME_CONTROL; break;
	}

	if (control == ME_CONTROL)
		midi_dbl_event(control, lastchan, type, val);
	else if (control != ME_NONE)
		midi_val_event(control, lastchan, val); 
	else if (ctl)
		ctl->cmsg(CMSG_INFO, 2, "unsupported control %d (%d) %d",
			  type, lastchan, val);

	return MID_OK;
}


/*----------------------------------------------------------------*/


/* Read a midi track into the linked list, either merging with any previous
   tracks or appending to them. */
static int read_track(FILE *fp, int append)
{
	int rc;
	int tracklen, endpos;

	init_midi_event(append);

	tracklen = 0;
	endpos = filepos;
	/* Check the formalities */
	/* read 4bytes header and track length */
	if (! (minfop->midi_mode == MODE_CMF||minfop->midi_mode == MODE_CMF2)) {
		char tmp[4];
		READID(tmp, fp);
		if (memcmp(tmp, "MTrk", 4)) {
			if (ctl)
				ctl->cmsg(CMSG_ERROR, -1, "No MTrk entry (filepos: %d)", filepos);
			return MID_FATAL;
		}
		tracklen = READLONG(fp); /* length */
		endpos = filepos + tracklen;
	}

	rc = MID_OK;
	while (rc == MID_OK && (tracklen == 0 || filepos < endpos)) {
		unsigned char me, a, b;
		int len;

		/* read timer tick */
		if ((len = getvl(fp)) < 0) {
			rc = len;
			break;
		}
		at += len; /* increase the time */

		if ((rc = readc(&me, fp)) != MID_OK)
			break;
		
		if (me == 0xF0 || me == 0xF7)
			/* SysEx event */
			rc = system_exclusive(fp);
		else if (me == 0xFF)
			/* Meta event */
			rc = meta_event(fp);
		else {
			a = me;
			if (a & 0x80) { /* status byte */
				lastchan = a & 0x0F;
				laststatus = (a>>4) & 0x07;
				if ((rc = readc(&a, fp)) != MID_OK) break;
				a &= 0x7F;
			}
			switch (laststatus) {
			case 0: /* Note off */
				if ((rc = readc(&b, fp)) != MID_OK) break;
				b &= 0x7F;
				midi_dbl_event(ME_NOTEOFF, lastchan, a, b);
				break;

			case 1: /* Note on */
				if ((rc = readc(&b, fp)) != MID_OK) break;
				b &= 0x7F;
				if (minfop->midi_mode == MODE_CMF2 &&
				    lastchan >= 11) {
					switch (lastchan) {
					case 11: a = 36; break; /* bassdrum */
					case 12: a = 38; break; /* snare */
					case 13: a = 47; break; /* tom */
					case 14: a = 49; break; /* cymbal */
					case 15: a = 44; break; /* hi-hat */
					}
				}
				midi_dbl_event(ME_NOTEON, lastchan, a, b);
				break;

			case 2: /* Key Pressure */
				if ((rc = readc(&b, fp)) != MID_OK) break;
				b &= 0x7F;
				midi_dbl_event(ME_KEYPRESSURE, lastchan, a, b);
				break;

			case 3: /* Control change */
				if ((rc = readc(&b, fp)) != MID_OK) break;
				b &= 0x7F;
				rc = control_change(fp, a, b);
				break;
				
			case 4: /* Program change */
				if (minfop->midi_mode == MODE_CMF2 &&
				    lastchan >= 11)
					break;
				midi_val_event(ME_PROGRAM, lastchan, a);
				break;
				
			case 5: /* Channel pressure */
				midi_val_event(ME_CHNPRESSURE, lastchan, a);
				break;
				
			case 6: /* Pitch wheel */
				if ((rc = readc(&b, fp)) != MID_OK) break;
				b &= 0x7F;
				midi_val_event(ME_PITCHWHEEL, lastchan,
					       (int)b * 128 + a);
				break;
				
			default: 
				if (ctl)
					ctl->cmsg(CMSG_WARNING, 0, "unknown status 0x%02X, channel 0x%02X",
					  laststatus, lastchan);
				break;
			}
		}
	}

	if (rc == MID_FATAL || rc == MID_EOF)
		return rc;

	/* skip the remainging track bytes */
	if (rc != MID_END && tracklen > 0 && filepos < endpos) {
		ctl->cmsg(CMSG_WARNING, 0, "#skip %d bytes", endpos-filepos);
		return skip(fp, endpos - filepos);
	}

	return MID_OK;
}

/* Free the linked event list from memory. */
static void free_midi_list(void)
{
	MidiEventList *meep, *next;
	for (meep = evlist; meep; meep = next) {
		next = meep->next;
		safe_free(meep);
		meep = next;
	}
	evlist=0;
}

/* merge linked list to an array together with calculating the csec event
   time */
static void convert_midi_list(MidiInfo *header)
{
	MidiEventList *meep;
	MidiEvent *evdata;
	int tempo;
	int prevt;
	int i;
	double curt, factor;

	evdata = (MidiEvent*)safe_malloc(sizeof(MidiEvent) * event_count);
	header->nlists = event_count;
	header->list = evdata;
	header->tempo = default_tempo;

	tempo = default_tempo;
	prevt = 0;
	curt = 0;
	i = 0;
	factor = (double)tempo / 10000.0 / header->divisions;
	for (meep = evlist->next; meep; meep = meep->next) {
		/* delta time in 1/100 second */
		double dtime = (double)(meep->event.time - prevt) * factor;
		curt += dtime;
		prevt = meep->event.time;
		if (meep->event.type == ME_TEMPO) {
			tempo = meep->event.p.val;
			factor = (double)tempo / 10000.0 / header->divisions;
		}
		evdata[i] = meep->event;
		/* reduce from 32 to 16 channels if necessary */
		if (evdata[i].channel >= 16 && !header->multi_part)
			evdata[i].channel -= 16;
		evdata[i].csec = (int)curt;
		i++;
		if (i >= event_count)
			break;
	}
}


/* read header info of a MIDI or CMF file */
static int read_header(FILE *fp, MidiInfo *header)
{
	int len, divisions;
	short format, tracks;
	char tmp[4];

	default_tempo = MIDI_DEFAULT_TEMPO;

	/* read 4bytes id */
	READID(tmp, fp);

	/* check RIFF format */
	if (memcmp(tmp, "RIFF", 4) == 0) {
		len = READLONG(fp);
		READID(tmp, fp);
		if (memcmp(tmp, "RMID", 4) != 0) {
			if (ctl)
				ctl->cmsg(CMSG_ERROR, -1, "not a MIDI file");
			return MID_FATAL;
		}
		for (;;) {
			READID(tmp, fp);
			if (memcmp(tmp, "data", 4) == 0) {
				len = READLONG(fp);
				READID(tmp, fp);
				break;
			}
			if (feof(fp)) {
				if (ctl)
					ctl->cmsg(CMSG_ERROR, -1, "not a MIDI file");
				return MID_FATAL;
			}
		}
	}

	if (memcmp(tmp, "CTMF", 4) == 0) { /* creative CMF file */
		short Version_Id;
		int offs1,offs2,offs3,offs4,len1,len2;
		char *s;
		if ((Version_Id=READSHORT(fp)) > 0x0101) {
			if (ctl)
				ctl->cmsg(CMSG_ERROR, -1, "Unknown version Id of CTMF file: 0x%x\n",Version_Id);
			return MID_FATAL;
		}
		offs4=READSHORT_LE(fp); /* skip 2 bytes -- offset of instrument block; */
					/* we should use this block */
		len = READSHORT_LE(fp); /* offset of music block */
		READSHORT_LE(fp); /* skip 2 bytes -- ticks per quarter note */
		divisions = READSHORT_LE(fp); /* ticks per sec */

		offs1=READSHORT_LE(fp); /* Music title */
		offs2=READSHORT_LE(fp); /* Composer name */
		offs3=READSHORT_LE(fp); /* Remarks */
#if 0
		if (offs1>offs2 || offs2>offs3 || offs3>offs4 || offs4>len || (offs1 && offs1<40) || len<40) {
			if (ctl) {
				ctl->cmsg(CMSG_ERROR, -1, "illegal CTMF file");
			}
			return MID_FATAL;
		}
#endif
		if (ctl && (offs1 || offs2 || offs3)) {
			len2=(offs1)? offs1: (offs2)? offs2: offs3;
			skip(fp,len2-20); /* skip the rest of the header */
			s = (char*)safe_malloc(len1=((offs4) ? offs4 : len)-len2);
			fread(s,1,len1,fp);
			if (offs1) {
				ctl->cmsg(CMSG_INFO, -1, "Title: %s", s);
				ctl->file_name(s);
			}
			if (offs2)
				ctl->cmsg(CMSG_INFO, -1, "Composer: %s", s+offs2-len2);
			if (offs3)
				ctl->cmsg(CMSG_INFO, -1, "Remarks: %s", s+offs3-len2);
			safe_free(s);
			skip(fp, len-offs4);
		} else
			skip(fp, len-20);

		default_tempo = 1000000;
		format = 0; /* always format 0 */
		tracks = 1;
		minfop->midi_mode = MODE_CMF;
	}
	else { /* normal MIDI file */
		len = READLONG(fp);
		if (memcmp(tmp, "MThd", 4) || len < 6) {
			if (ctl)
				ctl->cmsg(CMSG_ERROR, -1, "not a MIDI file");
			return MID_FATAL;
		}
		format = READSHORT(fp);
		tracks = READSHORT(fp);
		divisions = READSHORT(fp);

		if (divisions < 0)
			divisions = -(divisions/256) * (divisions & 0xFF);

		if (len > 6)
			skip(fp, len-6); /* skip the excess */
	}

	if (format < 0 || format >2) {
		if (ctl)
			ctl->cmsg(CMSG_ERROR, -1, "unknown MIDI file format %d", format);
		return MID_FATAL;
	}

	header->format = format;
	header->tracks = tracks;
	header->divisions = divisions;

	if (ctl)
		ctl->cmsg(CMSG_INFO, 1, "format=%d, tracks=%d, divs=%d",
			format, tracks, divisions);

	return MID_OK;
}


/* read a midi file and generate the event array */
MidiEvent *ReadMidiFile(FILE *fp, MidiInfo *header)
{
	int i, rc;
	int do_append;

	event_count = 0;
	at = 0;
	evlist = NULL;
	minfop = header;
	filepos = 0;
	text_printed = 0;
	clear_rpn();

	if (read_header(fp, header) != MID_OK)
		return NULL;

	/* Put a do-nothing event first in the list for easier processing */
	evlist = (MidiEventList*)safe_malloc(sizeof(MidiEventList));
	evlist->event.time = 0;
	evlist->event.type = ME_NONE;
	evlist->next = 0;

	channel_shifted = 0;
	switch(header->format) {
	case 0:
		if (read_track(fp, FALSE) == MID_FATAL) {
			free_midi_list();
			return 0;
		}
		break;

	case 1:
	case 2:
		if (header->format == 2)
			do_append = TRUE;
		else
			do_append = FALSE;

		for (i = 0; i < header->tracks; i++) {
			/* check multi MIDI part:
			 * normally track 0 is used for controls,
			 * track 1-16 for channel 1-16,
			 * track 17-32 for channel 17-32 */
			if (i > header->track_nums)
				channel_shifted = 16;

			if ((rc = read_track(fp, do_append)) != MID_OK) {
				if (rc == MID_EOF)
					break;
				else if (rc == MID_FATAL) {
					free_midi_list();
					return 0;
				}
			}
		}
		break;
	}

	convert_midi_list(header);
	free_midi_list();

	return header->list;
}

/* free event array */
void FreeMidiFile(MidiInfo *header)
{
	int i;
	for (i = 0; i < header->nlists; i++) {
		if (header->list[i].type == ME_LYRIC)
			safe_free(header->list[i].p.text);
	}
	safe_free(header->list);
}
