!**************************************************************************
!*
!* Boot-ROM-Code to load an operating system across a TCP/IP network.
!*
!* Module:  kernel.S
!* Purpose: Setup the stack and data areas, and then call the higher
!*          level routines.
!* Entries: Called by loader at offset 0.
!*
!**************************************************************************
!*
!* Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
!*
!*  This program is free software; you can redistribute it and/or modify
!*  it under the terms of the GNU General Public License as published by
!*  the Free Software Foundation; either version 2 of the License, or
!*  any later version.
!*
!*  This program is distributed in the hope that it will be useful,
!*  but WITHOUT ANY WARRANTY; without even the implied warranty of
!*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!*  GNU General Public License for more details.
!*
!*  You should have received a copy of the GNU General Public License
!*  along with this program; if not, write to the Free Software
!*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!*


!
!**************************************************************************
!
! Load assembler macros.
!
#include <macros.inc>
#include <memory.inc>
#include <version.inc>


!
!**************************************************************************
!
! Equates for this module:
!
STACKSIZE	equ	4096		! Size of stack
BIOS_SEG	equ	$0040		! BIOS data segment


!
!**************************************************************************
!
! Define the data segment
!
	.data

	extrn	_end_of_data


!
!**************************************************************************
!
! Define the BSS segment
!
	.bss

	extrn	_end_of_bss


!
!**************************************************************************
!
! The kernel starts here. At the beginning there is some
! miscellaneous data.
!
	.text

	entry	start

	extrn	_end_of_text
	extrn	_kernelmain
	extrn	_fatal
	extrn	prnstr
	extrn	term_packet


start:	jmp	krnstart		! the loader jumps to this address

	.align	DATAOFS			! Fill up

! The following value gets filled in by the binary patch program which
! concatenates the kernel with the packet driver programs.

prgadr:	.word	0			! Address of program pointer list

! These values can be used by the binary patch program to strip off the
! BSS segment from the final binary. The BSS segment just contains zeroes
! which dont have to be saved in the tight ROM.

	.word	_end_of_text
	.word	_end_of_data

! Put the version number into the final kernel image. This allows the
! binary patch program to verify that its using the correct kernel
! image.

	.byte	MAJOR_VER		! major version number
	.byte	MINOR_VER		! minor version number


!
!**************************************************************************
!
! Now start with the actual kernel code, which sets up the stack, copies
! the data segment into its place, and then calls the higher level code.
!
krnstart:

	cli
	cld
	mov	bx,#NEWDATA
	mov	dx,#_end_of_bss + STACKSIZE
	mov	ss,bx
	mov	sp,dx		! set new stack pointer

	xor	ax,ax
	mov	ds,ax
	mov	ax,#OLDINTS
	mov	es,ax		! save old interrupt vectors
	xor	si,si
	xor	di,di
	mov	cx,#VECTSIZE / 2
	rep
	movsw

	mov	ax,cs
	mov	ds,ax		! move data segment into lower ram
	mov	es,bx
	mov	si,#_end_of_text + $000F
	and	si,#$FFF0	! adjust to segment boundary
	mov	bx,si
	xor	di,di
	mov	cx,#_end_of_data
	mov	ax,cx
	rep
	movsb

	mov	cx,#_end_of_bss
	sub	cx,ax
	xor	al,al
	rep
	stosb			! clear BSS segment

! The boot rom code is completely within the OS loading area. In order
! to avoid overwriting it with the loaded OS we have to copy it into
! a secure memory area right at the end of the physical ram. However,
! dont copy the packet driver which gets loaded into the DOS ram
! area lateron.
! Note that DX should not be changed during all this copying.

	mov	cx,bx
	int	$12		! determine number of kB in the system
	shift	(shr,bx,10)	! compute the number of kB required
	add	bx,#(HIGHMEM >> 6) + 1
	cmp	ax,bx
	ja	memok		! check if enough memory available
	sti
	mov	bx,#memerr
	call	prnstr
	jmp	near _fatal	! print error message and terminate

memok:	mov	ax,cs
	mov	ds,ax		! copy the bootrom code into the secure
	mov	bx,#HIGHMEM	! high memory area
	mov	es,bx
	xor	si,si
	xor	di,di
	rep
	movsb

	mov	cx,#NEWDATA
	mov	es,cx		! set new data segment
#ifdef IS186
	push	bx
	push	#domain
#else
	mov	ax,#domain
	push	bx
	push	ax
#endif
	mov	ax,cs		! save for later
	retf			! call the new boot rom copy

! The main routine of the kernel gets three farpointers onto the
! stack: a pointer to the end of the data area and the two pointers
! at the beginning of the rom code. But before calling the main routine
! set the reboot flag to indicate warm boot. This is necessary so that
! a Ctrl-Alt-Del keypress reboots the system correctly.

domain:	xor	bx,bx			! set cleanup interrupt vector
	mov	ds,bx
	mov	word ptr [SERVICES * 4 + 0],#doservices
	mov	word ptr [SERVICES * 4 + 2],cs
	mov	bx,#BIOS_SEG
	mov	ds,bx
	mov	word ptr [$0072],#$1234	! indicates warm boot
	mov	ds,cx			! set new data segment

	sti
	push	ds
	add	dx,#2			! push the initial stack pointer
	push	dx			! the packet driver is still in the
	push	ax			! decompressed copy of the bootrom
	seg	cs			! in the OS loading area
	push	prgadr			! push the program pointer list address
	call	near _kernelmain	! now start the game
	add	sp,#8

! If the main program returns restore the interrupt vectors and call the
! bootstrap vector. Also deinitialize any resident program.

	mov	ah,#1
	int	SERVICES		! call the cleanup routine
	int	BOOTSTRAP		! call the bootstrap vector
dolop:	jmp	dolop


! Messages:

memerr:	.asciz	"Base mem err"


!
!**************************************************************************
!
! Interrupt routine to provide different services to the boot image.
! Input:  AH  -  function code
! Output: depends on the function called
! Registers changed: depends on the function called
!
doservices:

	sti
	or	ah,ah
	jnz	dosvc1

! Return installation information

	mov	ax,#$474B		! magic number
	mov	bx,#$0001		! service code
dosvc9:	iret


! Terminate all resident programs and the packet driver. The magic number
! given to the 2Fh interrupt is exactly what the name says: magic ;-))

dosvc1:	dec	ah			! check for function code 1
	jnz	dosvc9

	push	ds
	push	es
	mov	ax,#$CF45
	int	$2F
	mov	ax,#NEWDATA		! term_packet() needs correct data
	mov	ds,ax			! segment
	call	term_packet		! terminate packet driver

! Reset the interrupt vector table.

	cli				! make this copying atomic
	cld
	xor	ax,ax
	mov	es,ax
	mov	ax,#OLDINTS
	mov	ds,ax
	xor	si,si
	xor	di,di
	mov	cx,#VECTSIZE / 2
	rep
	movsw
	sti

! Return to caller

	pop	es
	pop	ds
	iret

!
!**************************************************************************
!
	end

