!**************************************************************************
!*
!* Boot-ROM-Code to load an operating system across a TCP/IP network.
!*
!* Module:  dosinit.S
!* Purpose: Initialize the DOS simulator, setup all interupt vectors etc.
!* Entries: _init_minidos, dosfatal, call21
!*
!**************************************************************************
!*
!* Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
!*
!*  This program is free software; you can redistribute it and/or modify
!*  it under the terms of the GNU General Public License as published by
!*  the Free Software Foundation; either version 2 of the License, or
!*  any later version.
!*
!*  This program is distributed in the hope that it will be useful,
!*  but WITHOUT ANY WARRANTY; without even the implied warranty of
!*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!*  GNU General Public License for more details.
!*
!*  You should have received a copy of the GNU General Public License
!*  along with this program; if not, write to the Free Software
!*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!*


!
!**************************************************************************
!
! Include assembler macros:
!
#include <macros.inc>
#include <memory.inc>
#include "./dospriv.inc"


!
!**************************************************************************
!
! Define macros to produce jump table entries.
!
macro tab

	extrn	dos?1			! define label as near external

	.byte	$?1			! save function number
	.word	dos?1			! save pointer to label

mend

TABSIZE	equ	3			! number of bytes per table entry


!
!**************************************************************************
!
! The DOS simulator needs its own stack when int$21 is called. This stack
! is placed in the BSS segment. Note that the real DOS uses three different
! stacks. This will not be completely simulated. The stack for function
! $59 is not necessary, so we only simulate two different stacks. The
! stack signature is contained in the registers structure, so this struc-
! ture has to come before the stack itself.
!
	.bss				! start BSS segment

	extrn	curr_psp		! segment of current psp
	extrn	first_mcb		! segment of first MCB in list

STACKSIZE	equ	512		! pretty small, so be careful.
STACKSIG	equ	$AA55		! signature for checking stack overflow

	.lcomm	dos1_regs,old_reg_size + STACKSIZE
	.lcomm	dos2_regs,old_reg_size + STACKSIZE

	.lcomm	temp_bp,2		! temporary storage for BP
	.lcomm	temp_ax,2		! temporary storage for AX
	.lcomm	temp_bx,2		! temporary storage for BX

	.comm	dos_active,1		! DOS active counter
	.lcomm	prognum,1		! current program number


!
!**************************************************************************
!
! Start code segment.
!
	.text

	public	_init_minidos		! define entry points
	public	call21
	public	dosfatal

! External routines:

	extrn	loadprog

! General library functions:

	extrn	_fatal
	extrn	prnstr
	extrn	prnword
	extrn	prnbyte
	extrn	prnchar


!
!**************************************************************************
!
! Interrupt handler table.
!
inttab:	.word	$20 * 4, int20		! terminate program
	.word	$21 * 4, int21		! dos function scheduler
	.word	$27 * 4, int27		! terminate and stay resident
	.word	$29 * 4, int29		! print character

FRSTINT	equ	$20			! first interrupt used by DOS
LSTINT	equ	$2F			! last interrupt used by DOS
INTNUM	equ	4			! number of interrupts in table


!
!**************************************************************************
!
! Initialize the mini-DOS simulator module.
! Input:  1. Arg on stack  -  far pointer to program pointer table
!         2. Arg on stack  -  far pointer to first usable memory address
! Output: None
!
_init_minidos:

	penter					! setup standard stack frame
	push	es
	push	di
	mov	word ptr dos1_regs + stack_sig,#STACKSIG	! stack sig
	mov	word ptr dos2_regs + stack_sig,#STACKSIG	! stack sig

! Initialize DOS memory by setting up the first two memory control blocks. The
! first block has a size of one paragraph and contains all zeros. It can be
! used as an empty environment for calling the DOS programs.

	getcarg	(dx,2)			! get second argument: offset
	getcarg	(bx,3)			! get second argument: segment
	shift	(shr,dx,4)		! adjust pointer to nearest paragraph
	add	dx,bx			! boundary
	add	dx,#2			! one additional paragraph for safety
	mov	bx,#LOWMEM
	sub	bx,dx			! check that we have enough memory
	jc	init3
	cmp	bx,#RAMSIZE / 16
	ja	init4
init3:	mov	ax,#FATAL_NOMEM		! fatal error -> not enough memory
	jmp	near dosfatal

init4:	cld
	sub	bx,#3			! We require three paragraphs of free
	push	ds			! memory
	mov	first_mcb,dx		! save pointer to first MCB
	mov	ds,dx
	mov	byte ptr [mcb_id],#MCB_MEM_ID
	mov	word ptr [mcb_owner],#$FFFF
	mov	word ptr [mcb_size],#1
	inc	dx
	mov	es,dx
	xor	di,di
	xor	ax,ax
	mov	cx,#8			! clear memory block
	rep
	stosw
	inc	dx
	mov	ds,dx
	mov	byte ptr [mcb_id],#MCB_END_ID
	mov	word ptr [mcb_owner],#0	! set MCB of first free memory block
	mov	word ptr [mcb_size],bx
	pop	ds

! Install interrupt handlers.

	cli
	xor	ax,ax
	mov	es,ax
	mov	di,#FRSTINT * 4		! first interrupt is $20
	mov	ax,#unused
	mov	cx,#LSTINT - FRSTINT + 1
init1:	seg	es
	mov	[di + 0],ax		! save pointer to 'unused' routine
	seg	es
	mov	[di + 2],cs
	add	di,#4			! go to next interrupt vector
	loop	init1

	mov	bx,#inttab		! pointer to interrupt table
	mov	cx,#INTNUM
init5:	seg	cs
	mov	di,[bx + 0]		! get pointer to interrupt vector
	seg	cs
	mov	ax,[bx + 2]
	seg	es
	mov	[di + 0],ax		! save offset
	seg	es
	mov	[di + 2],cs		! then segment
	add	bx,#4			! continue with next interrupt
	loop	init5
	sti

! Initialize the video screen. This ensures that the BIOS didnt leave
! us with some unusual video modes.

	mov	ah,#$0F
	int	$10
	cmp	al,#2			! check for modes 2, 3 or 7, e.g.
	je	init6			! 80x25 text mode
	cmp	al,#3
	je	init6
	cmp	al,#7
	je	init6
	mov	ax,#$0003		! set screen to mode 3
	int	$10

! Call all DOS programs sucessively

init6:	getcarg	(di,0)			! get address to program list from
	getcarg	(es,1)			! stack
init7:	seg	es
	mov	ax,[di+0]		! get far pointer to binary image
	or	ax,ax
	jz	init8			! check for end of list
	seg	es
	mov	si,[di+2]		! get far pointer to command line
	add	di,#4
	inc	byte ptr prognum
	push	di
	mov	di,ax
	call	near loadprog		! load and run the program
	pop	di
	jmp	init7

init8:	pop	di
	pop	es			! thats it
	pleave
	ret


!
!**************************************************************************
!
! Handle fatal errors by printing a message and then rebooting.
! Input:  AX  -  error number
! Output: routine does not return
!
dosfatal:

	sti
	push	ax
	mov	ax,#NEWDATA
	mov	ds,ax
	mov	bx,#errmsg
	call	prnstr			! print starting message
	pop	si
	push	si
	dec	si
	shl	si,#1
	seg	cs
	mov	bx,errtab[si]		! print error message
	call	prnstr
	pop	ax
	cmp	ax,#FATAL_PROG
	jne	fatal1
	mov	al,prognum		! print program number
	call	prnbyte
fatal1:
#ifdef DEBUG
	mov	al,dos_active
	or	al,al
	jz	fatal3			! use DOS stack
	seg	ds
	mov	es,word ptr old_stack + 2[bp]
	seg	ds
	mov	si,word ptr old_stack + 0[bp]
	jmp	fatal4

fatal3:	mov	ax,ss
	mov	es,ax			! use local stack
	mov	si,sp
fatal4:	mov	bx,#crlf
	call	prnstr
	mov	bx,#stkmsg		! print stack message
	call	prnstr
	mov	ax,es
	call	prnword
	mov	al,#$3A
	call	prnchar
	mov	ax,si			! print SS:SP:AX
	call	prnword
	mov	al,#$3A
	call	prnchar
	mov	ax,temp_ax
	call	prnword
	mov	bx,#crlf
	call	prnstr

	mov	cx,#16
fatal2:	seg	es
	lodsw
	call	prnword
	mov	bx,#crlf		! print last 16 words on stack
	call	prnstr
	loop	fatal2
#endif
fatal9:	jmp	near _fatal

! Error messages

errmsg:	.byte	$0D,$0A
	.asciz	"DOS ERROR: "

#ifdef DEBUG
stkmsg:	.asciz	"Stack: ss:sp:ax = "
crlf:	.byte	$0D,$0A,0
#endif


! The following table has to be identical with the error numbers in
! the dospriv include file:

errtab:	.word	errnom			! not enough memory
	.word	errstk			! stack overflow
	.word	errnop			! no active process
	.word	errmem			! memory corrupt
	.word	errprg			! program error

errnom:	.asciz	"no free mem"
errstk:	.asciz	"stack overflow"
errnop:	.asciz	"inv term"
errmem:	.asciz	"mem corrupt"
errprg:	.asciz	"corrupt prog "


!
!**************************************************************************
!
! Jump table for main dispatcher interrupt. Put it into code segment
! (instead const segment) to save space in lower ram area.
!
jmptab:

! functions for basic I/O:
	tab	(01)
	tab	(02)
	tab	(03)
	tab	(04)
	tab	(05)
	tab	(06)
	tab	(07)
	tab	(08)
	tab	(09)
	tab	(0A)
	tab	(0B)
	tab	(0C)

! functions for system management:
	tab	(0D)
	tab	(0E)
	tab	(19)
	tab	(1A)
	tab	(25)
	tab	(2E)
	tab	(2F)
	tab	(30)
	tab	(34)
	tab	(35)
	tab	(37)
	tab	(54)

! functions for time operations:
#ifdef NEEDTIME
	tab	(2A)
	tab	(2B)
	tab	(2C)
	tab	(2D)
#endif

! functions for memory and process management:
	tab	(00)
	tab	(31)
	tab	(48)
	tab	(49)
	tab	(4A)
	tab	(4C)
	tab	(4D)

! functions for file handling:
	tab	(3C)
	tab	(3D)
	tab	(3E)
	tab	(3F)
	tab	(40)

	.byte	$FF			! end of table

! Those functions which were only implemented before DOS2.0 return
! a different error code than those which were introduced with
! DOS2.0 and later. The first "later" function is 30h:

DOS20FN	equ	$30


!
!**************************************************************************
!
! Routine for main dispatcher interrupt 21h. This involves changing to a
! new stack, and then searching for the required function in the function
! table. The function handler has to get the callers data segment from
! old_ds if necessary.
! Input:  AH  -  DOS function number
!         others depending on function number
! Output: depending on function number
!
int21:

! First check for functions which dont use an internal stack.

int21s:	push	ds
	push	ax
	mov	ax,#NEWDATA		! switch to new data segment
	mov	ds,ax
	pop	ax
	cmp	ah,#$33			! function 33h is not implemented
	je	nostkr
	cmp	ah,#$64			! function 64h is not implemented
	je	nostkr
	cmp	ah,#$59
	jne	isno59
	mov	ax,#$00FF		! function 59h is not implemented,
	mov	bx,#$0D05		! so just return some dummy values
	mov	ch,#$01
	jmp	nostkr
isno59:	cmp	ah,#$50
	jne	isno50
	mov	curr_psp,bx		! implement function $50
	jmp	nostkr
isno50:	cmp	ah,#$51
	je	isf51			! function $51 is the same as 62h
	cmp	ah,#$62
	jne	int21r
isf51:	mov	bx,curr_psp		! implement function $51 and 62h
nostkr:	pop	ds
	iret

! For all other functions set new stack and then check in function table
! for the function handler. Select the correct stack for the function
! used. If changing anything in this code remember to also change the
! basicio functions!

int21r:	cli
	mov	temp_bp,bp
	mov	bp,#dos1_regs
	or	ah,ah
	jz	int21v
	cmp	ah,#$0C				! select correct stack
	jbe	int21u
int21v:	mov	bp,#dos2_regs
int21u:	seg	ds
	pop	old_ds[bp]
	push	ax
	mov	ax,sp
	add	ax,#2
	push	ax
	seg	ds
	pop	word ptr old_stack + 0[bp]
	seg	ds
	mov	word ptr old_stack + 2[bp],ss
	pop	temp_ax
	mov	temp_bx,bx
	mov	ax,temp_bp
	seg	ds
	mov	old_bp[bp],ax
	mov	ax,ds
	mov	bx,bp
	add	bx,#old_reg_size + STACKSIZE - 2
	mov	ss,ax				! set new stack
	mov	sp,bx

! Search for the function in the function table.

	mov	bx,#jmptab
	mov	ax,temp_ax
int21l:	seg	cs
	mov	al,[bx]
	cmp	al,#$0FF		! check for end of table
	je	notfnd
	cmp	al,ah			! found function value?
	je	int21f
	add	bx,#TABSIZE		! not found, loop to next
	jmp	int21l			! table entry

! Found table entry, so prepare registers and jump to the routine. This
! is done by putting its address onto the stack, and then executing a
! near return instruction.

int21f:	inc	byte ptr dos_active	! increment active counter
#ifdef IS186
	push	#int21t			! push return address onto stack
#else
	mov	ax,#int21t		! push return address onto stack
	push	ax
#endif
	seg	cs
	push	word ptr [bx+1]		! push jump address onto stack
	mov	bx,temp_bx
	mov	ax,temp_ax		! restore all registers except DS
	sti
	ret				! jump to handler routine

! Return to the caller by restoring the stack without changing any
! registers. Also check for stack overflow. Take care so that the
! flags dont get changed.

int21t:	cli
	pushf
	dec	byte ptr dos_active		! decrement active counter
	seg	ds				! check for stack overflow
	cmp	word ptr stack_sig[bp],#STACKSIG
	je	noerr
	mov	ax,#FATAL_STACK			! call fatal error handler
	jmp	near dosfatal

noerr:	mov	temp_bx,bx
int21e:	mov	temp_ax,ax
	pop	ax			! get flags
	seg	ds
	mov	ss,word ptr old_stack + 2[bp]
	seg	ds
	mov	sp,word ptr old_stack + 0[bp]
	mov	bx,sp
	seg	ss
	mov	byte ptr [bx+4],al	! this is the place where the INT
	mov	bx,temp_bx		! operation did put the flags
	mov	ax,temp_ax
	seg	ds
	push	old_bp[bp]
	seg	ds
	mov	ds,old_ds[bp]		! restore callers registers
	pop	bp
unused:	iret

! We couldnt find the requested function in the function table, so return
! with an error code in AX and the carry flag set. However, if it is a
! FCB related function, return with the proper error code for that.

notfnd:	cmp	ah,#DOS20FN		! check if its an old function
	jae	notfn1
	mov	al,#$0FF		! return with proper error code
	jmp	notfn2

notfn1:	mov	ax,#ERR_NOFN		! return error code
notfn2:	stc				! set carry flag to indicate error
	pushf
	jmp	int21e


!
!**************************************************************************
!
! INT20h handler: terminate current program.
! Input:  none
! Output: routine does not return
!
int20:

	xor	ah,ah
	jmp	near int21s		! just call DOS function 00h


!
!**************************************************************************
!
! INT27h handler: terminate current program and stay resident.
! Input:  DX  -  number of bytes in program memory block
! Output: routine does not return
!
int27:

	add	dx,#$000F
	rcr	dx,#1			! round byte count to nearest
	shift	(shr,dx,3)		! paragraph
	mov	ah,#$31			! just call DOS function 31h
	jmp	near int21s


!
!**************************************************************************
!
! INT29h handler: print character onto screen
! Input:  AL  -  character to print
! Output: none
!
int29:

	push	ax
	push	bx
	mov	ah,#$0E
	mov	bx,#$0007		! just print using the BIOS
	int	$10
	pop	bx
	pop	ax
	iret


!
!**************************************************************************
!
! Another way for a program to call the function dispatcher is to call
! address psp:0005. This will jump to the following address.
!
call21:

	cli
	push	bp
	mov	bp,sp
	push	ax
	mov	ax,[bp + 6]		! get return address to caller
	mov	[bp + 2],ax		! and put it into place
	pushf
	pop	[bp + 6]		! replace the old return address
	pop	ax			! with the flags. this makes the
	pop	bp			! call look like an INT.
	jmp	near int21s


!
!**************************************************************************
!
	end

