
/***

tcptable.c - table manipulation routines for the IP monitor
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
***/

#include "tcptable.h"
#include "stdwinset.h"
#include "deskman.h"
#include "attrs.h"
#include "log.h"
#include "rvnamed.h"

int revname(int lookup, struct in_addr *saddr, char *target);

void setlabels(WINDOW * win, int mode)
{
    wmove(win, 0, 42);
    whline(win, ACS_HLINE, 23);

    if (mode == 0) {
	wmove(win, 0, 47);
	wprintw(win, " Packets ");
	wmove(win, 0, 59);
	wprintw(win, " Bytes ");
    } else {
	wmove(win, 0, 45);
	wprintw(win, " Pkt Size ");
	wmove(win, 0, 56);
	wprintw(win, " Win Size ");
    }

}

void init_tcp_table(struct tcptable *table)
{
    table->borderwin = newwin(15, 80, 1, 0);
    table->borderpanel = new_panel(table->borderwin);

    wattrset(table->borderwin, BOXATTR);
    box(table->borderwin, ACS_VLINE, ACS_HLINE);
    wmove(table->borderwin, 0, 1);
    wprintw(table->borderwin, " Source ");
    wmove(table->borderwin, 0, 23);
    wprintw(table->borderwin, " Destination ");

    setlabels(table->borderwin, 0);	/* initially use mode 0 */

    wmove(table->borderwin, 0, 65);
    wprintw(table->borderwin, " Flags ");
    wmove(table->borderwin, 0, 72);
    wprintw(table->borderwin, " Iface ");
    update_panels();
    doupdate();

    table->head = table->tail = NULL;
    table->firstvisible = table->lastvisible = NULL;
    table->tcpscreen = newwin(13, 78, 2, 1);
    table->tcppanel = new_panel(table->tcpscreen);
    table->closedentries = NULL;
    wattrset(table->tcpscreen, BOXATTR);
    colorwin(table->tcpscreen);
    table->lastpos = 0;
    table->count = 0;

    wtimeout(table->tcpscreen, -1);
    stdwinset(table->tcpscreen);
}

/* add a new entry to the table */

struct tcptableent *addentry(struct tcptable *table,
			     unsigned long int saddr,
			     unsigned long int daddr,
			     unsigned int sport, unsigned int dport,
			     int protocol, char *ifname, int rev_lookup,
			     int *nomem)
{
    struct tcptableent *new_entry;
    struct closedlist *ctemp;

    /* allocate and attach a new node if no closed entries found */

    if (table->closedentries == NULL) {
	new_entry = malloc(sizeof(struct tcptableent));

	if (new_entry != NULL)
	    new_entry->oth_connection = malloc(sizeof(struct tcptableent));

	if ((new_entry->oth_connection == NULL) || (new_entry == NULL)) {
	    printnomem();
	    *nomem = 1;
	    return NULL;
	}
	new_entry->oth_connection->oth_connection = new_entry;

	if (table->head == NULL) {
	    new_entry->prev_entry = NULL;
	    table->head = new_entry;

	    table->firstvisible = new_entry;
	}
	if (table->tail != NULL) {
	    table->tail->next_entry = new_entry;
	    new_entry->prev_entry = table->tail;
	}
	table->lastpos++;
	new_entry->index = table->lastpos;
	table->lastpos++;
	new_entry->oth_connection->index = table->lastpos;

	table->tail = new_entry->oth_connection;
	new_entry->next_entry = new_entry->oth_connection;
	new_entry->next_entry->prev_entry = new_entry;
	new_entry->next_entry->next_entry = NULL;

	if (new_entry->oth_connection->index <= table->firstvisible->index + 12)
	    table->lastvisible = new_entry->oth_connection;
	else if (new_entry->index <= table->firstvisible->index + 12)
	    table->lastvisible = new_entry;

	new_entry->reused = new_entry->oth_connection->reused = 0;
	table->count++;

	wmove(table->borderwin, 14, 1);
	wprintw(table->borderwin, " TCP: %u entries ", table->count);
    } else {
	/*
	 * If we reach this point, we're allocating off the list of closed
	 * entries.  In this case, we take the top entry, let the new_entry
	 * variable point to whatever the top is pointing to.  The new_entry's
	 * oth_connection also points to the reused entry's oth_connection
	 */
	new_entry = table->closedentries->closedentry;
	new_entry->oth_connection = table->closedentries->pair;

	ctemp = table->closedentries;
	table->closedentries = table->closedentries->next_entry;
	free(ctemp);
	new_entry->reused = new_entry->oth_connection->reused = 1;
    }

    new_entry->saddr.s_addr = new_entry->oth_connection->daddr.s_addr = saddr;
    new_entry->daddr.s_addr = new_entry->oth_connection->saddr.s_addr = daddr;
    new_entry->protocol = protocol;
    new_entry->pcount = new_entry->bcount = 0;
    new_entry->win = new_entry->psize = 0;
    new_entry->oth_connection->pcount = new_entry->oth_connection->bcount = 0;
    new_entry->oth_connection->win = new_entry->oth_connection->psize = 0;
    strcpy(new_entry->ifname, ifname);
    strcpy(new_entry->oth_connection->ifname, ifname);

    new_entry->sport = new_entry->oth_connection->dport = ntohs(sport);
    new_entry->dport = new_entry->oth_connection->sport = ntohs(dport);

    new_entry->stat = new_entry->oth_connection->stat = 0;

    new_entry->s_fstat = revname(rev_lookup, &(new_entry->saddr), new_entry->s_fqdn);
    new_entry->d_fstat = revname(rev_lookup, &(new_entry->daddr), new_entry->d_fqdn);

    strcpy(new_entry->oth_connection->d_fqdn, new_entry->s_fqdn);
    strcpy(new_entry->oth_connection->s_fqdn, new_entry->d_fqdn);
    new_entry->oth_connection->s_fstat = new_entry->d_fstat;
    new_entry->oth_connection->d_fstat = new_entry->s_fstat; 

    if (new_entry->index < new_entry->oth_connection->index) {
	new_entry->half_bracket = ACS_ULCORNER;
	new_entry->oth_connection->half_bracket = ACS_LLCORNER;
    } else {
	new_entry->half_bracket = ACS_LLCORNER;
	new_entry->oth_connection->half_bracket = ACS_ULCORNER;
    }

    new_entry->inclosed = new_entry->oth_connection->inclosed = 0;
    new_entry->finack = new_entry->oth_connection->finack = 0;
    new_entry->finsent = new_entry->oth_connection->finsent = 0;
    new_entry->partial = new_entry->oth_connection->partial = 0;
    return new_entry;
}

void addtoclosedlist(struct tcptable *table, struct tcptableent *entry,
		     int *nomem)
{
    struct closedlist *ctemp;

    ctemp = malloc(sizeof(struct closedlist));

    if (ctemp == NULL) {
	printnomem();
	*nomem = 1;
	return;
    }
    ctemp->closedentry = entry;
    ctemp->pair = entry->oth_connection;
    ctemp->next_entry = table->closedentries;
    entry->inclosed = entry->oth_connection->inclosed = 1;
    table->closedentries = ctemp;
}

struct tcptableent *in_table(struct tcptable *table, unsigned long saddr,
			     unsigned long daddr, unsigned int sport,
			     unsigned int dport, int *isclosed,
			     unsigned int tm, int logging, FILE * logfile,
			     int *nomem)
{
    struct tcptableent *tblptr;
    char msgstring[120];

    int hastimeouts = 0;

    time_t now;

    if (table->head == NULL) {
	*isclosed = 0;
	return 0;
    }
    tblptr = table->head;

    do {
	if ((tblptr->saddr.s_addr == saddr) &&
	    (tblptr->daddr.s_addr == daddr) &&
	    (tblptr->sport == sport) &&
	    (tblptr->dport == dport))
	    break;

	now = time(NULL);

	/*
	 * Add the timed out entries to the closed list in case we didn't
	 * find any closed ones.
	 */

	if ((tm > 0) && ((now - tblptr->lastupdate) / 60 > tm) && (!tblptr->inclosed)) {
	    addtoclosedlist(table, tblptr, nomem);
	    if (!(*nomem))
		hastimeouts = 1;

	    if (logging) {
		sprintf(msgstring, "TCP: Connection %s:%u <-> %s:%u idle for %u minutes, marking for timeout",
			tblptr->s_fqdn, tblptr->sport, tblptr->d_fqdn, tblptr->dport,
			tm);
		writelog(logging, logfile, msgstring);
	    }
	}
	tblptr = tblptr->next_entry;
    } while (tblptr != NULL);

    if (tblptr != NULL) {	/* needed to avoid SIGSEGV */
	if (((tblptr->finsent == 2) || (tblptr->stat & FLAG_RST)) &&
	    ((tblptr->oth_connection->finsent == 2) ||
	     (tblptr->oth_connection->stat & FLAG_RST))) {
	    *isclosed = 1;
	    return NULL;
	} else {
	    *isclosed = 0;
	    return tblptr;
	}
    } else {
	*isclosed = hastimeouts;
	return NULL;
    }
}

/*
 * Update the TCP status record should an applicable packet arrive.
 */
 
void updateentry(struct tcptable *table, struct tcptableent *tableentry,
		 struct tcphdr *transpacket, unsigned long packetlength,
		 int logging, int revlook, FILE * logfile, int *nomem)
{
    char msgstring[160];

    if (tableentry->s_fstat != RESOLVED) {
        tableentry->s_fstat = revname(revlook,&(tableentry->saddr),
                                      tableentry->s_fqdn);
        strcpy(tableentry->oth_connection->d_fqdn,tableentry->s_fqdn);
        tableentry->oth_connection->d_fstat = tableentry->s_fstat;
    }
                                      
    if (tableentry->d_fstat != RESOLVED) {
        tableentry->d_fstat = revname(revlook,&(tableentry->daddr),
                                      tableentry->d_fqdn);
        strcpy(tableentry->oth_connection->s_fqdn,tableentry->d_fqdn);
        tableentry->oth_connection->s_fstat = tableentry->d_fstat;
    }
                                      
    tableentry->pcount++;

    if (tableentry->pcount == 1) {
	if ((transpacket->syn) || (transpacket->rst))
	    tableentry->partial = 0;
	else
	    tableentry->partial = 1;
    }
    tableentry->bcount += packetlength;
    tableentry->psize = packetlength;
    tableentry->win = ntohs(transpacket->window);

    tableentry->stat = 0;

    if (transpacket->syn)
	tableentry->stat |= FLAG_SYN;

    if (transpacket->ack) {
	tableentry->stat |= FLAG_ACK;

	/*
	 * The following sequences are used when the ACK is in response to
	 * a FIN (see comments for FIN below).  If the opposite direction
	 * already has its indicator set to 1 (FIN sent, not ACKed), and
	 * the incoming ACK has the same sequence number as the previously
	 * stored FIN's ack number (i.e. the ACK in response to the opposite
	 * flow's FIN), the opposite direction's state is set to 2 (FIN sent
	 * and ACKed).
	 */

	if ((tableentry->oth_connection->finsent == 1) &&
	   (ntohl(transpacket->seq) == tableentry->oth_connection->finack)) {
	    tableentry->oth_connection->finsent = 2;

	    if (logging) {
		sprintf(msgstring, "TCP: %s:%u to %s:%u: FIN acknowleged. %lu packets, %lu bytes sent",
			tableentry->s_fqdn, tableentry->sport,
			tableentry->d_fqdn, tableentry->dport,
			tableentry->pcount, tableentry->bcount);

		writelog(logging, logfile, msgstring);
	    }
	}
	/*
	 * Shall we add this entry to the closed entry list?  If both directions
	 * have their state indicators set to 2, that's it.
	 */

	if ((tableentry->finsent == 2) &&
	    (tableentry->oth_connection->finsent == 2) &&
	    (!(tableentry->inclosed)))
	    addtoclosedlist(table, tableentry, nomem);
    }
    /*
     * The closing sequence is similar, but not identical to the TCP close
     * sequence described in the RFC.  This sequence is primarily cosmetic.
     *
     * When a FIN is sent in a direction, a state indicator is set to 1,
     * to indicate a FIN sent, but not ACKed yet.  For comparison later,
     * the acknowlegement number is also saved in the entry.  See comments
     * in ACK above.
     */

    if (transpacket->fin) {

	/*
	 * First, we check if the opposite direction has no counts, in which
	 * case we simply mark the entire connection available for reuse.
	 * This is in case packets from a machine pass an interface, but
	 * on the return, completely bypasses any interface on our machine.
	 *
	 * Q: Could such a situation really happen in practice?  I managed to
	 * do it but under *really* ridiculous circumstances.
	 */

	if (tableentry->oth_connection->pcount == 0)
	    addtoclosedlist(table, tableentry, nomem);
	else {

	    /*
	     * That aside, mark the direction as being done, and make it
	     * ready for a complete close upon receipt of an ACK.  We save
	     * the acknowlegement number for identification of the proper
	     * ACK packet when it arrives in the other direction.
	     */

	    tableentry->finsent = 1;
	    tableentry->finack = ntohl(transpacket->ack_seq);
	}
	if (logging) {
	    sprintf(msgstring, "TCP: %s:%u to %s:%u: FIN sent pkt size=%u bytes",
		    tableentry->s_fqdn, tableentry->sport,
		    tableentry->d_fqdn, tableentry->dport,
		    tableentry->psize);

	    writelog(logging, logfile, msgstring);
	}
    }
    if (transpacket->rst) {
	tableentry->stat |= FLAG_RST;
	if (!(tableentry->inclosed))
	    addtoclosedlist(table, tableentry, nomem);

	if (logging) {
	    sprintf(msgstring, "TCP: %s:%u to %s:%u: Connection reset",
		    tableentry->s_fqdn, tableentry->sport,
		    tableentry->d_fqdn, tableentry->dport);
	    writelog(logging, logfile, msgstring);
	}
    }
    if (transpacket->psh)
	tableentry->stat |= FLAG_PSH;

    if (transpacket->urg)
	tableentry->stat |= FLAG_URG;

    tableentry->lastupdate = tableentry->oth_connection->lastupdate = time(NULL);
}

/*
 * Clears out the resolved IP addresses from the window.  This prevents
 * overlapping port numbers (in cases where the resolved DNS name is shorter
 * than its IP address), that may cause the illusion of large ports.  Plus,
 * such output, while may be interpreted by people with a little know-how,
 * is just plain wrong.
 *
 * Returns immediately if the entry is not visible in the window.
 */
 
void clearaddr(struct tcptable *table, struct tcptableent *tableentry,
                   unsigned int screen_idx)
{
    unsigned int target_row;
    
    if ((tableentry->index < screen_idx) ||
	(tableentry->index > screen_idx + 12))
	return;

    target_row = (tableentry->index) - screen_idx;

    wmove(table->tcpscreen, target_row, 1);
    wprintw(table->tcpscreen, "%44c",' ');
}
    
/*
 * Display a TCP connection line.  Returns immediately if the entry is
 * not visible in the window.
 */
 
void printentry(struct tcptable *table, struct tcptableent *tableentry,
		unsigned int screen_idx, int mode)
{
    char stat[7] = "";
    unsigned int target_row;

    if ((tableentry->index < screen_idx) ||
	(tableentry->index > screen_idx + 12))
	return;

    target_row = (tableentry->index) - screen_idx;

    /* clear the data if it's a reused entry */

    wattrset(table->tcpscreen, PTRATTR);
    wmove(table->tcpscreen, target_row, 2);
    if (tableentry->reused) {
        scrollok(table->tcpscreen, 0);
	wprintw(table->tcpscreen, "%76c", 32);
	scrollok(table->tcpscreen, 1);
	tableentry->reused = 0;
	wmove(table->tcpscreen, target_row, 1);
    }
 
    /* print half of connection indicator bracket */

    wmove(table->tcpscreen, target_row, 0);
    waddch(table->tcpscreen, tableentry->half_bracket);

    /* proceed with the actual entry */

    wattrset(table->tcpscreen, STDATTR);

    wmove(table->tcpscreen, target_row, 1);
    wprintw(table->tcpscreen, "%s:%u", tableentry->s_fqdn, tableentry->sport);

    wmove(table->tcpscreen, target_row, 23);
    wprintw(table->tcpscreen, "%s:%u", tableentry->d_fqdn, tableentry->dport);
    
    wattrset(table->tcpscreen, HIGHATTR);

    /* 
     * Print packet and byte counts or window size and packet size, depending
     * on the value of mode.
     */
     
    if (mode == 0) {
	wmove(table->tcpscreen, target_row, 45);
	if (tableentry->partial)
	    wprintw(table->tcpscreen, ">%8u %9u", tableentry->pcount, tableentry->bcount);
	else
	    wprintw(table->tcpscreen, " %8u %9u", tableentry->pcount, tableentry->bcount);
    } else {
	wmove(table->tcpscreen, target_row, 48);
	wprintw(table->tcpscreen, "%5u  %9u ", tableentry->psize, tableentry->win);
    }

    wattrset(table->tcpscreen, STDATTR);

    if (tableentry->finsent == 1)
	strcpy(stat, "DONE  ");
    else if (tableentry->finsent == 2)
	strcpy(stat, "CLOSED");
    else if (tableentry->stat & FLAG_RST)
	strcpy(stat, "RESET ");
    else {
	if (tableentry->stat & FLAG_SYN)
	    strcat(stat, "S");
	else
	    strcat(stat, "-");

	if (tableentry->stat & FLAG_PSH)
	    strcat(stat, "P");
	else
	    strcat(stat, "-");

	if (tableentry->stat & FLAG_ACK)
	    strcat(stat, "A");
	else
	    strcat(stat, "-");

	if (tableentry->stat & FLAG_URG)
	    strcat(stat, "U");
	else
	    strcat(stat, "-");

	strcat(stat, " ");
    }

    wmove(table->tcpscreen, target_row, 65);
    wprintw(table->tcpscreen, "%s", stat);
    wmove(table->tcpscreen, target_row, 72);
    wprintw(table->tcpscreen, "%s", tableentry->ifname);
}

void refreshtcpwin(struct tcptable *table, unsigned int idx, int mode)
{
    struct tcptableent *ptmp;
    int endloop = 0;

    setlabels(table->borderwin, mode);
    ptmp = table->firstvisible;

    if (ptmp != NULL)
	do {
	    printentry(table, ptmp, idx, mode);
	    ptmp = ptmp->next_entry;
	    if (ptmp == NULL)
		endloop = 1;
	    else if (ptmp->prev_entry == table->lastvisible)
		endloop = 1;
	} while (!endloop);

    update_panels();
    doupdate();
}

void destroytcptable(struct tcptable *table)
{
    struct tcptableent *ctemp;
    struct tcptableent *c_next_entry;
    struct closedlist *closedtemp;
    struct closedlist *closedtemp_next;

    if (table->head != NULL) {
	ctemp = table->head;
	c_next_entry = table->head->next_entry;

	while (ctemp != NULL) {
	    free(ctemp);
	    ctemp = c_next_entry;

	    if (c_next_entry != NULL)
		c_next_entry = c_next_entry->next_entry;
	}
    }
    if (table->closedentries != NULL) {
	closedtemp = table->closedentries;
	closedtemp_next = table->closedentries->next_entry;

	while (closedtemp != NULL) {
	    free(closedtemp);
	    closedtemp = closedtemp_next;
	    if (closedtemp_next != NULL)
		closedtemp_next = closedtemp_next->next_entry;
	}
    }
}
