/*
 * Various popups, such as the About... dialog.
 *
 *	create_about_popup()
 *					Create About info popup
 *	create_error_popup(widget, error, fmt, ...)
 *					Create error popup with Unix error
 *	create_nodaemon_popup()
 *					Create no-daemon warning popup
 *	create_multiple_popup()
 *					Create multiple-plan's warning popup
 *	multiple_writer_warning_popup()
 *					Create popup with write warning
 */

#include <stdio.h>
#include <errno.h>
#include <time.h>
#include <string.h>
#ifndef VARARGS
#include <stdarg.h>
#endif
#include <sys/types.h>
#include <unistd.h>
#include <Xm/Xm.h>
#include <Xm/MessageB.h>
#include <Xm/Protocols.h>
#include "cal.h"
#include "version.h"

#define NODAEMON_ONCE		/* if defined, the error popup that offers */
				/* to start pland comes up only once. */
static Boolean suppress_daemon_warning = False;

#ifndef __DATE__
#define __DATE__ ""
#endif

extern char		*progname;	/* argv[0] */
extern Display		*display;	/* everybody uses the same server */
extern Widget		mainwindow;	/* popup menus hang off main window */
extern BOOL		interactive;	/* interactive or fast appt entry? */
extern BOOL		startup_lock();


/*---------------------------------------------------------- about ----------*/
static char about_message[] = "\
\n\
Day Planner version %s\n\
Compiled %s\n\n\
Author: Thomas Driemeyer <thomas@bitrot.in-berlin.de>\n\n\
Send mail to thomas@bitrot.in-berlin.de if you would like to\n\
subscribe to the plan mailing list. Or check out my home page,\n\n\
http://www.in-berlin.de/User/bitrot\
\n";


void create_about_popup()
{
	char			msg[512];
	Widget			dialog;
	XmString		s;
	Arg			args[10];
	int			n;

	sprintf(msg, about_message, VERSION, __DATE__);
	s = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	n = 0;
	XtSetArg(args[n], XmNmessageString,	s);			n++;
	dialog = XmCreateInformationDialog(mainwindow, "About", args, n);
	XmStringFree(s);
	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));
	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON));
	(void)XmInternAtom(display, "WM_DELETE_WINDOW", False);
	XtManageChild(dialog);
}


/*---------------------------------------------------------- errors ---------*/
/*VARARGS*/
#ifndef VARARGS
void create_error_popup(Widget widget, int error, char *fmt, ...)
{
	va_list			parm;
#else
void create_error_popup(widget, error, fmt, a, b, c, d)
	Widget			widget;
	int			error, a, b, c, d;
	char			*fmt;
{
#endif
	char			msg[10240];
	Widget			dialog;
	XmString		string;
	Arg			args;

	if (!widget)
		widget = mainwindow;
	strcpy(msg, "ERROR:\n\n");

#ifndef VARARGS
	va_start(parm, fmt);
	vsprintf(msg + strlen(msg), fmt, parm);
	va_end(parm);
#else
	sprintf(msg + strlen(msg), fmt, a, b, c, d);
#endif
#	if defined(sgi) || defined(_sgi)
	if (error) {
		strcat(msg, "\n");
		strcat(msg, sys_errlist[error]);
	}
#	endif
	if (!interactive) {
		fputs(msg, stderr);
		return;
	}
	string = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args, XmNmessageString, string);
	dialog = XmCreateWarningDialog(widget, "Error", &args, 1);
	XmStringFree(string);
	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON));
	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));
	(void)XmInternAtom(display, "WM_DELETE_WINDOW", False);
	XtManageChild(dialog);
}


/*---------------------------------------------------------- no daemon ------*/
static char nodaemon_message[] = "\
Warning:\n\
There is no Scheduler daemon. Without a daemon,\n\
no action will be taken when an alarm triggers.\n\
Please start \"%s\" in your ~/.xsession or ~/.sgisession file.";

static void ok_callback(), cancel_callback();
static BOOL running;

void create_nodaemon_popup()
{
	char			msg[512];
	Widget			dialog;
	XmString		s1, s2, s3;
	Arg			args[10];
	int			n;

	if (running)
		return;
	running = TRUE;
#ifdef NODAEMON_ONCE
	if (suppress_daemon_warning)
		return;
#endif
	sprintf(msg, nodaemon_message, DAEMON_FN);
	s1 = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	s2 = XmStringCreateSimple("Start daemon now");
	s3 = XmStringCreateSimple("Continue without daemon");
	n = 0;
	XtSetArg(args[n], XmNmessageString,	s1);		n++;
	XtSetArg(args[n], XmNokLabelString,	s2);		n++;
	XtSetArg(args[n], XmNcancelLabelString,	s3);		n++;
	dialog = XmCreateWarningDialog(mainwindow, "Warning", args, n);
	XmStringFree(s1);
	XmStringFree(s2);
	XmStringFree(s3);
	XtAddCallback(dialog, XmNokCallback, ok_callback, (XtPointer)NULL);
	XtAddCallback(dialog, XmNcancelCallback, cancel_callback,
							  (XtPointer)NULL);
	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));
	(void)XmInternAtom(display, "WM_DELETE_WINDOW", False);
	XtManageChild(dialog);
}


/*
 * start the daemon. Nohup the daemon process so that it survives the death
 * of its parent.
 */

/*ARGSUSED*/
static void ok_callback(dialog)
	Widget			dialog;
{
	char			path[1024];
	PID_T			pid;

	running = FALSE;
	if (!find_file(path, DAEMON_FN, TRUE)) {
		fprintf(stderr, "%s: WARNING: can't find %s\n",
							progname, DAEMON_FN);
		create_nodaemon_popup();
		return;
	}
	pid = fork();
	if (pid > 0)
		return;
	if (pid == 0) {
		fprintf(stderr, "%s: starting %s\n", progname, DAEMON_FN);
#if defined(BSD) || defined(MIPS)
		setpgrp(0, 0);
#else
		setsid();
#endif
		execl(path, DAEMON_FN, 0);
	}
	fprintf(stderr, "%s: WARNING: can't start %s: ", progname, DAEMON_FN);
	perror("");
	create_nodaemon_popup();
}


/*
 * don't start the daemon, and set the flag that prevents the popup in the
 * future.
 */

/*ARGSUSED*/
static void cancel_callback(dialog)
	Widget			dialog;
{
	running = FALSE;
	suppress_daemon_warning = True;
}


/*---------------------------------------------------------- multiple plan's */
static char multiple_message[] = "\
Warning:\n\
Another %s program is running.\n\
%s\n\
Press Continue to start up anyway, or Kill to\n\
attempt to kill the other program. Continuing\n\
may cause loss of new appointment entries, and\n\
command-line entry of appointments will not work.";

static void kill_callback();


void create_multiple_popup(nolock)
	BOOL			nolock;		/* tried to kill competitor */
{
	char			msg[512];
	Widget			dialog;
	XmString		s1, s2, s3;
	Arg			args[10];
	int			n;

	sprintf(msg, multiple_message, progname,
		nolock ? "The other program could not be killed.\n" : "");
	s1 = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	s2 = XmStringCreateSimple("Kill");
	s3 = XmStringCreateSimple("Continue");
	n = 0;
	XtSetArg(args[n], XmNmessageString,	s1);		n++;
	XtSetArg(args[n], XmNokLabelString,	s2);		n++;
	XtSetArg(args[n], XmNcancelLabelString,	s3);		n++;
	dialog = XmCreateWarningDialog(mainwindow, "Warning", args, n);
	XmStringFree(s1);
	XmStringFree(s2);
	XmStringFree(s3);
	XtAddCallback(dialog, XmNokCallback, kill_callback, (XtPointer)NULL);
	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));
	(void)XmInternAtom(display, "WM_DELETE_WINDOW", False);
	XtManageChild(dialog);
}


/*
 * The user pressed Kill. Try to acquire a lock.
 */

/*ARGSUSED*/
static void kill_callback(dialog)
	Widget			dialog;
{
	if (!startup_lock(PLAN_PATH, TRUE))
		create_multiple_popup(mainwindow);
}


/*---------------------------------------------------------- multiple wr ----*/
void multiple_writer_warning_popup(widget)
	Widget			widget;		/* install near this widget */
{
	Widget			dialog;
	XmString		string;
	Arg			args;

	string = XmStringCreateLtoR(
"WARNING:\n\n\
You can now edit other users' appointments.\n\
plan will normally notice multiple writes at the same\n\
time (except over NFS), but will not notice if another\n\
user has modified the same appointment file while you\n\
were changing it. The older changes will get lost in\n\
this case. Watch out!", XmSTRING_DEFAULT_CHARSET);
	XtSetArg(args, XmNmessageString, string);
	dialog = XmCreateWarningDialog(widget, "Error", &args, 1);
	XmStringFree(string);
	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON));
	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));
	(void)XmInternAtom(display, "WM_DELETE_WINDOW", False);
	XtManageChild(dialog);
}
