;;;;
;;;; Symbol-Macrolet Facility for XLISP-STAT 2.1 Release 3.47 and later.
;;;; XLISP-STAT 2.1 Copyright (c) 1990-95, by Luke Tierney
;;;; Additions to Xlisp 2.1, Copyright (c) 1989 by David Michael Betz
;;;; You may give out copies of this software; for conditions see the file
;;;; COPYING included with this distribution.
;;;;

(in-package "XLISP")

;;;;;
;;;;;                   SYMBOL-MACROLET Macro
;;;;;
;;;;; This is a fairly simple implementation. Only special forms are
;;;;; handles; everything else is left to macro expansion. This
;;;;; produces rather inefficient interpreted code, but should not
;;;;; matter for compiled code. This system requires Release 3.47 or
;;;;; later. It has not been extensively tested yet.

;;**** need to do something with declarations in all special form handlers
;;**** need to do something about declarations in symbol-macrolet

(export 'symbol-macrolet)

(defmacro symbol-macrolet (defs &body body &environment env)
  (let ((form (if (consp (rest body)) `(progn ,@body) (first body)))
	(newenv (augment-environment env :symbol-macro defs)))
    (expand-symbol-macrolet form newenv)))


;;;;
;;;; Expansion and Utility Functions
;;;;

(defun lambda-expression-p (x) (and (consp x) (eq (first x) 'lambda)))

(defun symbol-macro-expand-function-call (form env)
  `(,(first form)
    ,@(mapcar #'(lambda (x) (expand-symbol-macrolet x env))
	      (rest form))))

(defun expand-symbol-macrolet (form env)
  (loop
   (cond
    ((symbolp form)
     (multiple-value-bind (newform changed) (macroexpand-1 form env)
       (if changed (setf form newform) (return form))))
    ((and (consp form) (symbolp (first form)))
     (let ((expander (get-symbol-macro-expander (first form) env)))
       (if expander
	   (multiple-value-bind
	    (newform again)
	    (funcall expander form env)
	    (if again (setf form newform) (return newform)))
	 (multiple-value-bind (newform changed) (macroexpand-1 form env)
	   (if changed
	       (setf form newform)
	       (return (symbol-macro-expand-function-call form env)))))))
    ((and (consp form) (lambda-expression-p form))
     (return (mapcar #'(lambda (x) (expand-symbol-macrolet x env)) form)))
    ((consp form) (error "bad form - ~s" form))
    (t (return form)))))

(defun symbol-macro-expand-body (form env)
  (mapcar #'(lambda (x) (expand-symbol-macrolet x env)) form))

(defun symbol-macro-expand-tagbody (form env)
  (mapcar #'(lambda (x) (if (consp x) (expand-symbol-macrolet x env) x)) form))

(defun symbol-macro-expand-lambda-list (llist env)
  (let ((args nil)
	(newllist nil))
    (flet ((get-supplied-p (x)
	     (let ((sym (first (rest (rest x)))))
	       (unless (symbolp sym) (error "bad-lambda-list - ~s" llist))
	       sym))
	   (get-and-fix-default (x)
	     (let ((dflt (second x))
		   (newenv (augment-environment env :variable args)))
	       (expand-symbol-macrolet dflt newenv))))
      (flet ((do-default-form (x)
	       (let* ((x1 (first x))
		      (sym (if (symbolp x1) x1 (second x1)))
		      (dflt (get-and-fix-default x))
		      (supp (get-supplied-p x)))
		 (push sym args)
		 (when supp (push supp args))
		 (push (if supp `(,sym ,dflt ,supp) `(,sym ,dflt)) newllist))))
        (dolist (e llist)
	  (cond
	   ((member e lambda-list-keywords) (push e newllist))
	   ((symbolp e) (push e args) (push e newllist))
	   ((and (consp e)
		 (or (symbolp (first e))
		     (and (consp (first e)) (symbolp (second (first e))))))
	    (do-default-form e))
	   (t (error "bad lambda list - ~s" llist))))))
    (values (nreverse newllist) (augment-environment env :variable args))))

(defun symbol-macro-expand-lambda-form (form env)
  (let ((name (first form))
	(llist (second form))
	(body (rest (rest form))))
    (multiple-value-bind
     (newllist newenv)
     (symbol-macro-expand-lambda-list llist env)
     `(,name ,newllist ,@(symbol-macro-expand-body body newenv)))))


;;;;
;;;; Expander Rule Implementation
;;;;

(let ((expander-table (make-hash-table :test 'eq)))

  (defun set-symbol-macro-expander (name fun)
    (setf (gethash name expander-table) fun))
    
  (defun get-symbol-macro-expander (name env)
    (unless (nth-value 1 (function-information name env))
	    (gethash name expander-table))))

(defmacro define-symbol-macro-expander (name args &body body)
  `(progn (set-symbol-macro-expander ',name #'(lambda ,args ,@body))
	  ',name))

(defmacro define-symbol-macro-expander-identity (name)
  `(define-symbol-macro-expander ,name (form env) form))

(defmacro define-symbol-macro-expander-function-like (name)
  `(define-symbol-macro-expander ,name (form env)
     (symbol-macro-expand-function-call form env)))


;;;;
;;;; Expanders for Non-Standard Special Forms
;;;;

(define-symbol-macro-expander case (form env)
  (flet ((mapexpand (x)
	   `(,(first x) ,@(symbol-macro-expand-body (rest x) env))))
    `(case ,(expand-symbol-macrolet (second form) env)
	   ,@(mapcar #'mapexpand (rest (rest form))))))

(define-symbol-macro-expander-function-like errset)
  
(define-symbol-macro-expander lambda (form env)
  (symbol-macro-expand-lambda-form form env))

(define-symbol-macro-expander-function-like nth-value)


;;;;
;;;; Expanders for Standard Special Forms
;;;;

(define-symbol-macro-expander block (form env)
  `(block ,(second form) ,@(symbol-macro-expand-body (rest (rest form)) env)))

(define-symbol-macro-expander-function-like catch)

(define-symbol-macro-expander-identity declare)

(define-symbol-macro-expander eval-when (form env)
  `(eval-when ,(second form)
	      ,@(symbol-macro-expand-body (rest (rest form)) env)))

(define-symbol-macro-expander flet (form env)
  (let* ((funs (second form))
	 (body (rest (rest form)))	       
	 (newenv (augment-environment env :function (mapcar #'first funs))))
    (flet ((expand-function (x) (symbol-macro-expand-lambda-form x env))
	   (expand-form (x) (expand-symbol-macrolet x newenv)))
    `(flet ,(mapcar #'expand-function funs) ,@(mapcar #'expand-form body)))))

(define-symbol-macro-expander function (form env)
  (if (lambda-expression-p (second form))
      `(function ,(expand-symbol-macrolet (second form) env))
      form))

(define-symbol-macro-expander-identity go)

(define-symbol-macro-expander-function-like if)

(define-symbol-macro-expander labels (form env)
  (let* ((funs (second form))
	 (body (rest (rest form)))	       
	 (newenv (augment-environment env :function (mapcar #'first funs))))
    (flet ((expand-function (x) (symbol-macro-expand-lambda-form x newenv))
	   (expand-form (x) (expand-symbol-macrolet x newenv)))
    `(labels ,(mapcar #'expand-function funs) ,@(mapcar #'expand-form body)))))

(define-symbol-macro-expander let (form env)
  (let* ((binds (mapcar #'(lambda (x) (if (symbolp x) (list x nil) x))
			(second form)))
	 (body (rest (rest form)))
	 (vars (mapcar #'first binds))
	 (vals (mapcar #'second binds))
	 (newvals (mapcar #'(lambda (x) (expand-symbol-macrolet x env)) vals))
	 (newenv (augment-environment env :variable vars)))
    `(let ,(mapcar #'list vars newvals)
       ,@(symbol-macro-expand-body body newenv))))

(define-symbol-macro-expander let* (form env)
  (let* ((binds (mapcar #'(lambda (x) (if (symbolp x) (list x nil) x))
			(second form)))
	 (body (rest (rest form)))
	 (vars (mapcar #'first binds))
	 (vals (mapcar #'second binds))
	 (newbinds nil))
    (do ((vars vars (cdr vars))
	 (vals vals (cdr vals)))
	((not (consp vars)))
	(let ((var (first vars))
	      (val (first vals)))
	  (push (list var (expand-symbol-macrolet val env)) newbinds)
	  (setf env (augment-environment env :variable (list var)))))
    `(let* ,(nreverse newbinds) ,@(symbol-macro-expand-body body env))))

(define-symbol-macro-expander-function-like locally)

(define-symbol-macro-expander macrolet (form env)
  (flet ((mkmacro (x)
	   (let ((name (first x))
		 (args (second x))
		 (body (rest (rest x))))
	     (list name (parse-macro name args body env)))))
    (let* ((macs (mapcar #'mkmacro (second form)))
	   (newenv (augment-environment env :macro macs))
	   (body (rest (rest form)))
	   (bform (if (consp (rest body)) `(progn ,@body) (first body))))
      (expand-symbol-macrolet bform newenv))))

(define-symbol-macro-expander-function-like multiple-value-call)

(define-symbol-macro-expander-function-like multiple-value-prog1)

(define-symbol-macro-expander-function-like progn)

(define-symbol-macro-expander-function-like progv)

(define-symbol-macro-expander-identity quote)

(define-symbol-macro-expander return-from (form env)
  `(return-from ,(second form) ,(expand-symbol-macrolet (third form) env)))

(define-symbol-macro-expander setq (form env)
  (destructuring-bind (name sym val &rest rest) form
    (cond
     (rest (values `(progn (setq ,sym ,val) (setq ,@rest)) t))
     (t (unless (symbolp sym) (error "not a symbol - ~s" sym))
	(let* ((place (expand-symbol-macrolet sym env))
	       (eval (expand-symbol-macrolet val env)))
	  (if (consp place)
	      (values `(setf ,place ,eval) t)
	      `(setq ,place ,eval)))))))

;; symbol-macrolet handles itself via macro expansion

(define-symbol-macro-expander tagbody (form env)
  (symbol-macro-expand-tagbody form env))

(define-symbol-macro-expander the (form env)
  `(the ,(second form) ,(expand-symbol-macrolet (third form) env)))

(define-symbol-macro-expander-function-like throw)

(define-symbol-macro-expander-function-like unwind-protect)
