# vfolder.tcl --
#
# This file contains commands which interacts with the user about performing
# different types of folder operations
#
#
#  TkRat software and its included text is Copyright 1996,1997,1998
#  by Martin Forssn
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.


# The ident of the next defined vfolder
set vFolderDefIdent 0

# The ident of the next defined vFolderStruct
set vFolderStructIdent 0

# VFolderRead --
#
# Reads the list of vfolders from disk. If needed it calls FixVFolderList
# which upgrades the list from the old format to the new.
#
# Arguments:

proc VFolderRead {} {
    global vFolderDef vFolderStruct vFolderStructIdent vFolderDefIdent \
	   vFolderVersion option vfolder_list vFolderInbox vFolderSave

    if [file readable $option(ratatosk_dir)/vfolderlist] {
	source $option(ratatosk_dir)/vfolderlist
    }

    # Upgrade old versions
    if { 0 == [info exists vFolderVersion]} {
	if [info exists vfolder_list] {
	    FixVFolderList
	}
	set vFolderVersion 4
    }
    if {3 == $vFolderVersion} {
	foreach id [array names vFolderDef] {
	    if { {} != [lindex $vFolderDef($id) 2]} {
		set vFolderDef($id) [lreplace $vFolderDef($id) 2 0 {}]
	    }
	}
	set vFolderVersion 4
    }
    if {0 == [llength [array names vFolderDef]]} {
	set vFolderDefIdent 1
	set vFolderDef(0) $option(default_folder)
	set vFolderStruct(0) {{vfolder 0 {} INBOX}}
	set vFolderVersion 4
	set vFolderInbox 0
	set vFolderSave {}
    }
    if ![info exists vFolderInbox] {
	set vFolderInbox 0
    }
    if ![info exists vFolderSave] {
	set vFolderSave {}
    }
    if ![info exists vFolderDef($vFolderInbox)] {
	set vFolderInbox [lindex [lsort -integer [array names vFolderDef]] 0]
    }
}

# SelectFileFolder --
#
# Presents a file selector to the user and opens the selected folder (if any)
#
# Arguments:
# handler -	The handler to the folder window which requested the open

proc SelectFileFolder {handler} {
    global idCnt t

    # Create identifier
    set id vfolderWinID[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0
    set oldFocus [focus]

    # Create toplevel
    toplevel $w
    wm transient $w .
    wm title $w $t(open_file)

    # Filechooser
    frame $w.chooser
    set fh [FileSelectorCreate $w.chooser ${id}(name) "set ${id}(done) 1" 0 0]

    # OK and cancel buttons
    frame $w.buttons
    button $w.buttons.ok -text $t(ok) -command "FileSelectorUpdate $fh 1"
    button $w.buttons.cancel -text $t(cancel) -command "set ${id}(done) 0"
    pack $w.buttons.ok \
	 $w.buttons.cancel -side left -expand 1

    # Pack it
    pack $w.buttons -side bottom -fill both
    pack $w.chooser -fill both -expand 1

    Place $w selectFileFolder
    grab $w
    tkwait variable ${id}(done)

    RecordPos $w selectFileFolder
    destroy $w
    focus $oldFocus

    # Do open
    if { 1 == $hd(done) } {
	FileSelectorDone $fh
	FolderRead $handler "RatOpenFolder std $hd(name)"
    }
    unset hd
}

# SelectDbasefolder --
#
# Lets the user search the database
#
# Arguments:
# handler -	The handler to the folder window which requested the open

proc SelectDbasefolder {handler} {
    global idCnt t b

    # Create identifier
    set id vfolderWinID[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0
    set hd(op) and
    set oldFocus [focus]

    # Create toplevel
    toplevel $w
    wm transient $w .
    wm title $w $t(open_dbase)

    # Populate window
    frame $w.op
    radiobutton $w.op.and -text $t(and) -variable ${id}(op) -value and
    radiobutton $w.op.or -text $t(or) -variable ${id}(op) -value or
    set b($w.op.and) dbs_and
    set b($w.op.or) dbs_or
    pack $w.op.and \
	 $w.op.or -side left -padx 5
    pack $w.op -side top

    foreach e {keywords subject all to from cc} {
	frame $w.$e
	label $w.$e.label -text $t($e):
	entry $w.$e.entry -textvariable ${id}($e) -relief sunken -width 20
	pack $w.$e.entry \
	     $w.$e.label -side right
	pack $w.$e -side top -fill both
	set b($w.$e.entry) dbs_$e
    }

    OkButtons $w $t(search) $t(cancel) "set ${id}(done)"
    pack $w.buttons -fill both

    Place $w selectDbaseFolder
    focus $w.keywords.entry
    grab $w
    tkwait variable ${id}(done)

    # Do search
    if { 1 == $hd(done) } {
	set exp $hd(op)
	foreach e {keywords subject all to from cc} {
	    if [string length $hd($e)] {
		set exp "$exp $e [list $hd($e)]"
	    }
	}
	if [string compare $hd(op) $exp] {
	    FolderRead $handler "RatOpenFolder dbase {} {} {} $exp"
	} else {
	    Popup $t(emptyexp)
	}
    }

    foreach bn [array names b $w.*] {unset b($bn)}
    RecordPos $w selectDbaseFolder
    destroy $w
    unset hd
    focus $oldFocus
}

# VFolderBuildMenu --
#
# Constructs a menu of vfolders. When one item in the menu is choosen
# $cmd is executed and the vfolder definition of the choosen folder is
# appended to $cmd. If the write argument is 1 then only those
# folders that can be written to are included in the menu.
#
# Arguments:
# m   -		The menu in which to insert the entries
# id  -		The id of the struct to start with (this should always be 0)
# cmd -		Command to execute when an item is choosen
# write -	If this argument is 1 the folders are going to be used for
#		inserting messages.

proc VFolderBuildMenu {m id cmd write} {
    global vFolderStruct vFolderDef

    if ![info exists vFolderStruct($id)] {
	return
    }
    foreach elem $vFolderStruct($id) {
	if ![string compare [lindex $elem 0] struct] {
	    set nm $m.m[lindex $elem 1]
	    $m add cascade -label [lindex $elem 2] -menu $nm
	    if ![winfo exists $nm] {
		menu $nm -postcommand "FixMenu $nm"
	    } else {
		$nm delete 1 end
	    }
	    VFolderBuildMenu $nm [lindex $elem 1] $cmd \
		    $write
	} else {
	    set vfolder $vFolderDef([lindex $elem 1])
	    set nm $m.m[lindex $elem 1]
	    if {![string compare dynamic [lindex $vfolder 1]] && 0 == $write} {
		$m add cascade -label [lindex $elem 2] -menu $nm
		if ![winfo exists $nm] {
		    menu $nm -postcommand "FixMenu $nm"
		} else {
		    $nm delete 1 end
		}
		set height [expr [winfo screenheight $nm]-20]
		foreach f [lsort [glob -nocomplain [lindex $vfolder 3]/*]] {
		    if [file isfile $f] {
			$nm add command -label [file tail $f] \
			        -command "$cmd {[list [file tail $f] file \
					              [lindex $vfolder 2] $f]}"
		    }
		}
	    } elseif {[string compare pop3 [lindex $vfolder 1]] || 0==$write} {
		$m add command -label [lindex $elem 2] \
			-command "$cmd [list $vfolder]"
	    }
	}
    }
}

# VFolderOpen --
#
# Opens the specified vfolder
#
# Arguments:
# handler -	The handler to the folder window which requested the open
# vfolder -	The definition of the vfolder to be opened

proc VFolderOpen {handler vfolder} {
    global t inbox vFolderDef vFolderInbox option
    upvar #0 $handler fh

    # Initialize browse mode
    switch $option(browse) {
	normal {set mode 0}
	browse {set mode 1}
	folder {
		array set features [lindex $vfolder 2]
		if ![info exists features(browse)] {
		    set features(browse) 0
		}
		set mode $features(browse)
	    }
    }
    set fh(browse) $mode

    switch [lindex $vfolder 1] {
    dbase {
	    set folder [FolderRead $handler \
		    [list RatOpenFolder dbase [lindex $vfolder 3] \
		     [lindex $vfolder 5] [lindex $vfolder 4] and keywords \
		     [lindex $vfolder 3]]]
	}
    imap {
	    set folder [FolderRead $handler "RatOpenFolder std \
		    [lindex $vfolder 3] [lindex $vfolder 4] IMAP"]
	}
    pop3 {
	    set folder [FolderRead $handler "RatOpenFolder std \
		    [lindex $vfolder 3] [lindex $vfolder 4] POP3"]
	}
    default {
	    set folder [FolderRead $handler "RatOpenFolder std \
		    [lindex $vfolder 3]"]
	}
    }
    if [string length $folder] {
	$folder setName [lindex $vfolder 0]
	set fh(folder_name) [lindex [$folder info] 0]
    }
    if ![string compare $vfolder $vFolderDef($vFolderInbox)] {
	set inbox $folder
    }
}

# VFolderInsert --
#
# Inserts the given messages into the given vfolder
#
# Arguments:
# handler  -	The handler to the folder window which requested the operation
# advance  -	1 if we should move to the next message on success
# messages -	The messages to move
# vfolder  -	Vfolder to insert the message into

proc VFolderInsert {handler advance messages vfolder} {
    upvar #0 $handler fh
    global option t

    if ![llength $vfolder] {
	return
    }
    foreach msg $messages {
	if [catch {$msg copy $vfolder} result] {
	    RatLog 4 "$t(insert_failed): $result"
	    break
	} else {
	    SetFlag $handler deleted 1 [$fh(folder_handler) find $msg]
	    if { 1 == $advance } {
		FolderNext $handler
	    }
	}
    }
}

# InsertIntoFile --
#
# Inserts the given message into a file. The user must specify the file.
#
# Arguments:

proc InsertIntoFile {} {
    global idCnt t option

    # Create identifier
    set id f[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0
    set oldFocus [focus]

    # Create toplevel
    toplevel $w
    wm transient $w .
    wm title $w $t(save_to_file)

    # Filechooser
    frame $w.chooser
    set fh [FileSelectorCreate $w.chooser ${id}(name) "set ${id}(done) 1" 0 0]

    # OK, perm and cancel buttons
    frame $w.buttons
    button $w.buttons.ok -text $t(ok) -command "FileSelectorUpdate $fh 1"  -bd 1
    CreateModeMenu $w.buttons.perm $id $option(permissions)
    button $w.buttons.cancel -text $t(cancel) -command "set ${id}(done) 0" -bd 1
    pack $w.buttons.ok \
	 $w.buttons.perm \
	 $w.buttons.cancel -side left -expand 1

    # Pack it
    pack $w.buttons -side bottom -fill both
    pack $w.chooser -fill both -expand 1

    Place $w insertIntoFile
    grab $w
    tkwait variable ${id}(done)

    if { 1 == $hd(done) } {
	FileSelectorDone $fh
	set result [list $hd(name) file {} $hd(name) [GetMode $id]]
    } else {
	set result {}
    }
    unset hd
    RecordPos $w insertIntoFile
    destroy $w
    focus $oldFocus

    return $result
}

# InsertIntoDBase --
#
# Inserts the given message into the database. The user must specify
# some of the arguments for the insert operation.
#
# Arguments:

proc InsertIntoDBase {} {
    global idCnt t b option

    # Create identifier
    set id f[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0
    set oldFocus [focus]

    # Create toplevel
    toplevel $w
    wm transient $w .
    wm title $w $t(insert_into_dbase)

    # Populate window
    frame $w.keywords
    label $w.keywords.label -text $t(keywords):
    entry $w.keywords.entry -textvariable ${id}(keywords) -relief sunken \
	    -width 20
    pack $w.keywords.entry \
	 $w.keywords.label -side right
    set b($w.keywords.entry) keywords
    frame $w.extype
    label $w.extype.label -text $t(extype):
    frame $w.extype.b
    radiobutton $w.extype.b.none -text $t(none) -variable ${id}(extype) \
	    -value none
    set b($w.extype.b.none) exp_none
    radiobutton $w.extype.b.remove -text $t(remove) -variable ${id}(extype) \
	    -value remove
    set b($w.extype.b.remove) exp_remove
    radiobutton $w.extype.b.incoming -text $t(add_incoming) \
	    -variable ${id}(extype) -value incoming
    set b($w.extype.b.incoming) exp_incoming
    radiobutton $w.extype.b.backup -text $t(backup) -variable ${id}(extype) \
	    -value backup
    set b($w.extype.b.backup) exp_backup
    pack $w.extype.b.none \
	 $w.extype.b.remove \
	 $w.extype.b.incoming \
	 $w.extype.b.backup -side top -anchor w
    pack $w.extype.b \
	 $w.extype.label -side right -anchor nw
    frame $w.exdate
    label $w.exdate.label -text $t(exdate):
    entry $w.exdate.entry -textvariable ${id}(exdate) -relief sunken \
	    -width 20
    pack $w.exdate.entry \
	 $w.exdate.label -side right
    set b($w.exdate.entry) exp_date
    frame $w.buttons
    frame $w.buttons.def -relief sunken -bd 1
    button $w.buttons.def.ok -text $t(insert) -command "set ${id}(done) 1"
    pack $w.buttons.def.ok -padx 4 -pady 4
    button $w.buttons.cancel -text $t(cancel) -command "set ${id}(done) 0"
    pack $w.buttons.def \
         $w.buttons.cancel -side left -expand 1 -pady 4
    pack $w.keywords \
	 $w.extype \
	 $w.exdate \
	 $w.buttons -side top -fill both

    set hd(extype) $option(def_extype)
    set hd(exdate) $option(def_exdate)
    bind $w <Return> "set ${id}(done) 1"
    Place $w insertIntoDbase
    focus $w.keywords.entry
    grab $w
    tkwait variable ${id}(done)

    # Do insert
    if { 1 == $hd(done) } {
	set result [list DBase dbase {} $hd(keywords) $hd(extype) $hd(exdate)]
    } else {
	set result {}
    }
    RecordPos $w insertIntoDbase
    foreach bn [array names b $w.*] {unset b($bn)}
    destroy $w
    focus $oldFocus
    unset hd
    return $result
}

# VFolderGetSave --
#
# Returns the default save folder specification or an empty string
#
# Arguments:

proc VFolderGetSave {} {
    global vFolderSave vFolderDef

    if {![string length $vFolderSave] 
	    || ![info exists vFolderDef($vFolderSave)]} {
	return ""
    }
    return $vFolderDef($vFolderSave)
}
