// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  SystemWindow.C

  Abstract System Window

  Stephane Rehel
  June 28, 1996
*/

#include "GL/opengl.h"

#include "SystemWindow.h"
#include "SoftWindow.h"
#include "MLEventManager.h"

#include "ScreenBlockBuffer.h"
#include "GraphicsSystem.h"
#include "WindowDriver.h"
#include "MLCursors.h"

/////////////////////////////////////////////////////////////////////////////

SystemWindow::SystemWindow()
{
  GraphicsSystem* graphicsSystem= GraphicsSystem::getGraphicsSystem();

  driver= graphicsSystem->newWindowDriver(this);
  parent= 0;
  firstRedraw= ITRUE;

  saveUnder= IFALSE;
  saveUnderBuffer= 0;
}

/////////////////////////////////////////////////////////////////////////////

SystemWindow::~SystemWindow()
{
  while( ! softChildren.empty() )
    {
    SIListElement<SoftWindow>* ile= softChildren.getLastElement();
    SoftWindow* sw= ile->element;
    softChildren.remove(ile);
    delete sw;
    }

  while( ! systemWindows.empty() )
    {
    SIListElement<SystemWindow>* ile= systemWindows.getLastElement();
    SystemWindow* sw= ile->element;
    systemWindows.remove(ile);
    delete sw;
    }

  map(IFALSE);

  delete saveUnderBuffer;
  saveUnderBuffer= 0;

  if( parent != 0 )
    {
    parent->systemWindows.remove(this);
    parent= 0;
    }

  delete driver;
  driver= 0;
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::setSaveUnder()
{
  saveUnder= ITRUE;
}

/////////////////////////////////////////////////////////////////////////////

// _parent may be NULL
IBOOL SystemWindow::create( SystemWindow* _parent,
                            int x, int y, int width, int height )
{
  if( parent != 0 || driver == 0 )
    return IFALSE;

  if( ! driver->create( _parent, x, y, width, height ) )
    return IFALSE;
  firstRedraw= ITRUE;
  isMapped= IFALSE;

  if( ! MLWindow::create( x, y, width, height ) )
    return IFALSE;

  parent= _parent;

  if( parent != 0 )
    parent->systemWindows.append(this);

  if( parent == 0 )
    setDefaultCursor();
   else
    {
    setCursor(parent->driver->cursor);

    driver->cursor= parent->driver->cursor;
    driver->pushedWaitCursor= parent->driver->pushedWaitCursor;
    driver->saveCursor= parent->driver->saveCursor;
    }

  return ITRUE;
}

/////////////////////////////////////////////////////////////////////////////

SystemWindow* SystemWindow::newSystemChild( int x, int y,
                                            int width, int height )
{
  SystemWindow* sw= new SystemWindow;

  if( ! sw->create( this, x, y, width, height ) )
    {
    delete sw;
    return 0;
    }

  return sw;
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::map( IBOOL yes /* = ITRUE */ )
{
  if( (yes?1:0) == (isMapped?1:0) )
    return;

  GraphicsSystem* graphicsSystem= GraphicsSystem::getGraphicsSystem();

  if( yes && saveUnder && !graphicsSystem->supportSaveUnder() )
    {
    delete saveUnderBuffer;
    saveUnderBuffer= new ScreenBlockBuffer;
    saveUnderBuffer->get(this);
    }

  IBOOL no_need_to_redraw= IFALSE;
  if( !yes )
    {
    if( saveUnderBuffer != 0 )
      {
      saveUnderBuffer->put(this);
      delete saveUnderBuffer;
      saveUnderBuffer= 0;
      no_need_to_redraw= ITRUE;
      }
     else
      {
      if( saveUnder )
        no_need_to_redraw= ITRUE;
      }
    }

  isMapped= yes;
  driver->map(yes);

  MLEventManager* eventManager= MLEventManager::getEventManager();

  if( eventManager != 0 )
    eventManager->removeRedraw(this);
  if( yes )
    firstRedraw= ITRUE;

  if( ! yes && ! no_need_to_redraw )
    graphicsSystem->invalidateRect(this);

  SIListIterator<SoftWindow> li1(softChildren);
  while( ! li1.eol() )
    li1.next()->map(yes);

  SIListIterator<SystemWindow> li2(systemWindows);
  while( ! li2.eol() )
    li2.next()->map(yes);
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::createContext()
{
  driver->createContext();
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::destroyContext()
{
  driver->destroyContext();
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::current()
{
  driver->current();

  glViewport( 0, 0, size.x(), size.y() );
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::setCursor( MLCursor* cursor,
                              IBOOL children_too /* = IFALSE */ )
{
  driver->setCursor(cursor);

  if( children_too )
    {
    SIListIterator<SystemWindow> li(systemWindows);
    while( ! li.eol() )
      (li++)->setCursor(cursor,children_too);
    }
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::setDefaultCursor()
{
  setCursor( MLCursors::normal );
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::pushWaitCursor()
{
  if( driver != 0 )
    driver->pushWaitCursor();

  SIListIterator<SystemWindow> li(systemWindows);
  while( ! li.eol() )
    (li++)->pushWaitCursor();

  glFlush();
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::popWaitCursor()
{
  SIListIterator<SystemWindow> li(systemWindows);
  while( ! li.eol() )
    (li++)->popWaitCursor();

  if( driver != 0 )
    driver->popWaitCursor();

  glFlush();
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::swapBuffers()
{
  if( driver != 0 )
    driver->swapBuffers();
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::setPosition( const IPoint& p )
{
  if( p == MLWindow::position )
    return;

  driver->setPosition(p);

  MLWindow::position= p;
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::setSize( const IVector& s )
{
  if( s == MLWindow::size )
    return;

  driver->setSize(s);

  MLWindow::size= s;
}

/////////////////////////////////////////////////////////////////////////////

void SystemWindow::drawXORFrame( int x1, int y1, int x2, int y2,
                                 IBOOL undo )
{
  if( x1 == x2 && y1 == y1 )
    return;
  driver->drawXORFrame(x1,y1,x2,y2,undo);
}

/////////////////////////////////////////////////////////////////////////////
