// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  MLEventManager.C

  Stephane Rehel
  June 29, 1996
*/

#include "MLEventManager.h"
#include "MLEventHandler.h"
#include "MLEventGrabber.h"
#include "MLKeyboardHandler.h"
#include "MLWindow.h"
#include "SystemWindow.h"
#include "SoftWindow.h"

/////////////////////////////////////////////////////////////////////////////

// used in dispatchEvents()
static MLWindow* pressWindow= 0;

/////////////////////////////////////////////////////////////////////////////

// static
MLEventManager* MLEventManager::eventManager= 0; // THE event manager

/////////////////////////////////////////////////////////////////////////////

MLEventManager::MLEventManager()
{
  assert( MLEventManager::eventManager == 0 );
  MLEventManager::eventManager= this;

  keyboardFocus= 0;
  eventGrabber= 0;

  // don't wait on events more than 1/20 second please
  wait_time_out= 1./20.;
}

/////////////////////////////////////////////////////////////////////////////

MLEventManager::~MLEventManager()
{
  SIList_destroyElements( MLEvent, events );
}

/////////////////////////////////////////////////////////////////////////////

// focus may be NULL for no special focus
// return previous one
MLKeyboardHandler* MLEventManager::setKeyboardFocus(
                                         MLKeyboardHandler* _keyboardFocus )
{
  MLKeyboardHandler* prev= keyboardFocus;

  keyboardFocus= _keyboardFocus;

  return prev;
}

/////////////////////////////////////////////////////////////////////////////

// grabber may be NULL for ungrabbing
// return previous one
MLEventGrabber* MLEventManager::setEventGrabber(
                                            MLEventGrabber* _eventGrabber )
{
  MLEventGrabber* prev= eventGrabber;

  eventGrabber= _eventGrabber;

  return prev;
}

/////////////////////////////////////////////////////////////////////////////

void MLEventManager::setWaitTimeout( double wto )
{
  if( wto < 0 )
    wto= 0.;
  wait_time_out= wto;
}

/////////////////////////////////////////////////////////////////////////////

#ifndef NDEBUG
#include "KeyCodes.h"
#include <stdlib.h>
#endif

// return ITRUE if only_one and it remains events in the list
IBOOL MLEventManager::dispatch( IBOOL only_one /* = IFALSE */ )
{
  while( ! events.empty() )
    {
    MLEvent* event= events.getFirst();
    events.removeFirst();

    if( event == 0 )
      continue;

    if( event->window == 0 )
      goto next;

#ifndef NDEBUG
if( event->type == MLEvent::KEY_RELEASE && event->key == KeyCodes::CTRL_X )
  exit(0);
if( event->type == MLEvent::KEY_PRESS &&
    event->key == KeyCodes::X.control().alt() )
  system("/usr/X11/bin/xmag");
#endif

    if( event->type == MLEvent::KEY_PRESS ||
        event->type == MLEvent::KEY_RELEASE )
      {
      if( eventGrabber == 0 && keyboardFocus != 0 )
        {
        if( keyboardFocus->handleMLKeyboard(*event) )
          goto next;
        }
      }

    dispatchEvent( event->window, *event );

next:
    delete event;

    if( only_one )
      return !events.empty();
    }

  return IFALSE;
}

/////////////////////////////////////////////////////////////////////////////

void MLEventManager::destroyWindow( MLWindow* window )
{
  if( window == 0 )
    return;

  removeEvents(window);

  if( pressWindow == window )
    pressWindow= 0;

  if( (void*)keyboardFocus == (void*)window )
    keyboardFocus= 0;

  if( (void*)eventGrabber == (void*)window )
    eventGrabber= 0;
}

/////////////////////////////////////////////////////////////////////////////

void MLEventManager::removeEvents( MLWindow* window )
{
  // remove all events to this window
  SIListIterator<MLEvent> li(events);
  SIListElement<MLEvent>* ile= li.eol() ? 0 : li.nextElement();
  while( ile != 0 )
    {
    SIListElement<MLEvent>* next= li.eol() ? 0 : li.nextElement();
    MLEvent* e= ile->element;
    if( e == 0 )
      continue;
    if( e->window == window )
      {
      events.remove(ile);
      delete e;
      }
    ile= next;
    }
}

/////////////////////////////////////////////////////////////////////////////

// return ITRUE if found any
IBOOL MLEventManager::removeTyped( MLWindow* window, MLEvent::Type eventType )
{
  if( window == 0 )
    return IFALSE;

  SIListIterator<MLEvent> li(events);

  SIListElement<MLEvent>* ile= li.eol() ? 0 : li.nextElement();

  IBOOL found= IFALSE;
  while( ile != 0 )
    {
    SIListElement<MLEvent>* next= li.eol() ? 0 : li.nextElement();

    MLEvent* e= ile->element;
    if( e->window == window && e->type == eventType )
      {
      events.remove(ile);
      delete e;
      found= ITRUE;
      }

    ile= next;
    }

  return found;
}

/////////////////////////////////////////////////////////////////////////////

void MLEventManager::removeRedraw( SystemWindow* window )
{
  if( window == 0 )
    return;

  SIListIterator<MLEvent> li(events);

  SIListElement<MLEvent>* ile= li.eol() ? 0 : li.nextElement();

  while( ile != 0 )
    {
    SIListElement<MLEvent>* next= li.eol() ? 0 : li.nextElement();

    MLEvent* e= ile->element;
    if( e->type == MLEvent::REDRAW )
      {
      IBOOL ok= IFALSE;
      if( e->window->isSystemWindow() )
        ok= (e->window == window);
       else
        ok= ((SoftWindow*)e->window)->systemParent == window;

      if(ok)
        {
        events.remove(ile);
        delete e;
        }
      }

    ile= next;
    }
}

/////////////////////////////////////////////////////////////////////////////

void MLEventManager::removeAnyRedraw()
{
  SIListIterator<MLEvent> li(events);

  SIListElement<MLEvent>* ile= li.eol() ? 0 : li.nextElement();

  while( ile != 0 )
    {
    SIListElement<MLEvent>* next= li.eol() ? 0 : li.nextElement();

    MLEvent* e= ile->element;
    if( e->type == MLEvent::REDRAW )
      {
      events.remove(ile);
      delete e;
      }

    ile= next;
    }
}

/////////////////////////////////////////////////////////////////////////////

IBOOL MLEventManager::findTyped( MLWindow* window, MLEvent::Type eventType )
{
  SIListIterator<MLEvent> li(events);
  while( ! li.eol() )
    {
    MLEvent* e= li.next();
    if( e->type == eventType )
      {
      if( window == 0 )
        return ITRUE;
       else
        if( e->window == window )
          return ITRUE;
      }
    }

  return IFALSE;
}

/////////////////////////////////////////////////////////////////////////////

MLEvent* MLEventManager::getTyped( MLWindow* window, MLEvent::Type eventType )
{
  SIListIterator<MLEvent> li(events);
  while( ! li.eol() )
    {
    SIListElement<MLEvent>* ile= li.nextElement();
    MLEvent* e= ile->element;
    if( e->window == window && e->type == eventType )
      {
      events.remove(ile);
      return e;
      }
    }

  return 0;
}

/////////////////////////////////////////////////////////////////////////////

IBOOL MLEventManager::sendEvent( MLWindow* window, MLEvent& event,
                                 IBOOL with_grab /* = ITRUE */ )
{
  if( window == 0 )
    return ITRUE;

  if( (event.type != MLEvent::KEY_PRESS && event.type != MLEvent::KEY_RELEASE) &&
      ! window->mapped() )
    return ITRUE;

  if( eventGrabber != 0 && with_grab )
    return eventGrabber->eventGrabber(window,event);

  if( window->_eventHandler != 0 )
    return window->_eventHandler->eventHandler(window,event);

  return window->handleEvent(event);
}

/////////////////////////////////////////////////////////////////////////////

// protected
void MLEventManager::postEvent( MLEvent* event,
                                IBOOL ignore_unmapped /* = IFALSE */ )
{
  if( event == 0 )
    return;
  if( event->window == 0 )
    {
    return;
    delete event;
    }

  if( event->window->mapped() || ignore_unmapped )
    events.append(event);
   else
    delete event;
}

/////////////////////////////////////////////////////////////////////////////

// protected
IBOOL MLEventManager::dispatchEvent( MLWindow* window, MLEvent& event )
{
  if( window == 0 )
    return IFALSE;

//  IBOOL key_force= (eventGrabber != 0) &&
//                   (event.type == MLEvent::KEY_PRESS ||
//                    event.type == MLEvent::KEY_RELEASE);
IBOOL key_force= IFALSE;
  if( !key_force && !window->mapped() )
    return IFALSE;

  switch( event.type )
    {
    case MLEvent::REDRAW:
      {
      if( window->isSystemWindow() )
        {
        if( !event.privateEvent && ((SystemWindow*)window)->firstRedraw )
          {
          // this is the draw event generated by XCreateWindow
          ((SystemWindow*)window)->firstRedraw= IFALSE;
          return ITRUE;
          }
//        if( ! findTyped(window,MLEvent::REDRAW) )
          sendEvent(window,event);
        }
       else
        {
        register SoftWindow* soft= (SoftWindow*) window;

        int x1= event.position.x() - soft->absPosition.x();
        int x2= x1 + event.size.x() - 1;
        int y1= event.position.y() - soft->absPosition.y();
        int y2= y1 + event.size.y() - 1;

        if( !( x2<0 || x1>=window->size.x() ||
               y2<0 || y1>=window->size.y() ) )
          {
          MLEvent e;
          e.type= MLEvent::REDRAW;
          e.window= window;
          e.position= IPoint(0,0);
          e.size= window->size; // full redraw...
          e.button= event.button;
          e.key= event.key;
          e.state= event.state;
          sendEvent(window,event);
          }
        }

//      if( event.privateEvent )
        return ITRUE;

      SIListIterator<SoftWindow> li(window->softChildren);
      while( ! li.eol() )
        dispatchEvent( li.next(), event );

      return ITRUE;
      }

    case MLEvent::MOUSE_PRESS:
    case MLEvent::MOUSE_RELEASE:
    case MLEvent::MOUSE_MOTION:
    case MLEvent::KEY_PRESS:
    case MLEvent::KEY_RELEASE:
      {
      static IPoint previousPosition(0,0);
#ifdef __WINDOWS__
      MLWindow* event_window= window;
#endif

      if( pressWindow != 0 )
        window= pressWindow;
       else
        {
        for( int i= window->softChildren.getNElements(); i >= 1; --i )
          {
          // event dispatching might have destroyed a window
          if( i > window->softChildren.getNElements() )
            i= window->softChildren.getNElements();
          SoftWindow* sw= window->softChildren.get(i);
          if( dispatchEvent( sw, event ) )
            return ITRUE;
          }
        }

      IPoint position;

      if( window->isSystemWindow() )
        position= event.position;
       else
        {
        register SoftWindow* soft= (SoftWindow*) window;

        position= IPoint( event.position.x() - soft->absPosition.x(),
                          event.position.y() - soft->absPosition.y() );

        if( pressWindow == 0 && !key_force )
          {
          if( position.x() < 0 ||
              position.x() >= window->size.x() ||
              position.y() < 0 ||
              position.y() >= window->size.y() )
            return IFALSE;
          }
        }

      if( pressWindow != 0 )
        {
#ifdef __WINDOWS__
        position= event.position + ( event_window->getAbsolutePosition()
                                    - pressWindow->getAbsolutePosition() );
#endif
        if( event.type == MLEvent::MOUSE_RELEASE )
          {
          IBOOL buttons;
          switch( event.button )
            {
            case 1: buttons= event.middle_pressed() || event.right_pressed();
                    break;
            case 2: buttons= event.left_pressed() || event.right_pressed();
                    break;
            case 3: buttons= event.left_pressed() || event.middle_pressed();
                    break;
            default: buttons= IFALSE;
            }
          if( ! buttons )
            pressWindow= 0;
          }
        }
       else
        {
        if( event.type == MLEvent::MOUSE_PRESS )
          {
          pressWindow= window;
          previousPosition= position;
          }
        }

      MLEvent e;
      e.type= event.type;
      e.time= event.time;
      e.window= window;
      e.position= position;
      if( pressWindow != 0 && event.type == MLEvent::MOUSE_MOTION )
        {
        e.size= position - previousPosition;
        previousPosition= position;
        }
       else
        e.size= event.size;
      e.button= event.button;
      e.key= event.key;
      e.state= event.state;

      sendEvent( e.window, e );

      return ITRUE;
      }

    default:
      {
      // private event?
      sendEvent(window,event);
      break;
      }
    }

  return IFALSE;
}

/////////////////////////////////////////////////////////////////////////////

SIList<SystemWindow>& MLEventManager::getSystemChildren( SystemWindow* sw )
{
  return sw->systemWindows;
}

/////////////////////////////////////////////////////////////////////////////
