/*
Copyright (c) 1991-1995 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: batching.c,v 1.6 1997/11/13 23:13:41 spreitze Exp $ */
/* Last edited by Mike Spreitzer November 13, 1997 2:28 pm PST */

#include "iluntrnl.h"

#include "connect.h"
#include "transprt.h"
#include "mooring.h"

#include "oscalls.h"

#define BUFFERSIZE	1024

typedef struct {
  ilu_TransportCreator	lower;
  ilu_cardinal		period;		/* microseconds */
  ilu_cardinal		buffersize;
}              *BatchingCreatorParms;

typedef struct {
  ilu_Mooring		lower;
  ilu_cardinal		period;		/* microseconds */
  ilu_cardinal		buffersize;
}              *BatchingMooringParms;

typedef struct {

  /* protected by "buffer_mutex", which also serves as the "xmu"
   * the transport layer underneath
   */

  ilu_Mutex		buffer_mutex;
  ilu_bytes		buffer;
  ilu_cardinal		buffersize;
  ilu_cardinal		buffernext;
  ilu_Error		error;
  ilu_boolean		alarm_set;

  /*L1, L2 unconstrained*/

  ilu_refany		alarm;
  ilu_FineTime		period;

  /* L2 unconstrained */

  ilu_Transport	lower;
  ilu_boolean	lower_boundaried;
  /*
   * The unboundaried transport from which (self) is constructed. We
   * don't want an exposed buffer in (lower).  lower's xmu = self's
   * xmu; same for ymu.
   */
}              *BatchingParms;
/* What goes in the data field of a ilu_Transport. */

/*L1, L2 unconstrained*/

#define ACCESS_BATCHING_PARMS(a) ((BatchingParms)(a))

/*L1.sup < trmu; L2 unconstrained*/
static          BatchingCreatorParms
_batching_InterpretInfo(ilu_TransportInfo info,
			ILU_ERRS((no_memory, inv_objref)) * err)
{
  BatchingCreatorParms    cp;
  ilu_TransportCreator lower;
  unsigned long period, buffersize;

  if (sscanf(info[0], "batching_%lu_%lu", &period, &buffersize) != 2 ||
      info[1] == NIL)
    return ILU_ERR_CONS1(inv_objref, err, minor, ilu_iom_ts, NIL);
  lower = _ilu_GetTransportCreator(info + 1, err);
  if (ILU_ERRNOK(*err))
    return NIL;
  if (!lower->tcr_reliable)
    return ILU_ERR_CONS1(inv_objref, err, minor, ilu_iom_ts, NIL);
  cp = (BatchingCreatorParms) ilu_MallocE(sizeof(*cp), err);
  if (cp == NIL)
    return NIL;
  cp->period = period;
  cp->buffersize = buffersize;
  cp->lower = lower;
  return cp;
}

/* L1.sup < trmu; L2 >= {xmu, ymu} */
static          ilu_boolean
_batching_SetInputHandler(ilu_Transport self, ilu_TIH tih,
			  ILU_ERRS((no_memory, internal, no_resources)) * err)
{
  BatchingParms   p = ACCESS_BATCHING_PARMS(transport_data(self));
  ilu_boolean val;
  ilu_AcquireMutex(p->buffer_mutex);
  val = _ilu_SetTransportInputHandler(p->lower, tih, err);
  ilu_ReleaseMutex(p->buffer_mutex);
  return val;
}

/* Main Invariant holds; L2 >= {ymu} */
static          ilu_boolean
_batching_WaitForInput(ilu_Transport self, int *disabled,
		       ilu_FineTime * limit,
		       ILU_ERRS((broken_locks, interrupted)) * err)
{
  BatchingParms   p = ACCESS_BATCHING_PARMS(transport_data(self));
  return (transport_wait_for_input(p->lower, disabled, limit, err));
}

/* L1.sup < trmu; L2 >= {xmu} */
static          ilu_boolean
_batching_InterruptST(ilu_Transport self, ILU_ERRS((bad_param)) * err)
{
  BatchingParms   p = ACCESS_BATCHING_PARMS(transport_data(self));
  ilu_boolean val;
  ilu_AcquireMutex(p->buffer_mutex);
  val = transport_interruptST(p->lower, err);
  ilu_ReleaseMutex(p->buffer_mutex);
  return val;
}

/* L1 >= {cmu}; L1.sup < trmu; L2 >= {xmu} */

static          ilu_boolean
_batching_DisableWait(ilu_Transport self,
		      ILU_ERRS((broken_locks, bad_param,
				internal)) * err)
{
  BatchingParms   p = ACCESS_BATCHING_PARMS(transport_data(self));
  ilu_boolean val;
  ilu_AcquireMutex(p->buffer_mutex);
  val = transport_disableWait(p->lower, err);
  ilu_ReleaseMutex(p->buffer_mutex);
  return val;  
}

static          ilu_boolean
_batching_EnableWait(ilu_Transport self,
		     ILU_ERRS((broken_locks, bad_param,
			       internal)) * err)
{
  BatchingParms   p = ACCESS_BATCHING_PARMS(transport_data(self));
  ilu_boolean val;
  ilu_AcquireMutex(p->buffer_mutex);
  val = transport_enableWait(p->lower, err);
  ilu_ReleaseMutex(p->buffer_mutex);
  return val;
}

/* L1.sup < trmu; L2 unconstrained */

static          ilu_integer
_batching_FdUsage(ilu_TransportCreator self, ilu_boolean mooring)
{
  BatchingCreatorParms    cp = (BatchingCreatorParms) self->tcr_data;
  return (*cp->lower->tcr_dfd) (cp->lower, mooring);
}

/* L1.sup < trmu; L2 >= {xmu, ymu} */
static          ilu_integer
_batching_CloseDFd(ilu_Transport self)
{
  BatchingParms   p = ACCESS_BATCHING_PARMS(transport_data(self));
  ilu_boolean val;
  ilu_AcquireMutex(p->buffer_mutex);
  val = (*p->lower->tr_class->tc_closeDFd) (p->lower);
  ilu_ReleaseMutex(p->buffer_mutex);
  return val;
}

/* Main Invariant holds; L2 >= {xmu}; input => L2 >= {ymu} */

static          ilu_ReadHeaderResultCode
_batching_BeginMessage(ilu_Transport self,
		       ilu_boolean input_p,
		       ILU_ERRS((IoErrs)) * err)
{
  return ILU_ERR_CONS1(internal, err, minor, ilu_im_tcNotBoundaried,
		       ilu_rhrc_error);
}

static          ilu_boolean
_batching_EndMessage(ilu_Transport self,
		     ilu_boolean flush,
		     ilu_Message * msgh,
		     ILU_ERRS((IoErrs)) * err)
{
  return ILU_ERR_CONS1(internal, err, minor, ilu_im_tcNotBoundaried,
		       ilu_rhrc_error);
}

/*Main Invariant holds; L2 >= {xmu}*/

static          ilu_boolean
_batching_SendWholeMessage(ilu_Transport self, ilu_Message * msgh,
			   ILU_ERRS((IoErrs)) * err)
{
  return ILU_ERR_CONS1(internal, err, minor, ilu_im_tcReliable, FALSE);
}

static void
  __batching_FlushBuffer (ilu_refany rock)
{
  ilu_Transport self = (ilu_Transport) rock;
  BatchingParms p = ACCESS_BATCHING_PARMS(transport_data(((ilu_Transport) rock)));
  ilu_AcquireMutex(p->buffer_mutex);
  if (p->alarm_set) {
    if (p->buffernext > 0) {
      if ((p->lower->tr_class->tc_write_bytes)(p->lower, p->buffer, p->buffernext, ilu_TRUE, &p->error)) {
	p->buffernext = 0;
      }
    }
    p->alarm_set = ilu_FALSE;
  }
  ilu_ReleaseMutex(p->buffer_mutex);
  return;
}

static          ilu_boolean
_batching_WriteBytes(ilu_Transport self, ilu_bytes b,
		     ilu_cardinal bufferSize,
		     ilu_boolean flush,
		     ILU_ERRS((IoErrs)) * err)
{
  BatchingParms   p = ACCESS_BATCHING_PARMS(transport_data(self));

  ilu_AcquireMutex(p->buffer_mutex);
  if (ILU_ERRNOK(p->error)) {
    *err = p->error;
    goto err1;
  };
  if (self->tr_outNext > 0) {
    if ((p->buffersize - p->buffernext) < self->tr_outNext) {
      if (!(p->lower->tr_class->tc_write_bytes(p->lower, p->buffer, p->buffernext, ilu_FALSE, err)))
	goto err1;
      p->buffernext = 0;
    };
    memcpy((void *) (p->buffer + p->buffernext),
	   (void *) self->tr_outBuff, self->tr_outNext);
    p->buffernext += self->tr_outNext;
    self->tr_outNext = 0;
  };

  if ((p->buffersize - p->buffernext) < bufferSize) {
    if (!(p->lower->tr_class->tc_write_bytes(p->lower, p->buffer, p->buffernext, ilu_FALSE, err)))
      goto err1;
    p->buffernext = 0;
  };
  if (bufferSize >= p->buffersize) {
    if (!(p->lower->tr_class->tc_write_bytes(p->lower, b, bufferSize, ilu_FALSE, err)))
      goto err1;
  } else {
    memcpy((void *) (p->buffer + p->buffernext), (void *) b, bufferSize);
    p->buffernext += bufferSize;
    ILU_CLER(*err);
  }

  if (p->alarm != NIL) {
    if ((p->buffernext > 0) && (!p->alarm_set)) {
      ilu_SetAlarm (p->alarm, ilu_FineTime_Add(ilu_FineTime_Now(), p->period),
		    __batching_FlushBuffer, self);
      p->alarm_set = ilu_TRUE;
    } else if ((p->buffernext == 0) && (p->alarm_set)) {
      ilu_UnsetAlarm (p->alarm);
      p->alarm_set = ilu_FALSE;
    }
  }

 err1:
  ilu_ReleaseMutex(p->buffer_mutex);
  return ILU_ERROK(*err);
}

/*Main Invariant holds; L2 >= {xmu, ymu}*/

static          ilu_cardinal
_batching_ReadBytes(ilu_Transport self,
		    ilu_bytes buffer,
		    ilu_cardinal len,
		    ilu_TransportReport * rpt,
		    ILU_ERRS((IoErrs)) * err)
{
  BatchingParms   p = ACCESS_BATCHING_PARMS(transport_data(self));
  ilu_cardinal	  val;
  if ((self->tr_inLimit - self->tr_inNext) > 0)
    return ILU_ERR_CONS1(internal, err, minor, ilu_im_tcInputSkipsBuff, 0);
  /* just get the bytes from the next layer... */
  ilu_AcquireMutex(p->buffer_mutex);
  self->tr_inLimit = transport_read_upto_bytes(p->lower, self->tr_inBuff, p->buffersize, rpt, err);
  val = MIN(len, self->tr_inLimit);
  memcpy((void *) buffer, (void *) self->tr_inBuff, val);
  self->tr_inNext = val;
  ilu_ReleaseMutex(p->buffer_mutex);
  return val;
}

/*L1.sup < trmu; L2 >= {xmu, ymu}*/

static          ilu_boolean
_batching_Close(ilu_Transport self, ilu_integer * dfd,
		ILU_ERRS((bad_locks, broken_locks, internal)) * err)
{
  BatchingParms   p = ACCESS_BATCHING_PARMS(transport_data(self));
  ilu_Transport   lt = p->lower;
  ilu_boolean     val;
  ilu_Error       lerr;
  ilu_AcquireMutex(p->buffer_mutex);
  val = transport_close(lt, dfd, err);
  ilu_ReleaseMutex(p->buffer_mutex);
  ilu_DestroyMutex(p->buffer_mutex, &lerr);
  if (ILU_ERRNOK(lerr)) {
    ILU_HANDLED(lerr);
  };
  if (p->alarm != NIL) {
    ilu_UnsetAlarm(p->alarm);
    ilu_DestroyAlarm(p->alarm);
  };
  if (p->buffer != NIL)
    ilu_free(p->buffer);
  if (ILU_ERRNOK(p->error)) {
    ILU_HANDLED(p->error);
  };
  ilu_free(self->tr_outBuff);	/* note also frees inBuff, since
				 * both are allocated with the same
				 * malloc() */
  ilu_free(p);
  ilu_free(self);
  return val;
}

/*L1, L2 unconstrained*/

static struct _ilu_TransportClass_s myclass = {
  ilu_FALSE,			/* boundaried */
  ilu_TRUE,			/* reliable */
  _batching_CloseDFd,
  _batching_SetInputHandler,
  _batching_WaitForInput,
  _batching_InterruptST,
  _batching_DisableWait,
  _batching_EnableWait,
  _batching_BeginMessage,
  _batching_EndMessage,
  _batching_SendWholeMessage,
  _batching_WriteBytes,
  _batching_ReadBytes,
  _batching_Close
};

/*L1.sup < trmu; L2 >= result's {xmu, ymu}*/
static          ilu_Transport
NewTrans(ilu_Transport lower, ilu_integer * dfd, ilu_cardinal period,
	 ilu_cardinal buffersize, ILU_ERRS((no_memory)) * err)
{
  ilu_Transport   ans = NIL;
  BatchingParms   parms = NIL;
  char buf[1000];
  parms = (BatchingParms) ilu_MallocE(sizeof(*parms), err);
  if (parms == NIL)
    return NIL;
  sprintf (buf, "_%lu_%lu_%p", period, buffersize, parms);
  parms->lower = lower;
  if ((parms->buffer_mutex = ilu_CreateMutex("batching", buf)) == NIL)
    goto fale;
  if ((parms->buffer = ilu_MallocE(buffersize, err)) == NIL)
    goto fale0;
  parms->buffersize = buffersize;
  parms->buffernext = 0;
  if (period > 0) {
    double temp = period;
    parms->period = ilu_FineTime_FromDouble(temp / 1000000.0);
    if ((parms->alarm = ilu_CreateAlarm()) == NIL)
      goto fale1;
  } else {
    parms->period = ilu_FineTime_FromDouble(0.0);
    parms->alarm = NIL;
  }
  parms->alarm_set = ilu_FALSE;
  ILU_CLER(parms->error);
  if ((ans = (ilu_Transport) ilu_MallocE(sizeof(*ans), err)) == NIL)
    goto fale2;
  if ((ans->tr_outBuff = (unsigned char *) ilu_MallocE(16 + parms->buffersize, err)) == NIL)
    goto fale3;
  ans->tr_inBuff = ans->tr_outBuff + 16;
  ans->tr_inNext = ans->tr_inLimit = 0;
  ans->tr_outNext = 0;
  ans->tr_outLimit = 16;
  ans->tr_class = &myclass;
  ans->tr_data = parms;
  ILU_CLER(*err);
  return ans;
 fale3:
  ilu_free(ans);
 fale2:
  if (parms->alarm != NIL)
    ilu_DestroyAlarm(parms->alarm);
 fale1:
  ilu_free(parms->buffer);
 fale0: {
   ILU_ERRS((internal, bad_params)) cerr;
   ilu_DestroyMutex(parms->buffer_mutex, &cerr);
   ILU_HANDLED(cerr);
 }
 fale:{
   ILU_ERRS((bad_locks, broken_locks, internal)) cerr;
   ilu_integer     cdfd = 0;
   ilu_free(parms);
   transport_close(lower, &cdfd, &cerr);
   ILU_HANDLED(cerr);
   *dfd += cdfd;
   return NIL;
  }
}

ilu_boolean
ilu_batching_Flush (ilu_Serializer si,
		    ILU_ERRS((bad_locks, broken_locks,
			      bad_param)) * err)
{
  ilu_Server      server;
  ilu_Error	lerr;
  ilu_Transport self;

  if (!si || !(server = si->si_server))
    return ILU_ERR_CONS1(bad_param, err, minor, ilu_bpm_duh, FALSE);
  if (!ilu_EnterServerMutex(server, TRUE, err))
    return FALSE;
  self = connection_transport(si->si_conn);
  if (transport_class(self) == &myclass) {
    BatchingParms p = ACCESS_BATCHING_PARMS(transport_data(self));
    ilu_AcquireMutex(p->buffer_mutex);
    if (p->buffernext > 0) {
      if ((p->lower->tr_class->tc_write_bytes)(p->lower, p->buffer, p->buffernext, ilu_TRUE, err)) {
	p->buffernext = 0;
	if ((p->alarm != NIL) && p->alarm_set) {
	  ilu_UnsetAlarm (p->alarm);
	  p->alarm_set = ilu_FALSE;
	};
      }
    }
  exitpoint:
    ilu_ReleaseMutex(p->buffer_mutex);
  } else {
    ILU_ERR_CONS1(bad_param, err, minor, ilu_bpm_non_batching, 0);
  }
  if (!ilu_ExitServerMutex(server, TRUE, &lerr)) {
    ILU_HANDLED(*err);
    *err = lerr;
    return FALSE;
  } else {
    return ILU_ERROK(*err);
  }
}

/* Main Invariant holds */
static          ilu_Transport
_batching_CreateTransport(ilu_TransportCreator self, ilu_boolean buffer,
			  ilu_integer *dfd, ilu_Passport pp,
			  ILU_ERRS((IoErrs)) * err)
{
  BatchingCreatorParms    cp = (BatchingCreatorParms) self->tcr_data;
  ilu_Transport   lower;
  lower = (*cp->lower->tcr_createTransport) (cp->lower, FALSE, dfd, pp,
					     err);
  if (ILU_ERRNOK(*err))
    return NIL;
  if (!transport_reliable(lower))
    /* He promised! (We checked earlier) */
    return ILU_ERR_CONS1(internal, err, minor, ilu_im_tcBug, NIL);
  /* pretend to acquire result's xmu, ymu */
  return NewTrans(lower, dfd, cp->period, cp->buffersize, err);
  /* pretend to release result's xmu, ymu */
}

/*L1.sup < trmu; L2 >= {xmu, ymu}*/
static ilu_integer
  _batching_MooringDFd(ilu_Mooring self, ilu_boolean add)
{
  BatchingMooringParms mp = (BatchingMooringParms) self->mo_data;
  return ((*mp->lower->mo_dfd) (mp->lower, add));
}

/*L1.sup < trmu; L2 >= {xmu, ymu}*/
static          ilu_boolean
_batching_SetReqHandler(ilu_Mooring self,
			ilu_TIH tih,
			ILU_ERRS((no_memory, imp_limit, no_resources,
				  broken_locks, internal)) * err)
{
  BatchingMooringParms mp = (BatchingMooringParms) self->mo_data;
  return ((*mp->lower->mo_set_req_handler) (mp->lower, tih, err));
}

/* Main Invariant holds; L2 >= {ymu} */
static          ilu_boolean
_batching_WaitForReq(ilu_Mooring self, int *disabled,
	   ILU_ERRS((interrupted, broken_locks)) * err)
{
  BatchingMooringParms mp = (BatchingMooringParms) self->mo_data;
  return ((*mp->lower->mo_wait_for_req) (mp->lower, disabled, err));
}

/* L1 >= {cmu}; L1.sup < trmu; L2 >= {xmu} */

static          ilu_boolean
_batching_DisableReqWait(ilu_Mooring self,
	       ILU_ERRS((broken_locks, bad_param, internal)) * err)
{
  BatchingMooringParms mp = (BatchingMooringParms) self->mo_data;
  return ((*mp->lower->mo_disableWait) (mp->lower, err));
}

static          ilu_boolean
_batching_EnableReqWait(ilu_Mooring self,
	       ILU_ERRS((broken_locks, bad_param, internal)) * err)
{
  BatchingMooringParms mp = (BatchingMooringParms) self->mo_data;
  return ((*mp->lower->mo_enableWait) (mp->lower, err));
}

/* Main Invariant holds; L2 >= self's {xmu, ymu} */

static          ilu_Transport
_batching_AcceptClient(ilu_Mooring self, ilu_string * tinfo_out,
		       ilu_integer *dfd, ilu_Passport pp,
		       ILU_ERRS((IoErrs)) * err)
{
  BatchingMooringParms mp = (BatchingMooringParms) self->mo_data;
  ilu_Transport lower;
  ilu_string      subtinfo = NIL;
  ilu_Transport   ans = NIL;
  lower = ((*mp->lower->mo_accept_connection)
	   (mp->lower, tinfo_out ? &subtinfo : NIL, dfd, pp, err));
  if (ILU_ERRNOK(*err) || lower == NIL)
    goto dun;
  if (tinfo_out) {
    char buf[1000];
    sprintf (buf, "batching_%lu_%lu", mp->period, mp->buffersize);
    *tinfo_out = ilu_Strcat3E(buf, " over ", subtinfo, err);
    if (ILU_ERRNOK(*err))
      goto dun;
  }
  ans = NewTrans(lower, dfd, mp->period, mp->buffersize, err);
dun:
  if (subtinfo)
    ilu_free(subtinfo);
  return ans;
}

/*L1.sup < trmu; L2 >= {xmu, ymu}*/
static          ilu_boolean
_batching_CloseMooring(ilu_Mooring self, ilu_integer * dfd,
		       ILU_ERRS((bad_locks, broken_locks,
				 internal)) * err)
{
  BatchingMooringParms mp = (BatchingMooringParms) self->mo_data;
  if (!(*mp->lower->mo_close) (mp->lower, dfd, err))
    return FALSE;
  ilu_free(mp);
  ilu_free(self);
  return TRUE;
}

/*L1, L2 unconstrained*/

static struct _ilu_Mooring_s mooringProto = {
  _batching_MooringDFd,
  _batching_SetReqHandler,
  _batching_WaitForReq,
  _batching_DisableReqWait,
  _batching_EnableReqWait,
  _batching_AcceptClient,
  _batching_CloseMooring,
  NIL				/* data */
};

/*L1.sup < trmu*/
static          ilu_Mooring
_batching_CreateMooring(ilu_TransportCreator self,
			ilu_TransportInfo * tinfo_out,
			ilu_boolean buffer,
			ilu_integer *dfd,
			ilu_Passport pp,	/* unused here */
			ILU_ERRS((no_memory)) * err)
{
  BatchingCreatorParms    cp = (BatchingCreatorParms) self->tcr_data;
  BatchingMooringParms	  mp;
  ilu_Mooring     lower, ans;
  ilu_TransportInfo      subtinfo = NIL;
  lower = ((*cp->lower->tcr_createMooring)
	   (cp->lower, tinfo_out ? &subtinfo : NIL, FALSE, dfd, pp, err));
  if (ILU_ERRNOK(*err))
    return NIL;
  ans = (ilu_Mooring) ilu_MallocE(sizeof(*ans), err);
  if (ans == NIL)
    return NIL;
  mp = (BatchingMooringParms) ilu_MallocE(sizeof(*mp), err);
  if (ans == NIL)
    return NIL;
  if (tinfo_out) {
    char buf[1000];
    sprintf (buf, "batching_%lu_%lu", cp->period, cp->buffersize);
    *tinfo_out = _ilu_ConcatTinfo (buf, subtinfo, err);
    if (ILU_ERRNOK(*err))
      return NIL;
    else
      ilu_free(subtinfo);
  }
  *ans = mooringProto;
  mp->buffersize = cp->buffersize;
  mp->period = cp->period;
  mp->lower = lower;
  ans->mo_data = (ilu_refany) mp;
  
  return (ans);
}

static void _batching_CloseCreator(ilu_TransportCreator self)
{
  BatchingCreatorParms    cp = (BatchingCreatorParms) self->tcr_data;
  (*cp->lower->tcr_close) (cp->lower);
  ilu_free(cp);
  ilu_free(self);
  return;
}

static struct _ilu_TransportCreator_s myCreatorProto = {
  ilu_FALSE,			/* boundaried */
  ilu_TRUE,			/* reliable */
  0,				/* tcr_holds */
  FALSE,			/* tcr_wantClose */
  _batching_FdUsage,
  _batching_CreateTransport,
  _batching_CreateMooring,
  _batching_CloseCreator,
  NIL				/* data */
};

/*L1.sup < trmu*/
ilu_TransportCreator
_ilu_batching_TransportCreator(ilu_TransportInfo tinfo,
			       ILU_ERRS((no_memory,
					 inv_objref)) * err)
{
  ilu_TransportCreator ans;
  BatchingCreatorParms    cp;
  cp = _batching_InterpretInfo(tinfo, err);
  if (ILU_ERRNOK(*err))
    return NIL;
  ans = (ilu_TransportCreator) ilu_MallocE(sizeof(*ans), err);
  if (ans == NIL)
    return NIL;
  *ans = myCreatorProto;
  ans->tcr_data = cp;
  ILU_CLER(*err);
  return ans;
}

