------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              B I N D G E N                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.77 $                             --
--                                                                          --
--          Copyright (C) 1992-1997 Free Software Foundation, Inc.          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

with ALI;      use ALI;
with Binde;    use Binde;
with Hostparm;
with Namet;    use Namet;
with Opt;      use Opt;
with Osint;    use Osint;
with Types;    use Types;
with Sdefault; use Sdefault;

with GNAT.OS_Lib;

package body Bindgen is

   Statement_Buffer : String (1 .. 1000);
   --  Buffer used for constructing output statements

   Last : Natural := 0;
   --  Last location in Statement_Buffer currently set

   With_Finalization : Boolean := False;
   --  Flag which indicates whether the program use finalization
   --  (presence of the unit System.Finalization_Implementation)

   With_Tasking : Boolean := False;

   Default_Object_Dir : String_Ptr := Object_Dir_Default_Name;
   Default_Object_Len : Integer := Default_Object_Dir.all'Length;

   -----------------------
   -- Local Subprograms --
   -----------------------

   function ABE_Boolean_Required (U : Unit_Id) return Boolean;
   --  Given a unit id value U, determines if the corresponding unit requires
   --  an access-before-elaboration check variable, i.e. it is a non-predefined
   --  body for which no pragma Elaborate, Elaborate_All or Elaborate_Body is
   --  present, and thus could require ABE checks.

   procedure Gen_Adainit (Main : Boolean);
   --  Generates the Adainit procedure. Main is true if main program is Ada.

   procedure Gen_Adafinal;
   --  Generate the Adafinal procedure as required by thr RM

   procedure Gen_Elab_Calls;
   --  Generate sequence of elaboration calls

   procedure Gen_Main_Program_File;
   --  Generate lines for output file in main program case

   procedure Gen_Non_Main_Program_File;
   --  Generate lines for output file in non-main program case

   procedure List_Object_Files_Options;
   --  Output a comment containing a list of the full names of the object
   --  files to be linked and the list of linker options supplised by
   --  Linker_Options pragmas in the source.

   procedure List_Versions;
   --  Output series of definitions for unit versions

   procedure Set_Char (C : Character);
   --  Set given character in Statement_Buffer at the Last + 1 position
   --  and increment Last by one to reflect the stored character.

   procedure Set_Int (N : Int);
   --  Set given value in decimal in Statement_Buffer with no spaces
   --  starting at the Last + 1 position, and updating Last past the value.
   --  A minus sign is output for a negative value.

   procedure Set_String (S : String);
   --  Sets characters of given string in Statement_Buffer, starting at the
   --  Last + 1 position, and updating last past the string value.

   procedure Set_Unit_Name;
   --  Given a unit name in the Name_Buffer, copies it to Statement_Buffer,
   --  starting at the Last + 1 position, and updating last past the value.
   --  changing periods to double underscores, and updating Last appropriately.

   procedure Tab_To (N : Natural);
   --  If Last is greater than or equal to N, no effect, otherwise store
   --  blanks in Statement_Buffer bumping Last, until Last = N.

   procedure Write_Statement_Buffer;
   --  Write out contents of statement buffer up to Last, and reset Last to 0

   --------------------------
   -- ABE_Boolean_Required --
   --------------------------

   function ABE_Boolean_Required (U : Unit_Id) return Boolean is
      Typ   : constant Unit_Type := Unit.Table (U).Utype;
      Units : Unit_Id;

   begin
      if Typ /= Is_Body then
         return False;

      else
         Units := U + 1;

         return (not Unit.Table (Units).Pure)
                   and then
                (not Unit.Table (Units).Preelab)
                   and then
                (not Unit.Table (Units).Elaborate_Body)
                   and then
                (not Unit.Table (Units).Predefined);
      end if;
   end ABE_Boolean_Required;

   -----------------
   -- Gen_Adainit --
   -----------------

   procedure Gen_Adainit (Main : Boolean) is
   begin

      Write_Binder_Info ("void adafinal ();");
      Write_Binder_Info ("void adainit ()");
      Write_Binder_Info ("{");

      if not Main then

         --  The flag ada__init_flag is used to ensure that only the first
         --  call to adainit has an effect (RM B.1(39)). It is false (zero)
         --  before the first call, and true thereafter.

         Write_Binder_Info ("   static int ada__init_flag = 0;");
         Write_Binder_Info ("   if (ada__init_flag) return;");
         Write_Binder_Info ("   ada__init_flag++;");
      end if;

      --  Generate call to set the runtime global variables defined in
      --  a-init.c. We define the varables in a-init.c, rather than in
      --  the binder generated file itself to avoid undefined externals
      --  when the runtime is linked as a shareable image library.

      --  We call the routine from inside adainit() because this works for
      --  both programs with and those without binder generated "main"
      --  functions. This means, of course, that adafinal() must be defined
      --  before adainit(), so we can pass its address.

      Write_Binder_Info
         ("   __gnat_set_globals (");

      Set_String ("      ");
      Set_Int (ALIs.Table (ALIs.First).Main_Priority);
      Set_Char (',');
      Tab_To (13);
      Set_String ("/* Main_Priority              */");
      Write_Statement_Buffer;

      Set_String ("      ");
      Set_Int (ALIs.Table (ALIs.First).Time_Slice_Value);
      Set_Char (',');
      Tab_To (13);
      Set_String ("/* Time_Slice_Value           */");
      Write_Statement_Buffer;

      Set_String ("      '");
      Set_Char (Locking_Policy);
      Set_String ("',");
      Tab_To (13);
      Set_String ("/* Locking_Policy             */");
      Write_Statement_Buffer;

      Set_String ("      '");
      Set_Char (Queuing_Policy);
      Set_String ("',");
      Tab_To (13);
      Set_String ("/* Queuing_Policy             */");
      Write_Statement_Buffer;

      Set_String ("      '");
      Set_Char (Task_Dispatching_Policy);
      Set_String ("',");
      Tab_To (13);
      Set_String ("/* Tasking_Dispatching_Policy */");
      Write_Statement_Buffer;

      Write_Binder_Info ("      adafinal);");
      Gen_Elab_Calls;
      Write_Binder_Info ("}");
   end Gen_Adainit;

   ------------------
   -- Gen_Adafinal --
   ------------------

   procedure Gen_Adafinal is
   begin
      Write_Binder_Info ("void adafinal () {");

      if With_Tasking then
         Write_Binder_Info ("   system__tasking__stages"
           & "__finalize_global_tasks ();");

      elsif With_Finalization then
         Write_Binder_Info ("   system__finalization_implementation"
           & "__finalize_global_list ();");
      end if;

      Write_Binder_Info ("}");
   end Gen_Adafinal;

   --------------------
   -- Gen_Elab_Calls --
   --------------------

   procedure Gen_Elab_Calls is
   begin
      for E in Elab_Order.First .. Elab_Order.Last loop
         Get_Name_String (Unit.Table (Elab_Order.Table (E)).Uname);

         --  if the program uses finalization we must make sure to finalize
         --  global objects too at the end of the program.

         if Name_Buffer (1 .. 34) = "system.finalization_implementation" then
            With_Finalization := True;
         end if;

         if Name_Buffer (1 .. 21) = "system.tasking.stages" then
            With_Tasking := True;
         end if;

         --  Passive units are excluded from elaboration

         if not Unit.Table (Elab_Order.Table (E)).Shared_Passive then

            --  Generate elaboration call if elaboration needed, and a comment
            --  if no elaboration is required.

            if Unit.Table (Elab_Order.Table (E)).No_Elab then
               Set_String ("/* ");
            else
               Set_String ("   ");
            end if;

            Set_Unit_Name;
            Set_String ("___elab");
            Set_Char (Name_Buffer (Name_Len)); -- 's' or 'b' for spec/body
            Set_String (" ();");

            if Unit.Table (Elab_Order.Table (E)).No_Elab then
               Set_String (" */");
            end if;

            Write_Statement_Buffer;
         end if;
      end loop;
   end Gen_Elab_Calls;

   ---------------------------
   -- Gen_Main_Program_File --
   ---------------------------

   procedure Gen_Main_Program_File is

      procedure Set_Main_Program_Name;
      --  Given the main program name in Name_Buffer (length in Name_Len)
      --  generate the name of the routine to be used in the call. The name
      --  is generated starting at Last + 1, and Last is updated past it.

      procedure Set_Main_Program_Name is
      begin
         --  Note that name has %b on the end which we ignore

         --  Output initial _ada_ if no dots in name

         for J in 1 .. Name_Len - 1 loop
            if J = Name_Len - 1 then
               Set_String ("_ada_");
            else
               exit when Name_Buffer (J) = '.';
            end if;
         end loop;

         --  Copy name, changing dots to double underscores

         for J in 1 .. Name_Len - 2 loop
            if Name_Buffer (J) = '.' then
               Set_String ("__");
            else
               Set_Char (Name_Buffer (J));
            end if;
         end loop;
      end Set_Main_Program_Name;

   --  Start of processing for Gen_Main_Program_File

   begin
      --  Write argv/argc stuff

      Write_Binder_Info ("extern int gnat_argc;");
      Write_Binder_Info ("extern char **gnat_argv;");
      Write_Binder_Info ("extern char **gnat_envp;");
      Write_Binder_Info ("extern int gnat_exit_status;");

      --  Generate adainit and adafinal

      Gen_Adainit (Main => True);
      Gen_Adafinal;

      --  Generate main

      if Bind_Alternate_Main_Name then
         Write_Binder_Info ("int gnat_main (argc, argv, envp)");
      else
         Write_Binder_Info ("int main (argc, argv, envp)");
      end if;

      Write_Binder_Info ("    int argc;");
      Write_Binder_Info ("    char **argv;");
      Write_Binder_Info ("    char **envp;");
      Write_Binder_Info ("{");
      Write_Binder_Info ("   gnat_argc = argc;");
      Write_Binder_Info ("   gnat_argv = argv;");
      Write_Binder_Info ("   gnat_envp = envp;");
      Write_Binder_Info (" ");

      Write_Binder_Info ("   __gnat_initialize();");
      Write_Binder_Info ("   adainit();");
      Write_Binder_Info (" ");

      --  Output main program name

      Get_Name_String (Unit.Table (First_Unit_Entry).Uname);

      --  Main program is procedure case

      if ALIs.Table (ALIs.First).Main_Program = Proc then
         Set_String ("   ");
         Set_Main_Program_Name;
         Set_String (" ();");
         Write_Statement_Buffer;

      --  Main program is function case

      else -- ALIs.Table (ALIs_First).Main_Program = Func
         Set_String ("   return (");
         Set_Main_Program_Name;
         Set_String (" ());");
         Write_Statement_Buffer;
      end if;

      Write_Binder_Info (" ");
      Write_Binder_Info ("   adafinal();");
      Write_Binder_Info ("   __gnat_finalize();");

      Write_Binder_Info ("   exit (gnat_exit_status);");
      Write_Binder_Info ("}");
      List_Versions;
      List_Object_Files_Options;
   end Gen_Main_Program_File;

   -------------------------------
   -- Gen_Non_Main_Program_File --
   -------------------------------

   procedure Gen_Non_Main_Program_File is
   begin
      Write_Binder_Info ("");
      Gen_Adainit (Main => False);
      Gen_Adafinal;
      List_Versions;
      List_Object_Files_Options;
   end Gen_Non_Main_Program_File;

   ---------------------
   -- Gen_Output_File --
   ---------------------

   procedure Gen_Output_File (Filename : String) is
   begin
      Create_Binder_Output (Filename);

      if Bind_Main_Program then
         Gen_Main_Program_File;
      else
         Gen_Non_Main_Program_File;
      end if;

      Close_Binder_Output;
   end Gen_Output_File;

   -------------------------------
   -- List_Object_Files_Options --
   -------------------------------

   procedure List_Object_Files_Options is
      Sptr : Natural;

   begin
      Write_Binder_Info ("/* BEGIN Object file/option list");

      for E in Elab_Order.First .. Elab_Order.Last loop

         --  If not spec that has an associated body, then generate a
         --  comment giving the name of the corresponding object file,
         --  except that we always skip shared passive units.

         if Unit.Table (Elab_Order.Table (E)).Utype /= Is_Spec
           and then not Unit.Table (Elab_Order.Table (E)).Shared_Passive
         then
            Get_Name_String
              (ALIs.Table
                (Unit.Table (Elab_Order.Table (E)).My_ALI).Ofile_Full_Name);

            --  If the presence of an object file is necessary or if it
            --  exists, then use it.

            if not Hostparm.Exclude_Missing_Objects
              or else
                GNAT.OS_Lib.Is_Regular_File (Name_Buffer (1 .. Name_Len))
            then
               Write_Binder_Info (Name_Buffer (1 .. Name_Len));
            end if;
         end if;
      end loop;

      --  Add a "-Ldir" for each directory in the object path:

      for J in 1 .. Nb_Dir_In_Obj_Search_Path loop
         declare
            Dir : String_Ptr := Dir_In_Obj_Search_Path (J);
            Str : String (1 .. Dir'Length + 2);

         begin
            Str (1 .. 2)        := "-L";
            Str (3 .. Str'Last) := Dir.all;
            Write_Binder_Info (Str);
         end;
      end loop;

      --  Add libgnat.a early since it references symbols in the thread
      --  library which is included as part of the linker options of the
      --  runtime

      --  There is already a -lgnat generated by gnatlink, and so this
      --  causes duplicate symbol errors on VMS where libgnat is a
      --  shareable image.

      if not Hostparm.OpenVMS then
         Write_Binder_Info ("-lgnat");
      end if;

      --  Write linker options

      Sptr := 0;
      for J in 1 .. Linker_Options.Last loop
         if Linker_Options.Table (J) = Ascii.Nul then
            Write_Binder_Info (Statement_Buffer (1 .. Sptr));
            Sptr := 0;
         else
            Sptr := Sptr + 1;
            Statement_Buffer (Sptr) := Linker_Options.Table (J);
         end if;
      end loop;

      Write_Binder_Info ("   END Object file/option list */");
   end List_Object_Files_Options;

   -------------------
   -- List_Versions --
   -------------------

   --  This routine generates a line of the form:

   --    unsigned unam = 0xhhhhhhhh;

   --  for each unit, where unam is the unit name suffixed by either B or
   --  S for body or spec, with dots replaced by double underscores.

   procedure List_Versions is
      Sptr : Natural;

   begin
      for U in Unit.First .. Unit.Last loop
         Statement_Buffer (1 .. 9) := "unsigned ";
         Sptr := 10;

         Get_Name_String (Unit.Table (U).Uname);

         for K in 1 .. Name_Len loop
            if Name_Buffer (K) = '.' then
               Statement_Buffer (Sptr) := '_';
               Sptr := Sptr + 1;
               Name_Buffer (K) := '_';

            elsif Name_Buffer (K) = '%' then
               exit;
            end if;

            Statement_Buffer (Sptr) := Name_Buffer (K);
            Sptr := Sptr + 1;
         end loop;

         if Name_Buffer (Name_Len) = 's' then
            Statement_Buffer (Sptr) := 'S';
         else
            Statement_Buffer (Sptr) := 'B';
         end if;

         Sptr := Sptr + 1;
         Statement_Buffer (Sptr .. Sptr + 4) := " = 0x";
         Sptr := Sptr + 5;
         Statement_Buffer (Sptr .. Sptr + 7) := Unit.Table (U).Version;
         Statement_Buffer (Sptr + 8) := ';';
         Write_Binder_Info (Statement_Buffer (1 .. Sptr + 8));
      end loop;

   end List_Versions;

   --------------
   -- Set_Char --
   --------------

   procedure Set_Char (C : Character) is
   begin
      Last := Last + 1;
      Statement_Buffer (Last) := C;
   end Set_Char;

   -------------
   -- Set_Int --
   -------------

   procedure Set_Int (N : Int) is
   begin
      if N < 0 then
         Set_String ("-");
         Set_Int (-N);

      else
         if N > 9 then
            Set_Int (N / 10);
         end if;

         Last := Last + 1;
         Statement_Buffer (Last) :=
           Character'Val (N mod 10 + Character'Pos ('0'));
      end if;
   end Set_Int;

   ----------------
   -- Set_String --
   ----------------

   procedure Set_String (S : String) is
   begin
      Statement_Buffer (Last + 1 .. Last + S'Length) := S;
      Last := Last + S'Length;
   end Set_String;

   -------------------
   -- Set_Unit_Name --
   -------------------

   procedure Set_Unit_Name is
   begin
      for J in 1 .. Name_Len - 2 loop
         if Name_Buffer (J) /= '.' then
            Set_Char (Name_Buffer (J));
         else
            Set_String ("__");
         end if;
      end loop;
   end Set_Unit_Name;

   ------------
   -- Tab_To --
   ------------

   procedure Tab_To (N : Natural) is
   begin
      while Last < N loop
         Set_Char (' ');
      end loop;
   end Tab_To;

   ----------------------------
   -- Write_Statement_Buffer --
   ----------------------------

   procedure Write_Statement_Buffer is
   begin
      Write_Binder_Info (Statement_Buffer (1 .. Last));
      Last := 0;
   end Write_Statement_Buffer;

end Bindgen;
