/*  SVGATextMode -- An SVGA textmode manipulation/enhancement tool
 *
 *  Copyright (C) 1995,1996,1997  Koen Gadeyne
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/***
 *** mach64.c, mode validation functions for mach64 cards
 *** Contributed by M. Grimrath (m.grimrath@tu-bs.de)
 ***/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>

#define STM_XFREE 1     /* avoids redefinition of SCREEN_ON */
#include "messages.h"
#include "chipset.h"
#include "Xmd.h"
#include "../common_hw/xf86_PCI.h"
#include "mach64_mem_access.h"
#include "mach64.h"
#include "regmach64.h"

#define VGA_MISC_R 0x3CC

/* Variables ************************************************************/
static unsigned ioCONFIG_CHIP_ID;
static unsigned ioCONFIG_CNTL;
static unsigned ioSCRATCH_REG0;
static unsigned ioSCRATCH_REG1;
static unsigned ioCONFIG_STAT0;
static unsigned ioMEM_CNTL;
static unsigned ioDAC_REGS;
static unsigned ioDAC_CNTL;
static unsigned ioGEN_TEST_CNTL;
static unsigned ioCLOCK_CNTL;   
static unsigned ioCRTC_GEN_CNTL;

static struct {   /* Information about clock chip */
  unsigned char ClockType;
  unsigned char pad0;
  unsigned short MinFreq;
  unsigned short MaxFreq;
  unsigned char CXClk;
  unsigned char pad1;
  unsigned short RefFreq;
  unsigned short RefDivider;
  unsigned short NAdj;
  unsigned short DRAMMemClk;
  unsigned short VRAMMemClk;
} info;
    

/************************************************************************/
/* Most code in here is (C) XFree86, Inc.                               */
static void
mach64PrintCTPLL()
{
#if 0
  int i;
  unsigned char pll[16];
  int R = 1432;
  int M, N, P;
  
  for (i = 0; i < 16; i++) {
    outb(ioCLOCK_CNTL + 1, i << 2);
    PDEBUG(("PLL register %2d: 0x%02x\n", i, pll[i] = inb(ioCLOCK_CNTL + 2)));
  }
  M = pll[PLL_REF_DIV];

  N = pll[VCLK0_FB_DIV];
  P = 1 << (pll[VCLK_POST_DIV] & VCLK0_POST);

  PDEBUG(("VCLK0: M=%d, N=%d, P=%d, Clk=%.2f\n", M, N, P,
	 (double)((2 * R * N)/(M * P)) / 100.0));
  N = pll[VCLK1_FB_DIV];
  P = 1 << ((pll[VCLK_POST_DIV] & VCLK1_POST) >> 2);

  PDEBUG(("VCLK1: M=%d, N=%d, P=%d, Clk=%.2f\n", M, N, P,
	 (double)((2 * R * N)/(M * P)) / 100.0));
  N = pll[VCLK2_FB_DIV];
  P = 1 << ((pll[VCLK_POST_DIV] & VCLK2_POST) >> 4);

  PDEBUG(("VCLK2: M=%d, N=%d, P=%d, Clk=%.2f\n", M, N, P,
	 (double)((2 * R * N)/(M * P)) / 100.0));
  N = pll[VCLK3_FB_DIV];
  P = 1 << ((pll[VCLK_POST_DIV] & VCLK3_POST) >> 6);

  PDEBUG(("VCLK3: M=%d, N=%d, P=%d, Clk=%.2f\n", M, N, P,
	 (double)((2 * R * N)/(M * P)) / 100.0));
  N = pll[MCLK_FB_DIV];
  P = 2;
  PDEBUG(("MCLK:  M=%d, N=%d, P=%d, Clk=%.2f\n", M, N, P,
	 (double)((2 * R * N)/(M * P)) / 100.0));
#endif
}


/************************************************************************/
/* Some code in here is (C) XFree86, Inc.                               */
void Mach64_verify_and_init(void)
{ int memdes;
  pciConfigPtr pcrp, *pcrpp;
  int BlockIO = 1;
  unsigned IOBase = 0;
  int found_clock   = CLKCHIP_NONE;
  int found_chipset = CS_NONE;
  char signature[10];
  unsigned long tmp;
  const unsigned long bios_base = 0xC0000;    
  
  
  /* Need access to BIOS ROM */
  if ((memdes = memacc_open()) < 0) {
    /* Actually cannot fail, since we already have IO port access */
    PERROR(("Cannot read BIOS ROM.\n"
	    "You must be superuser, or the program must be setuid root!\n"));
    return;
  }
  

  PDEBUG(("Mach64: Probing for PCI card\n"));
  pcrpp = xf86scanpci(0);
  if (pcrpp == NULL) {
    /* No PCI, up to now this is equal to 'not found' */
    PDEBUG(("No PCI found\n"));
    goto exit_mem;
  } else {
    /* Found PCI device(s), search for ATI */
    for (; *pcrpp != NULL; pcrpp++) {
      pcrp = *pcrpp;
      if (pcrp->_vendor == PCI_ATI_VENDOR_ID) {
	if (pcrp->_device == PCI_MACH64_CT) {
	  found_clock = CLKCHIP_MACH64CT;
	}
	/*
	 * The docs say check (pcrp->_user_config_0 & 0x04) 
	 * for BlockIO but this doesn't seem to be reliable.
	 * Instead check if pcrp->_base1 is non-zero.
	 */
	if (pcrp->_base1 & 0xfffffffc) {
	  BlockIO = 1;
	  IOBase = pcrp->_base1 & (pcrp->_base1 & 0x1 ?
				   0xfffffffc : 0xfffffff0);
	  /* If the Block I/O bit isn't set in userconfig, set it */
	  PDEBUG(("Mach64: PCI userconfig0 = 0x%02x\n", pcrp->_user_config_0));
	  if ((pcrp->_user_config_0 & 0x04) != 0x04) {
	    PDEBUG(("Mach64: Setting bit 0x04 in PCI userconfig for card %ld\n",
		    pcrp->_cardnum));
	    xf86writepci(0, pcrp->_bus,
			 pcrp->_cardnum, pcrp->_func,
			 PCI_REG_USERCONFIG, 0x04, 0x04);
	  }
	} else {
	  BlockIO = 0;
	  switch (pcrp->_user_config_0 & 0x03) {
	    case 0:  IOBase = 0x2EC; break;
	    case 1:  IOBase = 0x1CC; break;
	    case 2:  IOBase = 0x1C8; break;
	    default: 
	    PDEBUG(("Mach64: Cannot get IOBase\n"));
	    goto exit_pci;
	  }
	} /* if */
      } /* if */
    } /* for */
  } /* if PCI */
  if (found_clock != CLKCHIP_NONE) {
    PDEBUG(("Mach64: use %s, io 0x%04X\n",(BlockIO ? "block" : "sparse"),IOBase));
    if (BlockIO) {
      ioCONFIG_CHIP_ID = IOBase + CONFIG_CHIP_ID;
      ioCONFIG_CNTL    = IOBase + CONFIG_CNTL;
      ioSCRATCH_REG0   = IOBase + SCRATCH_REG0;
      ioSCRATCH_REG1   = IOBase + SCRATCH_REG1;
      ioCONFIG_STAT0   = IOBase + CONFIG_STAT0;
      ioMEM_CNTL       = IOBase + MEM_CNTL;
      /*ioDAC_REGS       = IOBase + DAC_REGS;*/
      /*ioDAC_CNTL       = IOBase + DAC_CNTL;*/
      ioGEN_TEST_CNTL  = IOBase + GEN_TEST_CNTL;
      ioCLOCK_CNTL     = IOBase + CLOCK_CNTL;
      ioCRTC_GEN_CNTL  = IOBase + CRTC_GEN_CNTL;
    } else {
      ioCONFIG_CHIP_ID = IOBase + (sioCONFIG_CHIP_ID << 10);
      ioCONFIG_CNTL    = IOBase + (sioCONFIG_CNTL << 10);
      ioSCRATCH_REG0   = IOBase + (sioSCRATCH_REG0 << 10);
      ioSCRATCH_REG1   = IOBase + (sioSCRATCH_REG1 << 10);
      ioCONFIG_STAT0   = IOBase + (sioCONFIG_STAT0 << 10);
      ioMEM_CNTL       = IOBase + (sioMEM_CNTL << 10);
      ioDAC_REGS       = IOBase + (sioDAC_REGS << 10);
      ioDAC_CNTL       = IOBase + (sioDAC_CNTL << 10);
      ioGEN_TEST_CNTL  = IOBase + (sioGEN_TEST_CNTL << 10);
      ioCLOCK_CNTL     = IOBase + (sioCLOCK_CNTL << 10);
      ioCRTC_GEN_CNTL  = IOBase + (sioCRTC_GEN_CNTL << 10);
    }
  } else {
    PWARNING(("Mach64: Cannot find ATI PCI videovard\n"));
  }


  PDEBUG(("Mach64: Probing for ATI BIOS\n"));
  memacc_fetch(memdes, bios_base + 0x30, signature, 10);
  if (memcmp(signature, " 761295520", 10)) {
    PWARNING(("Mach64: no ATI BIOS found\n"));
    goto exit_pci;
  }
	
  
  PDEBUG(("Mach64: Probing for\n"));
  tmp = inl(ioSCRATCH_REG0);
  outl(ioSCRATCH_REG0 ,0x55555555);
  if (inl(ioSCRATCH_REG0) != 0x55555555) {
    PWARNING(("Mach64: Probe failed\n"));
    goto exit_pci;
  }
  outl(ioSCRATCH_REG0, tmp);
  found_chipset = CS_ATIMACH64;


  PDEBUG(("Mach64: Probing for Mach64 CT (redundant if PCI)\n"));
  tmp = inl(ioCONFIG_CHIP_ID);
  if ((tmp & CFG_CHIP_TYPE) == MACH64_CT_ID) {
    if (found_clock != CLKCHIP_MACH64CT) {
      PWARNING(("Mach64: PCI and card disagree on clockchip type\n"
		"choosing card one\n"));
    }
    found_clock = CLKCHIP_MACH64CT;
  }

  
  /* Write found values if not overwritten in config file */
  if (chipset == CS_NONE) {
    PDEBUG(("Mach64: auto select for chipset %s\n",ChipsetRec[chipset].name_str));
    chipset = found_chipset;
  }
  if (clock_data.clockchiptype == CLKCHIP_NONE) {
    PDEBUG(("Mach64: auto select for clockchip %s\n",
	    ClockchipRec[clock_data.clockchiptype].name_str));
    clock_data.clockchiptype = found_clock;
  }
  

  /* Read out clock chip timings from BIOS */
  { unsigned short rom_table;
    unsigned short freq_table;

    PDEBUG(("Getting frequency values from BIOS ROM\n"));
    memacc_fetch(memdes, bios_base + 0x48, &rom_table, 2);
    memacc_fetch(memdes, bios_base + rom_table + 8*2, &freq_table, 2);
    memacc_fetch(memdes, bios_base + freq_table, &info, sizeof(info));
    clock_data.refclk   = info.RefFreq*10;
    if (clock_data.maxclock > info.MaxFreq*10) {
      clock_data.maxclock = info.MaxFreq*10;
    }
    PDEBUG(("MinFreq: %.2f\n",info.MinFreq*10 / 1000.0));
    PDEBUG(("MaxFreq: %.2f\n",clock_data.maxclock / 1000.0));
    PDEBUG(("RefFreq: %.2f\n",clock_data.refclk / 1000.0));
    
    if (info.ClockType != CLK_INTERNAL) {
      PWARNING(("Mach64: Error in BIOS? Mach64 CT w/o internal clock\n"));
    }
    mach64PrintCTPLL();
  }
  
  
  /* Check here if found chipset/clock match with config file */
  /* print out warning if not                                 */
  if ((found_chipset != CS_NONE) && (found_chipset != chipset)) {
    PWARNING(("Mach64: Didn't find %s chipset\n",ChipsetRec[chipset].name_str));
  }
  if ((found_clock != CLKCHIP_NONE) && (found_clock != clock_data.clockchiptype)) {
    PWARNING(("Mach64: Didn't find %s clockchip\n",
	      ClockchipRec[clock_data.clockchiptype].name_str));
  }
  

  exit_pci: xf86cleanpci();
  exit_mem: memacc_close(memdes);
}


/************************************************************************/
/* Most code in here is (C) XFree86, Inc.                               */
void Mach64_SetClock(int freq)
{ char old_crtc_ext_disp;
  int M, N, P, R;
  float Q;
  int postDiv;
  int mhz100 = freq / 10;
  unsigned char tmp1, tmp2;
  /*int ext_div = 0;*/
  int clkCntl;

  if (clock_data.clockchiptype != CLKCHIP_MACH64CT) {
    PERROR(("mach64ct: Sorry, can only handle Mach64CT so far\n"));
  }

  old_crtc_ext_disp = inb(ioCRTC_GEN_CNTL+3);
  outb(ioCRTC_GEN_CNTL+3, old_crtc_ext_disp | (CRTC_EXT_DISP_EN >> 24));

  M = info.RefDivider;
  R = info.RefFreq;

  /* Read out which clock for textmode is in use. Usually it is clock #1, */
  /* but you never know what strange configurations exist...              */
  clkCntl = (inb(VGA_MISC_R) >> 2) & 3;
  PDEBUG(("mach64ct: Programming clock #%d\n",clkCntl));
  
  if (mhz100 < info.MinFreq) mhz100 = info.MinFreq;
  if (mhz100 > clock_data.maxclock/10) mhz100 = clock_data.maxclock/10;

  Q = (mhz100 * M)/(2.0 * R);

  /* Not supported yet */
  /*if ((mach64ChipType == MACH64_VT || mach64ChipType == MACH64_GT) &&
      (mach64ChipRev & 0x01)) */
  if (Q > 255) {
    PWARNING(("mach64ct: Warning: Q > 255\n"));
    Q = 255;
    P = 0;
  }
  else if (Q > 127.5) {
    P = 0;
  }
  else if (Q > 63.75) {
    P = 1;
  }
  else if (Q > 31.875) {
    P = 2;
  }
  else if (Q >= 16) {
    P = 3;
  }
  else {
    PWARNING(("mach64ct: Warning: Q < 16\n"));
    P = 3;
  }
  postDiv = 1 << P;

  N = (int)(Q * postDiv + 0.5);

  PDEBUG(("mach64ct: Q = %f N = %d P = %d, postDiv = %d R = %d M = %d\n",
	  Q, N, P, postDiv, R, M));
  PDEBUG(("mach64ct: New freq: %.2f\n", 
	  (double)((2 * R * N)/(M * postDiv)) / 100.0));

  outb(ioCLOCK_CNTL + 1, PLL_VCLK_CNTL << 2);
  tmp1 = inb(ioCLOCK_CNTL + 2);
  outb(ioCLOCK_CNTL + 1, (PLL_VCLK_CNTL  << 2) | PLL_WR_EN);
  outb(ioCLOCK_CNTL + 2, tmp1 | 0x04);
  outb(ioCLOCK_CNTL + 1, VCLK_POST_DIV << 2);
  tmp2 = inb(ioCLOCK_CNTL + 2);
  outb(ioCLOCK_CNTL + 1, ((VCLK0_FB_DIV + clkCntl) << 2) | PLL_WR_EN);
  outb(ioCLOCK_CNTL + 2, N);
  outb(ioCLOCK_CNTL + 1, (VCLK_POST_DIV << 2) | PLL_WR_EN);
  outb(ioCLOCK_CNTL + 2,
       (tmp2 & ~(0x03 << (2 * clkCntl))) | (P << (2 * clkCntl)));
  outb(ioCLOCK_CNTL + 1, (PLL_VCLK_CNTL << 2) | PLL_WR_EN);
  outb(ioCLOCK_CNTL + 2, tmp1 & ~0x04);

  /* Not supported yet */
  /*if ((mach64ChipType == MACH64_VT || mach64ChipType == MACH64_GT) &&
      (mach64ChipRev & 0x01))*/

  usleep(5000);

  (void)inb(ioDAC_REGS); /* Clear DAC Counter */
  outb(ioCRTC_GEN_CNTL+3, old_crtc_ext_disp);
}
