/*************************************************************************
 *  TinyFugue - programmable mud client
 *  Copyright (C) 1993, 1994 Ken Keys
 *
 *  TinyFugue (aka "tf") is protected under the terms of the GNU
 *  General Public License.  See the file "COPYING" for details.
 ************************************************************************/
/* $Id: keyboard.c,v 35004.4 1995/09/24 21:58:57 hawkeye Exp $ */

/**************************************************
 * Fugue keyboard handling.
 * Handles all keyboard input and keybindings.
 **************************************************/

#include "config.h"
#include <ctype.h>
#include <errno.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "tfio.h"
#include "macro.h"	/* Macro, find_macro(), do_macro()... */
#include "keyboard.h"
#include "output.h"	/* iput(), idel(), redraw()... */
#include "history.h"	/* history_sub() */
#include "expand.h"	/* process_macro() */
#include "search.h"
#include "commands.h"

extern int errno;
static int literal_next = FALSE;
int input_is_complete = FALSE;
TIME_T keyboard_time = 0;

static int      FDECL(replace_input,(Aline *aline));
static int      NDECL(kill_input);
static void     FDECL(handle_input_string,(char *input, unsigned int len));


STATIC_BUFFER(scratch);                 /* buffer for manipulating text */
STATIC_BUFFER(cat_keybuf);              /* Total buffer for /cat */
STATIC_BUFFER(current_input);           /* unprocessed keystrokes */
static TrieNode *keytrie = NULL;        /* root of keybinding trie */

Stringp keybuf;                         /* input buffer */
int keyboard_pos = 0;                   /* current position in buffer */

/*
 * Some dokey operations are implemented internally with names like
 * DOKEY_FOO; others are implemented as macros in stdlib.tf with names
 * like /dokey_foo.  handle_dokey_command() looks first for an internal
 * function in efunc_table[], then for a macro, so all operations can be done
 * with "/dokey foo".  Conversely, internally-implemented operations should
 * have macros in stdlib.tf of the form "/def dokey_foo = /dokey foo",
 * so all operations can be performed with "/dokey_foo".
 */
static CONST char *efunc_table[] = {
    "DLINE"    ,
    "FLUSH"    ,
    "HPAGE"    ,
    "LINE"     ,
    "LNEXT"    ,
    "NEWLINE"  ,
    "PAGE"     ,
    "RECALLB"  ,
    "RECALLBEG",
    "RECALLEND",
    "RECALLF"  ,
    "REDRAW"   ,
    "REFRESH"  ,
    "SEARCHB"  ,
    "SEARCHF"  ,
    "SELFLUSH"
};

enum {
    DOKEY_DLINE    ,
    DOKEY_FLUSH    ,
    DOKEY_HPAGE    ,
    DOKEY_LINE     ,
    DOKEY_LNEXT    ,
    DOKEY_NEWLINE  ,
    DOKEY_PAGE     ,
    DOKEY_RECALLB  ,
    DOKEY_RECALLBEG,
    DOKEY_RECALLEND,
    DOKEY_RECALLF  ,
    DOKEY_REDRAW   ,
    DOKEY_REFRESH  ,
    DOKEY_SEARCHB  ,
    DOKEY_SEARCHF  ,
    DOKEY_SELFLUSH
};

void init_keyboard()
{
    Stringinit(keybuf);
}

/* Find the macro assosiated with <key> sequence. */
Macro *find_key(key)
    CONST char *key;
{
    return (Macro *)trie_find(keytrie, key);
}

int bind_key(spec)   /* install Macro's binding in key structures */
    Macro *spec;
{
    Macro *macro;
    int status;

    if ((macro = find_key(spec->bind))) {
        if (redef) {
            do_hook(H_REDEF, "%% Redefined %s %s", "%s %s",
                "binding", ascii_to_print(spec->bind));
            kill_macro(macro);
            /* intrie is guaranteed to succeed */
        } else {
            eprintf("Binding %s already exists.", ascii_to_print(spec->bind));
            return 0;
        }
    }

    status = intrie(&keytrie, spec, spec->bind);
    if (status > 0) return 1;

    eprintf("'%s' is %s an existing keybinding.",
        ascii_to_print(spec->bind),
        (status == TRIE_SUPER) ? "prefixed by" : "a prefix of");

    return 0;
}

void unbind_key(macro)
    Macro *macro;
{
    untrie(&keytrie, macro->bind);
}

void handle_keyboard_input()
{
    char *s, buf[64];
    int i, count;
    static TrieNode *n;
    static int key_start = 0;
    static int input_start = 0;
    static int place = 0;
    /* extern int quit_flag; */

    /* read a block of text */
    if ((count = read(0, buf, sizeof(buf))) < 0) {
        if (errno == EINTR) return;
        die("handle_keyboard_input: read", errno);
    }
    /* if (count == 0) quit_flag = 1; */
    keyboard_time = time(NULL);

    for (i = 0; i < count; i++) {
        if (!istrip && buf[i] & 0x80) Stringadd(current_input, '\033');
        Stringadd(current_input, safechar(buf[i]));
    }

    s = current_input->s;
    if (!s) return;       /* no legal chars; current_input not yet allocated */
    if (!n) n = keytrie;
    while (s[place]) {
        if (literal_next) {
            place++;
            key_start++;
            literal_next = FALSE;
            continue;
        }
        while (s[place] && n && n->children) n = n->u.child[s[place++]&0x7F];
        if (!n || !keytrie->children) {
            /* No match.  Try a suffix. */
            place = ++key_start;
            n = keytrie;
        } else if (!n->children) {
            /* Total match.  Process everything up to this point, */
            /* and call the macro. */
            handle_input_string(s + input_start, key_start - input_start);
            key_start = input_start = place;
            do_macro((Macro *)n->u.datum, NULL);
            n = keytrie;
        } /* else, partial match; just hold on to it for now. */
    }

    /* Process everything up to a possible match. */
    handle_input_string(s + input_start, key_start - input_start);

    /* Shift the window if there's no pending partial match. */
    if (!s[key_start]) {
        Stringterm(current_input, 0);
        place = key_start = 0;
    }
    input_start = key_start;
}

/* Update the input window and keyboard buffer. */
static void handle_input_string(input, len)
    char *input;
    unsigned int len;
{
#if 0
    int i, j;

    for (i = j = 0; i < len; i++) {
        if (isspace(input[i]))       /* convert newlines and tabs to spaces */
            input[j++] = ' ';
        else if (isprint(input[i]))
            input[j++] = input[i];
    }
    len = j;
#endif
    if (len == 0) return;

    if (keyboard_pos == keybuf->len) {                    /* add to end */
        Stringncat(keybuf, input, len);
    } else if (insert) {                                  /* insert in middle */
        Stringcpy(scratch, keybuf->s + keyboard_pos);
        Stringterm(keybuf, keyboard_pos);
        Stringncat(keybuf, input, len);
        SStringcat(keybuf, scratch);
    } else if (keyboard_pos + len < keybuf->len) {        /* overwrite */
        strncpy(keybuf->s + keyboard_pos, input, len);
    } else {                                              /* write past end */
        Stringterm(keybuf, keyboard_pos);
        Stringncat(keybuf, input, len);
    }                      
    keyboard_pos += len;
    iput(keybuf->s + keyboard_pos - len, len);
}


int handle_input_command(args)
    char *args;
{
    handle_input_string(args, strlen(args));
    return 1;
}


/*
 *  Builtin key functions.
 */

int handle_dokey_command(args)
    char *args;
{
    CONST char **ptr;
    STATIC_BUFFER(buffer);
    Macro *macro;

    ptr = (CONST char **)binsearch((GENERIC*)&args, (GENERIC*)efunc_table,
        sizeof(efunc_table)/sizeof(char*), sizeof(char*), gencstrcmp);

    if (!ptr) {
        Stringcat(Stringcpy(buffer, "dokey_"), args);
        if ((macro = find_macro(buffer->s))) return do_macro(macro, NULL);
        else eprintf("No editing function %s", args); 
        return 0;
    }

    switch (ptr - efunc_table) {

    case DOKEY_DLINE:      return kill_input();
    case DOKEY_FLUSH:      return screen_flush(FALSE);
    case DOKEY_HPAGE:      return dokey_hpage();
    case DOKEY_LINE:       return dokey_line();
    case DOKEY_LNEXT:      return literal_next = TRUE;

    case DOKEY_NEWLINE:
        reset_outcount();
        inewline();
        /* If we actually process the input now, weird things will happen with
         * current_command and mecho.  So we just set a flag and wait until the
         * end of handle_command(), when things are cleaner.
         */
        return input_is_complete = TRUE;  /* return value isn't really used */

    case DOKEY_PAGE:       return dokey_page();
    case DOKEY_RECALLB:    return replace_input(recall_input(-1, FALSE));
    case DOKEY_RECALLBEG:  return replace_input(recall_input(-2, FALSE));
    case DOKEY_RECALLEND:  return replace_input(recall_input(2, FALSE));
    case DOKEY_RECALLF:    return replace_input(recall_input(1, FALSE));
    case DOKEY_REDRAW:     return redraw();
    case DOKEY_REFRESH:    return logical_refresh(), keyboard_pos;
    case DOKEY_SEARCHB:    return replace_input(recall_input(-1, TRUE));
    case DOKEY_SEARCHF:    return replace_input(recall_input(1, TRUE));
    case DOKEY_SELFLUSH:   return screen_flush(TRUE);
    default:               return 0; /* impossible */
    }
}

static int replace_input(aline)
    Aline *aline;
{
    if (!aline) {
        bell(1);
        return 0;
    }
    if (keybuf->len) kill_input();
    handle_input_string(aline->str, aline->len);
    return 1;
}

static int kill_input()
{
    if (keybuf->len) {
        Stringterm(keybuf, keyboard_pos = 0);
        logical_refresh();
    } else {
        bell(1);
    }
    return keyboard_pos;
}

int do_kbdel(place)
    int place;
{
    if (place >= 0 && place < keyboard_pos) {
        Stringcpy(scratch, keybuf->s + keyboard_pos);
        SStringcat(Stringterm(keybuf, place), scratch);
        idel(place);
    } else if (place > keyboard_pos && place <= keybuf->len) {
        Stringcpy(scratch, keybuf->s + place);
        SStringcat(Stringterm(keybuf, keyboard_pos), scratch);
        idel(place);
    } else {
        bell(1);
    }
    return keyboard_pos;
}

#define isinword(c) (isalnum(c) || (wordpunct && strchr(wordpunct, (c))))

int do_kbword(dir)
    int dir;
{
    int stop = (dir < 0) ? -1 : keybuf->len;
    int place = keyboard_pos - (dir < 0);

    while (place != stop && !isinword(keybuf->s[place])) place += dir;
    while (place != stop && isinword(keybuf->s[place])) place += dir;
    return place + (dir < 0);
}

int do_kbmatch()
{
    static CONST char *braces = "(){}[]";
    char *type;
    int dir, stop, depth = 0, place = keyboard_pos;

    while (1) {
        if (place >= keybuf->len) return -1;
        if ((type = strchr(braces, keybuf->s[place]))) break;
        ++place;
    }
    dir = ((type - braces) % 2) ? -1 : 1;
    stop = (dir < 0) ? -1 : keybuf->len;
    do {
        if      (keybuf->s[place] == type[0])   depth++;
        else if (keybuf->s[place] == type[dir]) depth--;
        if (depth == 0) return place;
    } while ((place += dir) != stop);
    return -1;
}

int handle_input_line()
{
    String *line;
    extern int concat;

    SStringcpy(scratch, keybuf);
    Stringterm(keybuf, keyboard_pos = 0);
    input_is_complete = FALSE;

    if (concat) {
        if (scratch->s[0] == '.' && scratch->len == 1) {
            SStringcpy(scratch, cat_keybuf);
            Stringterm(cat_keybuf, 0);
            concat = 0;
        } else {
            SStringcat(cat_keybuf, scratch);
            if (concat == 2) Stringcat(cat_keybuf, "%;");
            return 0;
        }
    }

    if (*scratch->s == '^') {
        if (!(line = history_sub(scratch->s + 1))) {
            oputs("% No match.");
            return 0;
        }
        iput(line->s, line->len);
        inewline();
    } else line = scratch;

    if (kecho) tfprintf(tferr, "%s%S", kprefix, line);
    record_input(line->s);
    return process_macro(line->s, NULL, sub);
}

#ifdef DMALLOC
void free_keyboard()
{
    Stringfree(keybuf);
    Stringfree(scratch);
    Stringfree(cat_keybuf);
    Stringfree(current_input);
}
#endif
