;;
;; Extracted from M-x compile.
;;

;; Run compiler as inferior of Emacs, and parse its error messages.
;; Copyright (C) 1985, 1986 Free Software Foundation, Inc.

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

(provide 'lclint)

(defvar lclint-command "lclint")

(defvar lclint-process nil
  "Process created by lclint command, or nil if none exists now.
Note that the process may have been \"deleted\" and still
be the value of this variable.")

(defvar lclint-error-list nil
  "List of error message descriptors for visiting erring functions.
Each error descriptor is a list of length two.
Its car is a marker pointing to an error message.
Its cadr is a marker pointing to the text of the line the message is about,
  or nil if that is not interesting.
The value may be t instead of a list;
this means that the buffer of error messages should be reparsed
the next time the list of errors is wanted.")

(defvar lclint-parsing-end nil
  "Position of end of buffer when last error messages parsed.")

(defvar lclint-error-message nil
  "Message to print when no more matches for lclint-error-regexp are found"
)

;; The filename excludes colons to avoid confusion when error message
;; starts with digits.
(defvar lclint-error-regexp
  "\\([^ :\n]+\\(: *\\|(\\)[0-9]+\\)"
  "Regular expression for filename/linenumber in error in lclint log.")

;; works:  "\\([^ :\n]+\\(: *\\|(\\)[0-9]+\\)"
;;     "[^ ]+:[0-9]+\\(:\\|\\(,[0-9]+:\\)\\)"




(defun lclint (command)
  "Run lclint. Default: run `make'.
Runs COMMAND, a shell command, in a separate process asynchronously
with output going to the buffer *lclint*.
You can then use the command \\[next-error] to find the next error message
and move to the source code that caused it."
  (interactive (list (read-string "lclint command: " lclint-command)))
  (setq lclint-command command)
  (lclint1 lclint-command "No more messages"))

(defun grep (command)
  "Run grep, with user-specified args, and collect output in a buffer.
While grep runs asynchronously, you can use the \\[next-error] command
to find the text that grep hits refer to."
  (interactive "sRun grep (with args): ")
  (lclint1 (concat "grep -n " command " /dev/null")
            "No more grep hits" "grep"))

(defun lclint1 (command error-message &optional name-of-mode)
  (save-some-buffers)
  (if lclint-process
      (if (or (not (eq (process-status lclint-process) 'run))
              (yes-or-no-p "An lclint process is running; kill it? "))
          (condition-case ()
              (let ((comp-proc lclint-process))
                (interrupt-process comp-proc)
                (sit-for 1)
                (delete-process comp-proc))
            (error nil))
        (error "Cannot have two lclint processes")))
  (setq lclint-process nil)
  (lclint-forget-errors)
  (setq lclint-error-list t)
  (setq lclint-error-message error-message)
  (setq lclint-process
        (start-process "lclint" "*lclint*"
                       shell-file-name
                       "-c" (concat "exec " command)))
  (with-output-to-temp-buffer "*lclint*"
    (princ "cd ")
    (princ default-directory)
    (terpri)
    (princ command)
    (terpri))
  (set-process-sentinel lclint-process 'lclint-sentinel)
  (let* ((thisdir default-directory)
         (outbuf (process-buffer lclint-process))
         (outwin (get-buffer-window outbuf))
         (regexp lclint-error-regexp))
    (if (eq outbuf (current-buffer))
        (goto-char (point-max)))
    (save-excursion
      (set-buffer outbuf)
      (buffer-flush-undo outbuf)
      (let ((start (save-excursion (set-buffer outbuf) (point-min))))
        (set-window-start outwin start)
        (or (eq outwin (selected-window))
            (set-window-point outwin start)))
      (setq default-directory thisdir)
      (fundamental-mode)
      (make-local-variable 'lclint-error-regexp)
      (setq lclint-error-regexp regexp)
      (setq mode-name (or name-of-mode "lclint"))
      ;; Make log buffer's mode line show process state
      (setq mode-line-process '(": %s")))))

;; Called when lclint process changes state.

(defun lclint-sentinel (proc msg)
  (cond ((null (buffer-name (process-buffer proc)))
         ;; buffer killed
         (set-process-buffer proc nil))
        ((memq (process-status proc) '(signal exit))
         (let* ((obuf (current-buffer))
                omax opoint)
           ;; save-excursion isn't the right thing if
           ;;  process-buffer is current-buffer
           (unwind-protect
               (progn
                 ;; Write something in *lclint* and hack its mode line,
                 (set-buffer (process-buffer proc))
                 (setq omax (point-max) opoint (point))
                 (goto-char (point-max))
                 (insert ?\n mode-name " " msg)
                 (forward-char -1)
                 (insert " at "
                         (substring (current-time-string) 0 -5))
                 (forward-char 1)
                 (setq mode-line-process
                       (concat ": "
                               (symbol-name (process-status proc))))
                 ;; If buffer and mode line will show that the process
                 ;; is dead, we can delete it now.  Otherwise it
                 ;; will stay around until M-x list-processes.
                 (delete-process proc))
             (setq lclint-process nil)
             ;; Force mode line redisplay soon
             (set-buffer-modified-p (buffer-modified-p)))
           (if (and opoint (< opoint omax))
               (goto-char opoint))
           (set-buffer obuf)))))

(defun kill-lclint ()
  "Kill the process made by the \\[lclint] command."
  (interactive)
  (if lclint-process
      (interrupt-process lclint-process)))

(defun kill-grep ()
  "Kill the process made by the \\[grep] command."
  (interactive)
  (if lclint-process
      (interrupt-process lclint-process)))

(defun next-lclint-error (&optional argp)
  "Visit next lclint message and corresponding source code.
This operates on the output from the \\[lclint] command.
If all preparsed error messages have been processed,
the error message buffer is checked for new ones.
A non-nil argument (prefix arg, if interactive)
means reparse the error message buffer and start at the first error."
  (interactive "P")
  (message "Next lclint message...")

  (if (or (eq lclint-error-list t)
          argp)
      (progn (lclint-forget-errors)
             (setq lclint-parsing-end 1)))
  (if lclint-error-list
      nil
    (save-excursion
      (set-buffer "*lclint*")
      (set-buffer-modified-p nil)
      (lclint-parse-errors)))
  (let ((next-error (car lclint-error-list)))
    (if (null next-error)
        (error (concat lclint-error-message
                       (if (and lclint-process
                                (eq (process-status lclint-process)
                                    'run))
                           " yet" ""))))
    (setq lclint-error-list (cdr lclint-error-list))
    (if (null (car (cdr next-error)))
        nil
      (switch-to-buffer (marker-buffer (car (cdr next-error))))
      (goto-char (car (cdr next-error)))
      (set-marker (car (cdr next-error)) nil))
    (let* ((pop-up-windows t)
           (w (display-buffer (marker-buffer (car next-error)))))
      (set-window-point w (car next-error))
      (set-window-start w (car next-error)))
    (set-marker (car next-error) nil)))

;; Set lclint-error-list to nil, and
;; unchain the markers that point to the error messages and their text,
;; so that they no longer slow down gap motion.
;; This would happen anyway at the next garbage collection,
;; but it is better to do it right away.
(defun lclint-forget-errors ()
  (if (eq lclint-error-list t)
      (setq lclint-error-list nil))
  (while lclint-error-list
    (let ((next-error (car lclint-error-list)))
      (set-marker (car next-error) nil)
      (if (car (cdr next-error))
          (set-marker (car (cdr next-error)) nil)))
    (setq lclint-error-list (cdr lclint-error-list))))

(defun lclint-parse-errors ()
  "Parse the current buffer as error messages.
This makes a list of error descriptors, lclint-error-list.
For each source-file, line-number pair in the buffer,
the source file is read in, and the text location is saved in lclint-error-
list.
The function next-error, assigned to \\[next-error], takes the next error off the list
and visits its location."
  (setq lclint-error-list nil)
  (setq columnnum 0)
  (message "Parsing error messages...")
  (let (text-buffer
	last-filename last-linenum last-columnnum)
    ;; Don't reparse messages already seen at last parse.
    (goto-char lclint-parsing-end)
    ;; Don't parse the first two lines as error messages.
    ;; This matters for grep.
    (if (bobp)
	(forward-line 2))
    (while (re-search-forward lclint-error-regexp nil t)
      (let (linenum filename
	    error-marker text-marker)
	(setq columnnum 0)
	(save-restriction
	  (narrow-to-region (match-beginning 0) (match-end 0))
	  (goto-char (point-max))
	  (skip-chars-backward "[0-9]")
	  ;; If it's a lint message, use the last file(linenum) on the line.
	  ;; Normally we use the first on the line.
	  (if (= (preceding-char) ?\()
	      (progn
		(narrow-to-region (point-min) (1+ (buffer-size)))
		(end-of-line)
		(re-search-backward lclint-error-regexp)
		(skip-chars-backward "^ \t\n")
		(narrow-to-region (point) (match-end 0))
		(goto-char (point-max))
		(skip-chars-backward "[0-9]")))
	  ;; Are we looking at a "filename-first" or "line-number-first" form?
	  (if (looking-at "[0-9]")
	      (progn
		(setq linenum (read (current-buffer)))
		(goto-char (point-min)))
	    ;; Line number at start, file name at end.
	    (progn
	      (goto-char (point-min))
	      (setq linenum (read (current-buffer)))
	      (goto-char (point-max))
	      (skip-chars-backward "^ \t\n")))
	  (setq filename (lclint-grab-filename)))
	
	(save-excursion
	  (if (re-search-forward ",[0-9]+" nil t) 
	      (progn 
		(save-restriction
		  (narrow-to-region (match-beginning 0)
				    (match-end 0))
		  (goto-char (point-max))
		  (skip-chars-backward "[0-9]")
		  (if (looking-at "[0-9]")
		      (setq columnnum (1- (read (current-buffer))))
		    (setq columnnum 0))
		  ))))
	
	;; Locate the erring file and line.
	(if (and (equal filename last-filename)
		 (= linenum last-linenum)
		 (= columnnum last-columnnum))
	    nil
	  (beginning-of-line 1)
	  (setq error-marker (point-marker))
	  ;; text-buffer gets the buffer containing this error's file.
	  (if (not (equal filename last-filename))
	      (setq text-buffer
		    (and (file-exists-p (setq last-filename filename))
			 (find-file-noselect filename))
		    last-linenum 0
		    last-columnnum 0))
	  (if text-buffer
	      ;; Go to that buffer and find the erring line.
	      (save-excursion
		(set-buffer text-buffer)
		(if (zerop last-linenum)
		    (progn
		      (goto-char 1)
		      (setq last-linenum 1)))
		(setq last-linenum (- linenum ; In case hit eob
				      (forward-line (- linenum last-linenum))))
		(setq last-columnnum columnnum)
		(forward-char columnnum)
		(setq text-marker (point-marker))
		(setq lclint-error-list
		      (cons (list error-marker text-marker)
			    lclint-error-list)))))
	(forward-line 1)))
    (setq lclint-parsing-end (point-max)))
  (message "Parsing lclint messages...done")
  (setq lclint-error-list (nreverse lclint-error-list)))

(defun lclint-grab-filename ()
  "Return a string which is a filename, starting at point.
Ignore quotes and parentheses around it, as well as trailing colons."
  (if (eq (following-char) ?\")
      (save-restriction
        (narrow-to-region (point)
                          (progn (forward-sexp 1) (point)))
        (goto-char (point-min))
        (read (current-buffer)))
    (buffer-substring (point)
                      (progn
                        (skip-chars-forward "^ :,\n\t(")
                        (point)))))


