/****************************************************************************/
/*                                                                          */
/* IMG* Image Processing Tools Library                                      */
/* Program:   imgFFTGabor.c                                                 */
/* Author:    Simon A.J. Winder                                             */
/* Date:      Thu Oct 20 20:45:38 1994                                      */
/* Copyright (C) 1994 Simon A.J. Winder                                     */
/*                                                                          */
/****************************************************************************/

#include "tools.h"

#define PRGNAME "FFTGabor"
#define ERROR(e) imgError(e,PRGNAME)

void make_gabor(it_image *,double,double,double);
void copy_complex_image(it_image *,it_image *);
void real_multiply(it_image *,it_image *);
void imag_multiply(it_image *,it_image *);

static double sigma;

int main(int argc,char **argv)
{
  it_image *in1,*f1,*in2;
  int width,height;
  double angle,omega,anginc,bandwidth,tpb;

  IFHELP
    {
      fprintf(stderr,"img%s - Fourier directional Gabor filter\n",
	      PRGNAME);
      fprintf(stderr,"img%s [orient [freq [incr [bandwidth]]]]\n",
	      PRGNAME);
      fprintf(stderr,"  stdin: Complex\n");
      fprintf(stderr,"  stdout: Complex\n");
      fprintf(stderr,"  orient is the orientation in degrees from horizontal\n");
      fprintf(stderr,"  freq is the centre frequency in radians per pixel\n");
      fprintf(stderr,"  incr is the orientation increment in degrees per image\n");
      fprintf(stderr,"  bandwidth is the Gaussian half-height bandwidth in octaves\n");
      fprintf(stderr,"  defaults: 0 pi/2 0 1\n");
      exit(0);
    }

  imgStart(PRGNAME);

  bandwidth=1.0;
  angle=0.0;
  omega=M_PI_2;
  anginc=0.0;
  if(argc>5)
    ERROR("invalid arguments");
  if(argc>1)
    angle=atof(argv[1])*M_PI/180.0;    /* angle */
  if(argc>2)
    omega=atof(argv[2]);               /* centre frequency */
  if(argc>3)
    anginc=atof(argv[3])*M_PI/180.0;   /* angle increment */
  if(argc>4)
    bandwidth=atof(argv[4]);               /* gaussian width */
  
  tpb=pow(2.0,bandwidth);
  sigma=(tpb+1.0)/(tpb-1.0)*sqrt(2.0*M_LN2)/omega;
  
  /* Loop for all images */
  do {
    in1=i_read_image_file(stdin,IT_COMPLEX,IM_FRAGMENT);
    if(in1==NULL)
      ERROR("can't import image file");
    width=in1->width;
    height=in1->height;

    if((f1=i_create_image(width,height,IT_FLOAT,IM_FRAGMENT))==NULL)
      ERROR("out of memory");
    if((in2=i_create_image(width,height,IT_COMPLEX,IM_FRAGMENT))==NULL)
      ERROR("out of memory");

    copy_complex_image(in1,in2);

    /* Even Gabor function */
    make_gabor(f1,omega,angle,1.0);
    real_multiply(in1,f1);
    i_write_image_file(stdout,in1,IF_BINARY);
    i_destroy_image(in1);

    /* Odd Gabor function */
    make_gabor(f1,omega,angle,-1.0);
    imag_multiply(in2,f1);
    i_write_image_file(stdout,in2,IF_BINARY);
    i_destroy_image(in2);
    i_destroy_image(f1);

    angle+=anginc;
  } while(!feof(stdin));

  imgFinish(PRGNAME);
  return(0);
}

void make_gabor(it_image *f,double omega,double angle,double type)
{
  double wc,ws,cc,u,v,umwc,vmws,upwc,vpws,phwc,phhc;
  int x,y,width,height,halfwidth,halfheight;
  it_float *flt_ptr;

  width=f->width;
  height=f->height;
  halfwidth=width/2;
  halfheight=height/2;
  phwc=M_PI/((double) halfwidth);
  phhc=M_PI/((double) halfheight);
  wc=omega*cos(M_PI_2-angle);
  ws=omega*sin(M_PI_2-angle);

  cc= -sigma*sigma/2.0;

  for(y=0;y<height;y++)
    {
      flt_ptr=im_float_row(f,y);
      for(x=0;x<width;x++)
	{
	  u=phwc*(double) ((x<halfwidth)?(x):(x-width));
	  v=phhc*(double) ((y<halfheight)?(y):(y-height));
	  umwc=u-wc;
	  vmws=v-ws;
	  upwc=u+wc;
	  vpws=v+ws;
	  *flt_ptr++= (type*exp(cc*(umwc*umwc+vmws*vmws))
		       +exp(cc*(upwc*upwc+vpws*vpws)));
	}
    }
  /* Force DC to zero */
  im_float_value(f,0,0)=0.0;
}

void copy_complex_image(it_image *c1,it_image *c2)
{
  int x,y;
  it_complex *ptr1,*ptr2;
  
  for(y=0;y<(c1->height);y++)
    {
      ptr1=im_complex_row(c1,y);
      ptr2=im_complex_row(c2,y);
      for(x=0;x<(c1->width);x++)
	{
	  ptr2->Im=ptr1->Im;
	  (ptr2++)->Re=(ptr1++)->Re;
	}
    }
}

void real_multiply(it_image *c,it_image *f)
{
  int x,y;
  it_float *flt_ptr;
  it_complex *cpx_ptr;

  for(y=0;y<(c->height);y++)
    {
      flt_ptr=im_float_row(f,y);
      cpx_ptr=im_complex_row(c,y);
      for(x=0;x<(c->width);x++)
	{
	  cpx_ptr->Im *= *flt_ptr;
	  (cpx_ptr++)->Re *= *flt_ptr++;
	}
    }
}

void imag_multiply(it_image *c,it_image *f)
{
  int x,y;
  it_float *flt_ptr,save;
  it_complex *cpx_ptr;

  for(y=0;y<(c->height);y++)
    {
      flt_ptr=im_float_row(f,y);
      cpx_ptr=im_complex_row(c,y);
      for(x=0;x<(c->width);x++)
	{
	  save=cpx_ptr->Im;
	  cpx_ptr->Im = *flt_ptr * cpx_ptr->Re;
	  (cpx_ptr++)->Re = -(*flt_ptr++) * save;
	}
    }
}
/* Version 1.0 (Oct 1994) */
/* Version 1.1 (Nov 1994) */
