
/* colormap optimization filter for the Gimp
 *
 * cmapopti.c:	V1.00
 * Author: 	Jim GEUTHER
 * Date:    	13-May-96
 * Environment: Personal Power System 850 + AIX V4.1.3.0
 * This code was originally developed on the Amiga for ImageKnife.
 *
 * This plugin can be used to perform color optimization of an indexed
 * color image.
 *
 * My definition of colormap optimization is:
 *
 * Remove duplicate index entries, that is: indexes which have the same
 * RGB values.
 *
 * Reorder the colormap such that most frequently referenced indexes
 * are placed first.
 *
 * Usefullness: I will tell you when I have completed porting the
 * remaining code.
 *
 * History:
 * V1.00	Jim GEUTHER, original development for Gimp.
 * 
 */
 
/*
 * This is a plug-in for the GIMP.
 *
 * Copyright (C) 1993/94/95/96 Jim Geuther
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 */
 
#include "gimp.h"
#include "types.h"
#include "cmap_palette.h"

#define _DEBUG 

#ifdef	_DEBUG
#define	NULLP()	printf("%s.%ld: NULLPOINTER\n",__FILE__,__LINE__)
#define	DPRINTF(x) printf("%s.%ld: %s",__FILE__,__LINE__,x)
#else
#define	NULLP()	
#define	DPRINTF(x)
#endif

static void 	opti(Image input,Image output);
static char 	*prog_name;

int
main (argc, argv)
     int argc;
     char *argv[];
{
Image	input,output;


  /* Save the program name so we can use it later in reporting errors
   */
  prog_name = argv[0];

  /* Call 'gimp_init' to initialize this filter.
   * 'gimp_init' makes sure that the filter was properly called and
   *  it opens pipes for reading and writing.
   */
  if (gimp_init (argc, argv))
    {
	if(input=gimp_get_input_image(0)) {
		switch(gimp_image_type(input)) {
		case INDEXED_IMAGE :
				/* Seems that gimp_output_image() returns
				 * "black" image
				 */
				if(output=gimp_new_image(0, gimp_image_width(input),
						gimp_image_height(input),
						INDEXED_IMAGE)) {
					opti(input,output); 
					gimp_display_image(output);
					gimp_update_image(output);
					gimp_free_image(output);
				} else printf("%s: get output image failed\n",prog_name);
			break;
		default :
			gimp_message("cmapopti: can only operate on indexed color images");
			break;
		}
		gimp_free_image(input);
	}			
       /* Quit
       */
      gimp_quit ();
    }

  return 0;
}


static int /* found duplicate */ find_dup_colors(
struct ColourTriplet	*triplets,
int			colors,
Image	input,
Image	output
)
{

/*
** find_dup_colors:
**
** This function checks if a palette has duplicate colors and reports
** them.
**
*/

long		ix, ix0, ix1;

int		duplicate_colors=FALSE;

struct ColourTriplet	*newtriplets;

u_char			*newpens;
u_long			newcolors;
size_t		newsize;
int		height,width,channels,rowstride;
u_char		*src_row,*dest_row,*src,*dest,*cmap;

/*
** Allocate a table for the new triplets
*/
newsize=colors*sizeof(struct ColourTriplet);
if(newtriplets=calloc(newsize,1)) {
	if(newpens=calloc(colors*sizeof(u_char),1)) {
		newcolors=0;
		for(ix=0;ix<colors;ix++) {
			for(ix1=0;ix1<newcolors;ix1++) {
				if( (triplets[ix].Red==newtriplets[ix1].Red)
				&&  (triplets[ix].Green==newtriplets[ix1].Green)
				&&  (triplets[ix].Blue==newtriplets[ix1].Blue) ){
					newpens[ix]=ix1;
					duplicate_colors=TRUE;
					break;
				}
			}
			if(ix1==newcolors) {
				newtriplets[newcolors].Red=triplets[ix].Red;
				newtriplets[newcolors].Green=triplets[ix].Green;
				newtriplets[newcolors].Blue=triplets[ix].Blue;
				newcolors+=1;
				newpens[ix]=ix;
			}
		}

		if(duplicate_colors) {
			height    = gimp_image_height(input);
			width     = gimp_image_width(input);
			channels  = gimp_image_channels(input);
			rowstride = width * channels;
			src_row	  = gimp_image_data (input);
			dest_row  = gimp_image_data (output);

			for( ix0 = 0;  ix0 < height; ix0++ ) {
				dest = dest_row;
				src  = src_row;
				for( ix1 = 0; ix1 < width; ix1++ ) {
				*dest++ = newpens[ *src++ ];
				}
				src_row  += rowstride;
				dest_row += rowstride;
			}
			DPRINTF("duplicate colors done\n");
		} else DPRINTF("No duplicate colors\n"); 
		free(newpens);
	} else NULLP();
	free(newtriplets);
} else NULLP();


return( duplicate_colors );
}

static void colorhistory(
Image		input,
u_long		*counts_per_color,
struct ColourTriplet	*triplets
)
{

u_long			x,y;
long			colors;
long	width, height, channels, rowstride;
u_char	*src_row,*src;
u_char	*cmap, *cptr;

colors = gimp_image_colors( input );
width  = gimp_image_width( input );
height = gimp_image_height( input );
cmap   = gimp_image_cmap( input );
channels = gimp_image_channels( input );
rowstride = width * channels;

for( x = 0, cptr = cmap; x < colors ; x++ ) {
	triplets[ x ].Red   = *cptr++;
	triplets[ x ].Green = *cptr++;
	triplets[ x ].Blue  = *cptr++;
}

/* Read image data */
for( x = 0; x < colors; x++ ) counts_per_color[x]=0; 

src_row = gimp_image_data( input );

for( y = 0; y < height; y++ ) {
	src = src_row;
	for( x = 0; x < width; x++ ) {
			counts_per_color[*src++] += 1;
	}
	src_row += rowstride;
}

return;
}

static void remaphightolow(
Image		input,
Image		output,
u_long		counts_per_color[]
)
{

/*
** remaphightolow:
**
** Remap the palette from high reference count to low reference count.
** That is most frequently referenced pen numbers are put first.
**
*/

u_long	ix0,ix1,ix2,ix3,alt_colors,temp,isave;
u_char	*newpens,*convpens;
int	pens_changed;
int	height,width,channels,rowstride;
u_char	*src_row,*dest_row,*src,*dest,*cmap;

size_t	colors;

struct ColourTriplet	*triplets;

int	found;
struct palette	*palette;

colors =  gimp_image_colors(input);
if(newpens=calloc(colors*sizeof(u_char),1)){
	for(ix1=0;ix1<colors;ix1++) newpens[ix1]=ix1;
	pens_changed=FALSE;
	for(ix1=0;ix1<colors-1;ix1++) {
		isave=ix1;
		for(ix2=ix1+1;ix2<colors;ix2++) {
			if(counts_per_color[isave]<counts_per_color[ix2]) {
				isave=ix2;
			}
		}
		if(isave!=ix1){
			temp=counts_per_color[ix1];
			counts_per_color[ix1]=counts_per_color[isave];
			counts_per_color[isave]=temp;

			temp=newpens[ix1];
			newpens[ix1]=newpens[isave];
			newpens[isave]=temp;

			pens_changed=TRUE;
		}
	}
	
	if(pens_changed){
		/*
		** Build pen conversion table
		*/
		if(convpens=calloc(colors*sizeof(u_char),1))
		{
			for(ix1=0;ix1<colors;ix1++)
			{
				ix2=newpens[ix1];
				convpens[ix2]=ix1;
			}
			height    = gimp_image_height(input);
			width     = gimp_image_width(input);
			channels  = gimp_image_channels(input);
			rowstride = width * channels;
			src_row	  = gimp_image_data (input);
			dest_row  = gimp_image_data (output);

			for( ix0 = 0;  ix0 < height; ix0++ ) {
				dest = dest_row;
				src  = src_row;
				for( ix1 = 0; ix1 < width; ix1++ ) {
					*dest++ = convpens[ *src++ ];
				}
				src_row  += rowstride;
				dest_row += rowstride;
			}				
			free(convpens);
		}
		/*
		** Reorder palette
		*/
		if(triplets=calloc(colors*sizeof(struct ColourTriplet),1)){
			/* We could consider setting the unused colors
			 * to background color.
			 */
			/*
			** Save current palette
			*/
			if( palette = AllocPalette( colors, TRUE ) ) {
				cmap = gimp_image_cmap(input);
				cmap2palette(cmap,palette,colors);
				for(ix1=0;ix1<colors;ix1++){
					GetPaletteTriplet(palette,&triplets[ix1],ix1);
				}
				/*
				** Reorder palette according to new pens
				*/
				for(ix1=0;ix1<colors;ix1++){
					SetPaletteTriplet(palette,
							  triplets[newpens[ix1]].Red,
							  triplets[newpens[ix1]].Green,
							  triplets[newpens[ix1]].Blue,
							  ix1);
						 
				}
				palette2cmap(palette,cmap,colors);
				gimp_set_image_colors(output,cmap,colors);
				FreePalette( palette) ;
			} else NULLP();

			free(triplets);
		}
	}
	free(newpens);
}

return;
}

static void opti(
Image	input,
Image	output
)
{

/*
 * req_coloroptimize:
 */

struct image_info	*img_info;

u_long		*counts_per_color;

struct ColourTriplet	*triplets;

size_t	memsize,tripsize;

long	colors;
u_long	ix;
u_char	*cmap,*cptr;
return_code	rc=RETURN_ERROR;
int 		found_dups=FALSE;

/* Allocate counters */		
colors =  gimp_image_colors( input );
memsize = colors * sizeof(u_long);	

if( counts_per_color = calloc( memsize, 1 ) ) {
	tripsize = colors * sizeof(struct ColourTriplet);
	if( triplets = calloc( tripsize, 1 ) ) {
		if( cmap = gimp_image_cmap(input) ) {
			/* Convert indexed color map to triplets */
			for( ix = 0, cptr = cmap; ix < colors; ix++ ) {
				triplets[ix].Red = *cptr++;
				triplets[ix].Green = *cptr++;
				triplets[ix].Blue = *cptr++;
			}
			found_dups = find_dup_colors( &triplets[0], colors, input, output );
			colorhistory( found_dups ? output : input ,counts_per_color,triplets);
			remaphightolow( found_dups ? output : input, output, counts_per_color );
			free( cmap  );
			DPRINTF("Coloroptimization done\n");
			rc=RETURN_OK;
		} else NULLP();
		free( triplets );
	} else NULLP();
	free( counts_per_color );
} else NULLP();

return;
}






