/*
 * ProFTPD - FTP server daemon
 * Copyright (c) 1997, Public Flood Software
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Data transfer module for ProFTPD
 * $Id: mod_xfer.c,v 1.13 1997/12/29 20:22:01 flood Exp $
 */

/* History Log:
 *
 * 4/24/97 0.99.0pl1
 *   _translate_ascii was returning a buffer larger than the max buffer
 *   size causing memory overrun and all sorts of neat corruption.
 *   Status: Stomped
 *
 *   _list()/_nlist() consistantly overallocate their storage buffer
 *   (around 500 bytes or so for _list()), due to the conservative
 *   estimation algorithm that is used.  This needs to be recoded.
 *   Status: Needs Fix
 */

#include "conf.h"

static void _passive_cleanup(void *);

typedef struct dir_entry {
  struct dir_entry *next, *prev;

  char *d_name;
  char *d_link;
  int d_len;
  umode_t d_mode;
  char d_modestr[11];
  uid_t d_uid;
  gid_t d_gid;
  unsigned long d_size;
  unsigned long d_blocks;			/* # of 1K blocks */
  time_t d_time;				/* modification time */
  int d_nlinks;					/* Number of links */
} dir_t;

extern module auth_module;

/* From the auth module */
char *auth_map_uid(int);
char *auth_map_gid(int);

void xfer_abort(IOREQ*,int);

/* Variables for this module */
static FILE *retr_file = NULL;
static FILE *stor_file = NULL;
static int dconn_fd = -1;

module xfer_module;

static int _translate_ascii(char **buf, int *bufsize, int *adjlen)
{
  char *res = *buf;
  int newbufsize = 0;
  int thislen = *bufsize;

  if(**buf == '\n') {
    *--res = '\r';
    (*buf)++;
    newbufsize = 2;
    (*adjlen)++; thislen--;
  }

  while(thislen-- > 0 && newbufsize < *bufsize && **buf != '\n') {
    (*buf)++;
    newbufsize++;
  }

  *bufsize = newbufsize;
  *buf = res;
  return newbufsize;
}

static int _xfer_write_data(IOREQ *req, char *buf, int bufsize)
{
  int res,sbufsize,o_bufsize = bufsize;
  register int total = 0;

  if(!session.xfer.buflen) {
    if(session.xfer.get_data) {
      session.xfer.buf = session.xfer.bufstart;
     
      res = session.xfer.get_data(req,session.xfer.buf,session.xfer.bufsize);
      if(res > 0)
        session.xfer.buflen = res;
    }
 
    /* If the buflen is still 0, we're done */
    if(!session.xfer.buflen) {
      if(TimeoutNoXfer)
        reset_timer(TIMER_NOXFER,&xfer_module);

      if(session.xfer.complete)
        session.xfer.complete(req);

      register_cleanup(req->file->pool,(void*)session.d,
                       _passive_cleanup,_passive_cleanup);

      /* Not necessary to close both inf and outf here, as the
       * subpool management + _passive_cleanup will cause it to be
       * destroyed
       */

      io_close(req->file,IOR_REMOVE_FILE);
      session.d = NULL;

      if(session.xfer.p)
        destroy_pool(session.xfer.p);

      bzero(&session.xfer,sizeof(session.xfer));

      session.data_port = session.c->remote_port - 1;

      session.flags &= (SF_ALL^(SF_PASSIVE|SF_XFER|SF_ASCII_OVERRIDE));
      main_set_idle();

      return 0;
    }
  }

  if(bufsize > session.xfer.buflen)
    bufsize = session.xfer.buflen;

  while(bufsize > 0) {
    sbufsize = bufsize;
  
    if(session.flags & (SF_ASCII|SF_ASCII_OVERRIDE)) {
      if(o_bufsize > 1)
        _translate_ascii(&session.xfer.buf,&sbufsize,&session.xfer.buflen);
      else
        break;			/* Not enough client buffer to perform */
    }                           /* translation */

#ifdef HAVE_BCOPY
    bcopy(session.xfer.buf,buf,sbufsize);
#else
    memcpy(buf,session.xfer.buf,sbufsize);
#endif /* HAVE_BCOPY */
    session.xfer.buf += sbufsize;
    session.xfer.buflen -= sbufsize;
    buf += sbufsize;
    bufsize -= sbufsize; total += sbufsize;
    o_bufsize -= sbufsize;
  }

  session.xfer.total_bytes += total;

  if(total && TimeoutIdle)
    reset_timer(TIMER_IDLE,NULL);

  return total;
}

static int _xfer_read_data(IOREQ *req, char *buf, int bufsize)
{
  int res = 0;
  register int bskipped = 0;

  if(bufsize > 0 && TimeoutIdle)
    reset_timer(TIMER_IDLE,NULL);

  /* If in ASCII mode, strip out all '\r's */

  if(session.flags & SF_ASCII) {
    char *cp = buf,*dest = buf;
    register int i = bufsize;

    while(i--)
      if(*cp != '\r')
        *dest++ = *cp++;
      else {
        bufsize--;
        cp++; bskipped++;
      }
  }

  if(bufsize && session.xfer.get_data)
    res = session.xfer.get_data(req,buf,bufsize);

  if(res > 0)
    session.xfer.total_bytes += res;

  return res + bskipped;
}

static void _xfer_error(IOREQ *req,int err)
{
  char sbuf[256];

  sprintf(sbuf,R_426 " Data connection error, transfer aborted: %s\r\n",
          strerror(err));

  io_write_async(session.c->outf,sbuf,strlen(sbuf),NULL,NULL,NULL);
  xfer_abort(req,err);
}

static void _xfer_close(IOREQ *req)
{
  char sbuf[256];

  if(session.xfer.direction == IO_READ) {
    /* If we're reading, this indicates normal completion of transfer
     */
    if(TimeoutNoXfer)
      reset_timer(TIMER_NOXFER,&xfer_module);

    /* 0.99.0pl3, session.d was never closed */
    register_cleanup(req->file->pool,(void*)session.d,
                     _passive_cleanup,_passive_cleanup);
    io_close(req->file,IOR_REMOVE_FILE);

    if(session.xfer.complete)
      session.xfer.complete(req);

    if(session.xfer.p)
      destroy_pool(session.xfer.p);

    bzero(&session.xfer,sizeof(session.xfer));

    session.data_port = session.c->remote_port - 1;

    session.d = NULL;

    session.flags &= (SF_ALL^(SF_PASSIVE|SF_XFER|SF_ASCII_OVERRIDE));
  } else {
    strcpy(sbuf,R_426 " Data connection closed, transfer aborted.\r\n");
    io_write_async(session.c->outf,sbuf,strlen(sbuf),NULL,NULL,NULL);
    xfer_abort(req,0);
  }

  main_set_idle();
}

static void _passive_cleanup(void *cv)
{
  conn_t *c = (conn_t*)cv;

  c->inf = c->outf = NULL;
  inet_close(c->pool,c);
}

static int _passive_opened(IOREQ *req, char *buf, int bufsize)
{
  conn_t *c;
  char sbuf[256];

  inet_setblock(session.d->pool,session.d);
  c = inet_accept(session.pool,session.d,-1,-1,TRUE);

  if(c) {
    if(c->remote_ipaddr->s_addr == session.c->remote_ipaddr->s_addr) {
      register_cleanup(req->file->pool,(void*)session.d,_passive_cleanup,
                       _passive_cleanup);

      io_close(req->file,IOR_REMOVE_FILE);
      inet_setnonblock(session.pool,c);
      session.d = c;
      io_setbuf(c->inf,0);
      io_setbuf(c->outf,0);

      if(session.xfer.file_size)
        sprintf(sbuf,R_150 " Opening %s mode data connection for %s (%lu bytes).\r\n",
		(session.flags & (SF_ASCII|SF_ASCII_OVERRIDE) ? "ASCII" : "BINARY"),
                 session.xfer.filename,session.xfer.file_size);
      else 
        sprintf(sbuf,R_150 " Opening %s mode data connection for %s.\r\n",
               (session.flags & (SF_ASCII|SF_ASCII_OVERRIDE) ? "ASCII" : "BINARY"),
                session.xfer.filename);

      io_write_async(session.c->outf,sbuf,strlen(sbuf),
                     NULL,NULL,NULL);
  
      if(session.xfer.direction == IO_READ) {
        io_setbuf(c->inf,main_server->tcp_rwin);
        inet_setoptions(req->pool,c,main_server->tcp_rwin,0);
        session.d_req = io_read_async(c->inf,NULL,0,
					_xfer_read_data,
					_xfer_error,
					_xfer_close);
      } else {
        io_setbuf(c->outf,main_server->tcp_swin);
        inet_setoptions(req->pool,c,0,main_server->tcp_swin);
        session.d_req = io_write_async(c->outf,NULL,0,
                                        _xfer_write_data,
                                        _xfer_error,
                                        _xfer_close);
      }
      gettimeofday(&session.xfer.start_time,NULL);
      return 0;
    }
          
    inet_close(session.pool,c);
    sprintf(sbuf,R_425 " Can't build data connection: Transport endpoint address mismatch.\r\n");
    io_write_async(session.c->outf,sbuf,strlen(sbuf),NULL,NULL,NULL);
    xfer_abort(req,0);
    return 0;
  }

  inet_resetlisten(session.pool,session.d);
  inet_setnonblock(session.pool,session.d);
  io_repost_req(req);
  return 0;
}

static void _passive_close(IOREQ *req)
{
  xfer_abort(req,0);
}

static void _passive_error(IOREQ *req, int err)
{
  /* Some sort of error occured, abort entire xfer */
  xfer_abort(req,err);
}

static int _active_connect(IOREQ *req, char *buf, int bufsize)
{
  conn_t *c;
  char sbuf[256];


  if(session.d->mode == CM_CONNECT)
    switch(inet_connect_nowait(req->pool,session.d,session.c->remote_ipaddr,
                               session.data_port)) {
    case 0:
      io_repost_req(req);
      return 0;
    case -1:
      sprintf(sbuf,R_425 " Can't build data connection: %s\r\n",
              strerror(session.d->xerrno));
      io_write_async(session.c->outf,sbuf,strlen(sbuf),
                     NULL,NULL,NULL);
      xfer_abort(req,session.d->xerrno);
      return 0;
    }

  c = inet_openrw(session.pool,session.d,NULL,
                  session.d->listen_fd,-1,-1,TRUE);

  register_cleanup(req->file->pool,(void*)session.d,
                   _passive_cleanup,_passive_cleanup);
  io_close(req->file,IOR_REMOVE_FILE);

  session.d = c;

  if(c) {
    if(session.xfer.direction == IO_READ) {
      io_setbuf(c->inf,main_server->tcp_rwin);
      inet_setoptions(req->pool,c,main_server->tcp_rwin,0);
      session.d_req = io_read_async(c->inf,NULL,0,
					_xfer_read_data,
					_xfer_error,
					_xfer_close);
    } else {
      io_setbuf(c->outf,main_server->tcp_swin);
      inet_setoptions(req->pool,c,0,main_server->tcp_swin);
      session.d_req = io_write_async(c->outf,NULL,0,
                                        _xfer_write_data,
                                        _xfer_error,
                                        _xfer_close);
    }

    if(session.xfer.file_size)
      sprintf(sbuf,R_150 " Opening %s mode data connection for %s (%lu bytes).\r\n",
              (session.flags & (SF_ASCII|SF_ASCII_OVERRIDE) ? "ASCII" : "BINARY"),
	      session.xfer.filename,session.xfer.file_size);
    else 
      sprintf(sbuf,R_150 " Opening %s mode data connection for %s.\r\n",
              (session.flags & (SF_ASCII|SF_ASCII_OVERRIDE) ? "ASCII" : "BINARY"),
              session.xfer.filename);

    io_write_async(session.c->outf,sbuf,strlen(sbuf),NULL,NULL,NULL);
    gettimeofday(&session.xfer.start_time,NULL);
  }
  return 0;    
}

static void _active_error(IOREQ *req, int err)
{
  char sbuf[256];

  sprintf(sbuf,R_425 " Can't build data connection: %s\r\n",
               strerror(err));

  io_write_async(session.c->outf,sbuf,strlen(sbuf),NULL,NULL,NULL);
  xfer_abort(req,err);
}

void xfer_abort(IOREQ *req, int err)
{
  IOFILE *f;

  f = (req && req->file ? req->file : session.d->inf);

  session.flags |= SF_ABORT;

  if(session.xfer.abort)
    session.xfer.abort(req,err);

  if(f) {
    /* We don't need to close both inf and outf here,
     * because _passive_cleanup will cause the data conn_t to be
     * destroyed, thus destroying all sub-pools (thus outf and
     * any associated ioreqs)
     */
    if(session.d)
      register_cleanup(f->pool,(void*)session.d,_passive_cleanup,
                       _passive_cleanup);
    io_close(f,IOR_REMOVE_FILE);
  }

  if(session.xfer.p)
    destroy_pool(session.xfer.p);

  bzero(&session.xfer,sizeof(session.xfer));

  session.data_port = session.c->remote_port - 1;

  if(session.d)
    session.d = NULL;

  session.flags &= (SF_ALL^SF_PASSIVE);
  session.flags &= (SF_ALL^(SF_ABORT|SF_XFER|SF_PASSIVE|SF_ASCII_OVERRIDE));
  main_set_idle();
}

int xfer_open_data(int (*cl_io)(IOREQ*,char*,int),char *reason,
                   unsigned long size)
{
  if(session.flags & SF_PASSIVE) {
    if(!session.d) {
      log_pri(LOG_ERR,"Internal error: PASV mode set, but no data connection listening.");
      end_login(1);
    }

    if(session.d->mode != CM_OPEN) {
      io_kill_req(session.d_req);
      session.d_req = io_post_req(session.d->inf,IO_READ,
                                  _passive_opened,
                                  _passive_error,
				  _passive_close);

    } else {

      if(size)
        send_response(R_150,
                      "Opening %s mode data connection for %s (%lu bytes).",
                      (session.flags & (SF_ASCII|SF_ASCII_OVERRIDE) ? "ASCII" : "BINARY"),
                      reason, size);
      else
        send_response(R_150,"Opening %s mode data connection for %s.",
                      (session.flags & (SF_ASCII|SF_ASCII_OVERRIDE) ? "ASCII" : "BINARY"),
                      reason);

      gettimeofday(&session.xfer.start_time,NULL);
      if(session.xfer.direction == IO_READ) {
        io_setbuf(session.d->inf,main_server->tcp_rwin);
        inet_setoptions(session.d->pool,session.d,main_server->tcp_rwin,0);
        session.d_req = io_read_async(session.d->inf,NULL,0,
                          _xfer_read_data,
                          _xfer_error,
                          _xfer_close);
      } else {
        io_setbuf(session.d->outf,main_server->tcp_swin);
        inet_setoptions(session.d->pool,session.d,0,main_server->tcp_swin);
        session.d_req = io_write_async(session.d->outf,NULL,0,
                          _xfer_write_data,
                          _xfer_error,
                          _xfer_close);
      }
    }
  } else {
    if(session.d) {
      log_pri(LOG_ERR,"Internal error: non-PASV mode, data connection already exists!?!");
      end_login(1);
    }

    session.d = inet_create_dup_connection(session.pool,NULL,dconn_fd);
    if(inet_connect_nowait(session.d->pool,session.d,session.c->remote_ipaddr,
                        session.data_port) == -1) {
      send_response(R_425,"Can't build data connection: %s",
                    strerror(session.d->xerrno));
      inet_close(session.pool,session.d);
      session.d = NULL;
      xfer_abort(NULL,0);
      return -1;
    }

    if(session.d->mode == CM_OPEN) {
      /* Connection opened immediately */
      conn_t *c;

      c = inet_openrw(session.pool,session.d,NULL,
                      session.d->listen_fd,-1,-1,TRUE);

      inet_close(session.pool,session.d);

      session.d = c;

      if(c) {
        if(size)
          send_response(R_150,
                        "Opening %s mode data connection for %s (%lu bytes).",
                        (session.flags & (SF_ASCII|SF_ASCII_OVERRIDE) ? "ASCII" : "BINARY"),
                        reason,size);
        else
          send_response(R_150,"Opening %s mode data connection for %s.",
                       (session.flags & (SF_ASCII|SF_ASCII_OVERRIDE) ? "ASCII" : "BINARY"),
                       reason);

	gettimeofday(&session.xfer.start_time,NULL);

        if(session.xfer.direction == IO_READ) {
          io_setbuf(c->inf,main_server->tcp_rwin);
          inet_setoptions(c->pool,c,main_server->tcp_rwin,0);
          session.d_req = io_read_async(c->inf,NULL,0,
	  				_xfer_read_data,
					_xfer_error,
					_xfer_close);
        } else {
          io_setbuf(c->outf,main_server->tcp_swin);
          inet_setoptions(c->pool,c,0,main_server->tcp_swin);
          session.d_req = io_write_async(c->outf,NULL,0,
                                        _xfer_write_data,
                                        _xfer_error,
                                        _xfer_close);
        }

      } else {
        send_response(R_425,"Can't build data connection (internal error)");
        xfer_abort(NULL,0);
      }
      return(0);
    }


    /* Connection open is now in progress, so post a request to handle
     * everything once the connection becomes available
     */

    session.d->inf = io_open(session.d->pool,session.d->listen_fd,IO_READ);
    session.d_req = io_post_req(session.d->inf,IO_READ,
                                _active_connect,
                                _active_error,
				NULL);

  }
  return 0;
}
                           
static int _buffered_complete(IOREQ *req)
{
  char buf[] = R_226 " Transfer complete.\r\n";

  
  io_write_async(session.c->outf,buf,strlen(buf),
                 NULL,NULL,NULL);

  return 0;
}

char *xfer_setup_xfer(int direction, char *filename, int buflen,
                      int (*get_data)(IOREQ *,char*,int),
                      int (*complete)(IOREQ*))
{
  /* In case some other command as left data in session.xfer
   * (SF_XFER takes care of reentrancy, so it can't be us)
   */

  if(session.xfer.p) {
    destroy_pool(session.xfer.p);
    bzero(&session.xfer,sizeof(session.xfer));
  }

  session.xfer.p = make_sub_pool(session.pool);
  session.xfer.filename = pstrdup(session.xfer.p,filename);
  session.xfer.direction = direction;

  if(!get_data) {
    session.xfer.buf = (char*)palloc(session.xfer.p,buflen+1);
    /* Leave room for ASCII translation */
    session.xfer.bufstart = ++session.xfer.buf;
    session.xfer.bufsize = session.xfer.buflen = buflen;
    session.xfer.get_data = NULL;
    session.xfer.complete = _buffered_complete;
  } else {
    session.xfer.buf = (char*)palloc(session.xfer.p,buflen+1);
    /* Leave room for ASCII translation */
    session.xfer.bufstart = ++session.xfer.buf;
    session.xfer.bufsize = buflen;
    /* We set buflen to 0 so that the first call to _xfer_write_data will
     * result in get_data() being called and filling the buffer.
     */
    session.xfer.buflen = 0;
    session.xfer.get_data = get_data;
    session.xfer.complete = (complete ? complete : _buffered_complete);
  }

  if(TimeoutNoXfer)
    reset_timer(TIMER_NOXFER,&xfer_module);
  session.flags |= SF_XFER;
  return session.xfer.buf;
}

static char *_get_full_cmd(cmd_rec *cmd)
{
  pool *p;
  char *res = "";
  int i;

  p = cmd->tmp_pool;

  if(cmd->arg)
    res = pstrcat(p,cmd->argv[0]," ",cmd->arg,NULL);
  else {
    for(i = 0; i < cmd->argc; i++)
      res = pstrcat(p,res,cmd->argv[i]," ",NULL);

    while(res[strlen(res)-1] == ' ')
      res[strlen(res)-1] = '\0';
  }

  return res;
}

static int _compare_dir(dir_t *d1, dir_t *d2)
{
  return strcmp(d1->d_name,d2->d_name);
}

static xaset_t *_ls(cmd_rec *cmd, char *dir, char *spec, char *flags,
                    int *n_entries,int *total_bytes,int *total_blocks,
                    int *maxlen)
{
  pool *p;
  DIR *d;
  struct dirent *tmp;
  struct stat sbuf;
  dir_t *dent;
  int i,opt = 0,opt_all = 0,opt_classify = 0,opt_dir = 0;
  int showsymlinks = 0;
  char fullname[MAXPATHLEN],linkname[MAXPATHLEN];
  xaset_t *listing;
  char *cp,*pathglob,*realdir;

  *total_bytes = *total_blocks = 0;
  *n_entries = 0;
  *maxlen = 0;

  /* Unless explicitly set, symlinks are hidden in anonymous configurations
   * and visible for normal logins
   */

  showsymlinks = get_param_int( (session.anon_config ? session.anon_config->subset : main_server->conf), 
                                "ShowSymlinks",FALSE);

  if(showsymlinks == -1)
    showsymlinks = (session.anon_config ? 0 : 1);

  if(flags)
    do {
      opt++;
      switch(*flags) {
      case 'a':
        opt_all++;
        break;
      case 'F':
        opt_classify++;
        break;
      case 'd':
        opt_dir++;
        break;
      }
    } while(*flags++);

  p = cmd->tmp_pool;

  /* If spec is empty, user specified no wildcards */

  if(!*spec) {
    if(stat(dir,&sbuf) != -1) {
      if(opt_dir | !S_ISDIR(sbuf.st_mode)) {
        cp = rindex(dir,'/');
        if(cp) { spec = pstrdup(p,cp+1); *(cp+1) = '\0'; }
        else { spec = dir; dir = ""; }
      } else if(S_ISDIR(sbuf.st_mode))
        spec = "*";
    } else
      spec = "*";
  }

  if(!*dir)
    dir = "./";

  if((realdir = dir_realpath(p,dir)))
    dir = realdir;

  if(*(dir+strlen(dir)-1) != '/')
    dir = pstrcat(p,dir,"/",NULL);

  if(!dir_check(p,cmd->argv[0],cmd->group,dir)) 
    return NULL;

  /* If you think this is odd, that's because it is.  It seems as
   * though some FTP clients rely upon NLST <flags> <dir> returning
   * no data on a connection in order to determine if a directory
   * exists during a "mget" operation.  So, if we have no flags
   * we return an error, otherwise an empty listing.
   */

  d = opendir(dir);

  if(!d && !opt)
    return NULL;

  pathglob = pdircat(p,dir,spec,NULL);

  listing = xaset_create(p,(XASET_COMPARE)_compare_dir);

  if(!d)
    return listing;

  while((tmp = readdir(d)) != NULL) {
    if(tmp->d_name[0] == '.' && !opt_all)
      continue;

    strcpy(fullname,dir);
    strcat(fullname,tmp->d_name);

    if(fnmatch(pathglob,fullname,FNM_PATHNAME) != 0)
      continue;

    if((!showsymlinks && stat(fullname,&sbuf) != -1) ||
       (showsymlinks && lstat(fullname,&sbuf) != -1)) {
      /* Check for hidden entries */
      if(!dir_check_op_mode(p,fullname,OP_HIDE,(int)sbuf.st_uid,
                            (int)sbuf.st_gid,(int)sbuf.st_mode))
        continue;

      if(S_ISLNK(sbuf.st_mode) &&
           (i = readlink(fullname,linkname,MAXPATHLEN)) != -1) {
        linkname[i] = '\0';
      } else
        linkname[0] = '\0';

      dent = pcalloc(p,sizeof(dir_t));
      if(opt_classify) {
        dent->d_name = pstrcat(p,tmp->d_name," ",NULL);
        cp = dent->d_name + strlen(dent->d_name)-1;
      } else {
        dent->d_name = pstrdup(p,tmp->d_name);
        cp = NULL;
      }

      if(linkname[0]) {
        dent->d_link = pstrdup(p,linkname);
        *total_bytes += strlen(linkname) + 4;
      }

      dent->d_len = NAMLEN(tmp);
        
      dent->d_mode = sbuf.st_mode;
      if(S_ISREG(sbuf.st_mode))
        dent->d_modestr[0] = '-'; 
      else if(S_ISLNK(sbuf.st_mode))
        { dent->d_modestr[0] = 'l'; if(cp) *cp = '@'; }
      else if(S_ISDIR(sbuf.st_mode))
        { dent->d_modestr[0] = 'd'; if(cp) *cp = '/'; }
      else if(S_ISCHR(sbuf.st_mode))
        dent->d_modestr[0] = 'c';
      else if(S_ISBLK(sbuf.st_mode))
        dent->d_modestr[0] = 'b';
      else if(S_ISFIFO(sbuf.st_mode))
        { dent->d_modestr[0] = 'p'; if(cp) *cp = '|'; }
      else if(S_ISSOCK(sbuf.st_mode))
        { dent->d_modestr[0] = 's'; if(cp) *cp = '='; }
      else
        dent->d_modestr[0] = '?';

      if(cp && S_ISREG(sbuf.st_mode) && (sbuf.st_mode & 0111))
        *cp = '*';
 
      if(cp && *cp == ' ') {
        *cp = '\0';
        dent->d_len--;
      }

      /* Now fill in the rest of the mode */
      i = 0;
      while(i < 9) {
        dent->d_modestr[++i] = (sbuf.st_mode & 0400 ? 'r' : '-');
        dent->d_modestr[++i] = (sbuf.st_mode & 0200 ? 'w' : '-');
        dent->d_modestr[++i] = (sbuf.st_mode & 0100 ? 'x' : '-');
        sbuf.st_mode <<= 3;
      }

      dent->d_uid = sbuf.st_uid;
      dent->d_gid = sbuf.st_gid;
      dent->d_size = sbuf.st_size;
      dent->d_time = sbuf.st_mtime;
      dent->d_nlinks = sbuf.st_nlink;
      dent->d_blocks = (sbuf.st_size/1024) +
                       (sbuf.st_size % 1024 ? 1 : 0);

      /* Max size of returned line is 62 (include trailing \n) plus
       * length of filename and directory 
       */

      *total_bytes += dent->d_len;
      *total_blocks += dent->d_blocks;
      (*n_entries)++;
      if(dent->d_len > *maxlen)
        *maxlen = dent->d_len;

      xaset_insert_sort(listing,(xasetmember_t*)dent,TRUE);
    }
  }
  closedir(d);

  return listing;
}

static char *_make_spec(pool *p, char *dir)
{
  char *res;
  int last;

  last = strlen(dir)-1;
  
  while(last > 0 && *(dir+last) == '/')
    *(dir+last--) = '\0';

  if(!strchr(dir,'*'))
    return "";

  /* Find the right-most side of the dir, and chop it off, returning
     as a spec */

  res = rindex(dir,'/');
  if(res && *(res+1)) {
    if(res == dir) {
      res = pstrdup(p,dir+1);
      *(dir+1) = '\0';
    } else
      *res++ = '\0';

    return res;
  }

  res = pstrdup(p,dir);
  *dir = '\0';
  return res;
}

static char *_list(cmd_rec *cmd, char *dir, char *flags, int *size)
{
  char *cp,*res;
  int total_bytes,total_blocks;
  int n_entries,this_year,maxlen;
  xaset_t *listing;
  dir_t *dent;
  struct tm *mtime;
  time_t t;
  char fullname[256],tmstr[20];
  static char *mon[] = 
  { "Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec" };

  listing = _ls(cmd,dir,_make_spec(cmd->tmp_pool,dir),
                flags,&n_entries,&total_bytes,&total_blocks,&maxlen);

  if(!listing)
    return NULL;

  if(!listing->xas_list) {		/* No dirs to return */
    res = palloc(cmd->tmp_pool,1);
    *size = 0;
    return res;
  }

  total_bytes += (n_entries * 62)+30;

  time(&t);
  mtime = gmtime(&t);
  this_year = mtime->tm_year;

  res = palloc(cmd->tmp_pool,total_bytes);

  log_debug(DEBUG4,"allocate: %d bytes for _list()",total_bytes);

  *res = '\0'; cp = res;
  *size = 0;

  sprintf(fullname,"total %d\n",total_blocks);
  strcpy(cp,fullname);
  cp += strlen(fullname);
  *size += strlen(fullname);

  for(dent = (dir_t*)listing->xas_list; dent; dent=dent->next) {
    *tmstr = '\0';
    mtime = gmtime(&dent->d_time);

    if(mtime) {
      if(mtime->tm_year != this_year)
        sprintf(tmstr,"%s %2d %5d",mon[mtime->tm_mon],
                mtime->tm_mday, mtime->tm_year + 1900);
      else
        sprintf(tmstr,"%s %2d %02d:%02d",mon[mtime->tm_mon],
                mtime->tm_mday, mtime->tm_hour, mtime->tm_min);
    }

    sprintf(fullname,"%s %3d %-8s %-8s %8lu %s %s",
            dent->d_modestr,dent->d_nlinks,
            auth_map_uid((int)dent->d_uid),
            auth_map_gid((int)dent->d_gid),
            dent->d_size,tmstr,dent->d_name);

    if(dent->d_link) {
      strncat(fullname," -> ",sizeof(fullname)-1);
      strncat(fullname,dent->d_link,sizeof(fullname)-1);
    }
    strcat(fullname,"\n");

    strcpy(cp,fullname);
    cp += strlen(fullname);
    *size += strlen(fullname);
  }

  return res;
}

static char *_nlst(cmd_rec *cmd, char *dir, char *flags, int *size)
{
  char *cp,*res,*fl = flags;
  int total_bytes,total_blocks;
  int n_entries;
  xaset_t *listing;
  dir_t *dent;
  char fullname[256];
  int maxlen;
  int opt_cols = 0,opt_long = 0;

  if(fl)
    do {
      switch(*fl) {
      case 'C':
        opt_cols++;
        break;
      case 'l':
        opt_long++;
        break;
      }
    } while(*fl++);

  if(opt_long)
    return _list(cmd,dir,flags,size);

  listing = _ls(cmd,dir,_make_spec(cmd->tmp_pool,dir),
                flags,&n_entries,&total_bytes,&total_blocks,&maxlen);

  if(!listing)
    return NULL;

  if(!listing->xas_list) {		/* No dirs to return */
    *size = 0;
    res = palloc(cmd->tmp_pool,1);
    return res;
  }

  if(opt_cols) {
    array_header **cols;
    int ncols,i,j;
    int percol;
    dir_t **dentp;
    char fmtstr[10];

    ncols = 77 / (maxlen+2);
    percol = n_entries/ncols;
    if(n_entries % ncols)
      percol++;

    if(ncols > 1) {
      total_bytes = percol * 80;

      res = palloc(cmd->tmp_pool,total_bytes);

      log_debug(DEBUG4,"allocate: %d bytes for _nlst()",total_bytes);

      *res = '\0'; cp = res;
      *size = 0;

      cols = (array_header**)palloc(listing->mempool,
                                    ncols * sizeof(array_header*));

      dent = (dir_t*)listing->xas_list;
      for(i = 0; i < ncols; i++) {
        cols[i] = make_array(listing->mempool,percol,
                             sizeof(dir_t*));
        for(j = 0; j < percol && dent; j++) {
          *((dir_t**)push_array(cols[i])) = dent;
          dent = dent->next;
        }
      }

      sprintf(fmtstr,"%%-%ds  ",maxlen+1);
      for(j = 0; j < percol; j++) {
        for(i = 0; i < ncols; i++) {
          dentp = ((dir_t**)cols[i]->elts) + j;
          if(*dentp)
            sprintf(fullname,fmtstr,(*dentp)->d_name);
          else
            sprintf(fullname,fmtstr,"");
          strcpy(cp,fullname);
          cp += strlen(fullname);
          *size += strlen(fullname);
        }

        *cp++ = '\n';
        (*size)++;
      }
    
      return res;
    }
  }
    
  total_bytes += n_entries;		/* For \n */

  res = palloc(cmd->tmp_pool,total_bytes);
  *res = '\0'; cp = res;
  *size = 0;

  for(dent = (dir_t*)listing->xas_list; dent; dent=dent->next) {
    sprintf(fullname,"%s\n",dent->d_name);
    strcpy(cp,fullname);
    cp += strlen(fullname);
    *size += strlen(fullname);
  }

  return res;
}

static char *cmd_nlst(cmd_rec *cmd)
{
  char *lsbuf;
  char *tmp;
  char *dir = "",*flags = NULL;
  int lsbufsize = 0;

  if(cmd->argc > 1) {
    if(cmd->argv[1][0] == '-') {
      flags = cmd->argv[1];
      get_word(&cmd->arg);
    }

    dir = cmd->arg;
  }

  lsbuf = _nlst(cmd,dir,flags,&lsbufsize);

  log_debug(DEBUG4,"used: %d bytes for _nlst()",lsbufsize);

  if(!lsbuf) {
    send_response(R_550,"%s: %s",dir,strerror(errno));
    return NULL;
  }

  session.flags |= SF_ASCII_OVERRIDE;
  tmp = xfer_setup_xfer(IO_WRITE,"file list",lsbufsize,NULL,NULL);

  if(tmp)
    bcopy(lsbuf,tmp,lsbufsize);

  xfer_open_data(NULL,"file list",0);

  return NULL;
}

static char *cmd_stat(cmd_rec *cmd)
{
  char *rawdir,*dir,*flags,*cp;
  char tmstr[20];
  struct tm *mtime;
  xaset_t *listing;
  int n_entries,total_bytes,total_blocks,maxlen,this_year;
  int opt_dir = 0;
  time_t t;
  dir_t *dent;
  static char *mon[] = 
  { "Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec" };

  if(cmd->argc > 3 || cmd->argc < 2) {
    send_response(R_500,"'%s' not understood.",_get_full_cmd(cmd));
    return NULL;
  }

  time(&t);
  mtime = gmtime(&t);
  this_year = mtime->tm_year;

  if(cmd->argc > 1) {
    if(cmd->argv[1][0] == '-') {
      flags = pstrcat(cmd->tmp_pool,cmd->argv[1],"a",NULL);
      get_word(&cmd->arg);
      dir = pstrdup(cmd->tmp_pool,cmd->arg);
    } else
      flags = "a";
    rawdir = cmd->arg;
    dir = pstrdup(cmd->tmp_pool,cmd->arg);
  } else {
    dir = "";
    flags = "a";
    rawdir = "";
  }

  for(cp = flags; *cp; cp++)
    if(*cp == 'd')
      opt_dir++;

  listing = _ls(cmd,dir,_make_spec(cmd->tmp_pool,dir),
                flags,&n_entries,&total_bytes,&total_blocks,&maxlen);

  if(!listing || !listing->xas_list) {
    send_response(R_550,"%s: %s",rawdir,strerror(errno));
    return NULL;
  }

  send_response_ml_start(R_211,"status of %s:",rawdir);

  for(dent = (dir_t*)listing->xas_list; dent; dent=dent->next) {
      *tmstr = '\0';
      mtime = gmtime(&dent->d_time);

      if(mtime) {
        if(mtime->tm_year != this_year)
          sprintf(tmstr,"%s %2d %5d",mon[mtime->tm_mon],
                  mtime->tm_mday, mtime->tm_year + 1900);
        else
          sprintf(tmstr,"%s %2d %02d:%02d",mon[mtime->tm_mon],
                  mtime->tm_mday, mtime->tm_hour, mtime->tm_min);
      }

      send_response_raw("%s %3d %-8s %-8s %8lu %s %s",
            dent->d_modestr,dent->d_nlinks,
            auth_map_uid((int)dent->d_uid),
            auth_map_gid((int)dent->d_gid),
            dent->d_size,tmstr,dent->d_name);
  }

  send_response_ml_end("End of Status");
  return NULL;
}

static char *cmd_list(cmd_rec *cmd)
{
  char *lsbuf;
  char *tmp;
  char *dir = "",*flags = NULL;
  int lsbufsize = 0;


  if(cmd->argc > 1) {
    if(cmd->argv[1][0] == '-') {
      flags = cmd->argv[1];
      get_word(&cmd->arg);
      dir = cmd->arg;
    } else
      dir = cmd->arg;
  }

  lsbuf = _list(cmd,dir,flags,&lsbufsize);

  log_debug(DEBUG4,"used: %d bytes for _list()",lsbufsize);

  if(!lsbuf) {
    send_response(R_550,"%s: %s",dir,strerror(errno));
    return NULL;
  }

  session.flags |= SF_ASCII_OVERRIDE;
  tmp = xfer_setup_xfer(IO_WRITE,"file list",lsbufsize,NULL,NULL);

  if(tmp)
    bcopy(lsbuf,tmp,lsbufsize);
  
  xfer_open_data(NULL,"file list",0);

  return NULL;
}
  
static int _stor_write(IOREQ *req, char *buf, int len)
{
  int r;

  if(!stor_file)
    return -1;

  r = fwrite(buf,1,len,stor_file);

  if(r == -1)
    io_set_req_errno(req,ferror(stor_file));
  else
    session.xfer.tr_size += r;

  return r;
}
    
static int _retr_read(IOREQ *req, char *buf, int len)
{
  int r;

  if(feof(retr_file))
    return 0;

  r = fread(buf,1,len,retr_file);

  if(!r || r < len) {
    if(!feof(retr_file)) {
      r = -1;
      io_set_req_errno(req,ferror(retr_file));
    }
  }

  if(r != -1)
    session.xfer.tr_size += r;

  return r; 
}

static void _log_transfer(char direction)
{
  struct timeval end_time;
  char *fullpath;

  gettimeofday(&end_time,NULL);

  end_time.tv_sec -= session.xfer.start_time.tv_sec;
  if(end_time.tv_usec >= session.xfer.start_time.tv_usec)
    end_time.tv_usec -= session.xfer.start_time.tv_usec;
  else {
    end_time.tv_usec = 1000000L - (session.xfer.start_time.tv_usec -
                       end_time.tv_usec);
    end_time.tv_sec--;
  }

  fullpath = dir_abs_path(session.xfer.p,session.xfer.path);

  if((session.flags & SF_ANON) != 0) {
    log_xfer(end_time.tv_sec,session.c->remote_name,session.xfer.tr_size,
             fullpath,(session.flags & SF_ASCII ? 'a' : 'b'),
             direction,'a',session.anon_user);
  } else
    log_xfer(end_time.tv_sec,session.c->remote_name,session.xfer.tr_size,
             fullpath,(session.flags & SF_ASCII ? 'a' : 'b'),
             direction,'r',session.user);

  log_pri(LOG_DEBUG,"Transfer completed: %d bytes in %d.%02d seconds.",
                 session.xfer.total_bytes,end_time.tv_sec,
                 (end_time.tv_usec / 10000));
}

static int _stor_done(IOREQ *req)
{
  _log_transfer('i');

  fclose(stor_file);
  stor_file = NULL;

  if(session.fsgid && session.xfer.path) {
    struct stat sbuf;

    stat(session.xfer.path,&sbuf);
    if(chown(session.xfer.path,(uid_t)-1,(gid_t)session.fsgid) == -1)
      log_pri(LOG_WARNING,"chown(%s) failed: %s",
              session.xfer.path,strerror(errno));
    else
      chmod(session.xfer.path,sbuf.st_mode);
  }

  _buffered_complete(req);
  return 0;
}

static int _retr_done(IOREQ *req)
{
  _log_transfer('o');

  fclose(retr_file);
  retr_file = NULL;

  _buffered_complete(req);
  return 0;
}

static int _stor_abort(IOREQ *req, int err)
{
  fclose(stor_file);
  stor_file = NULL;
  if(session.xfer.path)
    unlink(session.xfer.path);

  return 0;
}

static int _retr_abort(IOREQ *req, int err)
{
  /* Isn't necessary to send anything here, just cleanup */
  fclose(retr_file);
  retr_file = NULL;
  return 0;
}

static char *cmd_stor(cmd_rec *cmd)
{
  char *dir;
  mode_t fmode;

  if(cmd->argc < 2) {
    send_response(R_500,"'%s' not understood.",_get_full_cmd(cmd));
    return NULL;
  }

  dir = dir_canonical_path(cmd->tmp_pool,cmd->arg);

  if(!dir || !dir_check(cmd->tmp_pool,cmd->argv[0],cmd->group,dir)) {
    send_response(R_550,"%s: %s",cmd->arg,strerror(errno));
    return NULL;
  }

  fmode = file_mode(dir);

  if(fmode && get_param_int(CURRENT_CONF,"AllowOverwrite",FALSE) != 1) {
    send_response(R_550,"%s: Overwrite permission denied",cmd->arg);
    return NULL;
  }

  if(fmode && !S_ISREG(fmode)) {
    send_response(R_553,"%s: Not a regular file",cmd->arg);
    return NULL;
  }

  /* If restarting, check permissions on this directory, if
   * AllowStoreRestart is set, permit it
   */

  if(fmode && session.restart_pos &&
     get_param_int(CURRENT_CONF,"AllowStoreRestart",FALSE) != TRUE) {
    send_response(R_451,"%s: Restart not permitted, try again.",
                  cmd->arg);
    session.restart_pos = 0L;
    return NULL;
  }

  stor_file = fopen(dir,(session.restart_pos ? "a" : "w"));

  if(stor_file && session.restart_pos) {
    if(fseek(stor_file,session.restart_pos,SEEK_SET) == -1) {
      int _errno = errno;
      fclose(stor_file);
      errno = _errno;
      stor_file = NULL;
    }

    session.restart_pos = 0L;
  }

  if(!stor_file)
    send_response(R_550,"%s: %s",cmd->arg,strerror(errno));
  else {
    xfer_setup_xfer(IO_READ,cmd->arg,main_server->tcp_rwin,
                    _stor_write,_stor_done);
    session.xfer.path = pstrdup(session.xfer.p,dir);
    session.xfer.abort = _stor_abort;
    session.xfer.file_size = session.restart_pos;
    xfer_open_data(NULL,cmd->arg,0);
  }
  return NULL;
}

static char *cmd_rest(cmd_rec *cmd)
{
  long int pos;
  char *endp;

  if(cmd->argc != 2) {
    send_response(R_500,"'%s': command not understood.",_get_full_cmd(cmd));
    return NULL;
  }

  pos = strtol(cmd->argv[1],&endp,10);
  if((endp && *endp) || pos < 0) {
    send_response(R_501,"REST requires a value greater than or equal to 0.");
    return NULL;
  }

  session.restart_pos = pos;
  send_response(R_350,"Restarting at %ld. Send STORE or RETRIEVE to initiate transfer.",
                pos);
  return NULL;
}

static char *cmd_retr(cmd_rec *cmd)
{
  char *dir;
  mode_t fmode;
  struct stat sbuf;

  if(cmd->argc < 2) {
    send_response(R_500,"'%s' not understood.",_get_full_cmd(cmd));
    return NULL;
  }

  dir = dir_realpath(cmd->tmp_pool,cmd->arg);

  if(!dir || !dir_check(cmd->tmp_pool,cmd->argv[0],cmd->group,dir)) {
    send_response(R_550,"%s: %s",cmd->arg,strerror(errno));
    return NULL;
  }


  fmode = file_mode(dir);

  if(!S_ISREG(fmode)) {
    if(!fmode)
      send_response(R_550,"%s: %s",cmd->arg,strerror(errno));
    else
      send_response(R_553,"%s: Not a regular file",cmd->arg);
    return NULL;
  }

  /* If restart is on, check to see if AllowRestartRetrieve
   * is off, in which case we disallow the transfer and
   * clear restart_pos
   */

  if(session.restart_pos &&
     get_param_int(CURRENT_CONF,"AllowRetrieveRestart",FALSE) == 0) {
    send_response(R_451,"%s: Restart not permitted, try again.",
                  cmd->arg);
    session.restart_pos = 0L;
    return NULL;
  }

  retr_file = fopen(dir,"r");

  if(session.restart_pos) {
    if(fseek(retr_file,session.restart_pos,SEEK_SET) == -1) {
      int _errno = errno;
      fclose(retr_file);
      errno = _errno;
      retr_file = NULL;
    }

    session.restart_pos = 0L;
  }

  if(!retr_file || stat(dir,&sbuf) == -1)
    send_response(R_550,"%s: %s",cmd->arg,strerror(errno));
  else {
    xfer_setup_xfer(IO_WRITE,cmd->arg,main_server->tcp_swin,
                    _retr_read,_retr_done);
    session.xfer.abort = _retr_abort;
    session.xfer.path = pstrdup(session.xfer.p,dir);
    session.xfer.file_size = (unsigned long)sbuf.st_size;
    xfer_open_data(NULL,cmd->arg,session.xfer.file_size - session.restart_pos);
  }
  return NULL;
}

static char *cmd_abor(cmd_rec *cmd)
{
  if(cmd->argc != 1) {
    send_response(R_500,"'%s' not understood.",_get_full_cmd(cmd));
    return NULL;
  }

  if(session.flags & SF_XFER)
    xfer_abort(NULL,0);

  return R_225 " " C_ABOR " command successful.";
}

static char *cmd_type(cmd_rec *cmd)
{
  if(cmd->argc != 2) {
    send_response(R_500,"'%s' not understood.",_get_full_cmd(cmd));
    return NULL;
  }

  cmd->argv[1][0] = toupper(cmd->argv[1][0]);

  if(!strcmp(cmd->argv[1],"A"))
    session.flags |= SF_ASCII;
  else if(!strcmp(cmd->argv[1],"I"))
    session.flags &= (SF_ALL^SF_ASCII);
  else {
    send_response(R_500,"'%s' not understood.",_get_full_cmd(cmd));
    return NULL;
  }

  send_response(R_200,"Type set to %s.",cmd->argv[1]);
  return NULL;
}

static int _noxfer_timeout(CALLBACK_FRAME)
{
  if(session.flags & SF_XFER)
    return TRUE;			/* Transfer in progress, ignore timeout */

  send_response_async(R_421,
           "No Transfer Timeout (%d seconds): closing control connection.",
           TimeoutNoXfer);
  schedule(main_exit,0,(void*)LOG_NOTICE,"FTP no transfer time out, disconnected.",
           (void*)0,NULL);
  remove_timer(TIMER_IDLE,ANY_MODULE);
  remove_timer(TIMER_LOGIN,ANY_MODULE);
  return 0;
}

int xfer_init_child()
{
  /* Setup TimeoutNoXfer timer */
  if(TimeoutNoXfer)
    add_timer(TimeoutNoXfer,TIMER_NOXFER,&xfer_module,_noxfer_timeout);
  return 0;
}

void xfer_set_data_port(in_addr_t *bind_addr, int port)
{
  dconn_fd = inet_prebind_socket(permanent_pool,bind_addr,port);

  log_debug(DEBUG4,"prebound socket is %d.\n",dconn_fd);
}

cmdtable xfer_commands[] = {
  { C_TYPE,	G_NONE,		cmd_type,	TRUE,	FALSE,	NULL },
  { C_LIST,	G_DIRS,		cmd_list,	TRUE,	FALSE,	NULL },
  { C_NLST,	G_DIRS,		cmd_nlst,	TRUE,	FALSE,	NULL },
  { C_STAT,	G_DIRS,		cmd_stat,	TRUE,	FALSE,	NULL },
  { C_RETR,	G_READ,		cmd_retr,	TRUE,	FALSE,	NULL },
  { C_STOR,	G_WRITE,	cmd_stor,	TRUE,	FALSE,	NULL },
  { C_ABOR,	G_NONE,		cmd_abor,	TRUE,	TRUE,	NULL },
  { C_REST,	G_NONE,		cmd_rest,	TRUE,	FALSE,	NULL },
  { NULL,	G_NONE,		NULL,		FALSE,	FALSE,	NULL }
};

module xfer_module = {
  NULL,NULL,				/* Always NULL */
  0x10,					/* API Version */
  "xfer",				/* Module name */
  NULL,					/* No config */
  xfer_commands,
  NULL,xfer_init_child
};

