/*
 * lftp and utils
 *
 * Copyright (c) 1996-1997 by Alexander V. Lukyanov (lav@yars.free.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <termios.h>

#include "CmdExec.h"
#include "GetJob.h"
#include "PutJob.h"
#include "CatJob.h"
#include "LsJob.h"
#include "LsCache.h"
#include "mgetJob.h"
#include "mputJob.h"
#include "mkdirJob.h"
#include "rmJob.h"
#include "mrmJob.h"
#include "SysCmdJob.h"
#include "QuoteJob.h"
#include "MirrorJob.h"
#include "mvJob.h"
#include "pgetJob.h"

#include "misc.h"
#include "alias.h"
#include "netrc.h"
#include "url.h"
#include "GetPass.h"
#include "vars.h"
#include "SignalHook.h"
#include "ProtoList.h"
#include "FileFeeder.h"
#include "xalloca.h"

const struct CmdExec::cmd_rec CmdExec::cmd_table[]=
{
   {"!",       &do_shell,   "!<shell_command>",
	 "Launch shell or shell command\n"},
   {"(",       &do_subsh,   "(commands)",
	 "Group commands together to be executed as one command\n"
	 "You can launch such a group in background\n"},
   {"?",       &do_help,	   0,"help"},
   {"alias",   &do_alias,   "alias [<name> [<value>]]",
	 "Define or undefine alias <name>. If <value> omitted,\n"
	 "the alias is undefined, else is takes the value <value>.\n"
         "If no argument is given the current aliases are listed.\n"},
   {"anon",    &do_anon,    "anon",
	 "anon - login anonymously (by default)\n"},
   {"bye",     &do_exit,	   0,"exit"},
   {"cache",   &do_cache,   "cache [SUBCMD]",
	 "cache command controls local memory cache\n\n"
	 "The following subcommands are recognized:\n"
	 "  stat        - print cache status (default)\n"
	 "  on|off      - turn on/off caching\n"
	 "  flush       - flush cache\n"
	 "  size <lim>  - set memory limit, -1 means unlimited\n"
	 "  expire <Nx> - set cache expiration time to N seconds (x=s)\n"
	 "                minutes (x=m) hours (x=h) or days (x=d)\n"},
   {"cat",     &do_cat,     "cat <files>",
	 "cat <files> - output remote files to stdout\n"},
   {"cd",      &do_cd,      "cd <rdir>",
	 "Change current remote directory\n"},
   {"connect", &do_open,	   0,"open"},
   {"debug",   &do_debug,   "debug <level>|off"},
   {"exit",    &do_exit,    "exit [<code>]",
	 "exit - exit from lftp or move to background if jobs are active\n\n"
	 "If no jobs active, the code is passed to operating system as lftp\n"
	 "termination status. If omitted, exit code of last command is used.\n"},
   {"get",     &do_get,  "get [-c] [-e] <rfile> [-o <lfile>]",
	 "Retrieve remote file <rfile> and store it to local file <lfile>.\n"
	 " -o <lfile> specifies local file name (default - basename of rfile)\n"
	 " -c  continue, reget\n"
	 " -e  delete remote files after successful transfer\n"},
   {"help",    &do_help,    "help [<cmd>]",
	 "Print help for command <cmd>, or list of available commands\n"},
   {"jobs",    &do_jobs,    "jobs [-v]",
	 "List running jobs. -v means verbose, several -v can be specified.\n"},
   {"kill",    &do_kill,    "kill all|<job_no>",
	 "Delete specified job with <job_no> or all jobs\n"},
   {"lcd",     &do_lcd,     "lcd <ldir>",
	 "Change current local directory <ldir>\n"},
   {"lftp",    &do_open,	   0,"open"},
   {"login",   &do_user,	   0,"user"},
   {"ls",      &do_ls,	   "ls [<args>]",
	 "List remote files. You can redirect output of this command to file\n"
	 "or via pipe to external command.\n"
	 "By default, ls output is cached, to see new listing use `rels' or\n"
	 "`cache flush'.\n"},
   {"mget",    &do_mget,    "mget [-c] [-d] [-e] <files>",
	 "Gets selected files with expanded wildcards\n"
	 " -c  continue, reget\n"
	 " -d  create directories the same as in file names and get the\n"
	 "     files into them instead of current directory\n"
	 " -e  delete remote files after successful transfer\n"},
   {"mirror",  &do_mirror,  "mirror [OPTS] [remote [local]]",
	 "\nMirror specified remote directory to local directory\n\n"
	 " -c, --continue         continue a mirror job if possible\n"
	 " -e, --delete           delete files not present at remote site\n"
	 " -s, --allow-suid       set suid/sgid bits according to remote site\n"
	 " -n, --only-newer       download only newer files\n"
	 " -r, --no-recursion     don't go to subdirectories\n"
	 " -p, --no-perms         don't set file permissions\n"
	 " -i RX, --include RX    include matching files (only one allowed)\n"
	 " -x RX, --exclude RX    exclude matching files (only one allowed)\n"
	 " -t Nx, --time-prec Nx  set time precision to N seconds (x=s)\n"
	 "                        minutes (x=m) hours (x=h) or days (x=d)\n"},

   {"mkdir",   &do_mkdir,  "mkdir <dirs>",
	 "Make remote directories\n"},
   {"more",    &do_cat,    "more <files>",
	 "Same as `cat <files> | more'. if PAGER is set, it is used as filter\n"},
   {"mput",    &do_mput,   "mput [-c] [-d] <files>",
	 "Upload files with wildcard expansion. -c means reput, -d means to create\n"
	 "remote directories and use the same remote name as local one. By default\n"
	 "It uses basename of local name as remote one.\n"},
   {"mrm",     &do_mrm,	   "mrm <files>",
	 "Removes specified files with wildcard expansion\n"},
   {"mv",      &do_mv,	   "mv <file1> <file2>",
	 "Rename <file1> to <file2>\n"},
   {"nlist",   &do_ls,	   "nlist [<args>]",
	 "List remote file names\n"},
   {"open",    &do_open,   "open <host>",
	 "open - select an ftp server\n"
	 "Usage: open [-e cmd] [-u user[,pass]] [-p port] <host|url>\n"},
   {"pget",    &do_get,	   "pget [OPTS] <rfile> [-o <lfile>]",
	 "Gets the specified file using several connections. This can speed up transfer,\n"
	 "but loads the net heavily impacting other users. Use only if you really\n"
	 "have to transfer the file ASAP, or some other user may go mad :)\n"
	 "\nOptions:\n"
	 " -n <maxconn>           Set maximum number of connections (default 5)\n"},
   {"put",     &do_put,	   "put [-c] <lfile> [-o <rfile>]",
	 "Upload <lfile> with remote name <rfile>. If -o omitted, basename\n"
	 "of <lfile> is used as remote name. -c means reput, which requires\n"
	 "you to have permission to overwrite remote files in current directory.\n"},
   {"pwd",     &do_pwd,    "pwd",
	 "Print current remote directory\n"},
   {"quit",    &do_exit,   0,"exit"},
   {"quote",   &do_quote,  "quote <cmd>",
	 "Send the command uninterpreted. Use with caution - it can lead to\n"
	 "unknown remote state and thus will cause reconnect. You cannot\n"
	 "be sure that any change of remote state because of quoted command\n"
	 "is solid - it can be reset by reconnect at any time.\n"},
   {"reget",   &do_get,	   "reget <rfile> [-o <lfile>]",
	 "Same as `get -c'\n"},
   {"rels",    &do_ls,	   "rels [<args>]",
	 "Same as `ls', but don't look in cache\n"},
   {"renlist", &do_ls,	   "renlist [<args>]",
	 "Same as `nlist', but don't look in cache\n"},
   {"reput",   &do_put,	   "reput <lfile> [-o <rfile>]",
	 "Same as `put -c'\n"},
   {"rm",      &do_rm,	   "rm <files>",
	 "Remove remote files\n"},
   {"rmdir",   &do_rm,	   "rmdir <dirs>",
	 "Remove remote directories\n"},
   {"scache",  &do_scache, "scache [<session_no>]",
	 "List cached sessions or switch to specified session number\n"},
   {"set",     &do_set,    "set [<var> [<val>]]",
	 "Set variable to given value. If the value is omitted, unset the variable.\n"
         "If called with no variable, currently set variables are listed.\n"},
   {"site",    &do_site,   "site <site_cmd>",
	 "Execute site command <site_cmd> and output the result\n"
	 "You can redirect its output\n"},
   {"source",  &do_source, "source <file>",
	 "Execute commands recorded in file <file>\n"},
   {"user",    &do_user,   "user <user> [<pass>]",
	 "Use specified info for remote login\n"},
   {"version", &do_ver,	   "version",
	 "Shows lftp version\n"},
   {"wait",    &do_wait,   "wait <jobno>",
	 "Wait for specified job to terminate.\n"},
   {"zcat",    &do_cat,    "zcat <files>",
	 "Same as cat, but filter each file through zcat\n"},
   {"zmore",   &do_cat,    "zmore <files>",
	 "Same as more, but filter each file through zcat\n"},

   {NULL,NULL}
};

CMD(lcd)
{
   if(args->count()<2)
   {
      eprintf("Usage: %s local-dir\n",args->getarg(0));
      return 0;
   }
   char *cd_to=expand_home_relative(args->getarg(1));
   RestoreCWD();
   int res=chdir(cd_to);
   if(res==-1)
   {
      perror(cd_to);
      exit_code=1;
      return 0;
   }
   SaveCWD();
   if(interactive)
      eprintf("lcd ok, local cwd=%s\n",cwd);
   exit_code=0;
   return 0;
}

CMD(ls)
{
   int mode=Ftp::LONG_LIST;
   if(strstr(args->a0(),"nlist"))
      mode=Ftp::LIST;
   if(mode==Ftp::LONG_LIST && args->count()==1)
      args->Append(var_ls);
   LsJob *j=new LsJob(Clone(),output,args->Combine(1),mode);
   output=0;
   if(!strncmp(args->a0(),"re",2))
      j->NoCache();
   return j;
}

CMD(cat)
{
   if(args->count()==1)
   {
      eprintf("Usage: %s files...\n",args->a0());
      return 0;
   }
   Job *j=new CatJob(Clone(),output,args);
   output=0;
   args=0;
   return j;
}

CMD(get)
{
   int opt;
   bool cont=false;
   char *opts="+ce";
   char *op=args->a0();
   ArgV	 *get_args=new ArgV(op);
   int n_conn=0;
   bool del=false;

   args->rewind();
   if(!strncmp(op,"re",2))
   {
      cont=true;
      opts="+e";
   }
   if(!strcmp(op,"pget"))
   {
      opts="+n:";
      n_conn=-1;
   }
   while((opt=args->getopt(opts))!=EOF)
   {
      switch(opt)
      {
      case('c'):
	 cont=true;
	 break;
      case('n'):
	 if(!isdigit(optarg[0]))
	 {
	    eprintf("%s: -n: Number expected. ",op);
	    goto err;
	 }
	 n_conn=atoi(optarg);
	 break;
      case('e'):
	 del=true;
	 break;
      case('?'):
      err:
	 eprintf("Try `help %s' for more information\n",op);
	 return 0;
      }
   }
   args->back();
   char *a=args->getnext();
   char *a1;
   if(a==0)
   {
      eprintf("File name missed. ");
      goto err;
   }
   while(a)
   {
      get_args->Append(a);
      a1=basename_ptr(a);
      a=args->getnext();
      if(a && !strcmp(a,"-o"))
      {
	 a=args->getnext();
	 if(a)
	 {
	    a=expand_home_relative(a);
	    struct stat st;
	    int res=stat(a,&st);
	    if(res!=-1 && S_ISDIR(st.st_mode))
	    {
	       char *comb=(char*)alloca(strlen(a)+strlen(a1)+2);
	       sprintf(comb,"%s/%s",a,a1);
	       get_args->Append(comb);
	    }
	    else
	    {
	       get_args->Append(a);
	    }
	 }
	 else
	    get_args->Append(a1);
	 a=args->getnext();
      }
      else
	 get_args->Append(a1);
   }

   if(bootstrap)
      want_terminate=true;

   if(n_conn==0)
   {
      GetJob *j=new GetJob(Clone(),get_args,cont);
      if(del)
	 j->DeleteFiles();
      return j;
   }
   else
   {
      pgetJob *j=new pgetJob(Clone(),get_args);
      if(n_conn!=-1)
	 j->SetMaxConn(n_conn);
      return j;
   }
}

CMD(put)
{
   int opt;
   bool cont=false;
   char *opts="+c";
   char *op=args->a0();
   ArgV	 *get_args=new ArgV(op);

   args->rewind();
   if(!strncmp(op,"re",2))
   {
      cont=true;
      opts="+";
   }
   while((opt=args->getopt(opts))!=EOF)
   {
      switch(opt)
      {
      case('c'):
	 cont=true;
	 break;
      case('?'):
      err:
	 eprintf("Try `help %s' for more information\n",op);
	 return 0;
      }
   }
   args->back();
   char *a=args->getnext();
   char *a1;
   if(a==0)
   {
      eprintf("File name missed. ");
      goto err;
   }
   while(a)
   {
      a=expand_home_relative(a);
      get_args->Append(a);
      a1=basename_ptr(a);
      a=args->getnext();
      if(a && !strcmp(a,"-o"))
      {
	 a=args->getnext();
	 if(a)
	    get_args->Append(a);
	 else
	    get_args->Append(a1);
	 a=args->getnext();
      }
      else
	 get_args->Append(a1);
   }

   PutJob *j=new PutJob(Clone(),get_args,cont);
   return j;
}

CMD(mget)
{
   Job *j=new mgetJob(Clone(),args);
   args=0;
   return j;
}

CMD(mput)
{
   Job *j=new mputJob(Clone(),args);
   args=0;
   return j;
}

CMD(shell)
{
   Job *j;
   if(args->count()==1)
      j=new SysCmdJob(0);
   else
      j=new SysCmdJob(args->getarg(1));
   return j;
}

CMD(mrm)
{
   Job *j=new mrmJob(Clone(),args);
   args=0;
   return j;
}
CMD(rm)
{
   int opt;
   while((opt=args->getopt("+"))!=EOF)
   {
      switch(opt)
      {
      case('?'):
      print_usage:
	 fprintf(stderr,"Usage: %s files...\n",args->a0());
	 return 0;
      }
   }
   args->back();
   char *curr=args->getnext();
   if(curr==0)
      goto print_usage;

   rmJob *j=(strcmp(args->a0(),"rmdir")
	     ?new rmJob(Clone(),new ArgV(args->a0()))
	     :new rmdirJob(Clone(),new ArgV(args->a0())));

   while(curr)
   {
      j->AddFile(curr);
      curr=args->getnext();
   }

   return j;
}
CMD(mkdir)
{
   Job *j=new mkdirJob(Clone(),args);
   args=0;
   return j;
}

CMD(source)
{
   if(args->count()<2)
   {
      eprintf("Usage: source <file>\n");
      return 0;
   }
   SetCmdFeeder(new FileFeeder(new FileStream(args->getarg(1),O_RDONLY)));
   return 0;
}

CMD(jobs)
{
   int opt;
   args->rewind();
   int v=1;
   while((opt=args->getopt("+v"))!=EOF)
   {
      switch(opt)
      {
      case('v'):
	 v++;
	 break;
      case('?'):
	 eprintf("Usage: jobs [-v] [-v] ...\n");
	 return 0;
      }
   }
   Job::ListJobs(v);
   exit_code=0;
   return 0;
}

CMD(cd)
{
   if(args->count()!=2)
   {
      eprintf("Usage: cd remote-dir\n");
      return 0;
   }

   char c;
   if(sscanf(args->getarg(1),"%*[a-z]://%*[^/]%c",&c)==1)
      return do_open();

   session->Chdir(args->getarg(1));
   builtin=BUILTIN_CD;
   return this;
}

CMD(pwd)
{
   const char *cwd=session->GetCwd();
   printf("%s\n",cwd[0]?cwd:"~");
   exit_code=0;
   return 0;
}

void  move_to_background()
{
   fflush(stdout);
   fflush(stderr);
   pid_t pid=fork();
   switch(pid)
   {
   case(0): // child
   {
      SignalHook::Ignore(SIGINT);
      SignalHook::Ignore(SIGQUIT);
      SignalHook::Ignore(SIGHUP);
      SignalHook::Ignore(SIGTSTP);

      char *home=getenv("HOME");
      if(!home) home=".";
      char *log=(char*)alloca(strlen(home)+1+9+1);
      sprintf(log,"%s/.lftp_log",home);
      int fd=open(log,O_WRONLY|O_APPEND|O_CREAT,0600);
      if(fd>=0)
      {
	 dup2(fd,1);
	 dup2(fd,2);
	 if(fd!=1 && fd!=2)
	    close(fd);
      }
      pid=getpid();
      time_t t=time(0);
      printf("[%lu] Started.  %s",(unsigned long)pid,ctime(&t));
      for(;;)
      {
	 SMTask::Schedule();
	 if(Job::NumberOfJobs()==0)
	    break;
	 SMTask::Block();
      }
      t=time(0);
      printf("[%lu] Finished. %s",(unsigned long)pid,ctime(&t));
      exit(0);
   }
   default: // parent
      printf("[%lu] Moving to background to complete transfers...\n",
	       (unsigned long)pid);
      fflush(stdout);
      _exit(0);
   case(-1):
      perror("fork()");
   }
}

CMD(exit)
{
   if(args->count()>=2)
   {
      if(sscanf(args->getarg(1),"%i",&prev_exit_code)!=1)
      {
	 eprintf("Usage: %s [<exit_code>]\n",args->a0());
	 return 0;
      }
   }
   while(!Done())
      RemoveFeeder();
   if(Job::NumberOfJobs()>0)
   {
      interactive=false;
      running=false; // allow re-entering
      move_to_background();
   }
   exit(prev_exit_code);
}

CmdExec *CmdExec::debug_shell=0;
void CmdExec::debug_callback(char *msg)
{
   if(!debug_shell)
      return;
   if(debug_shell->status_line==0)
      return;
   if(isatty(1) && tcgetpgrp(1)!=getpgrp())
      return;
   debug_shell->status_line->WriteLine(msg);
   if(debug_shell->waiting && debug_shell->interactive)
      debug_shell->ShowRunStatus(debug_shell->status_line);
}

CMD(debug)
{
   FILE	 *new_dfile=0;//stderr;
   void	 (*new_cb)(char *)=&debug_callback;
   int	 new_dlevel=9;

   debug_shell=this;

   if(args->count()>1)
   {
      if(!strcmp(args->getarg(1),"off"))
      {
	 new_cb=0;
	 new_dfile=0;
	 new_dlevel=0;
      }
      else
      {
	 new_dlevel=atoi(args->getarg(1));
      }
   }
   if(interactive)
   {
      if(new_cb)
	 printf("debug level %d\n",new_dlevel);
      else
	 printf("debug off\n");
   }
   session->SetDebug(new_dfile,new_dlevel,new_cb);
   exit_code=0;
   return 0;
}

CMD(user)
{
   char	 *pass;
   if(args->count()<2 || args->count()>3)
   {
      eprintf("Usage: %s userid [pass]\n",args->getarg(0));
      return 0;
   }
   if(args->count()==2)
      pass=GetPass("Password: ");
   else
      pass=args->getarg(2);
   if(pass)
      session->Login(args->getarg(1),pass);
   exit_code=0;
   return 0;
}
CMD(anon)
{
   session->AnonymousLogin();
   exit_code=0;
   return 0;
}

void unquote(char *buf,char *str)
{
   while(*str)
   {
      if(*str=='"' || *str=='\\')
	 *buf++='\\';
      *buf++=*str++;
   }
   *buf=0;
}

CMD(open)
{
   bool	 debug=false;
   char	 *port=NULL;
   char	 *host=NULL;
   char  *path=NULL;
   char	 *user=NULL;
   char	 *pass=NULL;
   int	 c;
   NetRC::Entry *nrc=0;
   char	 *cmd_to_exec=0;
   char  *file_to_exec=0;

   args->rewind();
   while((c=args->getopt("u:p:e:df:"))!=EOF)
   {
      switch(c)
      {
      case('p'):
	 port=optarg;
	 break;
      case('u'):
         user=optarg;
         pass=strchr(optarg,',');
	 if(pass==NULL)
	    pass=strchr(optarg,' ');
	 if(pass==NULL)
   	    break;
	 *pass=0;
	 pass++;
         break;
      case('d'):
	 debug=true;
	 break;
      case('e'):
	 cmd_to_exec=optarg;
	 break;
      case('f'):
	 file_to_exec=optarg;
	 break;
      case('?'):
	 eprintf("Usage: %s [-e cmd] [-p port] [-u user[,pass]] <host|url>\n",
	    args->a0());
	 return 0;
      }
   }

   if(optind<args->count())
      host=args->getarg(optind++);

   ParsedURL *url=0;

   if(host)
   {
      url=new ParsedURL(host);

      const ParsedURL &uc=*url;
      if(uc.host)
      {
	 FileAccess *new_session=0;
	 if(uc.proto && strcmp(uc.proto,session->GetProto()))
	 {
	    new_session=Protocol::NewSession(uc.proto);
	    if(!new_session)
	    {
	       eprintf("%s: %s - not supported protocol\n",
			args->getarg(0),uc.proto);
	       return 0;
	    }
	 }
	 else
	 {
	    new_session=session->Clone();
	    new_session->AnonymousLogin();
	 }
	 Reuse(session);
	 session=new_session;

	 if(uc.user && !user)
	    user=uc.user;
	 if(uc.pass && !pass)
	    pass=uc.pass;
	 host=uc.host;
	 if(uc.port!=0 && port==0)
	    port=uc.port;
	 if(uc.path && !path)
	    path=uc.path;
      }

      if(!strcmp(session->GetProto(),"ftp"))
      {
	 nrc=NetRC::LookupHost(host);
	 if(nrc)
	 {
	    if(nrc->user && !user)
	    {
	       user=nrc->user;
	       if(nrc->pass)
		  pass=nrc->pass;
	    }
	 }
      }
   }
   if(user)
   {
      if(!pass)
	 pass=GetPass("Password: ");
      if(!pass)
	 eprintf("%s: GetPass() failed: %s -- assume anonymous login\n",
	    args->getarg(0),strerror(errno));
      else
	 session->Login(user,pass);
   }
   if(host)
   {
      int port_num=0;
      if(port)
      {
	 if(!isdigit(port[0]))
	 {
	    struct servent *serv=getservbyname(port,"tcp");
	    if(serv==NULL)
	    {
	       eprintf("%s: %s - no such tcp service\n",args->a0(),port);
	       return 0;
	    }
	    port_num=serv->s_port;
	 }
	 else
	    port_num=atoi(port);
      }
      session->Connect(host,port_num);
      builtin=BUILTIN_OPEN;
   }
   if(nrc)
      delete nrc;

   if(cmd_to_exec)
      PrependCmd(cmd_to_exec);

   if(file_to_exec)
   {
      char *s=(char *)alloca(20+2*strlen(file_to_exec));
      strcpy(s,"source \"");
      unquote(s+strlen(s),file_to_exec);
      strcat(s,"\"\n");
      PrependCmd(s);
      want_terminate=true;
   }

   if(path)
   {
      char *s=(char*)alloca(strlen(path)*4+40);
      strcpy(s,"&& cd \"");
      unquote(s+strlen(s),path);
      strcat(s,"\"\n");
#if 0
      char *slash=strrchr(path,'/');
      if(slash && slash[1])
      {
	 *slash=0;
	 strcat(s,"|| (cd \"");
	 unquote(s+strlen(s),path);
	 strcat(s,"\" && get -- \"");
	 unquote(s+strlen(s),slash+1);
	 strcat(s,"\")\n");
      }
#endif
      PrependCmd(s);
   }

   if(debug)
      PrependCmd("debug\n");

   if(url)
      delete url;

   if(host)
      return this;

   exit_code=0;
   return 0;
}

CMD(kill)
{
   if(args->count()<2)
   {
      eprintf("Usage: %s <jobno> ... | all\n",args->getarg(0));
      return 0;
   }
   if(!strcmp(args->getarg(1),"all"))
   {
      Job::KillAll();
      exit_code=0;
      return 0;
   }
   args->rewind();
   for(;;)
   {
      char *arg=args->getnext();
      if(arg==0)
	 break;
      if(!isdigit(arg[0]))
      {
	 eprintf("%s: `%s' - not a number\n",args->getarg(0),arg);
      	 continue;
      }
      int n=atoi(arg);
      if(Job::Running(n))
	 Job::Kill(n);
      else
	 eprintf("%s: %d: No such job\n",args->getarg(0),n);
   }
   exit_code=0;
   return 0;
}

CMD(set)
{
   if(args->count()<2)
   {
      if(args->count()!=1)
      {
	 eprintf("Usage: %s <variable> [<value>]\n",args->getarg(0));
	 return 0;
      }
      ResMgr::Print(stdout);
      exit_code=0;
      return 0;
   }

   char *a=args->getarg(1);
   char *sl=strchr(a,'/');
   char *closure=0;
   if(sl)
   {
      *sl=0;
      closure=sl+1;
   }

   char *val=(args->count()<=2?0:args->Combine(2));
   const char *msg=
      ResMgr::Set(a,closure,val);
   xfree(val);

   if(msg)
   {
      eprintf("%s: %s. Use `set' to look at all variables.\n",a,msg);
      return 0;
   }
   exit_code=0;
   return 0;
}

CMD(alias)
{
   if(args->count()<2)
   {
      Alias::List();
   }
   else if(args->count()==2)
   {
      Alias::Del(args->getarg(1));
   }
   else
   {
      char *val=args->Combine(2);
      Alias::Add(args->getarg(1),val);
      free(val);
   }
   exit_code=0;
   return 0;
}

CMD(wait)
{
   if(args->count()!=2)
   {
      eprintf("Usage: wait <jobno>\n");
      return 0;
   }
   args->rewind();
   char *jn=args->getnext();
   if(!isdigit(jn[0]))
   {
      eprintf("wait: <jobno> must be a number\n");
      return 0;
   }
   Job *j=FindJob(atoi(jn));
   if(j==0)
   {
      eprintf("wait: no such job %s\n",jn);
      return 0;
   }
   if(j->parent && j->parent->waiting==j)
   {
      eprintf("wait: some other job waits for job %s\n",jn);
      return 0;
   }
   j->parent=0;
   return j;
}

CMD(quote)
{
   if(args->count()<=1)
   {
      eprintf("Usage: %s <raw_cmd>\n",args->a0());
      return 0;
   }
   Job *j=new QuoteJob(Clone(),args->a0(),args->Combine(1),
      output?output:new FDStream(1,"<stdout>"));
   output=0;
   return j;
}

CMD(site)
{
   if(args->count()<=1)
   {
      eprintf("Usage: site <site_cmd>\n");
      return 0;
   }
   char *cmd=args->Combine(1);
   cmd=(char*)xrealloc(cmd,strlen(cmd)+6);
   memmove(cmd+5,cmd,strlen(cmd)+1);
   memcpy(cmd,"SITE ",5);
   Job *j=new QuoteJob(Clone(),args->a0(),cmd,
      output?output:new FDStream(1,"<stdout>"));
   output=0;
   return j;
}

CMD(subsh)
{
   CmdExec *e=new CmdExec(Clone());

   char *c=args->getarg(1);
   e->FeedCmd(c);
   e->FeedCmd("\n");
   e->cmdline=(char*)xmalloc(strlen(c)+3);
   sprintf(e->cmdline,"(%s)",c);
   return e;
}

time_t decode_delay(const char *s)
{
   long prec;
   char ch;
   int n=sscanf(s,"%lu%c",&prec,&ch);
   if(n<1)
      return -1;
   if(n==1)
      ch='s';
   else if(ch=='m')
      prec*=60;
   else if(ch=='h')
      prec*=60*60;
   else if(ch=='d')
      prec*=24*60*60;
   else
      return -1;
   return prec;
}

CMD(mirror)
{
   static struct option mirror_opts[]=
   {
      {"delete",no_argument,0,'e'},
      {"allow-suid",no_argument,0,'s'},
      {"include",required_argument,0,'i'},
      {"exclude",required_argument,0,'x'},
      {"time-prec",required_argument,0,'t'},
      {"only-newer",no_argument,0,'n'},
      {"no-recursion",no_argument,0,'r'},
      {"no-perms",no_argument,0,'p'},
      {"continue",no_argument,0,'c'},
      {0}
   };

   char cwd[1024];
   int opt;
   int	 flags=0;
   char *include=0;
   char *exclude=0;
   time_t prec=0;

   args->rewind();
   while((opt=args->getopt_long("esi:x:t:nrpc",mirror_opts,0))!=EOF)
   {
      switch(opt)
      {
      case('e'):
	 flags|=MirrorJob::DELETE;
	 break;
      case('s'):
	 flags|=MirrorJob::ALLOW_SUID;
	 break;
      case('r'):
	 flags|=MirrorJob::NO_RECURSION;
	 break;
      case('n'):
	 flags|=MirrorJob::ONLY_NEWER;
	 break;
      case('p'):
	 flags|=MirrorJob::NO_PERMS;
	 break;
      case('c'):
	 flags|=MirrorJob::CONTINUE;
	 break;
      case('t'):
	 prec=decode_delay(optarg);
	 if(prec==(time_t)-1)
	 {
	    eprintf(
	       "mirror: --time-prec - invalid precision\n"
	       "mirror: Use `help mirror' to look at the syntax\n");
	    return 0;
	 }
	 break;
      case('x'):
	 exclude=optarg;
	 break;
      case('i'):
	 include=optarg;
	 break;
      case('?'):
	 return 0;
      }
   }

   args->back();
   const char *rcwd=args->getnext();
   if(!rcwd)
      rcwd=session->GetCwd();
   char *arg=args->getnext();
   if(arg)
   {
      strcpy(cwd,arg);
      create_directories(cwd);
   }
   else
   {
      if(getcwd(cwd,sizeof(cwd))==0)
      {
	 perror("getcwd()");
	 return 0;
      }
   }
   MirrorJob *j=new MirrorJob(Clone(),cwd,rcwd);
   j->SetFlags(flags,1);
   if(include)
   {
      if(j->SetInclude(include)==-1)
      {
	 delete j;
	 return 0;
      }
   }
   if(exclude)
   {
      if(j->SetExclude(exclude)==-1)
      {
	 delete j;
	 return 0;
      }
   }
   j->SetPrec((time_t)prec);
   return j;
}

CMD(mv)
{
   if(args->count()!=3)
   {
      eprintf("Usage: mv <file1> <file2>\n");
      return 0;
   }
   Job *j=new mvJob(Clone(),args->getarg(1),args->getarg(2));
   return j;
}

static char *const cache_subcmd[]={
   "status","flush","on","off","size","expire",
   NULL
};

CMD(cache)  // cache control
{
   args->rewind();
   const char *op=args->getnext();

   if(!op)
      op="status";
   else if(!find_var_name(op,cache_subcmd,&op))
   {
      eprintf("Invalid command. Try `help %s' for more information.\n",args->a0());
      return 0;
   }
   if(!op)
   {
      eprintf("Ambiguous command. Try `help %s' for more information.\n",args->a0());
      return 0;
   }

   exit_code=0;
   if(!op || !strcmp(op,"status"))
      LsCache::List();
   else if(!strcmp(op,"flush"))
      LsCache::Flush();
   else if(!strcmp(op,"on"))
      LsCache::On();
   else if(!strcmp(op,"off"))
      LsCache::Off();
   else if(!strcmp(op,"size"))
   {
      op=args->getnext();
      if(!op)
      {
	 eprintf("%s: Operand missed for size\n",args->a0());
	 exit_code=1;
	 return 0;
      }
      long lim=-1;
      if(strcmp(op,"unlim") && sscanf(op,"%ld",&lim)!=1)
      {
	 eprintf("%s: Invalid number for size\n",args->a0());
	 exit_code=1;
	 return 0;
      }
      LsCache::SetSizeLimit(lim);
   }
   else if(!strcmp(op,"expire"))
   {
      op=args->getnext();
      if(!op)
      {
	 eprintf("%s: Operand missed for `expire'\n",args->a0());
	 exit_code=1;
	 return 0;
      }
      time_t exp=decode_delay(op);
      if(exp==-1)
      {
	 eprintf("%s: Invalid expire period (use Ns - in sec, Nm - in min, Nh - in hours)\n",args->a0());
	 exit_code=1;
	 return 0;
      }
      LsCache::SetExpire(exp);
   }
   return 0;
}

CMD(scache)
{
   if(args->count()==1)
   {
      SessionPool::Print(stdout);
      exit_code=0;
   }
   else
   {
      char *a=args->getarg(1);
      if(!isdigit(a[0]))
      {
	 eprintf("%s: %s - should be a number\n",args->a0(),a);
	 return 0;
      }
      FileAccess *new_session=SessionPool::GetSession(atoi(a));
      if(new_session==0)
      {
	 eprintf("%s: %s - no such cached session. Use `scache' to look at session list.\n",args->a0(),a);
	 return 0;
      }
      Reuse(session);
      session=new_session;
   }
   return 0;
}

void CmdExec::print_cmd_help(const char *cmd)
{
   const cmd_rec *c;
   int part=find_cmd(cmd,&c);

   if(part==1)
   {
      if(c->long_desc==0)
      {
	 if(c->short_desc)
	    puts(c->short_desc);
	 else
	    printf("Sorry, no help for %s\n",cmd);
	 return;
      }
      if(!strchr(c->long_desc,' '))
      {
	 printf("%s is a built-in alias for %s\n",cmd,c->long_desc);
	 print_cmd_help(c->long_desc);
	 return;
      }
      puts(c->short_desc);
      printf("%s",c->long_desc);
      return;
   }
   const char *a=Alias::Find(cmd);
   if(a)
   {
      printf("%s is an alias to `%s'\n",cmd,a);
      return;
   }
   printf("%s command `%s'. Use `help' to see available commands.\n",
      part==0?"No such":"Ambiguous",cmd);
}

CMD(help)
{
   if(args->count()>1)
   {
      args->rewind();
      for(;;)
      {
	 char *cmd=args->getnext();
	 if(cmd==0)
	    break;
	 print_cmd_help(cmd);
      }
      return 0;
   }

   int i=0;
   const char *c1;
   while(cmd_table[i].name)
   {
      while(cmd_table[i].name && !cmd_table[i].short_desc)
	 i++;
      if(cmd_table[i].name)
      {
	 c1=cmd_table[i].short_desc;
	 i++;
	 while(cmd_table[i].name && !cmd_table[i].short_desc)
	    i++;
	 if(cmd_table[i].name)
	 {
	    printf("\t%-35s %s\n",c1,cmd_table[i].short_desc);
	    i++;
	 }
	 else
	    printf("\t%s\n",c1);
      }
   }
   return 0;
}

CMD(ver)
{
   printf("%s",
      "Lftp | Version " VERSION " | Copyright (c) 1996-97 Alexander V. Lukyanov\n"
      "This is free software with ABSOLUTELY NO WARRANTY. See COPYING for details.\n");
   return 0;
}
