/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1995, William Cheng.
 *
 * Permission limited to the use, copy, display, distribute without
 * charging for a fee, and produce derivative works of "tgif" and
 * its documentation for not-for-profit purpose is hereby granted by
 * the Author, provided that the above copyright notice appears in
 * all copies made of "tgif" and that both the copyright notice
 * and this permission notice appear in supporting documentation,
 * and that the name of the Author not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.  All other rights (including, but not limited to, the
 * right to sell "tgif", the right to sell derivative works of
 * "tgif", and the right to distribute "tgif" for a fee) are
 * reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /u/multimedia/william/X11/TGIF2/RCS/dialog.c,v 2.64 1995/05/15 01:21:26 william Exp $";
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include "const.h"
#include "types.h"

#include "auxtext.e"
#include "box.e"
#include "button.e"
#include "copypaste.e"
#include "cursor.e"
#ifndef _NO_EXTERN
#include "dialog.e"
#endif
#include "file.e"
#include "font.e"
#include "mainloop.e"
#include "mainmenu.e"
#include "msg.e"
#include "raster.e"
#include "setup.e"
#include "text.e"
#include "util.e"

char	gszMsgBox[1024];

static int	curX, curY;
static Window	dialogWindow;

unsigned int CornerLoop (OrigX, OrigY)
   int	* OrigX, * OrigY;
{
   XEvent	input;

   XGrabPointer (mainDisplay, rootWindow, False, ButtonPressMask,
         GrabModeAsync, GrabModeAsync, None, cornerCursor, CurrentTime);

   for (;;)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == ButtonPress)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         XSync (mainDisplay, False);
         *OrigX = input.xbutton.x;
         *OrigY = input.xbutton.y;
         return (input.xbutton.button);
      }
   }
}

unsigned int MagLoop (OrigX, OrigY)
   int	* OrigX, * OrigY;
{
   XEvent	input;

   XGrabPointer (mainDisplay, drawWindow, False, ButtonPressMask,
         GrabModeAsync, GrabModeAsync, None, magCursor, CurrentTime);

   for (;;)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == ButtonPress)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         XSync (mainDisplay, False);
         *OrigX = input.xbutton.x;
         *OrigY = input.xbutton.y;
         return (input.xbutton.button);
      }
   }
}

unsigned int PickAPoint (OrigX, OrigY)
   int	* OrigX, * OrigY;
{
   XEvent	input;

   XGrabPointer (mainDisplay, drawWindow, False, ButtonPressMask,
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);

   for (;;)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == ButtonPress)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         XSync (mainDisplay, False);
         *OrigX = input.xbutton.x;
         *OrigY = input.xbutton.y;
         return (input.xbutton.button);
      }
   }
}

#include "xbm/info.xbm"

#define ICON_W (info_width)
#define ICON_H (info_height)

#define X_MARGIN 20
#define Y_MARGIN 8
#define X_GAP 20
#define Y_GAP 20

#define SPACING 0

#define BTN_X_MARGIN 8
#define BTN_Y_MARGIN 2
#define BTN_XY_EXTRA 2
#define BTN_MIN_X_GAP 8

#define MAX_KEYSYMS 10

#define LF_BUTTON  0x01
#define MD_BUTTON  0x02
#define RT_BUTTON  0x03
#define XT_BUTTON  0x04

#define DEF_BUTTON 0x10

#define MAX_BUTTONS 3
#define MAX_BTN_STR_LEN 10

typedef struct BtnInfoRec {
   char		* str;
   struct BBRec	bbox;
   int		id, highlight;
   KeySym	key_sym[MAX_KEYSYMS];
} * BtnInfoRecPtr;

typedef struct MBRec {
   Window main_win, root_win, icon_win, msg_win, btn_win;

   int main_win_x, main_win_y, main_win_w, main_win_h;
   int icon_win_w, icon_win_h, msg_win_w, msg_win_h;
   int btn_win_w, btn_win_h, max_msg_win_w;
   int max_msg_str_len;

   char * msg_copy;

   Pixmap cur_bitmap;

   struct BtnInfoRec btn_info[MAX_BUTTONS+1];
} * MBRecPtr;

static struct MBRec	mbInfo;

static int	numButtons=MAX_BUTTONS;
static char	extraBtnChar='q';
static char	*btnStr[] = {
      "INVALID", "OK", "CANCEL", "YES", "NO", "Extra", NULL
};

static
void SetupMBButton(MBInfoPtr, BtnDesc, BtnCh, BtnID)
   struct MBRec	* MBInfoPtr;
   int		BtnDesc, BtnID;
   char		BtnCh;
{
   int btn_index=(BtnDesc & 0x0f)-1;
   int i=0;

   if (BtnID == MB_ID_FAILED)
      MBInfoPtr->btn_info[btn_index].str = NULL;
   else
      MBInfoPtr->btn_info[btn_index].str = btnStr[BtnID];

   MBInfoPtr->btn_info[btn_index].id = BtnID;
   switch (BtnCh)
   {
      case 'o':
         MBInfoPtr->btn_info[btn_index].key_sym[i++] = XK_o;
         MBInfoPtr->btn_info[btn_index].key_sym[i++] = XK_O;
         break;
      case 'y':
         MBInfoPtr->btn_info[btn_index].key_sym[i++] = XK_y;
         MBInfoPtr->btn_info[btn_index].key_sym[i++] = XK_Y;
         break;
      case 'n':
         MBInfoPtr->btn_info[btn_index].key_sym[i++] = XK_n;
         MBInfoPtr->btn_info[btn_index].key_sym[i++] = XK_N;
         break;
      case 'c':
         MBInfoPtr->btn_info[btn_index].key_sym[i++] = XK_c;
         MBInfoPtr->btn_info[btn_index].key_sym[i++] = XK_C;
         break;
   }
   if (BtnDesc & DEF_BUTTON)
   {
      MBInfoPtr->btn_info[btn_index].key_sym[i++] = XK_Return;
      MBInfoPtr->btn_info[btn_index].key_sym[i++] = XK_Linefeed;
      MBInfoPtr->btn_info[btn_index].highlight = TRUE;
   }
   else
      MBInfoPtr->btn_info[btn_index].highlight = FALSE;
   MBInfoPtr->btn_info[btn_index].key_sym[i] = (KeySym)0;
}

static
void CalcGeometry(MBInfoPtr, Message)
   struct MBRec	* MBInfoPtr;
   char		* Message;
{
   int len=strlen(Message), max_len=0, max_h=0, left, inc, i, a_btn_w, a_btn_h;
   char *dest_ptr, *c_ptr;

   MBInfoPtr->msg_copy = (char*)calloc((len<<1)+1, sizeof(char));
   if (MBInfoPtr->msg_copy == NULL) fprintf (stderr, "Can not calloc().\n");

   dest_ptr = MBInfoPtr->msg_copy;
   *dest_ptr = '\0';
   c_ptr = Message;
   while (c_ptr != NULL)
   {
      char	* lf_ptr=strchr(c_ptr, '\n');
      int	full_str_len;

      if (lf_ptr != NULL) *lf_ptr = '\0';
      full_str_len = strlen (c_ptr);
      if (full_str_len > max_len)
      {
         if (full_str_len > MBInfoPtr->max_msg_str_len)
         {  /* line too long for the message box */
            char	* line=c_ptr;

            max_len = MBInfoPtr->max_msg_str_len;
            while (line != NULL && *line != '\0')
            {
               int	line_len;

               while (*line == ' ') line++;
               line_len = strlen(line);
               if (line_len > MBInfoPtr->max_msg_str_len)
               {
                  int	index=MBInfoPtr->max_msg_str_len;

                  if (line[index] == ' ')
                  {
                     int	saved_index=index--;

                     while (index >=0 && line[index] == ' ') index--;
                     line[++index] = '\0';
                     sprintf(dest_ptr, "%s\n", line);
                     dest_ptr = &dest_ptr[strlen(dest_ptr)];
                     max_h += defaultFontHeight+SPACING;
                     line = &line[saved_index+1];
                  }
                  else
                  {
                     int	saved_index=index;

                     while (index >=0 && line[index] != ' ') index--;
                     if (index < 0)
                     {  /* very long word */
                        char	saved_ch=line[saved_index];

                        line[saved_index] = '\0';
                        sprintf(dest_ptr, "%s\n", line);
                        dest_ptr = &dest_ptr[strlen(dest_ptr)];
                        max_h += defaultFontHeight+SPACING;
                        line[saved_index] = saved_ch;
                        line = &line[saved_index];
                     }
                     else
                     {  /* find a good space */
                        saved_index = index--;
                        while (index >=0 && line[index] == ' ') index--;
                        line[++index] = '\0';
                        sprintf(dest_ptr, "%s\n", line);
                        dest_ptr = &dest_ptr[strlen(dest_ptr)];
                        max_h += defaultFontHeight+SPACING;
                        line = &line[saved_index+1];
                     }
                  }
               }
               else if (line_len > 0)
               {
                  sprintf(dest_ptr, "%s\n", line);
                  dest_ptr = &dest_ptr[strlen(dest_ptr)];
                  max_h += defaultFontHeight+SPACING;
                  break;
               }
            }
         }
         else
         {
            max_len = full_str_len;
            sprintf(dest_ptr, "%s\n", c_ptr);
            dest_ptr = &dest_ptr[strlen(dest_ptr)];
            max_h += defaultFontHeight+SPACING;
         }
      }
      else
      {
         sprintf(dest_ptr, "%s\n", c_ptr);
         dest_ptr = &dest_ptr[strlen(dest_ptr)];
         max_h += defaultFontHeight+SPACING;
      }
      max_h -= SPACING;
      if (lf_ptr != NULL)
      {
         *lf_ptr = '\n';
         c_ptr = &lf_ptr[1];
      }
      else
         break;
   }
   MBInfoPtr->msg_win_w = max_len*defaultFontWidth;
   MBInfoPtr->msg_win_h = max_h;
   MBInfoPtr->icon_win_w = ICON_W;
   MBInfoPtr->icon_win_h = ICON_H;
   if (MBInfoPtr->msg_win_h > MBInfoPtr->icon_win_h)
      MBInfoPtr->icon_win_h = MBInfoPtr->msg_win_h;
   else
      MBInfoPtr->msg_win_h = MBInfoPtr->icon_win_h;

   a_btn_w = MAX_BTN_STR_LEN*defaultFontWidth + (BTN_XY_EXTRA<<1);
   a_btn_h = defaultFontHeight + (BTN_Y_MARGIN<<1) + (BTN_XY_EXTRA<<1);

   MBInfoPtr->btn_win_w = numButtons*a_btn_w+BTN_MIN_X_GAP*(numButtons-1)+2;
   MBInfoPtr->btn_win_h = a_btn_h+2;

   if (MBInfoPtr->btn_win_w >
         MBInfoPtr->msg_win_w+MBInfoPtr->icon_win_w+X_GAP)
      MBInfoPtr->msg_win_w =
            MBInfoPtr->btn_win_w-MBInfoPtr->icon_win_w-X_GAP;
   else
      MBInfoPtr->btn_win_w =
            MBInfoPtr->msg_win_w+MBInfoPtr->icon_win_w+X_GAP;

   MBInfoPtr->main_win_w = MBInfoPtr->btn_win_w + (X_MARGIN<<1) +
         (brdrW<<1);
   MBInfoPtr->main_win_h = MBInfoPtr->icon_win_h + MBInfoPtr->btn_win_h +
         (Y_MARGIN<<1) + Y_GAP + (brdrW<<1);
   left = ((MBInfoPtr->btn_win_w - numButtons*a_btn_w -
         BTN_MIN_X_GAP*(numButtons-1))>>1);
   inc = a_btn_w + ((MBInfoPtr->btn_win_w-(left<<1)-numButtons*a_btn_w) /
         (numButtons-1));
   for (i=0; i < numButtons; i++)
   {
      MBInfoPtr->btn_info[i].bbox.ltx = left+BTN_XY_EXTRA;
      MBInfoPtr->btn_info[i].bbox.lty = BTN_XY_EXTRA;
      MBInfoPtr->btn_info[i].bbox.rbx = left +
            MAX_BTN_STR_LEN*defaultFontWidth + BTN_XY_EXTRA;
      MBInfoPtr->btn_info[i].bbox.rby = BTN_XY_EXTRA + defaultFontHeight +
            (BTN_Y_MARGIN<<1);
      left += inc;
   }
   MBInfoPtr->main_win_x = ((DisplayWidth(mainDisplay, mainScreen) -
         MBInfoPtr->main_win_w)>>1);
   MBInfoPtr->main_win_y = ((DisplayHeight(mainDisplay, mainScreen) -
         MBInfoPtr->main_win_h)/3);
   if (MBInfoPtr->main_win_x < 0) MBInfoPtr->main_win_x = 0;
   if (MBInfoPtr->main_win_y < 0) MBInfoPtr->main_win_y = 0;
}

static
int SetupMBWindow (MBInfoPtr, Message, Title, IconAndBtns)
   struct MBRec	* MBInfoPtr;
   char		* Message, * Title;
   int		IconAndBtns;
{
   XWMHints		wmhints;
   XSizeHints		sizehints;
   XSetWindowAttributes	win_attrs;

   MBInfoPtr->msg_copy = NULL;
   MBInfoPtr->max_msg_win_w = (DisplayWidth(mainDisplay,mainScreen)>>1);
   MBInfoPtr->max_msg_str_len = MBInfoPtr->max_msg_win_w / defaultFontWidth;

   numButtons = MAX_BUTTONS;
   if (IconAndBtns & MB_BTN_EXTRA) numButtons++;
   CalcGeometry(MBInfoPtr, Message);

   switch (IconAndBtns & MB_BTNMASK)
   {
      case MB_BTN_OK:
         SetupMBButton(MBInfoPtr, LF_BUTTON, '\0', MB_ID_FAILED);
         SetupMBButton(MBInfoPtr, MD_BUTTON | DEF_BUTTON, 'o', MB_ID_OK);
         SetupMBButton(MBInfoPtr, RT_BUTTON, '\0', MB_ID_FAILED);
         break;
      case MB_BTN_YESNOCANCEL:
         SetupMBButton(MBInfoPtr, LF_BUTTON | DEF_BUTTON, 'y', MB_ID_YES);
         SetupMBButton(MBInfoPtr, MD_BUTTON, 'n', MB_ID_NO);
         SetupMBButton(MBInfoPtr, RT_BUTTON, 'c', MB_ID_CANCEL);
         break;
      case MB_BTN_OKCANCEL:
         SetupMBButton(MBInfoPtr, LF_BUTTON | DEF_BUTTON, 'o', MB_ID_OK);
         SetupMBButton(MBInfoPtr, MD_BUTTON, '\0', MB_ID_FAILED);
         SetupMBButton(MBInfoPtr, RT_BUTTON, 'c', MB_ID_CANCEL);
         break;
      case MB_BTN_YESNO:
         SetupMBButton(MBInfoPtr, LF_BUTTON | DEF_BUTTON, 'y', MB_ID_YES);
         SetupMBButton(MBInfoPtr, MD_BUTTON, '\0', MB_ID_FAILED);
         SetupMBButton(MBInfoPtr, RT_BUTTON, 'n', MB_ID_NO);
         break;
      default:
         SetupMBButton(MBInfoPtr, LF_BUTTON, '\0', MB_ID_FAILED);
         SetupMBButton(MBInfoPtr, MD_BUTTON, '\0', MB_ID_FAILED);
         SetupMBButton(MBInfoPtr, RT_BUTTON, '\0', MB_ID_FAILED);
         break;
   }
   if (IconAndBtns & MB_BTN_EXTRA)
   {
      if ((IconAndBtns & MB_BTNMASK) != 0)
         SetupMBButton(MBInfoPtr, XT_BUTTON, extraBtnChar, MB_ID_EXTRA);
      else
         SetupMBButton(MBInfoPtr, XT_BUTTON | DEF_BUTTON, extraBtnChar,
               MB_ID_EXTRA);
   }
   else
      SetupMBButton(MBInfoPtr, XT_BUTTON, '\0', MB_ID_FAILED);

   switch (IconAndBtns & MB_ICONMASK)
   {
      case MB_ICON_STOP:
         MBInfoPtr->cur_bitmap = msgBoxPixmap[MB_PIXMAP_STOP];
         break;
      case MB_ICON_QUESTION:
         MBInfoPtr->cur_bitmap = msgBoxPixmap[MB_PIXMAP_QUESTION];
         break;
      case MB_ICON_INFORMATION:
         MBInfoPtr->cur_bitmap = msgBoxPixmap[MB_PIXMAP_INFORMATION];
         break;
      default: MBInfoPtr->cur_bitmap = None; break;
   }

   if ((MBInfoPtr->main_win = XCreateSimpleWindow (mainDisplay, rootWindow,
         MBInfoPtr->main_win_x, MBInfoPtr->main_win_y,
         MBInfoPtr->main_win_w, MBInfoPtr->main_win_h, brdrW,
         myBorderPixel, myBgPixel)) == 0)
   {
      fprintf (stderr, "Fatal error!  Can not create MsgBox window!\n");
      return (FALSE);
   }
   if ((MBInfoPtr->icon_win = XCreateSimpleWindow (mainDisplay,
         MBInfoPtr->main_win, X_MARGIN, Y_MARGIN,
         MBInfoPtr->icon_win_w, MBInfoPtr->icon_win_h, 0,
         myBorderPixel, myBgPixel)) == 0)
   {
      fprintf (stderr, "Fatal error!  Can not create MsgBox window!\n");
      return (FALSE);
   }
   if ((MBInfoPtr->msg_win = XCreateSimpleWindow (mainDisplay,
         MBInfoPtr->main_win, X_MARGIN+MBInfoPtr->icon_win_w+X_GAP, Y_MARGIN,
         MBInfoPtr->msg_win_w, MBInfoPtr->msg_win_h, 0,
         myBorderPixel, myBgPixel)) == 0)
   {
      fprintf (stderr, "Fatal error!  Can not create MsgBox window!\n");
      return (FALSE);
   }
   if ((MBInfoPtr->btn_win = XCreateSimpleWindow (mainDisplay,
         MBInfoPtr->main_win, X_MARGIN, Y_MARGIN+Y_GAP+MBInfoPtr->icon_win_h,
         MBInfoPtr->btn_win_w, MBInfoPtr->btn_win_h, 0,
         myBorderPixel, myBgPixel)) == 0)
   {
      fprintf (stderr, "Fatal error!  Can not create MsgBox window!\n");
      return (FALSE);
   }
   win_attrs.save_under = True;
   win_attrs.colormap = mainColormap;
   XChangeWindowAttributes (mainDisplay, MBInfoPtr->main_win,
         CWSaveUnder | CWColormap, &win_attrs);

   wmhints.flags = InputHint | StateHint;
   wmhints.input = True;
   wmhints.initial_state = NormalState;
   XSetWMHints (mainDisplay, MBInfoPtr->main_win, &wmhints);
   wmhints.flags = InputHint;
   XSetWMHints (mainDisplay, MBInfoPtr->icon_win, &wmhints);
   XSetWMHints (mainDisplay, MBInfoPtr->msg_win, &wmhints);
   XSetWMHints (mainDisplay, MBInfoPtr->btn_win, &wmhints);

   sizehints.flags = PPosition | PSize | USPosition | PMinSize | PMaxSize;
   sizehints.x = MBInfoPtr->main_win_x;
   sizehints.y = MBInfoPtr->main_win_y;
   sizehints.width = sizehints.max_width = sizehints.min_width =
         MBInfoPtr->main_win_w;
   sizehints.height = sizehints.max_height = sizehints.min_height =
         MBInfoPtr->main_win_h;
#ifdef NOTR4MODE
   XSetNormalHints (mainDisplay, MBInfoPtr->main_win, &sizehints);
#else
   XSetWMNormalHints(mainDisplay, MBInfoPtr->main_win, &sizehints);
#endif /* NOTR4MODE */
   XStoreName(mainDisplay, MBInfoPtr->main_win, Title);

#ifdef MAPBEFORESELECT
   XMapWindow(mainDisplay, MBInfoPtr->main_win);
   XMapWindow(mainDisplay, MBInfoPtr->icon_win);
   XMapWindow(mainDisplay, MBInfoPtr->msg_win);
   XMapWindow(mainDisplay, MBInfoPtr->btn_win);
   XSelectInput(mainDisplay, MBInfoPtr->main_win,
         ButtonPressMask | KeyPressMask |
         KeyPressMask | StructureNotifyMask | VisibilityChangeMask);
   XSelectInput(mainDisplay, MBInfoPtr->icon_win,
         ButtonPressMask | KeyPressMask | ExposureMask);
   XSelectInput(mainDisplay, MBInfoPtr->msg_win,
         ButtonPressMask | KeyPressMask | ExposureMask);
   XSelectInput(mainDisplay, MBInfoPtr->btn_win, ButtonReleaseMask |
         ButtonPressMask | KeyPressMask |
         ButtonPressMask | KeyPressMask | ExposureMask);
#else /* !MAPBEFORESELECT */
   XSelectInput(mainDisplay, MBInfoPtr->main_win,
         ButtonPressMask | KeyPressMask |
         KeyPressMask | StructureNotifyMask | VisibilityChangeMask);
   XSelectInput(mainDisplay, MBInfoPtr->icon_win,
         ButtonPressMask | KeyPressMask | ExposureMask);
   XSelectInput(mainDisplay, MBInfoPtr->msg_win,
         ButtonPressMask | KeyPressMask | ExposureMask);
   XSelectInput(mainDisplay, MBInfoPtr->btn_win, ButtonReleaseMask |
         ButtonPressMask | KeyPressMask |
         ButtonPressMask | KeyPressMask | ExposureMask);
   XMapWindow(mainDisplay, MBInfoPtr->main_win);
   XMapWindow(mainDisplay, MBInfoPtr->icon_win);
   XMapWindow(mainDisplay, MBInfoPtr->msg_win);
   XMapWindow(mainDisplay, MBInfoPtr->btn_win);
#endif /* !MAPBEFORESELECT */
   if (warpToWinCenter)
      XWarpPointer (mainDisplay, None, MBInfoPtr->main_win, 0, 0, 0, 0,
            (MBInfoPtr->main_win_w>>1), (MBInfoPtr->main_win_h>>1));
   XSync (mainDisplay, False);
   return (TRUE);
}

static XComposeStatus	c_stat;

static
int HandleMsgBoxKeyEvent (MBInfoPtr, input)
   struct MBRec	* MBInfoPtr;
   XEvent	* input;
{
   XKeyEvent *key_ev=(&(input->xkey));
   KeySym key_sym;
   char buf[80];
   int i, j;

   XLookupString(key_ev, buf, 80-1, &key_sym, &c_stat);
   TranslateKeys(buf, &key_sym);
   if (buf[0]=='\033' && (key_sym & 0xff)=='\033')
      return (MB_ID_CANCEL);

   for (i=0; i < numButtons; i++)
   {
      if (MBInfoPtr->btn_info[i].str != NULL)
      {
         for (j=0; MBInfoPtr->btn_info[i].key_sym[j] != (KeySym)0; j++)
         {
            if (MBInfoPtr->btn_info[i].key_sym[j] == key_sym)
            {
               DisplayButtonInBBox(MBInfoPtr->btn_win,
                     MBInfoPtr->btn_info[i].str,
                     strlen(MBInfoPtr->btn_info[i].str),
                     &MBInfoPtr->btn_info[i].bbox, BUTTON_INVERT,
                     MBInfoPtr->btn_info[i].highlight, BTN_XY_EXTRA);
               XSync (mainDisplay, False);
               return MBInfoPtr->btn_info[i].id;
            }
         }
      }
   }
   return INVALID;
}

static
int HandleMsgBoxBtnEvent (MBInfoPtr, input)
   struct MBRec	* MBInfoPtr;
   XEvent	* input;
{
   XButtonEvent	* button_ev=(&(input->xbutton));
   int		i, x=button_ev->x, y=button_ev->y;

   for (i=0; i < numButtons; i++)
   {
      if (MBInfoPtr->btn_info[i].str != NULL)
      {
         if (x >= MBInfoPtr->btn_info[i].bbox.ltx &&
               x < MBInfoPtr->btn_info[i].bbox.rbx &&
               y >= MBInfoPtr->btn_info[i].bbox.lty &&
               y < MBInfoPtr->btn_info[i].bbox.rby)
         {
            int inside=TRUE;

            DisplayButtonInBBox(MBInfoPtr->btn_win, MBInfoPtr->btn_info[i].str,
                  strlen(MBInfoPtr->btn_info[i].str),
                  &MBInfoPtr->btn_info[i].bbox, BUTTON_INVERT,
                  MBInfoPtr->btn_info[i].highlight, BTN_XY_EXTRA);

            XGrabPointer(mainDisplay, MBInfoPtr->btn_win, FALSE,
                  PointerMotionMask | ButtonReleaseMask,
                  GrabModeAsync, GrabModeAsync, None, defaultCursor,
                  CurrentTime);
            while (TRUE)
            {
               XEvent ev;

               XNextEvent(mainDisplay, &ev);
               if (ev.type == ButtonRelease)
               {
                  XUngrabPointer(mainDisplay, CurrentTime);
                  XSync(mainDisplay, False);
                  button_ev = &(ev.xbutton);
                  x = button_ev->x;
                  y = button_ev->y;
                  if (x >= MBInfoPtr->btn_info[i].bbox.ltx &&
                        x < MBInfoPtr->btn_info[i].bbox.rbx &&
                        y >= MBInfoPtr->btn_info[i].bbox.lty &&
                        y < MBInfoPtr->btn_info[i].bbox.rby)
                  {
                     DisplayButtonInBBox(MBInfoPtr->btn_win,
                           MBInfoPtr->btn_info[i].str,
                           strlen(MBInfoPtr->btn_info[i].str),
                           &MBInfoPtr->btn_info[i].bbox,
                           BUTTON_NORMAL,
                           MBInfoPtr->btn_info[i].highlight, BTN_XY_EXTRA);
                     return MBInfoPtr->btn_info[i].id;
                  }
                  break;
               }
               else if (ev.type == MotionNotify)
               {
                  XEvent	tmp_ev;
                  XMotionEvent	* motion_ev=(&(ev.xmotion));

                  while (XCheckMaskEvent(mainDisplay, PointerMotionMask,
                        &tmp_ev)) ;
                  x = motion_ev->x;
                  y = motion_ev->y;
                  if (inside)
                  {
                     if (!(x >= MBInfoPtr->btn_info[i].bbox.ltx &&
                           x < MBInfoPtr->btn_info[i].bbox.rbx &&
                           y >= MBInfoPtr->btn_info[i].bbox.lty &&
                           y < MBInfoPtr->btn_info[i].bbox.rby))
                     {
                        DisplayButtonInBBox(MBInfoPtr->btn_win,
                              MBInfoPtr->btn_info[i].str,
                              strlen(MBInfoPtr->btn_info[i].str),
                              &MBInfoPtr->btn_info[i].bbox, BUTTON_NORMAL,
                              MBInfoPtr->btn_info[i].highlight, BTN_XY_EXTRA);
                        inside = FALSE;
                     }
                  }
                  else
                  {
                     if (x >= MBInfoPtr->btn_info[i].bbox.ltx &&
                           x < MBInfoPtr->btn_info[i].bbox.rbx &&
                           y >= MBInfoPtr->btn_info[i].bbox.lty &&
                           y < MBInfoPtr->btn_info[i].bbox.rby)
                     {
                        DisplayButtonInBBox(MBInfoPtr->btn_win,
                              MBInfoPtr->btn_info[i].str,
                              strlen(MBInfoPtr->btn_info[i].str),
                              &MBInfoPtr->btn_info[i].bbox, BUTTON_INVERT,
                              MBInfoPtr->btn_info[i].highlight, BTN_XY_EXTRA);
                        inside = TRUE;
                     }
                  }
               }
            }
            return INVALID;
         }
      }
   }
   return INVALID;
}

static
void RefreshMsgBox (MBInfoPtr)
   struct MBRec	* MBInfoPtr;
{
   int		i;
   XEvent	ev;

   if (MBInfoPtr->msg_copy != NULL && *MBInfoPtr->msg_copy != '\0')
   {
      int	y=0;
      char	* c_ptr=MBInfoPtr->msg_copy;

      while (c_ptr != NULL)
      {
         char	* c_ptr1=strchr(c_ptr, '\n');
         int	len;

         if (c_ptr1 != NULL) *c_ptr1 = '\0';
         len = strlen(c_ptr);
         XDrawString(mainDisplay, MBInfoPtr->msg_win, defaultGC,
               0, y+defaultFontAsc, c_ptr, len);
         y += defaultFontHeight+SPACING;
         if (c_ptr1 != NULL)
         {
            *c_ptr1 = '\n';
            c_ptr = &c_ptr1[1];
         }
         else
            break;
      }
   }
   for (i=0; i < numButtons; i++)
   {
      if (MBInfoPtr->btn_info[i].str != NULL)
      {
         DisplayButtonInBBox(MBInfoPtr->btn_win, MBInfoPtr->btn_info[i].str,
               strlen(MBInfoPtr->btn_info[i].str),
               &MBInfoPtr->btn_info[i].bbox, BUTTON_NORMAL,
               MBInfoPtr->btn_info[i].highlight, BTN_XY_EXTRA);
      }
   }
   if (MBInfoPtr->cur_bitmap != None)
   {
      int y=((MBInfoPtr->icon_win_h-ICON_H)>>1);

      XSetTSOrigin(mainDisplay, defaultGC, 0, y);
      XSetFillStyle(mainDisplay, defaultGC, FillOpaqueStippled);
      XSetStipple(mainDisplay, defaultGC, MBInfoPtr->cur_bitmap);
      XFillRectangle(mainDisplay, MBInfoPtr->icon_win, defaultGC, 0, y,
            ICON_W, ICON_H);
      XSetFillStyle(mainDisplay, defaultGC, FillSolid);
      XSetTSOrigin(mainDisplay, defaultGC, 0, 0);
   }
   while (XCheckWindowEvent(mainDisplay,MBInfoPtr->main_win,ExposureMask,&ev)) ;
   while (XCheckWindowEvent(mainDisplay,MBInfoPtr->icon_win,ExposureMask,&ev)) ;
   while (XCheckWindowEvent(mainDisplay,MBInfoPtr->msg_win,ExposureMask,&ev)) ;
   while (XCheckWindowEvent(mainDisplay,MBInfoPtr->btn_win,ExposureMask,&ev)) ;
}

int MsgBox (Message, Title, IconAndBtns)
   char	* Message, * Title;
   int	IconAndBtns;
{
   char	* dup_msg=UtilStrDup(Message);
   int	rc=MB_ID_FAILED, looping=TRUE, exposed=FALSE;

   if (dup_msg == NULL)
   {
      fprintf (stderr, "Can not calloc().\n");
      return (rc);
   }
   if (!SetupMBWindow (&mbInfo, dup_msg, Title, IconAndBtns))
   {
      fprintf (stderr, "Invalid parameters passed to MsgBox().\n");
      Msg ("Invalid parameters passed to MsgBox().\n");
      if (mbInfo.msg_copy != NULL) cfree (mbInfo.msg_copy);
      cfree (dup_msg);
      return (rc);
   }
   while (looping)
   {
      XEvent	input, ev;

      XNextEvent (mainDisplay, &input);
      if ((input.type==MapNotify && input.xany.window==mbInfo.main_win) ||
            (input.type==Expose && (input.xany.window==mbInfo.main_win ||
            input.xany.window==mbInfo.icon_win ||
            input.xany.window==mbInfo.msg_win ||
            input.xany.window==mbInfo.btn_win)) ||
            (!exposed &&
            (XCheckWindowEvent (mainDisplay,mbInfo.main_win,ExposureMask,&ev) ||
            XCheckWindowEvent (mainDisplay,mbInfo.main_win,StructureNotifyMask,
            &ev))))
      {
         RefreshMsgBox (&mbInfo);
         exposed = TRUE;
         XSync (mainDisplay, False);
         if (input.xany.window == mbInfo.main_win ||
               input.xany.window==mbInfo.icon_win ||
               input.xany.window==mbInfo.msg_win ||
               input.xany.window==mbInfo.btn_win)
            continue;
      }
      if (input.type == Expose)
         ExposeEventHandler (&input, FALSE);
      else if (input.type == VisibilityNotify &&
            input.xany.window==mainWindow &&
            input.xvisibility.state==VisibilityUnobscured)
      {
         int	i;

         while (XCheckWindowEvent (mainDisplay, mainWindow,
               VisibilityChangeMask, &ev)) ;
         if (pinnedMainMenu) XMapRaised (mainDisplay, mainMenuWindow);
         for (i = 0; i < numExtraWins; i++)
            if (extraWinInfo[i].mapped && extraWinInfo[i].raise &&
                  extraWinInfo[i].window != None)
               XMapRaised (mainDisplay, extraWinInfo[i].window);
         XMapRaised (mainDisplay, mbInfo.main_win);
      }
      else if (input.type == KeyPress)
      {
         if ((rc=HandleMsgBoxKeyEvent(&mbInfo, &input)) != INVALID)
            break;
      }
      else if (input.type == ButtonPress && input.xany.window==mbInfo.btn_win)
      {
         if ((rc=HandleMsgBoxBtnEvent(&mbInfo, &input)) != INVALID)
            break;
      }
   }
   if (mbInfo.msg_copy != NULL) cfree (mbInfo.msg_copy);
   cfree (dup_msg);

   XDestroyWindow (mainDisplay, mbInfo.main_win);
   if (warpToWinCenter)
      XWarpPointer (mainDisplay, None, drawWindow, 0, 0, 0, 0,
            (int)(ZOOMED_SIZE(drawWinW)>>1), (int)(ZOOMED_SIZE(drawWinH)>>1));
   return (rc);
}

int Dialog (Message, Comment, ReturnStr)
   char	* Message, * Comment, * ReturnStr;
   /* returns INVALID if <ESC> is types */
   /* returns FALSE otherwise */
   /* if Comment is NULL, "( <CR> or <ESC> to continue )" is assumed */
   /* if ReturnStr is NULL, hitting any key will return INVALID */
{
   int		i, w, h, str_w, left, top, fore_draw_pixel, fore_erase_pixel;
   int		win_x, win_y, usr_str_w=0, comment_left=0;
   int		dialoging=TRUE, index=0, dsp_w, dsp_h, exposed=FALSE;
   int		rc=FALSE;
   char		buf[81], def_comment[81];
   XEvent	input, ev;
   XKeyEvent	* key_ev;
   KeySym	key_sym;
   XWMHints	wmhints;
   XSizeHints	sizehints;
   XSetWindowAttributes	win_attrs;

   if (Comment == NULL)
      strcpy (def_comment, "( <CR> or <ESC> to continue )");
   else
      *def_comment = '\0';
   str_w = defaultFontWidth * strlen (Message);
   if (Comment == NULL)
   {
      int	comment_w = defaultFontWidth * strlen (def_comment);

      h = 8 * defaultFontHeight;
      w = max(max(str_w,comment_w)+6*defaultFontWidth,600);
      comment_left = (w - comment_w) / 2;
   }
   else if (*Comment == '\0')
   {
      h = 7 * defaultFontHeight;
      w = max(str_w+6*defaultFontWidth,600);
   }
   else
   {
      int	comment_w = defaultFontWidth * strlen (Comment);

      h = 8 * defaultFontHeight;
      w = max(max(str_w,comment_w)+6*defaultFontWidth,600);
      comment_left = (w - comment_w) / 2;
   }
   left = (w - str_w) / 2;
   top = 2 * defaultFontHeight;

   fore_draw_pixel = myFgPixel;
   fore_erase_pixel = myBgPixel;

   dsp_w = DisplayWidth (mainDisplay, mainScreen);
   dsp_h = DisplayHeight (mainDisplay, mainScreen);

   win_x = (w > dsp_w) ? 0 : (dsp_w - w)/2;
   win_y = (h > dsp_h) ? 0 : (dsp_h - h)/3;

   if (w > dsp_w-2*(brdrW+1)) w = dsp_w-2*(brdrW+1);

   if ((dialogWindow = XCreateSimpleWindow (mainDisplay, rootWindow, win_x,
         win_y, w, h, brdrW, myBorderPixel, myBgPixel)) == 0)
   {
      fprintf (stderr, "Could not create dialog window!\n");
      if (fileModified) EmergencySave (0);
      exit (-1);
   }
   win_attrs.save_under = True;
   win_attrs.colormap = mainColormap;
   XChangeWindowAttributes (mainDisplay, dialogWindow,
         CWSaveUnder | CWColormap, &win_attrs);

   wmhints.flags = InputHint | StateHint;
   wmhints.input = True;
   wmhints.initial_state = NormalState;
   XSetWMHints (mainDisplay, dialogWindow, &wmhints);

   sizehints.flags = PPosition | PSize | USPosition | PMinSize | PMaxSize;
   sizehints.x = win_x;
   sizehints.y = win_y;
   sizehints.width = sizehints.min_width = sizehints.max_width = w;
   sizehints.height = sizehints.min_height = sizehints.max_height = h;
#ifdef NOTR4MODE
   XSetNormalHints (mainDisplay, dialogWindow, &sizehints);
#else
   XSetWMNormalHints (mainDisplay, dialogWindow, &sizehints);
#endif
   sprintf (buf, "%s - Information", TOOL_NAME);
   XStoreName (mainDisplay, dialogWindow, buf);

   XSetTransientForHint (mainDisplay, dialogWindow, mainWindow);
#ifdef MAPBEFORESELECT
   XMapWindow (mainDisplay, dialogWindow);
   XSelectInput (mainDisplay, dialogWindow,
         ButtonPressMask | KeyPressMask | ExposureMask | StructureNotifyMask);
#else
   XSelectInput (mainDisplay, dialogWindow,
         ButtonPressMask | KeyPressMask | ExposureMask | StructureNotifyMask);
   XMapWindow (mainDisplay, dialogWindow);
#endif
   if (warpToWinCenter)
      XWarpPointer (mainDisplay, None, dialogWindow, 0, 0, 0, 0,
            (int)(w/2), (int)(h/2));

   XSync (mainDisplay, False);

   curX = 200;
   curY = (Comment==NULL || *Comment!='\0') ? 5*defaultFontHeight :
         4*defaultFontHeight;

   while (dialoging)
   {
      XNextEvent (mainDisplay, &input);

      if ((input.type==MapNotify && input.xany.window==dialogWindow) ||
            (input.type==Expose && input.xany.window==dialogWindow) ||
            (!exposed &&
            (XCheckWindowEvent (mainDisplay,dialogWindow,ExposureMask,&ev) ||
            XCheckWindowEvent (mainDisplay,dialogWindow,StructureNotifyMask,
            &ev))))
      {
         while (XCheckWindowEvent (mainDisplay,dialogWindow,ExposureMask,&ev)) ;
         while (XCheckWindowEvent (mainDisplay,dialogWindow,StructureNotifyMask,
               &ev)) ;

         XDrawRectangle (mainDisplay, dialogWindow, defaultGC, 0, 0, w-1, h-1);
         MyBox (dialogWindow, defaultGC, 10, 10, w-9, h-9);
         MyBox (dialogWindow, defaultGC, 11, 11, w-10, h-10);
         MyBox (dialogWindow, defaultGC, 12, 12, w-11, h-11);
         XDrawString (mainDisplay, dialogWindow, defaultGC, left, top, Message,
               strlen(Message));
         if (Comment == NULL)
            XDrawString (mainDisplay, dialogWindow, defaultGC, comment_left,
                  top+defaultFontHeight, def_comment, strlen(def_comment));
         else if (*Comment != '\0')
            XDrawString (mainDisplay, dialogWindow, defaultGC, comment_left,
                  top+defaultFontHeight, Comment, strlen(Comment));
         if (index != 0)
            XDrawString (mainDisplay, dialogWindow, defaultGC,
                  (int)((w-usr_str_w)/2), curY+defaultFontAsc,
                  (ReturnStr == NULL ? "" : ReturnStr), index);

         PutCursor (dialogWindow, (int)((w-usr_str_w)/2)+usr_str_w+1, curY,
               fore_draw_pixel);

         exposed = TRUE;
         XSync (mainDisplay, False);

         if ((input.type==MapNotify && input.xany.window==dialogWindow) ||
               (input.type==Expose && input.xany.window==dialogWindow))
            continue;
      }

      if (input.type == VisibilityNotify &&
            input.xany.window==mainWindow &&
            input.xvisibility.state==VisibilityUnobscured)
      {
         while (XCheckWindowEvent (mainDisplay, mainWindow,
               VisibilityChangeMask, &ev)) ;
         if (pinnedMainMenu) XMapRaised (mainDisplay, mainMenuWindow);
         for (i = 0; i < numExtraWins; i++)
            if (extraWinInfo[i].mapped && extraWinInfo[i].raise &&
                  extraWinInfo[i].window != None)
               XMapRaised (mainDisplay, extraWinInfo[i].window);
         XMapRaised (mainDisplay, dialogWindow);
      }
      else if (input.type == ButtonPress)
      {
         XButtonEvent *button_ev=(&(input.xbutton));

         if (button_ev->button == Button2) {
            int buf_len=0;
            char *cut_buffer=FetchCutBuffer(&buf_len);

            if (cut_buffer != NULL) {
               if ((unsigned char)(*cut_buffer) != TGIF_HEADER &&
                     ReturnStr != NULL) {
                  unsigned char *c_ptr=(unsigned char *)cut_buffer;

                  if (exposed) {
                     PutCursor (dialogWindow,
                           (int)((w-usr_str_w)/2)+usr_str_w+1, curY,
                           fore_erase_pixel);
                     XClearArea (mainDisplay, dialogWindow,
                           (int)((w-usr_str_w)/2), curY, usr_str_w+1,
                           defaultFontHeight+1, FALSE);
                  }
                  for ( ; index < 80 && *c_ptr != '\0'; c_ptr++) {
                     if (*c_ptr >= (unsigned char)('\040') &&
                           *c_ptr < (unsigned char)('\377')) {
                        curX += defaultFontWidth;
                        usr_str_w += defaultFontWidth;
                        ReturnStr[index++] = (char)(*c_ptr);
                     } else {
                        break;
                     }
                  }
                  if (exposed) {
                     XDrawString (mainDisplay, dialogWindow, defaultGC,
                           (int)((w-usr_str_w)/2), curY+defaultFontAsc,
                           ReturnStr, index);
                     PutCursor (dialogWindow,
                           (int)((w-usr_str_w)/2)+usr_str_w+1, curY,
                           fore_draw_pixel);
                  }
               }
               XFree(cut_buffer);
            }
         }
      }
      else if (input.type == KeyPress)
      {
         /* erase the old cursor */
         if (exposed)
            PutCursor (dialogWindow, (int)((w-usr_str_w)/2)+usr_str_w+1, curY,
                  fore_erase_pixel);

         key_ev = &(input.xkey);
         XLookupString (key_ev, buf, 80, &key_sym, &c_stat);
         TranslateKeys (buf, &key_sym);
         if ((buf[0]=='\033' && (key_sym & 0xff)=='\033') ||
               (buf[0]=='\r' && (key_sym & 0xff)=='\r') ||
               (buf[0]=='\n' && (key_sym & 0xff)=='\n') ||
               (buf[0]=='\b' && (key_sym & 0xff)=='\b') ||
               (buf[0]=='\b' && (key_sym & 0xff)=='h' &&
               (key_ev->state & ControlMask)) ||
               (buf[0]=='\177' && (key_sym & 0x7f)=='\177') ||
               ((key_ev->state & ControlMask)==0 &&
               key_sym>='\040' && key_sym<='\177'))
         {
            if (ReturnStr == NULL)
            {
               rc = INVALID;
               break;
            }
            switch (buf[0])
            {
               case '\033':
                  ReturnStr[0] = '\0';
                  dialoging = FALSE;
                  rc = INVALID;
                  break;
               case '\r':
               case '\n':
                  ReturnStr[index] = '\0';
                  dialoging = FALSE;
                  break;
               case '\177': /* <DEL> */
               case '\b': /* <BS> */
                  if (index != 0)
                  {
                     if (exposed)
                        XClearArea (mainDisplay, dialogWindow,
                              (int)((w-usr_str_w)/2), curY, usr_str_w+1,
                              defaultFontHeight+1, FALSE);
                     index--;
                     curX -= defaultFontWidth;
                     usr_str_w -= defaultFontWidth;
                     if (index != 0 && exposed)
                        XDrawString (mainDisplay, dialogWindow, defaultGC,
                              (int)((w-usr_str_w)/2), curY+defaultFontAsc,
                              ReturnStr, index);
                  }
                  break;
               default:
                  if (buf[0] >= '\040' && index < 80)
                  {
                     if (exposed)
                        XClearArea (mainDisplay, dialogWindow,
                              (int)((w-usr_str_w)/2), curY, usr_str_w+1,
                              defaultFontHeight+1, FALSE);
                     curX += defaultFontWidth;
                     usr_str_w += defaultFontWidth;
                     ReturnStr[index++] = buf[0];
                     if (exposed)
                        XDrawString (mainDisplay, dialogWindow, defaultGC,
                              (int)((w-usr_str_w)/2), curY+defaultFontAsc,
                              ReturnStr, index);
                  }
                  break;
            }
         }
         if (exposed)
            PutCursor (dialogWindow, (int)((w-usr_str_w)/2)+usr_str_w+1, curY,
                  fore_draw_pixel);
      }
      else if (input.type == Expose && input.xany.window != dialogWindow)
         ExposeEventHandler (&input, FALSE);
   }

   XDestroyWindow (mainDisplay, dialogWindow);
   if (warpToWinCenter)
      XWarpPointer (mainDisplay, None, drawWindow, 0, 0, 0, 0,
            (int)(ZOOMED_SIZE(drawWinW)>>1), (int)(ZOOMED_SIZE(drawWinH)>>1));
   return (rc);
}
