/* lastcomm.c */

/* Copyright (C) 1993 Free Software Foundation, Inc.

This file is part of the GNU Accounting Utilities

The GNU Accounting Utilities are free software; you can redistribute
them and/or modify them under the terms of the GNU General Public
License as published by the Free Software Foundation; either version
2, or (at your option) any later version.

The GNU Accounting Utilities are distributed in the hope that they will
be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with the GNU Accounting Utilities; see the file COPYING.  If
not, write to the Free Software Foundation, 675 Mass Ave, Cambridge,
MA 02139, USA.  */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#include <math.h>
#include <pwd.h>
#include "common.h"
#include "uid_hash.h"
#include "getopt.h"

#if defined(DIRENT) || defined(_POSIX_VERSION)
#include <dirent.h>
#define NLENGTH(dirent) (strlen((dirent)->d_name))
#else /* not (DIRENT or _POSIX_VERSION) */
#define dirent direct
#define NLENGTH(dirent) ((dirent)->d_namlen)
#ifdef USG
#ifdef SYSNDIR
#include <sys/ndir.h>
#else /* not SYSNDIR */
#include <ndir.h>
#endif /* not SYSNDIR */
#else /* not USG */
#include <sys/dir.h>                /* Assume SYSDIR, or not if you want.  */
#endif /* not USG */
#endif /* not (DIRENT or _POSIX_VERSION) */

#define BUFFERED_RECS 256	/* how many records we buffer on input */

#define DEV_NAME_LEN 8		/* length of device names */

#define DEV_TABLE_SIZE 51	/* hash table size for device list (prime) */

struct dev_entry {
  struct dev_entry *next;	/* gee, could it be a linked list? */
  char name[DEV_NAME_LEN];	/* name of the device */
  long type;			/* type number of this device */
};

/* globals */

struct dev_entry *dev_list[DEV_TABLE_SIZE];

static char rcsid[] = "$Id: lastcomm.c,v 1.11 1994/03/23 02:51:38 noel Exp noel $";

int debugging_enabled;           /* print internal information relating
				  * to entries read, data structures, etc.
				  */
int strict_match_flag;           /* nonzero if each record has to match
				  * all items on the command line
				  */

#include "files.h"

struct name_entry *name_list;	/* lookup table for ttys */

char *program_name;              /* name of the program, for usage & errs */
FILE *Afile;                     /* pointer to the acct file */
char default_acct_file[] = ACCT_FILE;  /* self-explanatory */
char *acct_file_name = default_acct_file;    /* the file name */
long line_number = 0;            /* the line number of the file,
				  * for error reporting purposes
				  */
struct acct *rec_buffer;         /* buffer for read records so that
				  * we don't have to slow things down
				  * with a million fseek calls!
				  */
int buf_last;                    /* number of last item in the record buffer
				  */

/* protos */

void main (int, char *[]);
void give_usage (void);
void parse_entries (void);
void init_flags_and_data (void);
void setup_devices (void);
char *devname (long);
int get_entry (struct acct **);
char *uid_name (int);
int desired_entry (char *, char *, char *);


/* code */

void
main (int argc, char *argv[])
{
  int c;

  init_flags_and_data ();
  program_name = argv[0];
  
  while (1)
    {
      int option_index = 0;
      
      static struct option long_options[] = {
	{ "debug", 0, 0, 1 },
	{ "version", 0, 0, 2 },
	{ "help", 0, 0, 3 },
	{ "other-file", 1, 0, 4 },
	{ "strict-match", 0, 0, 5 },
	{ 0, 0, 0, 0 }
      };
      
      c = getopt_long (argc, argv, "", long_options, &option_index);
      
      if (c == EOF)
	break;

      switch (c)
	{
	case 1:
	  debugging_enabled = 1;
	  break;
	case 2:
	  fprintf (stderr, "%s\n", rcsid);
	  break;
	case 3:
	  give_usage ();
	  exit (1);
	  break;
	case 4:
	  acct_file_name = optarg;
	  break;
	case 5:
	  strict_match_flag = 1;
	  break;
	default:
	  {
	    /* char ts[255];
	    sprintf (ts, "?? getopt returned char code 0%o ??", c);
	    fatal (ts); */
	    give_usage ();
	    exit (1);
	  }
	  break;
	}
    }

  while (optind < argc)
    {
      /* if we get here, we expect everything else to be a
       * username, terminal name, or command name
       */
      struct name_entry *node;
      
      node = (struct name_entry *) xmalloc (sizeof (struct name_entry));
      node->name = argv[optind++];
      node->next = name_list;
      name_list = node;
    }
  
  if (debugging_enabled)
    {
      struct name_entry *np;
      for (np = name_list; np != NULL; np = np->next)
	fprintf (stddebug, "%s\n", np->name);
    }
  
  Afile = open_binary (acct_file_name);

  setup_devices ();
  parse_entries ();
  
  fclose (Afile);

  exit (0);			/* guarantee the proper return value */
}


/* guess what this does... */
void
give_usage (void)
{
  char *usage = "\
Usage: %s [command-name] ... [user-name] ... [terminal-name] ...\n\
       [--other-file <file>] [--strict-match] [--debug]\n\
       [--version] [--help]\n\
";
  
  fprintf (stderr, usage, program_name);
}

/* parse the entries in an acct file */
void
  parse_entries (void)
{
  struct acct *rec;             /* the current record */

  fseek (Afile, 0, SEEK_END);      /* go to end of file */

  /* loop while there are entries to be had */
  while (get_entry (&rec))
    {
      char *this_uid = uid_name (rec->ac_uid);
      char *this_dev = devname ((long) rec->ac_tty);
      char *this_comm = rec->ac_comm;

      if (desired_entry (this_uid, this_dev, this_comm))
	{
#ifdef HAVE_COMP_T
	  double ut = comp_t_2_double (rec->ac_utime);
	  double st = comp_t_2_double (rec->ac_stime);
#else
	  double ut = (double) rec->ac_utime;
	  double st = (double) rec->ac_stime;
#endif

	  printf ("%-10.*s ", CMD_LEN, this_comm);

#ifdef ASU
	  if (rec->ac_flag & ASU)
	    putchar ('S');
	  else
#endif
	    putchar (' ');
	  
#ifdef AFORK
	  if (rec->ac_flag & AFORK)
	    putchar ('F');
	  else
#endif
	    putchar (' ');
	  
#ifdef ACOMPAT
	  if (rec->ac_flag & ACOMPAT)
	    putchar ('C');
	  else
#endif
	    putchar (' ');
	  
#ifdef ACORE
	  if (rec->ac_flag & ACORE)
	    putchar ('D');
	  else
#endif
	    putchar (' ');
	  
#ifdef AXSIG
	  if (rec->ac_flag & AXSIG)
	    putchar ('X');
	  else
#endif
	    putchar (' ');
	  
	  printf (" %-8.8s %-8.8s %6.2f secs %-16.16s\n",
		  this_uid, this_dev,
		  ((ut + st) / (double) AHZ),
		  ctime (&rec->ac_btime));
	}
    }
}

/* decide whether or not to print an entry based on the name_list
 */
int
desired_entry (char *uid, char *dev, char *comm)
{
  struct name_entry *ne;

  if (name_list == NULL)
    return 1;

  if (strict_match_flag)
    {
      for (ne = name_list; ne != NULL; ne = ne->next)
	{
	  if (strcmp (uid, ne->name) != 0
	      && strcmp (dev, ne->name) != 0
	      && strcmp (comm, ne->name) != 0)    /* at least one must match */
	    return 0;
	}
      /* found all of them */
      return 1;
    }
  else
    {
      for (ne = name_list; ne != NULL; ne = ne->next)
	{
	  if (strcmp (uid, ne->name) == 0
	      || strcmp (dev, ne->name) == 0
	      || strcmp (comm, ne->name) == 0)
	    return 1;
	}
      /* didn't match anything */
      return 0;
    }
}




/* read the '/dev' directory entries and make a linked list of ttys
 * (so we can search through it later)
 */
void
setup_devices (void)
{
  DIR *dirp;              /* the directory itself */
  struct dirent *dp;      /* directory entry structure */
  struct stat sp;         /* file information structure */
  char name[255];         /* string to hold the full name of the device */
  struct dev_entry *dev;  /* temporary ptr to device entry */

  if (debugging_enabled)
    fprintf (stddebug, "device     devnum\n-----------------\n");

  dirp = opendir ("/dev");
  
  for (dp = readdir (dirp); dp != NULL; dp = readdir (dirp))
    {
      sprintf (name, "/dev/%s", dp->d_name);
      stat (name, &sp);
      
      if ((sp.st_mode & S_IFMT) != S_IFCHR)     /* skip if not a tty */
	continue;
      
      if (debugging_enabled)
	fprintf (stddebug, "%-8.8s %8d\n", dp->d_name, sp.st_rdev);

      /* entry OK, so put it into the list */

      dev = (struct dev_entry *) xmalloc (sizeof (struct dev_entry));
      strncpy (dev->name, dp->d_name, DEV_NAME_LEN);
      dev->type = sp.st_rdev;
      dev->next = dev_list[HASH (sp.st_rdev)];
      dev_list[HASH (sp.st_rdev)] = dev;
    }
  closedir (dirp);
}


/* return the name of the device associated with DEV_NUM.  The
   argument passed was originally a dev_t, but that data type is too
   limited on some systems (won't let us pass -1 because it's an
   unsigned short or other). */

char *
devname (long dev_num)
{
  struct dev_entry *de;

  /* special case */

  if (dev_num == -1)
    return "__";

  for (de = dev_list[HASH (dev_num)]; de != NULL; de = de->next)
    {
      if (de->type == dev_num)         /* found it! */
	return de->name;
    }
  
  /* didn't find it */

  return "??";
}


/* Do a buffered read of the file and return the next record in REC.
 * Return 0 if no more entries.
 */
int
get_entry (struct acct **rec)
{
  /* if BUF_LAST == 0, we know that there's nothing in the buffer,
   * so we should read another block of records from the file.  If we can't
   * read BUFFERED_RECS, read as many as possible and set up BUF_LAST
   * correctly.
   */
  
  if (buf_last == 0)
    {
      long new_offset, this_offset, max_recs, recs_read;
      
      this_offset = ftell (Afile);
      if (this_offset == 0)
	return 0;                    /* no more records */
      
      max_recs = this_offset / (long) sizeof (struct acct);
  
      recs_read = (max_recs < BUFFERED_RECS) ? max_recs : BUFFERED_RECS;
    
      fseek (Afile, -sizeof (struct acct) * recs_read, SEEK_CUR);
      new_offset = ftell (Afile);
      
      if (debugging_enabled)
	fprintf (stddebug, "Did seek in file %ld --> %ld (%d)\n",
		 this_offset, new_offset, sizeof (struct acct));
      
      if (fread ((void *) rec_buffer, sizeof (struct acct),
		 recs_read, Afile) == -1)
	fatal ("get_entry: couldn't read from file");
      
      if (debugging_enabled)
	fprintf (stddebug, "Got %ld records from file\n", recs_read);
  
      /* don't need to check this, because the above read was fine */
      fseek (Afile, -sizeof (struct acct) * recs_read, SEEK_CUR);
      
      buf_last = recs_read - 1;
    }		 
  else
    {
      buf_last--;
    }
  
  *rec = rec_buffer + buf_last;
  
  return 1;
}


/* clear all global flags & data */
void
init_flags_and_data (void)
{
  int a;

  debugging_enabled = 0;
  strict_match_flag = 0;
  name_list = NULL;

  clear_uid_table ();

  /* clear the hash tables */
  for (a = 0; a < DEV_TABLE_SIZE; a++)
    {
      dev_list[a] = NULL;
    }
  
  buf_last = 0;

  rec_buffer = (struct acct *) xmalloc (sizeof (struct acct) * BUFFERED_RECS);
}

